"This module provides functionality for compilation of strings as dolfin SubDomains."

__author__ = "Martin Sandve Alnes (martinal@simula.no)"
__date__ = "2008-07-01 -- 2009-12-16"
__copyright__ = "Copyright (C) 2008-2008 Martin Sandve Alnes"
__license__  = "GNU LGPL Version 2.1"

import re
import os
import hashlib
import numpy
import instant

# Import local compile_extension_module
from dolfin.compilemodules import (compile_extension_module,
                                   expression_to_code_fragments,
                                   math_header)

__all__ = ["compile_subdomains",]

_map_args = ["x", "y"]

_subdomain_template = """
class %(classname)s: public SubDomain
{
public:
%(members)s

  %(classname)s()
  {
%(constructor)s
  }

  /// Return true for points inside the sub domain
  bool inside(const double* x, bool on_boundary) const
  {
    %(inside)s
  }

};
"""

# TODO: Support implementation of map as well
"""
  /// Map coordinate x in domain H to coordinate y in domain G (used for periodic boundary conditions)
  void map(const double* x, double* y) const
  {
    %(map)s
  }
"""

def expression_to_subdomain(expr, classname):
    "Generates code for a dolfin::SubDomain subclass for a single expression."
    
    # Assure we have a simple string expression
    assert isinstance(expr, str)
    
    # Extract code fragments from the expr and defaults
    fragments = expression_to_code_fragments([expr], {}, ["x", "on_boundary","DOLFIN_EPS"])
    
    # Generate code for inside()
    insidecode = "  return %s;" % expr
    
    # Generate code for map()
    #mapcode = "..."
    
    # Connect the code fragments using the function template code
    fragments["classname"] = classname
    fragments["inside"]    = insidecode
    #fragments["map"]       = mapcode
    code = _subdomain_template % fragments
    return code

def compile_subdomain_code(code, classnames = None):
    # Autodetect classnames:

    _classnames = re.findall(r"class[ ]+([\w]+).*", code)
    
    # Just a little assertion for safety:
    if classnames is None:
        classnames = _classnames
    else:
        assert all(a == b for (a,b) in zip(classnames, _classnames))

    # Complete the code
    code = math_header + \
"""
namespace dolfin
{
""" + code + \
"""
}
"""
    
    # Compile the extension module
    compiled_module = compile_extension_module(\
                                code,
                                dolfin_module_import=["common","mesh","function"],
#                                additional_declarations=_numpy_typemaps,
                                use_numpy_typemaps=False)
    
    # Construct instances of the compiled subdomain classes
    subdomains = [getattr(compiled_module, name)() for name in classnames]
    return subdomains

def compile_subdomains(expressions):
    """Compiles C++ string expressions into dolfin::SubDomain instances.
    
    The variable 'expressions' can either be a str or a list.
    
    If 'expressions' is a str, it is interpreted as a C++ string
    with complete implementations of subclasses of dolfin::SubDomain.
    The compiled subdomains returned will be in the same order 
    as they are defined in this code.

    If it is a list, each item of the list is interpreted as
    a logical 'inside' expression, and the compiled subdomains 
    returned will be in the same order as they occur in this list.
    
    If an expression string contains a name, it is assumed to
    be a scalar variable name, and is added as a public member
    of the generated subdomain. The exceptions are set in the
    variable dolfin.compile_subdomains._builtins."""
    #, which contains:
    #    %s
    #""" % "\n".join("        " + b for b in _builtins)
    if not isinstance(expressions, (list, str)):
        raise TypeError, "expected a list of 'str' or a 'str'"
    if isinstance(expressions, str):
        expressions = [expressions]
    if not all(isinstance(expr, str) for expr in expressions):
        raise TypeError, "expected a list of 'str' or a 'str'"
    code = ""
    classnames = []
    for i, expr in enumerate(expressions):
        classname = "SubDomain_" + hashlib.md5(expr).hexdigest()
        code += expression_to_subdomain(expr, classname)
        classnames.append(classname)
    subdomains = compile_subdomain_code(code, classnames)

    if len(subdomains) == 1:
        return subdomains[0]
    return subdomains


if __name__ == "__main__":
    subdomains = compile_subdomains(["x[0] >= 1.0-DOLFIN_EPS", "on_boundary && x[1] < xlen+DOLFIN_EPS"])
    print subdomains


