__author__ = "Marie E. Rognes (meg@simula.no)"
__copyright__ = "Copyright (C) 2009 - Marie E. Rognes"
__license__  = "GNU GPL version 3 or any later version"

# Last changed: 2010-05-09

from dolfin import FunctionSpace, Function, Expression

__all__ = ["Bubble", "FacetBubble", "SpecialFacetFunction"]

class Bubble(Function):
   """
   The Bubble is a polynomial (of minimal degree) that equals 1 at an
   appropriate point in the interior of a cell and equals 0 at the
   boundary of the cell
   """

   def __init__(self, mesh):
      """ Initialize """

      # Create bubble function space
      B = FunctionSpace(mesh, "B", mesh.topology().dim()+1)
      Function.__init__(self, B)

      # Initialize vector
      self.vector()[:] = 1.0

class FacetBubble(Function):
   """
   The FacetBubble is a polynomial (of minimal degree) that equals 1
   at an appropriate point on a given facet of equals 0 at the other
   facets.
   """

   def __init__(self, mesh, e):
      """ Initialize """

      # Create appropriate function space
      q = mesh.topology().dim()
      V = FunctionSpace(mesh, "DG", q)
      Function.__init__(self, V)

      self.e = e

      # Set coefficients corresponding to interior dof for this facet
      # for each cell. Note: This depends on the ordering of dofs
      n = V.element().space_dimension()
      interior_facet_dofs = [n*(i+1) - (q+1) + e
                             for i in range(mesh.num_cells())]
      self.vector()[interior_facet_dofs] = 1.0

class SpecialFacetFunction(Expression):
    """
    A SpecialFacetFunction imitates a function in the space of
    polynomials that belong to a space V_h on each facet
    """

    def set(self, R_e):
        self.R_e = R_e

    def eval_data(self, values, data):
        """ If x is on a facet, evaluate the function associated
        with that facet
        """
        # Note: Maybe slow?
        values[0] = 0.0
        if data.on_facet():
            self.R_e[data.facet()].eval_data(values, data)
