/* 
     This file is part of doodle.
     (C) 2004 Christian Grothoff (and other contributing authors)

     doodle is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published
     by the Free Software Foundation; either version 2, or (at your
     option) any later version.

     doodle is distributed in the hope that it will be useful, but
     WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
     General Public License for more details.

     You should have received a copy of the GNU General Public License
     along with doodle; see the file COPYING.  If not, write to the
     Free Software Foundation, Inc., 59 Temple Place - Suite 330,
     Boston, MA 02111-1307, USA.
*/

/**
 * @file doodle/doodle.c
 * @brief main method of doodle (binary).
 * @author Christian Grothoff
 *
 * doodle links against libdoodle to use the code factored into
 * tree.c.  doodle uses GNU getopt and for portability ships with its
 * own version of the getopt code (getopt*).<p>
 *
 * doodle requires libextractor (http://ovmj.org/libextractor/).
 */

#include "config.h"
#include <string.h>
#include <unistd.h>
#include <stdio.h>
#include <ctype.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/mman.h>
#include <fcntl.h>
#include <errno.h>
#include <netinet/in.h>
#include <stdarg.h>
#include <dirent.h>
#include <extractor.h>
#include <locale.h>

#include "gettext.h"
#define _(String) gettext (String)
#define gettext_noop(String) String

#ifndef MINGW
 #define DIR_SEPARATOR '/'
 #define DIR_SEPARATOR_STR "/"
#else
 #define DIR_SEPARATOR '\\'
 #define DIR_SEPARATOR_STR "\\"
#endif

#include "doodle.h"
#include "getopt.h"
#include "grow.c"

/**
 * Maximum search-string length.  If a search-string is more than
 * MAX_LENGTH/2 characters long, it may not find a match even if one
 * exists.  This value can be enlarged arbitrarily, but the cost is
 * that building the database will become more expensive IF for
 * keywords that are that long (the cost for search-strings is
 * quadratic in the length of the search-string; with MAX_LENGTH it is
 * bounded to become linear after a certain size by breaking the
 * string into smaller sections).
 */
#define MAX_LENGTH 128

typedef struct {
  char shortArg;
  char * longArg;
  char * mandatoryArg;
  char * description;
} Help;

#define BORDER 29

static int verbose = 0;
static int very_verbose = 0;

/**
 * Produce nicely formatted output for --help.  Generic
 * method also used in libextractor and GNUnet.
 */
static void formatHelp(const char * general,
		       const char * description,
		       const Help * opt) {
  int slen;
  int i;
  int j;
  int ml;
  int p;
  char * scp;
  const char * trans;
	   
  printf(_("Usage: %s\n%s\n\n"),
	 gettext(general),
	 gettext(description));
  printf(_("Arguments mandatory for long options are also mandatory for short options.\n"));
  slen = 0;
  i = 0;
  while (opt[i].description != NULL) {
    if (opt[i].shortArg == 0)
      printf("      ");
    else
      printf("  -%c, ",
	     opt[i].shortArg);
    printf("--%s",
	   opt[i].longArg);
    slen = 8 + strlen(opt[i].longArg);
    if (opt[i].mandatoryArg != NULL) {
      printf("=%s",
	     opt[i].mandatoryArg);
      slen += 1+strlen(opt[i].mandatoryArg);
    }    
    if (slen > BORDER) {
      printf("\n%*s", BORDER, "");
      slen = BORDER;
    }
    if (slen < BORDER) {
      printf("%*s", BORDER-slen, "");
      slen = BORDER;
    }
    trans = gettext(opt[i].description);
    ml = strlen(trans);
    p = 0;
  OUTER:
    while (ml - p > 78 - slen) {
      for (j=p+78-slen;j>p;j--) {
	if (isspace(trans[j])) {
	  scp = MALLOC(j-p+1);
	  memcpy(scp,
		 &trans[p],
		 j-p);
	  scp[j-p] = '\0';
	  printf("%s\n%*s",
		 scp,
		 BORDER+2,
		 "");
	  free(scp);
	  p = j+1;
	  slen = BORDER+2;
	  goto OUTER;
	}
      }
      /* could not find space to break line */
      scp = MALLOC(78 - slen + 1);
      memcpy(scp,
	     &trans[p],
	     78 - slen);
      scp[78 - slen] = '\0';
      printf("%s\n%*s",
	     scp,
	     BORDER+2,
	     "");	
      free(scp);
      slen = BORDER+2;
      p = p + 78 - slen;
    }
    /* print rest */
    if (p < ml)
      printf("%s\n",
	     &trans[p]);
    i++;
  }
}

/**
 * Print the doodle-specific text for --help.
 */
static void printHelp () {
  static Help help[] = {
    { 'a', "approximate", "DISTANCE",
      gettext_noop("consider strings to match if DISTANCE letters are different") },
    { 'b', "build", NULL,
      gettext_noop("build database (default is to search)") },
    { 'B', "binary", "LANG",
      gettext_noop("use the generic plaintext extractor for the language with the 2-letter language code LANG (use when building database)") }, 
    { 'd', "database", "FILENAME",
      gettext_noop("use location FILENAME to store doodle database") },
    { 'e', "extract", NULL,
      gettext_noop("for each matching file, print the extracted keywords") },
    { 'f', "filenames", NULL,
      gettext_noop("add the filename to the list of keywords (use when building database)") },
    { 'h', "help", NULL,
      gettext_noop("print this help page") },
    { 'H', "hash", "ALGORITHM",
      gettext_noop("compute hash using the given ALGORITHM (currently sha1 or md5)") },
    { 'i', "ignore-case", NULL,
      gettext_noop("be case-insensitive (use when searching)") },
    { 'l', "library", "LIBRARY",
      gettext_noop("load an extractor plugin named LIBRARY") }, 
    { 'L', "log", "FILENAME",
      gettext_noop("log keywords to a file named FILENAME") }, 
    { 'n', "nodefault", NULL,
      gettext_noop("do not load default set of extractor plugins") },
    { 'm', "memory", "SIZE",
      gettext_noop("set the memory limit to SIZE MB (for the tree).") },
    { 'p', "print", NULL,
      gettext_noop("print suffix tree (for debugging)") },
    { 'P', "prunepaths", NULL,
      gettext_noop("exclude given paths from building or searching") },
    { 'v', "version", NULL,
      gettext_noop("print the version number") },
    { 'V', "verbose", NULL,
      gettext_noop("be verbose") },
    { 0, NULL, NULL, NULL },
  };
  formatHelp(_("doodle [OPTIONS] ([FILENAMES]*|[KEYWORDS]*)"),
	     _("Index and search meta-data of files."),
	     help);  
}

static int do_extract = 0;
static int do_default = 1;
static int do_print   = 0;
static int do_filenames = 0;
static int ignore_case = 0;
static unsigned int do_approx = 0;
static char * prunepaths = "/tmp /usr/tmp /var/tmp /dev /proc";

/* *************** directory traversals **************** */

static int isPruned(const char * filename) {
  int i;
  int last;
 
  i = strlen(prunepaths);
  last = i;
  while (i > 0) {
    while ( (prunepaths[i] != ' ') && (i >= 0) )
      i--;
    if (0 == strncmp(&prunepaths[i+1],
		     filename,
		     last - (i+1))) {
      if (very_verbose)
	printf(_("Pruned: %s\n"), filename);
      return 1;
    }
    last = i;
    i--;
  }
  return 0;
}

typedef int (*ScannerCallback)(const char * filename,
			       void * arg);

/**
 * Scan a directory for files. The name of the directory
 * must be expanded first (!).
 * @param dirName the name of the directory
 * @param callback the method to call for each file,
 *        can be NULL, in that case, we only count
 * @param data argument to pass to callback
 * @return the number of files found, -1 on error
 */
static int scanDirectory(char * dirName,
			 int verbose,
			 int very_verbose,
			 ScannerCallback callback,
			 void * cb_arg) {
  DIR * dinfo;
  struct dirent *finfo;
  struct stat istat;
  int count = 0;

  if (dirName == NULL) {
    /* dirName==NULL */
    printf(_("Assertion failed at %s:%d.\n"),
	   __FILE__, __LINE__);
    return -1;
  }
  if (isPruned(dirName))
    return 0;
  if (0 != lstat(dirName, &istat)) {
    if (verbose) 
      printf(_("Call to '%s' for file '%s' failed: %s\n"),
	     "lstat",
	     dirName, 
	     strerror(errno));
    return 0;
  }
#ifdef S_ISLNK
  if (S_ISLNK(istat.st_mode))
    return 0;
#endif
#ifdef S_ISSOCK
  if (S_ISSOCK(istat.st_mode))
    return 0;
#endif
  if (S_ISCHR(istat.st_mode) || 
      S_ISBLK(istat.st_mode) ||
      S_ISFIFO(istat.st_mode) )
    return 0;
  if (S_ISREG(istat.st_mode)) {
    if (-1 == callback(dirName,
		       cb_arg))
      return -1;
    else
      return 1;
  }
  if (! S_ISDIR(istat.st_mode))
    return 0;
  if (verbose)
    printf(_("Scanning '%s'\n"), dirName);
  errno = 0;
  dinfo = opendir(dirName);
  if ((errno == EACCES) || (dinfo == NULL)) {
    if ( (verbose) &&
	 (errno == EACCES) )
      printf(_("Access to directory '%s' was denied.\n"),
	     dirName);
    return 0;
  }
  while ((finfo = readdir(dinfo)) != NULL) {
    char * dn;
    int ret;

    if (finfo->d_name[0] == '.')
      continue;
    dn = MALLOC(strlen(dirName) + 
		strlen(finfo->d_name) + 2);
    strcpy(dn, dirName);
    if (dirName[strlen(dirName)-1] != DIR_SEPARATOR)
      strcat(dn, DIR_SEPARATOR_STR);
    strcat(dn, finfo->d_name);
    ret = scanDirectory(dn,
			verbose,
			very_verbose,
			callback,
			cb_arg);   
    free(dn);
    if (ret >= 0) 
      count += ret;
    else
      return -1;
  }
  closedir(dinfo);
  return count;
}


/**
 * Complete filename (a la shell) from abbrevition.
 * @param fil the name of the file, may contain ~/ or 
 *        be relative to the current directory
 * @returns the full file name, 
 *          NULL is returned on error
 */
static char * expandFileName(const char * fil) {
  char buffer[512];
  char * fn;
#ifndef MINGW
  char * fm;
  const char *fil_ptr;
#else
  long lRet;
#endif

  if (fil == NULL)
    return NULL;

#ifndef MINGW
  if (fil[0] == DIR_SEPARATOR) {
    /* absolute path, just copy */
    return STRDUP(fil);
  }
  if (fil[0] == '~') {
    fm = getenv("HOME");
    if (fm == NULL) {
      /* keep it symbolic to show error to user! */
      fm = "$HOME";
    }

    /* do not copy '~' */
    fil_ptr = fil + 1;

	/* skip over dir seperator to be consistent */    
    if (fil_ptr[0] == DIR_SEPARATOR) 
      fil_ptr++;    
  } else {
    fil_ptr = fil;
    if (getcwd(buffer, 512) != NULL)
      fm = buffer;
    else
      fm = "$PWD";
  }
  fn = MALLOC(strlen(fm) + 1 + strlen(fil_ptr) + 1);
  
  sprintf(fn, "%s/%s", fm, fil_ptr);
#else
  fn = MALLOC(MAX_PATH + 1);

  if ((lRet = conv_to_win_path(fil, buffer)) != ERROR_SUCCESS)
  {
  	SetErrnoFromWinError(lRet);

    return NULL;
  }
  /* is the path relative? */
  if ((strncmp(buffer + 1, ":\\", 2) != 0) && 
      (strncmp(buffer, "\\\\", 2) != 0))
  {
    char szCurDir[MAX_PATH + 1];
    lRet = GetCurrentDirectory(MAX_PATH + 1, szCurDir);
    if (lRet + strlen(fn) + 1 > (_MAX_PATH + 1))
    {
      SetErrnoFromWinError(ERROR_BUFFER_OVERFLOW);
      
      return NULL;
    }
    sprintf(fn, "%s\\%s", szCurDir, buffer);
  }
  else
  {
    strcpy(fn, buffer);
  }
#endif
  return fn;
}

/**
 * Scan location for files.  Call callback on each file.  Abort if
 * callback returns -1.
 *
 * @return -1 on error
 */
static int scan(char * location,
		ScannerCallback callback,
		void * cb_arg) {
  int ret;
  char * exp;
  
  exp = expandFileName(location);
  ret = scanDirectory(exp,
		      verbose,
		      very_verbose,
		      callback,
		      cb_arg);
  free(exp);
  return ret;
}



/* ************** actual functionality code ************* */

/**
 * Find keywords in the given file and append
 * the string describing these keywords to the
 * long string (for the suffix tree).
 */
static int buildIndex(EXTRACTOR_ExtractorList * elist,
		      const char * filename,
		      FILE * logFile,
		      struct DOODLE_SuffixTree * tree) {
  EXTRACTOR_KeywordList * head;
  EXTRACTOR_KeywordList * pos;

  head = EXTRACTOR_getKeywords(elist,
			       filename);
  head = EXTRACTOR_removeDuplicateKeywords
    (head,
     EXTRACTOR_DUPLICATES_REMOVE_UNKNOWN);
  pos  = head;
  if (do_extract) {
    printf(_("Keywords for '%s':\n"),
	   filename);
    EXTRACTOR_printKeywords(stdout,
			    head);
  }
  while (pos != NULL) {
    char * cpos = pos->keyword;
    size_t slen;

    if (logFile != NULL)
      fprintf(logFile, "%s\n", cpos);
    slen = strlen(cpos);
    if (slen > MAX_LENGTH) {
      char section[MAX_LENGTH+1];
      char * xpos;
      int j;

      for (j=0;j<slen;j+=MAX_LENGTH/2) {
	strncpy(section,
		&cpos[j],
		MAX_LENGTH);
	xpos = &section[0];
	while (xpos[0] != '\0') {
	  if (0 != DOODLE_tree_expand(tree,
				      xpos,
				      filename)) {
	    EXTRACTOR_freeKeywords(head);
	    return 0;
	  }
	  xpos++;
	}
      }
    } else {
      while (cpos[0] != '\0') {
	if (0 != DOODLE_tree_expand(tree,
				    cpos,
				    filename)) {
	  EXTRACTOR_freeKeywords(head);
	  return 0;
	}
	cpos++;
      }
    }
    pos = pos->next;
  }
  EXTRACTOR_freeKeywords(head);
  if (do_filenames) {
    const char * cpos = filename;
    while (cpos[0] != '\0') {
      if (0 != DOODLE_tree_expand(tree,
				  cpos,
				  filename)) 
	return 0;
      cpos++;
    }
  }    
  return 1;
}

/**
 * Print log-messages to stdout.
 */
static void my_log(void * unused,
		   unsigned int level,
		   const char * msg,
		   ...) {
  va_list args;
  if ( (level == 0) ||
       (verbose && (level == 1)) ||
       (very_verbose && (level == 2) ) ) {    
    va_start(args, msg);
    vfprintf(stdout, msg, args);
    va_end(args);
  }    
}

/**
 * @brief closure for do_index
 */ 
typedef struct {
  EXTRACTOR_ExtractorList * elist;
  struct DOODLE_SuffixTree * tree;
  FILE * logFile;
} DIC;


static int do_index(const char * filename,
		    DIC * dic) {
  int i;
  int j;
  struct stat sbuf;
  
  j = -1;
  for (i=DOODLE_getFileCount(dic->tree)-1;i>=0;i--) 
    if (0 == strcmp(filename,
		    DOODLE_getFileAt(dic->tree,i)->filename)) {
      j = i;
      break;
    }
  if (j != -1) {
    if (0 != stat(filename,
		  &sbuf)) {
      printf(_("Call to '%s' for file '%s' failed: %s\n"),
	     "stat",
	     filename,
	     strerror(errno));
      return -1;
    }
    if (DOODLE_getFileAt(dic->tree,j)->mod_time == (unsigned int) sbuf.st_mtime) {
      return 0; /* already processed! */
    } else {
      /* remove old keywords, file changed! */
      DOODLE_tree_truncate(dic->tree,
			   filename);
    }
  }
  if (very_verbose)
    printf(_("Processing '%s'\n"),
	   filename);
  return buildIndex(dic->elist,
		    filename,
		    dic->logFile,
		    dic->tree);
}

static int build(const char * libraries,
		 const char * dbName,
		 size_t mem_limit,
		 const char * log,
		 int argc,
		 char * argv[]) {
  int i;
  int ret;
  DIC cls;
  char * ename;

  if (dbName == NULL) {
    printf(_("No database specified.  Aborting.\n"));
    return -1;
  }
  for (i=strlen(dbName);i>=0;i--) {
    if (dbName[i] == ':') {
      printf(_("'%s' is an invalid database filename (has a colon) for building database (option '%s').\n"),
	     dbName,
	     "-b");
      return -1;
    }
  }
  ename = expandFileName(dbName);
  if (ename == NULL)
    return -1;
  /* unlink(ename); */
  cls.tree = DOODLE_tree_create(&my_log,
				NULL,
				ename);
  free(ename);
  if (cls.tree == NULL)
    return -1;
  if (mem_limit != 0)
    DOODLE_tree_set_memory_limit(cls.tree,
				 mem_limit);

  /* first check if all indexed files still exist and are
     accessible */
  if (verbose)
    printf(_("Checking integrity of existing database.\n"));
  for (i=DOODLE_getFileCount(cls.tree)-1;i>=0;i--) {
    struct stat sbuf;
    if ( (0 != lstat(DOODLE_getFileAt(cls.tree,i)->filename,
		     &sbuf)) &&
	 ( (errno == ENOENT) ||
	   (errno == ENOTDIR) ||
	   (errno == ELOOP) ||
	   (errno == EACCES) ) ) {
      printf(_("File '%s' could not be accessed: %s. Removing file from index.\n"),
	     DOODLE_getFileAt(cls.tree,i)->filename,
	     strerror(errno));
      DOODLE_tree_truncate(cls.tree,
			   DOODLE_getFileAt(cls.tree,i)->filename);
    }
    if (! S_ISREG(sbuf.st_mode)) {
      printf(_("File '%s' is not a regular file. Removing file from index.\n"),
	     DOODLE_getFileAt(cls.tree,i)->filename);
      DOODLE_tree_truncate(cls.tree,
			   DOODLE_getFileAt(cls.tree,i)->filename);
    }
  }

  if (do_default) {
    if (verbose)
      printf(_("Loading default set of libextractor plugins.\n"));
    cls.elist = EXTRACTOR_loadDefaultLibraries();
  } else 
    cls.elist = NULL;
  if (libraries != NULL) {
    if (verbose)
      printf(_("Loading libextractor plugins: '%s'\n"),
	     libraries);
    cls.elist = EXTRACTOR_loadConfigLibraries(cls.elist,
					      libraries);  
  }
  if (cls.elist == NULL) {
    printf(_("Failed to load any libextractor plugins. Aborting.\n"));
    DOODLE_tree_destroy(cls.tree);
    return -1;
  }
  cls.logFile = NULL;
  if (log != NULL) {
    cls.logFile = fopen(log, "w+");
    if (cls.logFile == NULL)
      printf(_("Could not open '%s' for logging: %s.\n"),
	     log,
	     strerror(errno));
  }

  ret = 0;
  for (i=0;i<argc;i++) {
    if (verbose)
      printf(_("Indexing '%s'\n"),
	     argv[i]);
    if (-1 == scan(argv[i],
		   (ScannerCallback) &do_index,
		   &cls)) {
      ret = -1;
      break;
    }
  }
  EXTRACTOR_removeAll(cls.elist);
  DOODLE_tree_destroy(cls.tree);
  if (cls.logFile != NULL)
    fclose(cls.logFile);
  return ret;
}

/**
 * @brief closure for printIt
 */
typedef struct {
  EXTRACTOR_ExtractorList * list;
  char ** filenames_seen;
  int seen_size;
  int seen_count;
} PrintItArgs;



static void printIt(const DOODLE_FileInfo * fileinfo,
		    PrintItArgs * args) {
  EXTRACTOR_ExtractorList * list = args->list;
  EXTRACTOR_KeywordList * head;
  const char * filename;
  int i;
  
  filename = fileinfo->filename;
  if (isPruned(filename))
    return;
  for (i=args->seen_count-1;i>=0;i--)
    if (0 == strcmp(filename,
		    args->filenames_seen[i]))
      return;
  if (args->seen_count == args->seen_size) {    
    GROW(args->filenames_seen,
	 args->seen_size,
	 args->seen_size*2 + 128);
  }
  args->filenames_seen[args->seen_count++] = STRDUP(filename);
  if (! access(filename, R_OK | F_OK)) {
    if (do_extract) {
      head = EXTRACTOR_getKeywords(list,
				   filename);
      head = EXTRACTOR_removeDuplicateKeywords
	(head,
	 EXTRACTOR_DUPLICATES_REMOVE_UNKNOWN);
    
      /* print */
      printf(_("Keywords for matching file '%s':\n"),
	     filename);
      EXTRACTOR_printKeywords(stdout,
			      head);
      EXTRACTOR_freeKeywords(head);
    } else {
      printf("%s\n",
	     filename);
    }
    /* There could be an else,
       but we do not want to give hints to matching files
       that cannot be accessed.
       Either they are not readable by the user calling doodle
       or they are not existant any more.
    */
    /*  } else {
     printf(_("Unable to access file '%s': %s\n"), filename, strerror(errno)); */
  }
}

static int print(const char * dbName) {
  struct DOODLE_SuffixTree * tree;
  char * ename;
  struct stat buf;
  int ret;
 
 if (dbName == NULL) {
    printf(_("No database specified. Aborting.\n"));
    return -1;
  }
  ename = expandFileName(dbName);
  if (0 != stat(ename, &buf)) {
    printf(_("Call to '%s' for file '%s' failed: %s.\n"),
	   "stat",
	   dbName,
	   strerror(errno));
    free(ename);
    return -1;
  }
  tree = DOODLE_tree_create(&my_log,
			    NULL,
			    ename);
  free(ename);
  if (tree == NULL) 
    return -1;
  ret = DOODLE_tree_dump(stdout,
			 tree);
  DOODLE_tree_destroy(tree);
  return ret;
}

static int search(const char * libraries,
		  const char * dbName,
		  size_t mem_limit,
		  int argc,
		  char * argv[]) {
  int ret;
  struct stat buf;  
  char * ename;
  struct DOODLE_SuffixTree * tree;
  EXTRACTOR_ExtractorList * extractors;
  int i;
  PrintItArgs args;

  if (dbName == NULL) {
    printf(_("No database specified. Aborting.\n"));
    return -1;
  }
  ename = expandFileName(dbName);
  if (0 != stat(ename, &buf)) {
    printf(_("Call to '%s' for file '%s' failed: %s.\n"),
	   "stat",
	   dbName,
	   strerror(errno));
    free(ename);
    return -1;
  }
  tree = DOODLE_tree_create(&my_log,
			    NULL,
			    ename);
  if (mem_limit != 0)
    DOODLE_tree_set_memory_limit(tree,
				 mem_limit);

  free(ename);
  if (tree == NULL) 
    return -1;
  if (do_extract) {
    if (do_default)
      extractors = EXTRACTOR_loadDefaultLibraries();
    else 
      extractors = NULL;
    if (libraries != NULL) 
      extractors = EXTRACTOR_loadConfigLibraries(extractors,
						 libraries);  
  } else
    extractors = NULL;

  ret = 0;
  args.list = extractors;
  args.filenames_seen = NULL;
  args.seen_size = 0;
  args.seen_count = 0;

  for (i=0;i<argc;i++) {
    printf(_("Searching for '%s':\n"),
	   argv[i]);
    if (strlen(argv[i]) > MAX_LENGTH) {
      printf(_("Warning: search string is longer than %d characters, search will not work.\n"),
	     MAX_LENGTH);
      continue; /* no need to even try... */
    }
    if (strlen(argv[i]) > MAX_LENGTH/2) {
      printf(_("Warning: search string is longer than %d characters, search may not work properly.\n"),
	     MAX_LENGTH/2);
    }
    if ( (do_approx == 0) &&
	 (ignore_case == 0) ) {
      if (0 == DOODLE_tree_search(tree,
				  argv[i],
				  (DOODLE_ResultCallback) &printIt,
				  &args)) {
	printf(_("\tNot found!\n"));
	ret++;
      } 
    } else {
      if (0 == DOODLE_tree_search_approx(tree,
					 do_approx,
					 ignore_case,
					 argv[i],
					 (DOODLE_ResultCallback) &printIt,
					 &args)) {
	printf(_("\tNot found!\n"));
	ret++;
      }
    }
  }  
  DOODLE_tree_destroy(tree);
  EXTRACTOR_removeAll(extractors);

  for (i=0;i<args.seen_count;i++)
    free(args.filenames_seen[i]);
  GROW(args.filenames_seen,
       args.seen_size,
       0);
  return ret;
}

int main(int argc,
	 char * argv[]) {
  int c;
  int option_index;
  char * libraries = NULL;
  int do_build = 0;
  size_t mem_limit = 0;
  char * dbName;
  char * binary = NULL;
  char * hash = NULL;
  char * tmp;
  char * log = NULL;
  int ret;

  setlocale (LC_ALL, "");
  bindtextdomain (PACKAGE, LOCALEDIR);
  textdomain (PACKAGE);

  dbName = getenv("DOODLE_PATH");
  if (NULL == dbName)
    dbName = "~/.doodle";
  tmp = getenv("PRUNEPATHS");
  if (tmp != NULL)
    prunepaths = tmp;

  while (1) {
    static struct option long_options[] = {
      {"approximate", 1, 0, 'a'},
      {"binary", 1, 0, 'B'},
      {"build", 0, 0, 'b'},
      {"database", 1, 0, 'd'},
      {"extract", 0, 0, 'e'},
      {"filenames", 0, 0, 'f'} ,
      {"help", 0, 0, 'h'},
      {"hash", 1, 0, 'H'}, 
      {"ignore-case", 0, 0, 'i'},
      {"library", 1, 0, 'l'},
      {"log", 1, 0, 'L'},
      {"memory", 1, 0, 'm'},
      {"nodefault", 1, 0, 'n'},
      {"print", 0, 0, 'p'},
      {"prunepaths", 1, 0, 'P' },
      {"verbose", 0, 0, 'V'},
      {"version", 0, 0, 'v'},
      {NULL, 0, 0, 0}
    };
    option_index = 0;
    c = getopt_long(argc,
		    argv, "VvhbB:eH:nm:il:L:vd:fa:p", 
		    long_options, 
		    &option_index);
    
    if (c == -1)
      break; /* No more flags to process */
    switch (c) {
    case 'a': 
      if (1 != sscanf(optarg, "%ud", &do_approx)) {
	printf(_("You must pass a number to the '%s' option.\n"),
	       "-a");
	return -1;
      }
      if (do_build == 1) {
	printf(_("The options '%s' and '%s' cannot be used together!\n"),
	       "-a", "-b");
	return -1;
      }	
      break;
    case 'B':
      binary = optarg;
      break;
    case 'b':
      do_build = 1;
      if (do_approx != 0) {
	printf(_("The options '%s' and '%s' cannot be used together!\n"),
	       "-a", "-b");
	return -1;
      }	
      if (do_print == 1) {
	printf(_("The options '%s' and '%s' cannot be used together!\n"),
	       "-b", "-p");
	return -1;
      }	
      if (ignore_case == 1) {
	printf(_("The options '%s' and '%s' cannot be used together!\n"),
	       "-b", "-i");
	return -1;
      }    
      break;
    case 'd':
      dbName = optarg;
      break;
    case 'e':
      do_extract = 1;
      break;
    case 'f':
      do_filenames = 1;
      break;
    case 'h':
      printHelp();
      return 0;
    case 'H':
      hash = optarg;
      break;
    case 'i':
      ignore_case = 1;
      if (do_build == 1) {
	printf(_("The options '%s' and '%s' cannot be used together!\n"),
	       "-b", "-i");
	return -1;
      }
      break;      
    case 'l':
      libraries = optarg;
      break;
    case 'L':
      log = optarg;
      break;
    case 'm': 
      if (1 != sscanf(optarg, "%ud", &mem_limit)) {
	printf(_("You must pass a number to the '%s' option.\n"),
	       "-m");
	return -1;
      }
      if (mem_limit > 0xFFFFFFFF / 1024 / 1024) {
	printf(_("Specified memory limit is too high.\n"));
	return -1;
      }
      mem_limit *= 1024 * 1024;
      break;
    case 'n':
      do_default = 0;
      break;      
    case 'p':
      do_print = 1;
      if (do_build == 1) {
	printf(_("The options '%s' and '%s' cannot be used together!\n"),
	       "-b", "-p");
	return -1;
      }	
      break;
    case 'P':
      prunepaths = optarg;
      break;
    case 'V':
      if (verbose == 1)
	very_verbose = 1;
      verbose = 1;
      break;
    case 'v':
      printf(_("Version %s\n"),
	     PACKAGE_VERSION);
      return 0;
    default:
      fprintf(stderr,
	      _("Use '--help' to get a list of options.\n"));
      return -1;
    }  /* end of parsing commandline */
  } /* while (1) */
  
  if ( (do_print == 0) &&
       (argc - optind < 1) ) {    
    fprintf(stderr,
	    do_build ? 
	    _("Invoke with filenames or directories to index!\n") :
	    _("Invoke with query terms to search for!\n"));
    return -1;
  }
  
  if (libraries != NULL)
    libraries = strdup(libraries);
  if (binary != NULL) {    
    if (libraries == NULL)
      tmp = MALLOC(strlen("libextractor_printable_") + 1 + 
		   strlen(binary));
    else
      tmp = MALLOC(strlen(libraries) + 
		   strlen(":libextractor_printable_") + 1 + 
		   strlen(binary));
    if (libraries == NULL) {
      strcpy(tmp, "libextractor_printable_");
    } else {
      strcpy(tmp, libraries);
      strcat(tmp, ":libextractor_printable_");
    }
    strcat(tmp, binary);
    if (libraries != NULL)
      free(libraries);
    libraries = tmp;
  } 

  if (hash != NULL) {    
    if (libraries == NULL)
      tmp = MALLOC(strlen("libextractor_hash_") + 1 + 
		   strlen(hash));
    else
      tmp = MALLOC(strlen(libraries) + 
		   strlen(":libextractor_hash_") + 1 + 
		   strlen(hash));
    if (libraries == NULL) {
      strcpy(tmp, "libextractor_hash_");
    } else {
      strcpy(tmp, libraries);
      strcat(tmp, ":libextractor_hash_");
    }
    strcat(tmp, hash);
    if (libraries != NULL)
      free(libraries);
    libraries = tmp;
  } 

  if (do_build) {
    ret = build(libraries,
		dbName,
		mem_limit,
		log,
		argc - optind,
		&argv[optind]);
    if (libraries != NULL)
      free(libraries);
    return ret;    
  } else if (do_print) {
    int i;
    char * name;
    
    ret = 0;
    name = strdup(dbName);    
    for (i=strlen(name)-1;i>=0;i--) {
      if (name[i] == ':') {
	ret += print(&name[i+1]);		 
	name[i] = '\0';
      }
    }
    ret = print(name);
    free(name);
    if (libraries != NULL)
      free(libraries);
    return ret;
  } else {
    int i;
    char * name;
    
    ret = 0;
    name = strdup(dbName);    
    for (i=strlen(name)-1;i>=0;i--) {
      if (name[i] == ':') {
	ret += search(libraries,
		      &name[i+1],
		      mem_limit,
		      argc - optind,
		      &argv[optind]);
	name[i] = '\0';
      }
    }
    ret += search(libraries,
		  name,
		  mem_limit,
		  argc - optind,
		  &argv[optind]);
    free(name);
    if (libraries != NULL)
      free(libraries);
    return ret;
  }
}

/* end of doodle.c */
