/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * Copyright 2005 Todd Kulesza
 *
 * Authors:
 * 		Todd Kulesza <todd@dropline.net>
 */

#include <config.h>

#include <string.h>
#include <libxml/parser.h>
#include <glib.h>

#include "atom.h"

/* Create a new AtomEntry item */
AtomEntry*
atom_entry_new (void)
{
	AtomEntry *entry;
	
	entry = g_new0 (AtomEntry, 1);
	entry->content = NULL;
	entry->id = NULL;
	entry->issued = NULL;
	entry->link = NULL;
	entry->title = NULL;
	
	return entry;
}

/* Free an AtomEntry item */
void
atom_entry_free (AtomEntry *entry)
{
	g_return_if_fail (entry);
	
	g_free (entry->content);
	g_free (entry->id);
	g_free (entry->issued);
	g_free (entry->link);
	g_free (entry->title);
	g_free (entry);
	
	return;
}

/* Build an XML packet from the AtomEntry data */
gchar*
atom_build_packet (AtomEntry *entry)
{
	xmlDocPtr doc;
	xmlNodePtr node, child;
	xmlChar *xml;
	gint bufsize;
	
	/* build the root entry node */
	doc = xmlNewDoc ("1.0");
	doc->encoding = xmlStrdup("UTF-8");
	doc->standalone = 1;
	node = xmlNewNode (NULL, "entry");
	xmlNewProp (node, "xmlns", "http://purl.org/atom/ns#");
	xmlDocSetRootElement (doc, node);
	
	/* add the AtomEntry elements */
	child = xmlNewTextChild (node, NULL, "title", entry->title);
	xmlNewProp (child, "type", "text/html");
	xmlNewProp (child, "mode", "escaped");
	xmlNewTextChild (node, NULL, "issued", entry->issued);
	child = xmlNewTextChild (node, NULL, "generator", "Drivel");
	xmlNewProp (child, "version", VERSION);
	xmlNewProp (child, "url", "http://www.dropline.net/drivel/");
	child = xmlNewTextChild (node, NULL, "content", entry->content);
	xmlNewProp (child, "type", "text/html");
	xmlNewProp (child, "mode", "escaped");
	
	/* output the XML */
	xmlDocDumpFormatMemoryEnc (doc, &xml, &bufsize, "UTF-8", 0);
	
	xmlFreeDoc (doc);
	
	return xml;
}

/* Remove the <?xml> and <div> tags and convert <br/> to line-breaks */
static gchar*
tidy_xhtml (const gchar *xhtml)
{
	gchar *text, *xml, *bracket;
	gchar **lines;
	const gchar *tidy, *tag;
	
	tidy = xhtml;
	
	/* Remove the <?xml> tag */
	xml = strstr (tidy, "<?xml");
	if (xml)
	{
		bracket = strstr (xml, ">");
		if (bracket)
			tidy = ++bracket;
	}
	
	/* Remove the surrounding <div> tags */
	xml = strstr (tidy, "<div xmlns=\"http://www.w3.org/1999/xhtml\">");
	if (xml)
	{
		bracket = strstr (xml, ">");
		if (bracket)
			tidy = ++bracket;
	}
	/* Search the last 8 characters for closing </div> tag */
	tag = tidy + (strlen (tidy) - 8);
	xml = strstr (tag, "</div>");
	text = g_strdup (tidy);
	if (xml)
		text[xml - tidy] = '\0';
	
	/* Convert <br/> to line breaks */
	lines = g_strsplit (text, "<br/>", 0);
	
	g_free (text);
	text = g_strjoinv ("\n", lines);
	g_strfreev (lines);
	
	return text;
}

/* Parse an XML packet into an AtomEntry */
AtomEntry*
atom_entry_parse (xmlDocPtr doc, xmlNodePtr node)
{
	AtomEntry *entry;
	
	entry = atom_entry_new ();
	
	while (node)
	{
		if (!xmlStrcasecmp (node->name, "content"))
		{
			xmlDocPtr content;
			xmlNodePtr root;
			xmlChar *buffer;
			gint size;
			
			content = xmlNewDoc ("1.0");
			root = xmlDocCopyNode (node->xmlChildrenNode, content, 1);
			xmlDocSetRootElement (content, root);
			xmlDocDumpFormatMemoryEnc (content, &buffer, &size, "UTF-8", 0);
			entry->content = tidy_xhtml (buffer);
			
			xmlFree (buffer);
			xmlFreeDoc (content);
		}
		
		if (!xmlStrcasecmp (node->name, "id"))
			entry->id = xmlNodeListGetString (doc, node->xmlChildrenNode, 1);
		
		if (!xmlStrcasecmp (node->name, "issued"))
			entry->issued = xmlNodeListGetString (doc, node->xmlChildrenNode, 1);
		
		if (!xmlStrcasecmp (node->name, "link"))
		{
			gchar *rel = xmlGetProp (node, "rel");
			if (!xmlStrcasecmp (rel, "service.edit"))
				entry->link = xmlGetProp (node, "href");
			g_free (rel);
		}
		
		if (!xmlStrcasecmp (node->name, "title"))
			entry->title = xmlNodeListGetString (doc, node->xmlChildrenNode, 1);
		
		node = node->next;
	}
	
	return entry;
}
