/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * Copyright 2004 Todd Kulesza
 *
 * Authors:
 * 		Todd Kulesza <todd@dropline.net>
 */

#include <config.h>

#include <string.h>
#include <glib.h>
#include <curl/curl.h>

#include "drivel_request.h"
#include "drivel.h"
#include "journal.h"
#include "dialogs.h"
#include "login.h"
#include "network.h"
#include "tray.h"
#include "utils.h"
#include "blog_lj.h"

static void
parse_friendgroup_data (DrivelRequest *dr, GtkListStore *store)
{
	const gchar *mesg;
	
	/* fetch the user's security groups */
	mesg = drivel_request_value_lookup (dr, "frgrp_maxnum");
	if (mesg)
	{
		gchar group_sort[32], group_name[32], group_public[32];
		const gchar *name, *sort, *public;
		gboolean is_public;
		gint i, groups;
		
		groups = (gint)g_strtod (mesg, NULL);
		
		for (i = 1; i <= groups; i++)
		{
			g_snprintf (group_name, 32, "frgrp_%d_name", i);
			g_snprintf (group_sort, 32, "frgrp_%d_sortorder", i);
			g_snprintf (group_public, 32, "frgrp_%d_public", i);
			
			name = drivel_request_value_lookup (dr, group_name);
			sort = drivel_request_value_lookup (dr, group_sort);
			public = drivel_request_value_lookup (dr, group_public);
			
			if (public)
				is_public = (gboolean)g_strtod (public, NULL);
			else
				is_public = FALSE;
			
			if (name && sort)
			{
				gint sortorder;
				
				sortorder = (gint)g_strtod (sort, NULL);
				store_security_append (store, "protected.png", name, i, 
						sortorder, is_public);
			}
		}
	}
	
	return;
}

DrivelRequest*
blog_lj_build_login_request (const gchar *username, const gchar *uri, 
		const gchar *version, gint n_moods)
{
	DrivelRequest *dr;
	
	dr = drivel_request_new_with_items (REQUEST_TYPE_LOGIN, 
			REQUEST_PROTOCOL_POST, 
			BLOG_API_LJ,
			uri,
			g_strdup ("mode"), g_strdup ("login"),
			g_strdup ("user"), curl_escape (username, 0),
			g_strdup ("ver"), g_strdup ("1"),
			g_strdup ("clientversion"), curl_escape (version, 0),
			g_strdup ("getpickws"), g_strdup ("1"),
			g_strdup ("getpickwurls"), g_strdup ("1"),
			g_strdup ("getmenus"), g_strdup ("1"),
			g_strdup ("getmoods"), g_strdup_printf ("%d", n_moods),
			NULL);
	
	return dr;
}

void
blog_lj_parse_login_request (DrivelClient *dc, DrivelRequest *dr)
{
	const gchar *mesg, *key, *value;
	gchar static_key[64];
	gint pictures, i, j, offset, menus;
	GtkWidget *dialog;
	DrivelRequest *dr2;
	LJMenuItem *menu_item;
	DrivelJournal *dj;
	
	add_account_to_list (dc);
	
	/* clear out the list of security groups and add the defaults */
	fill_security_menu (dc, dc->security_store);
	
	mesg = drivel_request_value_lookup (dr, "success");
	if (!mesg || !strcmp (mesg, "FAIL"))
	{
		mesg = drivel_request_value_lookup (dr, "errmsg");
		display_error_dialog (dc, _("Server error"), mesg);
		return;
	}

	/* if lj sent us a message, display it to the user */
	mesg = drivel_request_value_lookup (dr, "message");
	if (mesg)
	{
		dialog = gtk_message_dialog_new (
					GTK_WINDOW (dc->current_window),
					GTK_DIALOG_DESTROY_WITH_PARENT,
					GTK_MESSAGE_INFO,
					GTK_BUTTONS_OK,
					mesg);
		
		gtk_dialog_run (GTK_DIALOG (dialog));
		
		gtk_widget_destroy (dialog);
	}
	
	/* are we able to use the fast servers? */
	mesg = drivel_request_value_lookup (dr, "fastserver");
	if (mesg && g_strtod (mesg, NULL))
		dc->use_fast_servers = TRUE;
	else
		dc->use_fast_servers = FALSE;
	
	/* clear or create the hash tables for picture filenames and keywords */
	if (dc->picture_keywords != NULL)
		hash_table_clear (dc->picture_keywords);
	else
		dc->picture_keywords = g_hash_table_new_full (g_str_hash, g_str_equal,
				hash_table_item_free, hash_table_item_free);
	if (dc->picture_filenames != NULL)
		hash_table_clear (dc->picture_filenames);
	else
		dc->picture_filenames = g_hash_table_new_full (g_str_hash, g_str_equal,
				hash_table_item_free, hash_table_item_free);
	dc->default_picture_file = NULL;
	g_hash_table_insert (dc->picture_keywords, g_strdup ("pickw_0"),
				g_strdup_printf ("[%s]", _("default")));
	
	/* fill the picture keywords hash table */
	mesg = drivel_request_value_lookup (dr, "pickw_count");
	if (mesg)
	{
		pictures = (gint) g_strtod (mesg, NULL);
		dc->pictures = pictures;
		
		for (i = 1; i < pictures + 1; i++)
		{
			gchar *localkey;
			localkey = g_strdup_printf ("pickw_%d", i);
			g_hash_table_insert (dc->picture_keywords, g_strdup (localkey), 
					g_strdup (drivel_request_value_lookup (dr, localkey)));
			g_free (localkey);
		}
	}
	else
		dc->pictures = 0;
	
	/* fill the picture filenames hash table */
	mesg = drivel_request_value_lookup (dr, "pickwurl_count");
	if (mesg && (dc->pictures > 0))
	{
		pictures = (gint) g_strtod (mesg, NULL);
		/* if we have a different number of picture urls vs picture keywords,
		 * something went wrong.  just use the keywords, ignore the pictures. */
		if (pictures == dc->pictures)
		{
			for (i = 1; i < pictures + 1; i++)
			{
				gchar *pic_file, *key, *no_user_id, *user_id, *pic_id;
				const gchar *url;
				DrivelRequest *dr2;
				
				key = g_strdup_printf ("pickwurl_%d", i);
				url = drivel_request_value_lookup (dr, key);
				user_id = g_path_get_basename (url);
				no_user_id = g_path_get_dirname (url);
				pic_id = g_path_get_basename (no_user_id);
				pic_file = g_strdup_printf ("%s_%s", user_id, pic_id);
				g_free (no_user_id);
				g_free (user_id);
				g_free (pic_id);
				g_free (key);
				key = g_strdup_printf ("pickw_%d", i);
				g_hash_table_insert (dc->picture_filenames, g_strdup (key),
						g_strdup (pic_file));
				dr2 = blog_lj_build_getpicture_request (url, pic_file);
				if (!picture_exists (dc->config_directory, pic_file))
					net_enqueue_request (dc, dr2);
				
				g_free (pic_file);
				g_free (key);
			}
		}
	}
	mesg = drivel_request_value_lookup (dr, "defaultpicurl");
	if (mesg)
	{
		gchar *pic_file, *user_id, *pic_id, *no_user_id;
		DrivelRequest *dr2;
		
		user_id = g_path_get_basename (mesg);
		no_user_id = g_path_get_dirname (mesg);
		pic_id = g_path_get_basename (no_user_id);
		pic_file = g_strdup_printf ("%s_%s", user_id, pic_id);
		dc->default_picture_file = g_build_filename (dc->config_directory, 
				"pictures", pic_file, NULL);
		dr2 = blog_lj_build_getpicture_request (mesg, pic_file);
		if (!picture_exists (dc->config_directory, pic_file))
			net_enqueue_request (dc, dr2);
		
		g_free (no_user_id);
		g_free (user_id);
		g_free (pic_id);
		g_free (pic_file);
	}
	
	/* get any new moods */
	mesg = drivel_request_value_lookup (dr, "mood_count");
	if (mesg)
	{
		gchar mood_id [32], mood_name [32];
		gint new_moods;
		
		new_moods = (gint) g_strtod (mesg, NULL);
		offset = 0;
		value = NULL;
		
		for (i = 1; i <= new_moods; i++)
		{			
			g_snprintf (mood_id, 32, "mood_%d_id", i);
			g_snprintf (mood_name, 32, "mood_%d_name", i);

			value = drivel_request_value_lookup (dr, mood_id);
			key = drivel_request_value_lookup (dr, mood_name);
			
			g_snprintf (static_key, 64, "/apps/drivel/moods/mood_%s", value);
			
			g_hash_table_insert (dc->mood_icons, g_strdup (key), g_strdup (value));
			gconf_client_set_string (dc->client, static_key, key, NULL);
			dc->mood_list = g_slist_append (dc->mood_list, g_strdup (key));
		}
		
		if (value)
			dc->moods = (gint) g_strtod (value, NULL);

		dc->mood_list = g_slist_sort (dc->mood_list, string_compare);
		
		gconf_client_set_int (dc->client, "/apps/drivel/moods/moods", dc->moods, NULL);
		gconf_client_set_list (dc->client, "/apps/drivel/moods/mood_list", GCONF_VALUE_STRING, dc->mood_list, NULL);
	}
	
	/* fetch the user's security groups */
	parse_friendgroup_data (dr, dc->security_store);
	
	/* # of journals this user has access to */
	mesg = drivel_request_value_lookup (dr, "access_count");
	if (mesg)
	{
		dc->journals = (gint) g_strtod (mesg, NULL);

		for (i = 0; i < dc->journals; i++)
		{
			g_snprintf (static_key, 64, "access_%d", i + 1);
			mesg = drivel_request_value_lookup (dr, static_key);
			if (strncmp (mesg, dc->user->username, strlen (mesg)))
			{
				dj = drivel_journal_new ();
				dj->name = g_strdup (mesg);
				dj->uri_view = g_strdup_printf ("%s/users/%s",
						dc->user->server, mesg);
				dj->type = JOURNAL_TYPE_COMMUNITY;
				dc->journal_list = g_slist_prepend (dc->journal_list, dj);
			}
		}
	}
	else
		dc->journals = 0;
	
	/* put our login name into the list of journals we can post to */
	dj = drivel_journal_new ();
	dj->name = g_strdup (dc->user->username);
	dj->uri_view = g_strdup_printf ("%s/users/%s",
						dc->user->server, dj->name);
	/* get the user's journal name */
	mesg = drivel_request_value_lookup (dr, "name");
	if (mesg)
		dj->description = g_strdup (mesg);
	dc->journal_list = g_slist_prepend (dc->journal_list, dj);
	dc->journal_list = g_slist_sort (dc->journal_list, (GCompareFunc)sort_journals);
	dc->journals++;

	/* fetch recent entries for our journal */
	dr2 = blog_lj_build_getevents_request (dc->user->username, 
			dc->user->server, 0, FALSE, FALSE, "lastn", NULL, 0, 0, 0, 
			DRIVEL_N_RECENT_POSTS, NULL, 0, dj);
	net_enqueue_request (dc, dr2);
	
	/* create the web links menu */
	for (i = 0; ; i++)
	{
		g_snprintf (static_key, 64, "menu_%d_count", i);
		mesg = drivel_request_value_lookup (dr, static_key);

		if (mesg)
		{			
			menus = (gint) g_strtod (mesg, NULL);
			for (j = 1; j < menus + 1; j++)
			{
				menu_item = g_new0 (LJMenuItem, 1);
				
				g_snprintf (static_key, 64, "menu_%d_%d_text", i, j);
				mesg = drivel_request_value_lookup (dr, static_key);
				menu_item->label =g_strdup (mesg);
				
				g_snprintf (static_key, 64, "menu_%d_%d_url", i, j);
				mesg = drivel_request_value_lookup (dr, static_key);
				menu_item->url =g_strdup (mesg);
				
				g_snprintf (static_key, 64, "menu_%d_%d_sub", i, j);
				mesg = drivel_request_value_lookup (dr, static_key);
				if (mesg)
					menu_item->sub_menu = (gint) g_strtod (mesg, NULL);
				
				menu_item->menu_index = i;
				menu_item->item_index = j;
				
				dc->menu_list = g_slist_append (dc->menu_list, menu_item);
			}
		}
		else
			break;
	}

	gtk_widget_hide (dc->login_window);
	journal_window_build (dc);
		
	return;
}

DrivelRequest*
blog_lj_build_getpicture_request (const gchar *url, const gchar *filename)
{
	DrivelRequest *dr;

	dr = drivel_request_new_with_items (REQUEST_TYPE_GETPICTURE, 
			REQUEST_PROTOCOL_GET, 
			BLOG_API_LJ,
			url,
			g_strdup ("filename"), g_strdup (filename),
			NULL);
	
	return dr;
}

void
blog_lj_parse_getpicture_request (DrivelClient *dc, DrivelRequest *dr)
{
	gchar *path;
	GnomeVFSHandle *handle;
	GnomeVFSResult result;
	const gchar *filename;
	
	if (!dc->config_directory)
	{
		g_warning ("You do not have a valid config directory.");
		return;
	}
	
	filename = drivel_request_item_lookup (dr, "filename");
	if (!filename)
	{
		g_warning ("I don't know what filename to save this image as.");
		return;
	}
	
	path = g_build_filename (dc->config_directory, "pictures", 
			filename, NULL);
	result = gnome_vfs_create (&handle, path, GNOME_VFS_OPEN_WRITE, FALSE,
			GNOME_VFS_PERM_USER_READ | GNOME_VFS_PERM_USER_WRITE);
	g_free (path);
	
	if (result != GNOME_VFS_OK)
	{
		g_warning ("%s", gnome_vfs_result_to_string (result));
		return;
	}
	
	gnome_vfs_write (handle, dr->data->data, dr->data->len, NULL);
	gnome_vfs_close (handle);
	
	fill_picture_menu (dc, dc->picture_store);
	gtk_combo_box_set_active (GTK_COMBO_BOX (dc->journal_picture),
			gconf_client_get_int (dc->client, dc->gconf->default_picture, NULL));
	
	return;
}

DrivelRequest*
blog_lj_build_checkfriends_request (const gchar *username, const gchar *uri,
		const gchar* lastupdate)
{
	DrivelRequest *dr;
	
	dr = drivel_request_new_with_items (REQUEST_TYPE_CHECKFRIENDS, 
			REQUEST_PROTOCOL_POST, 
			BLOG_API_LJ,
			uri,
			g_strdup ("mode"), g_strdup ("checkfriends"),
			g_strdup ("user"), curl_escape (username, 0),
			g_strdup ("ver"), g_strdup ("1"),
			g_strdup ("lastupdate"), curl_escape (lastupdate, 0),
			NULL);
	
	return dr;
}

void
blog_lj_parse_checkfriends_request (DrivelClient *dc, DrivelRequest *dr)
{
	const gchar *mesg;
	
	dc->checking_friends = FALSE;
	
	mesg = drivel_request_value_lookup (dr, "success");
	if (!mesg || !strcmp (mesg, "FAIL"))
		return;
	
	mesg = drivel_request_value_lookup (dr, "lastupdate");
	if (mesg)
	{
		g_free (dc->lastupdate);
		dc->lastupdate = g_strdup (mesg);
	}
	
	mesg = drivel_request_value_lookup (dr, "new");
	if (mesg)
	{
		if ((gint) g_strtod (mesg, NULL))
		{
			dc->friends_update = TRUE;
			tray_display (TRUE);
		}
		else
			dc->friends_update = FALSE;
	}
	
	mesg = drivel_request_value_lookup (dr, "interval");
	if (mesg)
		dc->time_since_checkfriends = (gint) g_strtod (mesg, NULL);
	
	return;
}

DrivelRequest*
blog_lj_build_postevent_request (const gchar *username, const gchar *uri,
		const gchar *event, const gchar *music, const gchar *mood,
		gint moodid, const gchar *subject, const gchar *security, gint mask,
		const gchar *picture, const gchar *tags, gint year, gint month, gint day, gint hour,
		gint minute, gint nocomments, gint preformatted, 
		const DrivelJournal *dj, gint backdate)
{
	DrivelRequest *dr;
	const gchar *journal;
	gchar text[8];
	
	if (dj && dj->type == JOURNAL_TYPE_COMMUNITY)
		journal = dj->name;
	else
		journal = NULL;
		
	dr = drivel_request_new_with_items (REQUEST_TYPE_POSTEVENT, 
			REQUEST_PROTOCOL_POST, 
			BLOG_API_LJ,
			uri,
			g_strdup ("mode"), g_strdup ("postevent"),
			g_strdup ("user"), curl_escape (username, 0),
			g_strdup ("ver"), g_strdup ("1"),
			g_strdup ("lineendings"), g_strdup ("unix"),
			g_strdup ("event"), curl_escape (event, 0),
			g_strdup ("subject"), curl_escape (subject, 0),
			g_strdup ("security"), curl_escape (security, 0),
			curl_escape ("prop_current_music", 0), curl_escape (music, 0),
			curl_escape ("prop_current_mood", 0), curl_escape (mood, 0),
			curl_escape ("prop_picture_keyword", 0), curl_escape (picture, 0),
			curl_escape ("prop_taglist", 0), curl_escape (tags, 0),
			NULL);
	
	
	g_snprintf (text, 8, "%d", moodid);
	drivel_request_add_items (dr, 
			curl_escape ("prop_current_moodid", 0), g_strdup (text), NULL);
	
	g_snprintf (text, 8, "%d", mask);
	drivel_request_add_items (dr, 
			g_strdup ("allowmask"), g_strdup (text), NULL);
	
	g_snprintf (text, 8, "%04d", year);
	drivel_request_add_items (dr, 
			g_strdup ("year"), g_strdup (text), NULL);
	
	g_snprintf (text, 8, "%02d", month);
	drivel_request_add_items (dr, 
			g_strdup ("mon"), g_strdup (text), NULL);
	
	g_snprintf (text, 8, "%02d", day);
	drivel_request_add_items (dr, 
			g_strdup ("day"), g_strdup (text), NULL);
	
	g_snprintf (text, 8, "%02d", hour);
	drivel_request_add_items (dr, 
			g_strdup ("hour"), g_strdup (text), NULL);
	
	g_snprintf (text, 8, "%02d", minute);
	drivel_request_add_items (dr, 
			g_strdup ("min"), g_strdup (text), NULL);
	
	g_snprintf (text, 8, "%d", nocomments);
	drivel_request_add_items (dr, 
			curl_escape ("prop_opt_nocomments", 0), g_strdup (text), NULL);
	
	g_snprintf (text, 8, "%d", preformatted);
	drivel_request_add_items (dr, 
			curl_escape ("prop_opt_preformatted", 0), g_strdup (text), NULL);
	
	g_snprintf (text, 8, "%d", backdate);
	drivel_request_add_items (dr,
			curl_escape ("prop_opt_backdated", 0), g_strdup (text), NULL);
	
	if (journal)
	{
		drivel_request_add_items (dr, 
				g_strdup ("usejournal"), curl_escape (journal, 0), NULL);
	}
	
	return dr;
}

DrivelRequest*
blog_lj_build_editevent_request (const gchar *username, const gchar *uri,
		const gchar *itemid, const gchar *event, const gchar *music, 
		const gchar *mood, gint moodid, const gchar *subject, 
		const gchar *security, gint mask, const gchar *picture, const gchar *tags,
		gint year, gint month, gint day, gboolean newdate, gint nocomments, 
		gint preformatted, const DrivelJournal *dj)
{
	DrivelRequest *dr;
	const gchar *journal;
	gchar text[8];
	
	if (dj && dj->type == JOURNAL_TYPE_COMMUNITY)
		journal = dj->name;
	else
		journal = NULL;
	
	dr = drivel_request_new_with_items (REQUEST_TYPE_EDITEVENT, 
			REQUEST_PROTOCOL_POST, 
			BLOG_API_LJ,
			uri,
			g_strdup ("mode"), g_strdup ("editevent"),
			g_strdup ("user"), curl_escape (username, 0),
			g_strdup ("ver"), g_strdup ("1"),
			g_strdup ("lineendings"), g_strdup ("unix"),
			g_strdup ("itemid"), curl_escape (itemid, 0),
			g_strdup ("event"), curl_escape (event, 0),
			g_strdup ("subject"), curl_escape (subject, 0),
			g_strdup ("security"), curl_escape (security, 0),
			curl_escape ("prop_current_music", 0), curl_escape (music, 0),
			curl_escape ("prop_current_mood", 0), curl_escape (mood, 0),
			curl_escape ("prop_picture_keyword", 0), curl_escape (picture, 0),
			curl_escape ("prop_taglist", 0), curl_escape (tags, 0),
			NULL);
	
	
	g_snprintf (text, 8, "%d", moodid);
	drivel_request_add_items (dr, 
			curl_escape ("prop_current_moodid", 0), g_strdup (text), NULL);
	
	g_snprintf (text, 8, "%d", mask);
	drivel_request_add_items (dr, 
			g_strdup ("allowmask"), g_strdup (text), NULL);
	
	g_snprintf (text, 8, "%d", nocomments);
	drivel_request_add_items (dr, 
			curl_escape ("prop_opt_nocomments", 0), g_strdup (text), NULL);
	
	g_snprintf (text, 8, "%d", preformatted);
	drivel_request_add_items (dr, 
			curl_escape ("prop_opt_preformatted", 0), g_strdup (text), NULL);
	
	if (newdate)
	{
		g_snprintf (text, 8, "%04d", year);
		drivel_request_add_items (dr, 
				g_strdup ("year"), g_strdup (text), NULL);
		
		g_snprintf (text, 8, "%02d", month);
		drivel_request_add_items (dr, 
				g_strdup ("mon"), g_strdup (text), NULL);
		
		g_snprintf (text, 8, "%02d", day);
		drivel_request_add_items (dr, 
				g_strdup ("day"), g_strdup (text), NULL);
	}
	
	if (journal)
	{
		drivel_request_add_items (dr, 
				g_strdup ("usejournal"), curl_escape (journal, 0), NULL);
	}
	
	return dr;
}

void
blog_lj_parse_postevent_request (DrivelClient *dc, DrivelRequest *dr)
{
	const gchar *mesg;
	
	mesg = drivel_request_value_lookup (dr, "success");
	if (!mesg || !strcmp (mesg, "FAIL"))
	{
		mesg = drivel_request_value_lookup (dr, "errmsg");
		display_error_dialog (dc, _("Server error"), mesg);
		return;
	}
	
	net_ping_technorati (dc);
	journal_finished_post (dc);
	
	return;
}

DrivelRequest*
blog_lj_build_getevents_request (const gchar *username, const gchar *uri,
		gint truncate, gboolean prefersubject, gboolean noprops, 
		const gchar *selecttype, const gchar *lastsync, gint year, gint month, 
		gint day, gint howmany, const gchar *beforedate, gint itemid, 
		const DrivelJournal *dj)
{
	DrivelRequest *dr;
	const gchar *journal;
	gchar text[8];
	
	if (dj && dj->type == JOURNAL_TYPE_COMMUNITY)
		journal = dj->name;
	else
		journal = NULL;
	
	dr = drivel_request_new_with_items (REQUEST_TYPE_GETEVENTS, 
			REQUEST_PROTOCOL_POST, 
			BLOG_API_LJ,
			uri,
			g_strdup ("mode"), g_strdup ("getevents"),
			g_strdup ("user"), curl_escape (username, 0),
			g_strdup ("ver"), g_strdup ("1"),
			g_strdup ("lineendings"), g_strdup ("unix"),
			g_strdup ("prefersubject"), g_strdup_printf ("%d", prefersubject),
			g_strdup ("noprops"), g_strdup_printf ("%d", noprops),
			g_strdup ("selecttype"), g_strdup (selecttype),
			NULL);
	
	/* handle the different modes of getting events */
	if (!strcmp (selecttype, "day"))
	{
		g_snprintf (text, 8, "%04d", year);
		drivel_request_add_items (dr, 
				g_strdup ("year"), g_strdup (text), NULL);
		
		g_snprintf (text, 8, "%02d", month);
		drivel_request_add_items (dr, 
				g_strdup ("month"), g_strdup (text), NULL);
		
		g_snprintf (text, 8, "%02d", day);
		drivel_request_add_items (dr, 
				g_strdup ("day"), g_strdup (text), NULL);
	}
	else if (!strcmp (selecttype, "lastn"))
	{
		drivel_request_add_items (dr, 
				g_strdup ("howmany"), g_strdup_printf ("%d", howmany), NULL);
		
		if (beforedate)
		{
			drivel_request_add_items (dr, 
					g_strdup ("beforedate"), g_strdup (beforedate), NULL);
		}
	}
	else if (!strcmp (selecttype, "one"))
	{
		drivel_request_add_items (dr, 
				g_strdup ("itemid"), g_strdup_printf ("%d", itemid), NULL);
	}
	else if (!strcmp (selecttype, "syncitems"))
	{
		drivel_request_add_items (dr, 
				g_strdup ("lastsync"), g_strdup (lastsync), NULL);
	}
	
	if (truncate)
	{
		drivel_request_add_items (dr, 
				g_strdup ("truncate"), g_strdup_printf ("%d", truncate), NULL);
	}
	
	if (journal)
	{
		drivel_request_add_items (dr, 
				g_strdup ("usejournal"), curl_escape (journal, 0), NULL);
	}
	
	if (!strcmp (selecttype, "lastn"))
	{
		drivel_request_add_items (dr,
				g_strdup ("lastn"), g_strdup ("true"), NULL);
	}
	
	return dr;
}

void
blog_lj_parse_getevents_request (DrivelClient *dc, DrivelRequest *dr)
{
	const gchar *mesg;
	gchar **itemid, **new_event, key [32];
	gchar **old_event, **security, **allowmask, **subject, **tags;
	gchar **eventtime, **comments = NULL, **autoformat = NULL;
	gchar **mood = NULL, **music = NULL, **picture = NULL;
	gint prop_count, events_count, i, j;
	
	mesg = drivel_request_value_lookup (dr, "success");
	if (!mesg || !strcmp (mesg, "FAIL"))
	{
		mesg = drivel_request_value_lookup (dr, "errmsg");
		display_error_dialog (dc, _("Server error"), mesg);
		return;
	}

	mesg = drivel_request_value_lookup (dr, "events_count");
	if (mesg)
		events_count = (gint) g_strtod (mesg, NULL);
	else
		events_count = 0;

	itemid = g_new (gchar *, events_count + 1);
	old_event = g_new (gchar *, events_count + 1);
	new_event = g_new (gchar *, events_count + 1);
	security = g_new (gchar *, events_count + 1);
	allowmask = g_new (gchar *, events_count + 1);
	subject = g_new (gchar *, events_count + 1);
	tags = g_new (gchar *, events_count + 1);
	eventtime = g_new (gchar *, events_count + 1);
	picture = g_new (gchar *, events_count + 1);
	mood = g_new (gchar *, events_count + 1);
	music = g_new (gchar *, events_count + 1);
	comments = g_new (gchar *, events_count + 1);
	autoformat = g_new (gchar *, events_count + 1);
	
	for (i = 0; i <= events_count; i++)
	{
		itemid [i] = NULL;
		old_event [i] = NULL;
		new_event [i] = NULL;
		security [i] = NULL;
		allowmask [i] = NULL;
		subject [i] = NULL;
		tags [i] = NULL;
		eventtime [i] = NULL;
		picture [i] = NULL;
		mood [i] = NULL;
		music [i] = NULL;
		comments [i] = NULL;
		autoformat [i] = NULL;
	}
	
	for (i = 0; i < events_count; i++)
	{
		g_snprintf (key, 32, "events_%d_itemid", i + 1);
		itemid [i] = g_strdup (drivel_request_value_lookup (dr, key));
		
		g_snprintf (key, 32, "events_%d_event", i + 1);
		old_event [i] = g_strdup (drivel_request_value_lookup (dr, key));
		new_event [i] = unescape_text (old_event [i]);
		
		g_snprintf (key, 32, "events_%d_security", i + 1);
		security [i] = g_strdup (drivel_request_value_lookup (dr, key));
		
		g_snprintf (key, 32, "events_%d_allowmask", i + 1);
		allowmask [i] = g_strdup (drivel_request_value_lookup (dr, key));
		
		g_snprintf (key, 32, "events_%d_subject", i + 1);
		subject [i] = g_strdup (drivel_request_value_lookup (dr, key));
		
		g_snprintf (key, 32, "events_%d_eventtime", i + 1);
		eventtime [i] = g_strdup (drivel_request_value_lookup (dr, key));
		
		mesg = drivel_request_value_lookup (dr, "prop_count");
		if (mesg)
			prop_count = (gint) g_ascii_strtod (mesg, NULL);
		else
			prop_count = 0;
		
		for (j = 0; j < prop_count; j++)
		{
			g_snprintf (key, 32, "prop_%d_itemid", j + 1);
			mesg = drivel_request_value_lookup (dr, key);
			if (strcmp (mesg, itemid[i]))
				continue;

			g_snprintf (key, 32, "prop_%d_name", j + 1);
			mesg = drivel_request_value_lookup (dr, key);

			g_snprintf (key, 32, "prop_%d_value", j + 1);

			if (!strcmp (mesg, "picture_keyword"))
				picture [i] = g_strdup (drivel_request_value_lookup (dr, key));
			else if (!strcmp (mesg, "current_mood"))
				mood [i] = g_strdup (drivel_request_value_lookup (dr, key));
			else if (!strcmp (mesg, "current_music"))
				music [i] = g_strdup (drivel_request_value_lookup (dr, key));
			else if (!strcmp (mesg, "opt_nocomments"))
				comments [i] = g_strdup (drivel_request_value_lookup (dr, key));
			else if (!strcmp (mesg, "opt_preformatted"))
				autoformat [i] = g_strdup (drivel_request_value_lookup (dr, key));
			else if (!strcmp (mesg, "taglist"))
				tags [i] = g_strdup (drivel_request_value_lookup (dr, key));
		}
	}
	
	if (drivel_request_item_lookup (dr, "lastn"))
	{
		for (i = 0; i < events_count; i++)
		{
			DrivelJournalProp *prop ;
			DrivelJournalEntry *entry = journal_entry_new ();
			
			entry->content = g_strdup (new_event[i]);
			if (subject[i])
				entry->subject = g_strdup (subject[i]);
			entry->userid = g_strdup (drivel_request_item_lookup (dr, "user"));
			entry->postid = g_strdup (itemid[i]);
			entry->security = g_strdup (security[i]);
			entry->security_mask = g_strdup (allowmask[i]);
			if (picture[i])
			{
				prop = journal_prop_new ();
				prop->name = g_strdup ("picture");
				prop->value = g_strdup (picture[i]);
				g_ptr_array_add (entry->properties, prop);
			}
			if (music[i])
			{
				prop = journal_prop_new ();
				prop->name = g_strdup ("music");
				prop->value = g_strdup (music[i]);
				g_ptr_array_add (entry->properties, prop);
			}
			if (mood[i])
			{
				prop = journal_prop_new ();
				prop->name = g_strdup ("mood");
				prop->value = g_strdup (mood[i]);
				g_ptr_array_add (entry->properties, prop);
			}
			if (comments[i])
			{
				prop = journal_prop_new ();
				prop->name = g_strdup ("comments");
				prop->value = g_strdup (comments[i]);
				g_ptr_array_add (entry->properties, prop);
			}
			if (autoformat[i])
			{
				prop = journal_prop_new ();
				prop->name = g_strdup ("autoformat");
				prop->value = g_strdup (autoformat[i]);
				g_ptr_array_add (entry->properties, prop);
			}
			if (tags[i])
			{
				prop = journal_prop_new ();
				prop->name = g_strdup ("taglist");
				prop->value = g_strdup (tags[i]);
				g_ptr_array_add (entry->properties, prop);
			}
			prop = journal_prop_new ();
			prop->name = g_strdup ("eventtime");
			prop->value = g_strdup (eventtime[i]);
			g_ptr_array_add (entry->properties, prop);
			
			g_ptr_array_add (dc->recent_entries, entry);
		}
		
		journal_refresh_recent_entries (dc);
	}
	else if (dc->edit_entry)
	{	
		journal_edit_entry (dc, itemid [0], new_event [0], security [0], 
				allowmask [0], subject [0],	mood [0], music [0], picture [0], 
				eventtime [0], comments [0], autoformat [0], NULL, tags [0]);
	}
	else
		update_history_list (dc, itemid, new_event, eventtime, events_count);
	
	g_strfreev (itemid);
	g_strfreev (new_event);
	g_strfreev (old_event);
	g_strfreev (security);
	g_strfreev (allowmask);
	g_strfreev (subject);
	g_strfreev (eventtime);
	g_strfreev (picture);
	g_strfreev (mood);
	g_strfreev (music);
	g_strfreev (comments);
	g_strfreev (autoformat);
	g_strfreev (tags);
	
	return;
}

DrivelRequest*
blog_lj_build_getdaycounts_request (const gchar *username, const gchar *uri,
		const DrivelJournal *dj)
{
	DrivelRequest *dr;
	const gchar *journal;
	
	if (dj && dj->type == JOURNAL_TYPE_COMMUNITY)
		journal = dj->name;
	else
		journal = NULL;
	
	dr = drivel_request_new_with_items (REQUEST_TYPE_GETDAYCOUNTS, 
			REQUEST_PROTOCOL_POST, 
			BLOG_API_LJ,
			uri,
			g_strdup ("mode"), g_strdup ("getdaycounts"),
			g_strdup ("user"), curl_escape (username, 0),
			g_strdup ("ver"), g_strdup ("1"),
			NULL);
	
	if (journal)
	{
		drivel_request_add_items (dr, 
				g_strdup ("usejournal"), curl_escape (journal, 0), NULL);
	}
	
	return dr;
}

void
blog_lj_parse_getdaycounts_request (DrivelClient *dc, DrivelRequest *dr)
{
	const gchar *mesg;
	gchar date [12];
	gint8 days [31];
	guint month, day, year, i, count;
	
	mesg = drivel_request_value_lookup (dr, "success");
	if (mesg == NULL || !strcmp (mesg, "FAIL"))
	{
		mesg = drivel_request_value_lookup (dr, "errmsg");
		display_error_dialog (dc, _("Server error"), mesg);
		return;
	}
	
	gtk_calendar_get_date (GTK_CALENDAR (dc->history_calendar),
			&year, &month, &day);
	
	for (i = 0; i < 31; i++)
	{
		g_snprintf (date, 11, "%04d-%02d-%02d", year, month + 1, i + 1);

		mesg = drivel_request_value_lookup (dr, date);
		if (mesg)
			count = g_strtod (mesg, NULL);
		else
			count = 0;
		
		days [i] = count;
	}
	
	update_history_marks (dc, days);
	
	return;
}

DrivelRequest*
blog_lj_build_editfriends_request (const gchar *username, const gchar *uri,
		const gchar *friend, gboolean delete, gboolean new, const gchar *fg, 
		const gchar *bg)
{
	DrivelRequest *dr;
	gchar *text;
	
	dr = drivel_request_new_with_items (REQUEST_TYPE_EDITFRIENDS, 
			REQUEST_PROTOCOL_POST, 
			BLOG_API_LJ,
			uri,
			g_strdup ("mode"), g_strdup ("editfriends"),
			g_strdup ("user"), curl_escape (username, 0),
			g_strdup ("ver"), g_strdup ("1"),
			NULL);
	
	if (delete || !new)
	{
		text = g_strdup_printf ("editfriend_delete_%s", friend);
		drivel_request_add_items (dr, 
				curl_escape (text, 0), g_strdup ("1"), NULL);
		g_free (text);
	}
	
	if (!delete)
	{
		drivel_request_add_items (dr,
				curl_escape ("editfriend_add_1_user", 0), curl_escape (friend, 0), 
				NULL);
	}
	
	if (!delete && fg)
	{
		drivel_request_add_items (dr,
				curl_escape ("editfriend_add_1_fg", 0), curl_escape (fg, 0), 
				NULL);
	}
	
	if (! delete && bg)
	{
		drivel_request_add_items (dr,
				curl_escape ("editfriend_add_1_bg", 0), curl_escape (bg, 0),
				NULL);
	}
	
	return dr;
}

void
blog_lj_parse_editfriends_request (DrivelClient *dc, DrivelRequest *dr)
{
	const gchar *mesg, *uri;
	DrivelRequest *dr_new;
	
	mesg = drivel_request_value_lookup (dr, "success");
	if (!mesg || !strcmp (mesg, "FAIL"))
	{
		mesg = drivel_request_value_lookup (dr, "errmsg");
		display_error_dialog (dc, _("Server error"), mesg);
		return;
	}
	
	uri = drivel_request_get_uri (dr);
	dr_new = blog_lj_build_getfriends_request (dc->user->username, uri, TRUE, FALSE);
	net_enqueue_request (dc, dr_new);

	return;
}

DrivelRequest*
blog_lj_build_getfriends_request (const gchar *username, const gchar *uri, 
		gboolean friendof, gboolean groups)
{
	DrivelRequest *dr;
	DrivelRequestType type;
	
	if (groups)
		type = REQUEST_TYPE_GETFRIENDGROUPS;
	else
		type = REQUEST_TYPE_GETFRIENDS;
	
	dr = drivel_request_new_with_items (type, 
			REQUEST_PROTOCOL_POST, 
			BLOG_API_LJ,
			uri,
			g_strdup ("mode"), g_strdup ("getfriends"),
			g_strdup ("user"), curl_escape (username, 0),
			g_strdup ("ver"), g_strdup ("1"),
			g_strdup ("includefriendof"), g_strdup_printf ("%d", friendof),
			g_strdup ("includegroups"), g_strdup_printf ("%d", groups),
			NULL);
	
	return dr;
}

void
blog_lj_parse_getfriends_request (DrivelClient *dc, DrivelRequest *dr)
{
	const gchar *mesg;
	gchar key [32];
	gint friends, friends_of, i;
	LJFriend *friend;
	GSList *list_item;
	
	mesg = drivel_request_value_lookup (dr, "success");
	if (!mesg || !strcmp (mesg, "FAIL"))
	{
		mesg = drivel_request_value_lookup (dr, "errmsg");
		display_error_dialog (dc, _("Server error"), mesg);
		return;
	}
	
	if (dc->friends_list)
	{
		g_slist_foreach (dc->friends_list, friends_list_free_item, NULL);
		g_slist_free (dc->friends_list);
		dc->friends_list = NULL;
	}
	
	mesg = drivel_request_value_lookup (dr, "friend_count");
	friends = (gint) g_strtod (mesg, NULL);
	for (i = 1; i < friends + 1; i++)
	{
		friend = g_new0 (LJFriend, 1);
		
		g_snprintf (key, 32, "friend_%d_user", i);
		friend->username = g_strdup (drivel_request_value_lookup (dr, key));
		g_snprintf (key, 32, "friend_%d_name", i);
		friend->name = g_strdup (drivel_request_value_lookup (dr, key));
		g_snprintf (key, 32, "friend_%d_bg", i);
		friend->bg = g_strdup (drivel_request_value_lookup (dr, key));
		g_snprintf (key, 32, "friend_%d_fg", i);
		friend->fg = g_strdup (drivel_request_value_lookup (dr, key));
		g_snprintf (key, 32, "friend_%d_groupmask", i);
		mesg = drivel_request_value_lookup (dr, key);
		if (mesg)
			friend->groupmask = (gint) g_strtod (mesg, NULL);
		else
			friend->groupmask = 0;
		g_snprintf (key, 32, "friend_%d_type", i);
		mesg = drivel_request_value_lookup (dr, key);
		if (!mesg)
			friend->type = FRIEND_TYPE_USER;
		else if (!strcmp (mesg, "community"))
			friend->type = FRIEND_TYPE_COMMUNITY;
		else if (!strcmp (mesg, "syndicated"))
			friend->type = FRIEND_TYPE_FEED;
		else
			friend->type = FRIEND_TYPE_USER;
		friend->friend = TRUE;
		friend->friend_of = FALSE;
		
		dc->friends_list = g_slist_prepend (dc->friends_list, friend);
	}
	
	mesg = drivel_request_value_lookup (dr, "friendof_count");
	friends_of = (gint) g_strtod (mesg, NULL);
	for (i = 1; i <= friends_of; i++)
	{
		g_snprintf (key, 32, "friendof_%d_user", i);
		mesg = drivel_request_value_lookup (dr, key);
		list_item = g_slist_find_custom (dc->friends_list, mesg, compare_usernames);
		if (list_item)
		{
			friend = list_item->data;
			friend->friend_of = TRUE;
		}
		else
		{
			friend = g_new0 (LJFriend, 1);
		
			g_snprintf (key, 32, "friendof_%d_user", i);
			friend->username = g_strdup (drivel_request_value_lookup (dr, key));
			g_snprintf (key, 32, "friendof_%d_name", i);
			friend->name = g_strdup (drivel_request_value_lookup (dr, key));
			g_snprintf (key, 32, "friendof_%d_bg", i);
			friend->bg = g_strdup (drivel_request_value_lookup (dr, key));
			g_snprintf (key, 32, "friendof_%d_fg", i);
			friend->fg = g_strdup (drivel_request_value_lookup (dr, key));
			friend->groupmask = 1;
			g_snprintf (key, 32, "friendof_%d_type", i);
			mesg = g_strdup (drivel_request_value_lookup (dr, key));
			if (mesg)
				friend->type = TRUE;
			else
				friend->type = FALSE;
			friend->friend = FALSE;
			friend->friend_of = TRUE;
		
			dc->friends_list = g_slist_prepend (dc->friends_list, friend);
		}
	}
	
	mesg = drivel_request_item_lookup (dr, "includegroups");
	if (mesg && (gboolean)g_strtod (mesg, NULL))
	{
		/* fetch the user's security groups */
		gtk_list_store_clear (dc->security_store_filtered);
		parse_friendgroup_data (dr, dc->security_store_filtered);
		display_security_dialog (dc);
	}
	else
		display_edit_friends_dialog (dc);
	
	return;
}
