#	Programmer:	Daniel Pozmanter
#	E-mail:		drpython@bluebottle.com
#	Note:		You must reply to the verification e-mail to get through.
#
#	Copyright 2003-2004 Daniel Pozmanter
#
#	Distributed under the terms of the GPL (GNU Public License)
#
#    DrPython is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
	
#The Prompt

import sys, locale, os.path, re
import wx
import wx.stc
from drProperty import *
import drPopUp
import drKeywords
import drShortcuts, drShortcutsFile
from drDragAndDrop import drDropTarget

class DrPrompt(wx.stc.StyledTextCtrl):
	def __init__(self, parent, id, grandparent):
		wx.stc.StyledTextCtrl.__init__(self, parent, id)
		
		#Maximum Number of Commands to Keep Track of in Prompt
		self.MAX_PROMPT_COMMANDS = 25

		self.grandparent = grandparent

		self.stclabelarray = drShortcutsFile.GetSTCShortcutList()
		self.stcactionarray = drShortcuts.GetSTCCommandList()

		#Encoding (Dan, 3.6.2)
		self.locale = grandparent.defaultlocale

		self.ID_POPUP_BASE = 33000

		self.SetDropTarget(drDropTarget(self))

		#Right Click Menu
		self.UsePopUp(0)

		self.CommandArray = []
		self.CommandArrayPos = -1
		
		self.editpoint = 0
		
		#Speed Optimization Submitted by Franz (Added here by Dan)
		self.SetModEventMask(wx.stc.STC_PERFORMED_UNDO | wx.stc.STC_PERFORMED_REDO |\
		wx.stc.STC_MOD_DELETETEXT | wx.stc.STC_MOD_INSERTTEXT)
		
		#Process
		self.process = None
		self.pid = -1
		self.pythonintepreter = 0
		
		self.SetMarginWidth(0, 0)
		self.SetMarginWidth(1, 0)
		self.SetMarginWidth(2, 0)
		
		#Goto Traceback:
		self.reTFilename = re.compile('\".*\"')
		self.reTLinenumber = re.compile('line.*\d')
		
		self.SetMarginType(1, wx.stc.STC_MARGIN_NUMBER)

		self.Bind(wx.stc.EVT_STC_MODIFIED, self.OnModified, id=id)
		self.Bind(wx.EVT_KEY_DOWN, self.OnKeyDown)
		self.Bind(wx.EVT_KEY_UP, self.OnKeyUp)
		self.Bind(wx.EVT_UPDATE_UI,  self.RunCheck, id=id)
		self.Bind(wx.EVT_RIGHT_DOWN, self.OnPopUp)
		self.Bind(wx.EVT_LEFT_DOWN, self.OnLeftDown)
		
		self.Bind(wx.EVT_IDLE, self.OnIdle)

		self.Bind(wx.EVT_LEFT_DCLICK, self.OnGotoTraceback)

	def GetEditPoint(self):
		return self.editpoint

	def GetEndOfLineCharacter(self):
		emode = self.GetEOLMode()
		if emode == wx.stc.STC_EOL_CR:
			return '\r'
		elif emode == wx.stc.STC_EOL_CRLF:
			return '\r\n'
		return '\n'

	def OnIdle(self, event):
		if (self.process is not None):
			if self.inputstream.CanRead():
				text = self.inputstream.read()
				self.AddText(text)
				self.EmptyUndoBuffer()
				self.editpoint = self.GetLength()
				self.GotoPos(self.editpoint)
				self.ScrollToLine(self.LineFromPosition(self.editpoint))
			if self.errorstream.CanRead():
				text = self.errorstream.read()
				self.AddText(text)
				self.EmptyUndoBuffer()
				self.editpoint = self.GetLength()		
				self.GotoPos(self.editpoint)
				self.ScrollToLine(self.LineFromPosition(self.editpoint))

	def OnLeftDown(self, event):
		if self.grandparent.prefs.draganddropmode != 1:
			pos = self.PositionFromPoint(wx.Point(event.GetX(), event.GetY()))
			s, e = self.GetSelection()
			if (pos > s) and (pos < e):
				self.SetSelection(pos, pos)
				return
		event.Skip()

	def OnGotoTraceback(self, event):
		line = self.GetLine(self.GetCurrentLine())
		fn = self.reTFilename.search(line)
		ln = self.reTLinenumber.search(line)
		if (fn is not None) and (ln is not None):
			filename = fn.group().strip('\"').replace('\\', '/')			
			try:
				linenumber = int(ln.group().strip('line ')) - 1
			except:
				linenumber = 0
			if os.path.exists(filename):
				alreadyopen = map(lambda x: x.filename, self.grandparent.txtDocumentArray)
				if filename in alreadyopen:
					i = alreadyopen.index(filename)
					self.grandparent.setDocumentTo(i)
				else:
					self.grandparent.OpenFile(filename, (len(self.grandparent.txtDocument.filename) > 0))
				self.grandparent.txtDocument.ScrollToLine(linenumber)
				self.grandparent.txtDocument.GotoLine(linenumber)
				self.grandparent.txtDocument.EnsureCaretVisible()
				self.grandparent.txtDocument.SetFocus()

	def OnKeyDown(self, event):
		if self.pid == -1:
			self.grandparent.RunShortcuts(event)
			event.Skip()
			return
		if (not self.grandparent.RunShortcuts(event)):
			keycode = event.GetKeyCode()
			pos = self.GetCurrentPos()
			if (not self.pid == -1):
				if (pos >= self.editpoint) and (keycode == wx.WXK_RETURN):
					text = self.GetTextRange(self.editpoint, self.GetLength())
					l = len(self.CommandArray)
					if (l < self.MAX_PROMPT_COMMANDS):
						self.CommandArray.insert(0, text)
						self.CommandArrayPos = -1
					else:
						self.CommandArray.pop()
						self.CommandArray.insert(0, text)
						self.CommandArrayPos = -1
					if self.grandparent.prefs.autodetectencoding:
						#Encoding:  Edited by Dan for 3.6.2
						try:
							self.outputstream.write((text + '\n').encode(self.locale))
						except:
							self.AddText("\nDrPython Prompt ERROR:  Encoding: " + str(sys.exc_info()[1]) + "\nTrying without encoding...\n")
							self.outputstream.write(text + '\n')
					else:
						self.outputstream.write(text + '\n')
					self.GotoPos(self.GetLength())
				if (keycode == wx.WXK_UP):
					l = len(self.CommandArray)
					if (len(self.CommandArray) > 0):
						if (self.CommandArrayPos + 1) < l:		
							self.GotoPos(self.editpoint)
							self.SetTargetStart(self.editpoint)
							self.SetTargetEnd(self.GetLength())
							self.CommandArrayPos = self.CommandArrayPos + 1
							self.ReplaceTarget(self.CommandArray[self.CommandArrayPos])
					
				elif (keycode == wx.WXK_DOWN):
					if (len(self.CommandArray) > 0):
						self.GotoPos(self.editpoint)
						self.SetTargetStart(self.editpoint)
						self.SetTargetEnd(self.GetLength())
						if (self.CommandArrayPos - 1) > -1:					
							self.CommandArrayPos = self.CommandArrayPos - 1
							self.ReplaceTarget(self.CommandArray[self.CommandArrayPos])
						else:	
							if (self.CommandArrayPos - 1) > -2:		
								self.CommandArrayPos = self.CommandArrayPos - 1
							self.ReplaceTarget("")
			if ((pos > self.editpoint) and (not keycode == wx.WXK_UP)) or ((not keycode == wx.WXK_BACK) and (not keycode == wx.WXK_LEFT) and (not keycode == wx.WXK_UP) and (not keycode == wx.WXK_DOWN)):
				if (pos < self.editpoint):
					if (not keycode == wx.WXK_RIGHT):
						event.Skip()
				else:
					event.Skip()

	def OnKeyUp(self, event):
		if self.pid == -1:
			event.Skip()
			return
		keycode = event.GetKeyCode()
		#franz: pos was not used
		if keycode == wx.WXK_HOME:
			if (self.GetCurrentPos() < self.editpoint):
				self.GotoPos(self.editpoint)
			return
		elif keycode == wx.WXK_PRIOR:
			if (self.GetCurrentPos() < self.editpoint):
				self.GotoPos(self.editpoint)
			return
		event.Skip()

	def OnModified(self, event):
		if not (self.grandparent.prefs.promptwordwrap):
			ll = self.TextWidth(wx.stc.STC_STYLE_DEFAULT, "OOO")
			x = 0
			spaces = ""
			while (x < self.grandparent.prefs.tabwidth):
				spaces = spaces + " "
				x = x + 1
			current_width = self.GetScrollWidth()
			line = self.GetCurLine()[0].replace('\t', spaces)
			actual_width = self.TextWidth(wx.stc.STC_STYLE_DEFAULT, line)
			if (current_width < actual_width):
				self.SetScrollWidth(actual_width + ll)
				
	def OnPopUp(self, event):
		drPopUp.OnPopUp(self, event)

	def OnPopUpMenu(self, event):
		drPopUp.OnPopUpMenu(self, event)
							
	def RunCheck(self, event):
		if (self.GetCurrentPos() < self.editpoint) or (self.pid == -1):
			self.SetReadOnly(1)
		else:
			self.SetReadOnly(0)
			
	def SetupPrefsPrompt(self, notmdiupdate = 1):
		self.SetEndAtLastLine(not self.grandparent.prefs.promptscrollextrapage)
		

		if notmdiupdate:
			self.SetViewWhiteSpace(self.grandparent.prefs.promptwhitespaceisvisible)
			
		if (self.grandparent.prefs.promptwordwrap):
			self.SetWrapMode(wx.stc.STC_WRAP_WORD)
		else:
			self.SetWrapMode(wx.stc.STC_WRAP_NONE)	
		if (self.grandparent.prefs.eolmode == 1):
			self.SetEOLMode(wx.stc.STC_EOL_CRLF)
		elif (self.grandparent.prefs.eolmode == 2):
			self.SetEOLMode(wx.stc.STC_EOL_CR)
		else:
			self.SetEOLMode(wx.stc.STC_EOL_LF)
		self.SetTabWidth(self.grandparent.prefs.tabwidth)
		self.SetUseTabs(self.grandparent.prefs.promptusetabs)
		self.SetMarginWidth(1, self.grandparent.prefs.promptmarginwidth)
		
		if self.grandparent.prefs.promptusestyles:			
				
			self.SetKeyWords(0, drKeywords.GetKeyWords(0))
				
			self.SetLexer(drKeywords.GetLexer(0))	
		
			self.StyleSetSpec(wx.stc.STC_STYLE_DEFAULT, self.grandparent.prefs.txtPromptStyleArray[0])
					
			self.StyleClearAll()

			self.StartStyling(0, 0xff)
			
			self.SetCaretWidth(self.grandparent.prefs.promptcaretwidth)
			
			self.SetCaretForeground(self.grandparent.prefs.txtPromptStyleArray[13])
			
			if (self.grandparent.prefs.promptusestyles < 2):
				self.StyleSetSpec(wx.stc.STC_STYLE_LINENUMBER, self.grandparent.prefs.txtPromptStyleArray[1])
				self.StyleSetSpec(wx.stc.STC_STYLE_BRACELIGHT, self.grandparent.prefs.txtPromptStyleArray[2])
				self.StyleSetSpec(wx.stc.STC_STYLE_BRACEBAD, self.grandparent.prefs.txtPromptStyleArray[3])
				self.StyleSetSpec(wx.stc.STC_P_CHARACTER, self.grandparent.prefs.txtPromptStyleArray[4])
				self.StyleSetSpec(wx.stc.STC_P_CLASSNAME, self.grandparent.prefs.txtPromptStyleArray[5])
				self.StyleSetSpec(wx.stc.STC_P_COMMENTBLOCK, self.grandparent.prefs.txtPromptStyleArray[6])
				self.StyleSetSpec(wx.stc.STC_P_COMMENTLINE, self.grandparent.prefs.txtPromptStyleArray[6])
				self.StyleSetSpec(wx.stc.STC_P_DEFNAME, self.grandparent.prefs.txtPromptStyleArray[7])
				self.StyleSetSpec(wx.stc.STC_P_WORD, self.grandparent.prefs.txtPromptStyleArray[8])
				self.StyleSetSpec(wx.stc.STC_P_NUMBER, self.grandparent.prefs.txtPromptStyleArray[9])
				self.StyleSetSpec(wx.stc.STC_P_OPERATOR, self.grandparent.prefs.txtPromptStyleArray[10])
				self.StyleSetSpec(wx.stc.STC_P_STRING, self.grandparent.prefs.txtPromptStyleArray[11])
				self.StyleSetSpec(wx.stc.STC_P_STRINGEOL, self.grandparent.prefs.txtPromptStyleArray[11])
				self.StyleSetSpec(wx.stc.STC_P_TRIPLE, self.grandparent.prefs.txtPromptStyleArray[12])
				self.StyleSetSpec(wx.stc.STC_P_TRIPLEDOUBLE, self.grandparent.prefs.txtPromptStyleArray[12])
				
				self.SetSelForeground(1, getStyleProperty("fore", self.grandparent.prefs.txtPromptStyleArray[14]))
				self.SetSelBackground(1, getStyleProperty("back", self.grandparent.prefs.txtPromptStyleArray[14]))
	
	def SetText(self, text):		
		ro = self.GetReadOnly()
		self.SetReadOnly(0)
		wx.stc.StyledTextCtrl.SetText(self, text)
		self.SetReadOnly(ro)
				
	def SetSelectedText(self, text):
		ro = self.GetReadOnly()
		self.SetReadOnly(0)
		self.SetTargetStart(self.GetSelectionStart())
		self.SetTargetEnd(self.GetSelectionEnd())
		self.ReplaceTarget(text)
		self.SetReadOnly(ro)