/* -*- Mode: C++; tab-width: 2; indent-tabs-mode: nil; c-basic-offset: 2 -*- */
/***************************************************************************
 *            plugin.h
 *
 *  Sun Feb  7 14:11:40 CET 2016
 *  Copyright 2016 Bent Bisballe Nyeng
 *  deva@aasimon.org
 ****************************************************************************/

/*
 *  This file is part of PluginGizmo.
 *
 *  PluginGizmo is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation; either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  PluginGizmo is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with PluginGizmo; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA.
 */
#pragma once

#include <vector>
#include <string>

#include <cstdlib>

class MidiEvent;

//! Abstract base-class for plugin implementations.
class Plugin {
public:
	//! Implement this to create a new plugin instance.
	static Plugin* create();

	//! Init function for setting up plugin parameters.
	virtual void init() = 0;

	//! Get current free-wheel mode.
	virtual bool getFreeWheel() const = 0;

	//! This method is called by the host when the free-wheel mode changes.
	virtual void onFreeWheelChange(bool freewheel) = 0;


	//! Call this to get current samplerate.
	virtual float getSamplerate() = 0;

	//! This method is called by the host when the samplerate changes.
	virtual void onSamplerateChange(float samplerate) = 0;


	//! Call this to get current frame-size.
	virtual std::size_t getFramesize() = 0;

	//! This method is called by the host when the frame-size changes.
	virtual void onFramesizeChange(std::size_t framesize) = 0;


	//! Call this to get current active state
	virtual bool getActive() = 0;

	//! This method is called by the host when the active state changes.
	virtual void onActiveChange(bool active) = 0;


	//! This method is called by the host to get the current state for storing.
	virtual std::string onStateSave() = 0;

	//! This method is called by the host when a new state has been loaded.
	virtual void onStateRestore(const std::string& config) = 0;


	//! This is method is called by the host to get the current latency.
	//! \param The latency in samples.
	virtual float getLatency() = 0;

	//! Call this method to signal a latency change to the host.
	//! \param latency The latency in samples.
	virtual void setLatency(float latency) = 0;


	//! Called by the the host to get the number of midi input channels.
	//! This must remain constant during the lifespan of the plugin instance.
	virtual std::size_t getNumberOfMidiInputs() = 0;

	//! Called by the the host to get the number of midi output channels.
	//! This must remain constant during the lifespan of the plugin instance.
	virtual std::size_t getNumberOfMidiOutputs() = 0;

	//! Called by the the host to get the number of audio input channels.
	//! This must remain constant during the lifespan of the plugin instance.
	virtual std::size_t getNumberOfAudioInputs() = 0;

	//! Called by the the host to get the number of audio output channels.
	//! This must remain constant during the lifespan of the plugin instance.
	virtual std::size_t getNumberOfAudioOutputs() = 0;


	//! Get unique plugin id.
	virtual std::string getId() = 0;

	//! Process callback.
	virtual void process(std::size_t pos,
	                     const std::vector<MidiEvent>& input_events,
	                     std::vector<MidiEvent>& output_events,
	                     const std::vector<const float*>& input_samples,
	                     const std::vector<float*>& output_samples,
	                     std::size_t count) = 0;


	//
	// GUI (optional)
	//

	//! Return true if a GUI implementation is to be used.
	virtual bool hasGUI()
	{
		return false;
	}

	//! Create new window.
	virtual void createWindow(void *parent) {}

	//! Destroy window.
	virtual void onDestroyWindow() {}

	//! Show window.
	virtual void onShowWindow() {}

	//! Hide window.
	virtual void onHideWindow() {}

	//! Called regularly by host; process ui events.
	virtual void onIdle() {}

	//! Signal new window size to host.
	virtual void resizeWindow(std::size_t width, std::size_t height) = 0;

	//! Signal close window event to the host.
	virtual void closeWindow() = 0;
};
