/*  dvdisaster: Additional error correction for optical media.
 *  Copyright (C) 2004,2005 Carsten Gnoerlich.
 *  Project home page: http://www.dvdisaster.com
 *  Email: carsten@dvdisaster.com  -or-  cgnoerlich@fsfe.org
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA,
 *  or direct your browser at http://www.gnu.org.
 */

#include "dvdisaster.h"

#include <time.h>

/***
 *** Debugging functions.
 ***/

/*
 * Debugging function to seed the image with random correctable errors
 */

void RandomError(char *prefix)
{  ImageInfo *ii;
   gint64 block_idx[255];
   gint64 s,si;
   int block_sel[255];
   int i,percent,last_percent = 0;
   int n_data,n_eras;
   double eras_scale, blk_scale;

   srandom((int)time(NULL));

   if(!Closure->redundancy || !strcmp(Closure->redundancy, "normal")) n_eras = 32; 
   else if(!strcmp(Closure->redundancy, "high")) n_eras = 64;
   else n_eras = atoi(Closure->redundancy);

   n_data = 255-n_eras;
   eras_scale = (n_eras+1)/((double)RAND_MAX+1.0);
   blk_scale = (double)n_data/((double)RAND_MAX+1.0);


   /*** Open the image file */

   OpenImageAndEcc(&ii, NULL, WRITEABLE_IMAGE);

   /*** Setup block pointers */

   s = (ii->sectors+n_data-1)/n_data;

   for(si=0, i=0; i<n_data; si+=s, i++)
     block_idx[i] = si;

   PrintLog(_("\nGenerating random correctable erasures (max = %d).\n"),n_eras);

   /*** Randomly delete the blocks */

   for(si=0; si<s; si++)
   {  int n_erasures = (int)(eras_scale*(double)random());

      /* Reset the block selector */

      for(i=0; i<n_data; i++)  
	block_sel[i] = 0;

      /* Randomly pick n blocks */

      for(i=0; i<n_erasures; i++)
      {  int idx;
      
         do
	 {  idx = (int)(blk_scale*(double)random());
	 } while(block_sel[idx]);

	 block_sel[idx] = 1;
      }

      /* Delete the randomly picked blocks */

      for(i=0; i<n_data; i++)
      {  if(block_sel[i] && block_idx[i]<ii->sectors)
	  {  if(!LargeSeek(ii->file, (gint64)(2048*block_idx[i])))
	      Stop(_("Could not seek to image sector %lld:\n%s\n"),block_idx[i],strerror(errno));
	     if(LargeWrite(ii->file, Closure->deadSector, 2048) != 2048)
	       Stop(_("Could not write image sector %lld:\n%s\n"),block_idx[i],strerror(errno));
	  }

          block_idx[i]++;
      }

      percent = (100*si)/s;
      if(last_percent != percent) 
      {  PrintProgress(_("Progress: %3d%%"),percent);
	 last_percent = percent;
      }
   }

   PrintProgress(_("Progress: 100%%\n"
	"Recover the image using the --fix option before doing another --random-errors run.\n"
	"Otherwise you'll accumulate >= %d erasures/ECC block and the image will be lost.\n"), 
	n_eras);

   FreeImageInfo(ii);
}

/*
 * Debugging function to simulate images with single
 * byte errors (except for faulty cabling and/or controllers,
 * this should never happen)
 */

void Byteset(char *arg)
{  ImageInfo *ii;
   gint64 s;
   int i,byte;
   char *cpos = NULL;

   /*** Open the image file */

   OpenImageAndEcc(&ii, NULL, WRITEABLE_IMAGE);

   /*** See which byte to set */

   cpos = strchr(arg,',');
   if(!cpos) Stop(_("2nd argument is missing"));
   *cpos = 0;

   s = atoi(arg);
   arg = cpos+1;

   cpos = strchr(arg,',');
   if(!cpos) Stop(_("3rd argument is missing"));
   *cpos = 0;

   i = atoi(arg);
   byte = atoi(cpos+1);

   if(s<0 || s>ii->sectors-1)
     Stop(_("Sector must be in range [0..%lld]\n"),ii->sectors-1);

   if(i<0 || i>=2048)
     Stop(_("Byte position must be in range [0..2047]"));

   if(byte<0 || byte>=256)
     Stop(_("Byte value must be in range [0..255]"));

   PrintLog(_("Setting byte %d in sector %lld to value %d.\n"), i, s, byte); 

   /*** Set the byte */
   
   s = 2048*s + i;
   
   if(!LargeSeek(ii->file, (gint64)s))
     Stop(_("Could not seek to first image sector:\n%s\n"),strerror(errno));

   if(LargeWrite(ii->file, &byte, 1) != 1)
     Stop(_("Could not write the new byte value"));

   FreeImageInfo(ii);
}

/*
 * Debugging function to simulate medium with unreadable sectors
 */

void Erase(char *arg)
{  ImageInfo *ii;
   gint64 start,end,s;
   char *dashpos = NULL;

   /*** Open the image file */

   OpenImageAndEcc(&ii, NULL, WRITEABLE_IMAGE);

   /*** See which sectors to erase */

   dashpos = strchr(arg,'-');
   if(dashpos)
   {  *dashpos = 0;
      start = atoi(arg);
      end  = atoi(dashpos+1);
   }
   else start = end = atoi(arg);

   if(start>end || start < 0 || end >= ii->sectors)
     Stop(_("Sectors must be in range [0..%lld].\n"),ii->sectors-1);

   PrintLog(_("Erasing sectors [%lld,%lld]\n"),start,end);

   /*** Erase them. */

   if(!LargeSeek(ii->file, (gint64)(2048*start)))
     Stop(_("Could not seek to first image sector:\n%s\n"),strerror(errno));

   for(s=start; s<=end; s++)
   {  int m = (end == ii->sectors-1) ? ii->inLast : 2048;
      int n = LargeWrite(ii->file, Closure->deadSector, m);

      if(n != m)
	Stop(_("Could not write image sector %lld:\n%s\n"),s,strerror(errno));
   }

   /*** Clean up */

   FreeImageInfo(ii);
}

/*
 * Debugging function for truncating images
 */

void TruncateImage(char *arg)
{  ImageInfo *ii;
   gint64 end;

   /*** Open the image file */

   OpenImageAndEcc(&ii, NULL, WRITEABLE_IMAGE);

   /*** Determine last sector */

   end = atoi(arg);

   if(end >= ii->sectors)
     Stop(_("New length must be in range [0..%lld].\n"),ii->sectors-1);

   PrintLog(_("Truncating image to %lld sectors.\n"),end);

   /*** Truncate it. */

   if(!LargeTruncate(ii->file, (gint64)(2048*end)))
     Stop(_("Could not truncate %s: %s\n"),Closure->imageName,strerror(errno));

   /*** Clean up */

   FreeImageInfo(ii);
}

/*
 * Debugging function to create an image filled with random numbers
 */

void RandomImage(char *image_name, char *n_sectors)
{  LargeFile *image;
   gint64 sectors,s = 0;
   int percent, last_percent = 0;
   guint32 invert;

   sectors = atoi(n_sectors);
   if(sectors < 0) sectors = 1;

   /*** Open the image file */

   if(!(image = LargeOpen(image_name, O_RDWR | O_CREAT, IMG_PERMS)))
     Stop(_("Can't open %s:\n%s"),image_name,strerror(errno));

   /*** Print banner */

   PrintLog(_("\nCreating random image with %lld sectors.\n\n"
	      "There is no need for permanently storing this image;\n" 
              "you can always reproduce it by calling\n"
	      "dvdisaster --debug --random-image %lld --random-seed %d\n\n"),
	      sectors, sectors, Closure->randomSeed);

   if(Closure->randomSeed >= 0)
   {  SRandom(Closure->randomSeed);
      invert = 0;
   }
   else
   {  SRandom(-Closure->randomSeed);
      invert = 0xffffffff;
   }

   /*** Create it */

   while(s<sectors)
   {  guint32 buf[512];
      int i=511;
      int n;

      do buf[i--] = (Random32() ^ invert); while(i>=0);
      
      n = LargeWrite(image, buf, 2048);
      s++;

      if(n != 2048)
	Stop(_("Could not write image sector %lld:\n%s\n"),s,strerror(errno));

      percent = (100*s)/sectors;
      if(last_percent != percent) 
      {  PrintProgress(_("Progress: %3d%%"),percent);
	 last_percent = percent;
      }
   }

   /*** Clean up */

   if(!LargeClose(image))
     Stop(_("Error closing image file:\n%s"), strerror(errno));
}

/*
 * Replaces the "unreadable sector" marker with zeros.
 */

void ZeroUnreadable(void)
{  ImageInfo *ii;
   unsigned char buf[2048],zeros[2048];
   gint64 s,cnt=0;
   int percent, last_percent = 0;

   OpenImageAndEcc(&ii, NULL, WRITEABLE_IMAGE);
   PrintLog(_("Replacing the \"unreadable sector\" markers with zeros.\n"));
   memset(zeros, 0, 2048);   

   for(s=0; s<ii->sectors; s++)
   {  int n = LargeRead(ii->file, buf, 2048);

      if(n != 2048)
	Stop(_("Could not read image sector %lld:\n%s\n"),s,strerror(errno));

      /* Replace the dead sector marker */

      if(!memcmp(buf,Closure->deadSector,2048)) 
      {
	if(!LargeSeek(ii->file, (gint64)(2048*s)))
	  Stop(_("Could not seek to image sector %lld:\n%s\n"),s,strerror(errno));

	n = LargeWrite(ii->file, zeros, 2048);

	if(n != 2048)
	  Stop(_("Could not write image sector %lld:\n%s\n"),s,strerror(errno));

	cnt++;
      }

      percent = (100*s)/ii->sectors;
      if(last_percent != percent) 
      {  PrintProgress(_("Progress: %3d%%"),percent);
	 last_percent = percent;
      }
   }

   PrintProgress(_("%lld \"unreadable sector\" markers replaced.\n"), cnt);

   FreeImageInfo(ii);
}

/*
 * Debugging function to show contents of a given sector
 */

static void hexdump(unsigned char *buf, int len, int step)
{  int i,j;

   for(i=0; i<len; i+=step)
   {  g_printf("%04x: ",i);
      for(j=0; j<step; j++)
	if(i+j >= len) g_printf((j&0x07) == 0x07 ? "    " : "   ");
	else           g_printf("%02x%s", buf[i+j], (j&0x07) == 0x07 ? "  " : " ");

      for(j=0; j<step; j++)
      { if(i+j >= len) break;
	if((j&0x07) == 0x07)
	      g_printf("%c ", isprint(buf[i+j]) ? buf[i+j] : '.');
	else  g_printf("%c", isprint(buf[i+j]) ? buf[i+j] : '.');
      }
    
      g_printf("\n");
   }
}

void ShowSector(char *arg)
{  ImageInfo *ii;
   gint64 sector;
   int n;
   unsigned char buf[2048];

   /*** Open the image file */

   OpenImageAndEcc(&ii, NULL, READABLE_IMAGE);

   /*** Determine sector to show */

   sector =  atoi(arg);

   if(sector < 0 || sector >= ii->sectors)
     Stop(_("Sector must be in range [0..%lld]\n"),ii->sectors-1);

   PrintLog(_("Contents of sector %lld:\n\n"),sector);

   /*** Show it. */

   if(!LargeSeek(ii->file, (gint64)(2048*sector)))
     Stop(_("Could not seek to image sector %lld:\n%s\n"),sector,strerror(errno));

   n = LargeRead(ii->file, buf, 2048);
   if(n != 2048)
     Stop(_("Could not read image sector %lld:\n%s\n"),sector,strerror(errno));

   hexdump(buf, 2048, 32);

   g_printf("CRC32 = %04x\n", Crc32(buf, 2048));

   /*** Clean up */

   FreeImageInfo(ii);
}

/***
 *** Send a CDB to the drive and report what happens
 ***
 * Sending ill-formed cdbs may kill your system
 * and/or damage yout drive permanently.
 *
 * Example command line call for sending an inquiry:
 *
 * ./dvdisaster --debug --send-cdb 12,00,00,00,24,00:24
 *
 * The first six bytes make up the cdb; cdbs with upto 12 bytes are possible.
 * The :24 arg is the allocation length. 
 * Note that the allocation length must match those specified in the cdb;
 * differing values may crash the system.
 */

enum {  SHIFT0, SHIFT4, ALLOC };

void SendCDB(char *cdb_raw)
{  int cdb_len = 0;
   int alloc_len = 0;
   unsigned char cdb[16];
   int mode = SHIFT4;
   int nibble=0;
   int status;
   char *c = cdb_raw;

   while(*c && cdb_len<16)
   {  if(*c == ',' || *c== ':')
      {  if(*c == ':')
	   mode = ALLOC;
	 c++; continue; 
      }

      if(*c >= '0' && *c <= '9')
	nibble = *c - '0';
      else if(*c >= 'a' && *c <= 'f')
	nibble = *c - 'a' + 10;
      else if(*c >= 'A' && *c <= 'F')
	nibble = *c - 'A' + 10;
      else Stop("illegal char '%c' in cdb \"%s\"\n",*c,cdb_raw);

      switch(mode)
      {  case SHIFT0:
	   cdb[cdb_len] |= nibble;
	   mode = SHIFT4;
	   cdb_len++;
	   break;

         case SHIFT4:
	   cdb[cdb_len] = nibble << 4;
	   mode = SHIFT0;
	   break;

         case ALLOC:
	   alloc_len = (alloc_len << 4) | nibble;
	   break;
      }

      c++;
   }

   PrintLog("\n");
   status = SendReadCDB(Closure->device, cdb, cdb_len, alloc_len);

   if(!status)
   {  g_printf("\nDrive returned:\n\n");
      hexdump(Closure->scratchBuf, alloc_len, 16);
   }
}
