/*  dvdisaster: Additional error correction for optical media.
 *  Copyright (C) 2004,2005 Carsten Gnoerlich.
 *  Project home page: http://www.dvdisaster.com
 *  Email: carsten@dvdisaster.com  -or-  cgnoerlich@fsfe.org
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA,
 *  or direct your browser at http://www.gnu.org.
 */

#ifndef DVDISASTER_H
#define DVDISASTER_H

/* "Dare to be gorgeous and unique. 
 *  But don't ever be cryptic or otherwise unfathomable.
 *  Make it unforgettably great."
 *
 *  From "A Final Note on Style", 
 *  Amiga Intuition Reference Manual, 1986, p. 231
 */

/***
 *** I'm too lazy to mess with #include dependencies.
 *** Everything #includeable is rolled up herein...
 */

#define _GNU_SOURCE

//#include <glib.h>

#include <gtk/gtk.h>
#include <glib/gprintf.h>
#include <glib/gstrfuncs.h>

#include <ctype.h>
#include <errno.h>
#include <fcntl.h>
#include <getopt.h>
#ifdef WITH_NLS_YES
 #include <libintl.h>
 #include <locale.h>
#endif
#include <math.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <stdarg.h>
#include <stddef.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#ifdef SYS_MINGW
#include <io.h>
#endif

#include "md5.h"
#include "version.h"

#ifndef G_THREADS_ENABLED
 #error "need multithreading glib2"
#endif

/* Phrase extraction for gettext() 
   Note that these functions are even required when
   WITH_NLS_NO is set! */

#define _(string) sgettext(string)
#define _utf(string) sgettext_utf8(string)

/* File permissions for images */

#ifdef SYS_MINGW
#define IMG_PERMS (S_IRUSR | S_IWUSR)
#else
#define IMG_PERMS (S_IRUSR | S_IWUSR | S_IRGRP | S_IWGRP | S_IROTH | S_IWOTH)
#endif

/***
 *** Our global closure (encapsulation of global variables)
 ***/

typedef struct _GlobalClosure
{  int version;         /* Integer number representing current program version */
   char *cookedVersion; /* version string formatted for GUI use */
   char *device;        /* currently selected device to read from */
   GPtrArray *deviceNames;  /* List of drive names */
   GPtrArray *deviceNodes;  /* List of device nodes (C: or /dev/foo) */
   char *imageName;     /* complete path of current image file */
   char *eccName;       /* complete path of current ecc file */
   GPtrArray *methodList; /* List of available methods */
   char *methodName;    /* Name of currently selected codec */
   gint64 readStart;    /* Range to read */
   gint64 readEnd;
   int cacheMB;         /* Cache setting for the parity codec, in megabytes */
   int sectorSkip;      /* Number of sectors to skip after read error occurs */
   char *redundancy;    /* Error correction code redundancy */
   int adaptiveRead;    /* Use optimized strategy for reading defective images */
   int speedWarning;    /* Print warning if speed changes by more than given percentage */
   int fillUnreadable;  /* Byte value for filling unreadable sectors or -1 */
   int spinupDelay;     /* Seconds to wait for drive to spin up */
   int truncate;        /* confirms truncation of large images */
   int noTruncate;      /* do not truncate image at the end */
   int unlinkImage;     /* delete image after ecc file creation */
   int driveSpeed;      /* currently unused */
   int debugMode;       /* may activate additional messages */
   int splitFiles;      /* limit image files to 2GB */
   int autoSuffix;      /* automatically extend files with suffices .img/.ecc */
   int parseUDF;        /* use information from the ISO/UDF filesystems */
   int readAndCreate;   /* automatically create .ecc file after reading an image */
   int enableCurveSwitch; /* TRUE in readAndCreateMode after reading is complete */
  
   char *deadSector;    /* Copy of our "dead sector" marker. */
   char *dotFile;       /* path to .dvdisaster file */
   char *binDir;        /* place where the binary resides */
   char *docDir;        /* place where our documentation resides */
   char *browser;       /* Name of preferred WEB browser */

   char bs[256];        /* A string of 255 backspace characters */

   GThread *mainThread; /* Thread of the main() routine */
   void (*cleanupProc)(gpointer);  /* Procedure to cleanup running threads after an error condition */
   gpointer cleanupData;
   char *errorTitle;    /* Title to show in error dialogs */
   gint32 randomSeed;   /* for the random number generator */

#ifdef SYS_MINGW
   /*** Hooks into the ASPI library */

   void *aspiLib;
   unsigned long (*GetASPI32SupportInfo)(void);
   unsigned long (*SendASPI32Command)(void*);
#endif

   unsigned char *scratchBufBase;  /* scratch buffer for use with SCSI commands; */
   unsigned char *scratchBuf;      /* 32768 bytes aligned at a 4096 byte boundary */

   /*** GUI-related things */

   int guiMode;         /* TRUE if GUI is active */
   int keepStyle;       /* TRUE = don't tinker with style */
   int stopActions;     /* crude method to stop ongoing action(s) */

   GtkWidget *logWidget;     /* Dialog for the log display */
   GtkScrolledWindow *logScroll; /* and its scrolled window */
   GtkTextBuffer *logBuffer; /* Text buffer for the log output */
   GString *logString;       /* holds logging output for current action */

   /*** Widgets of the main window */

   GtkWindow *window;        /* main window */
   GtkTooltips *tooltips;    /* our global tooltips structure */
   GdkPixbuf *windowIcon;    /* main window icon */

   GtkWidget *imageFileSel;  /* image file selector */
   GtkWidget *imageEntry;    /* image name entry field */
   GtkWidget *eccFileSel;    /* ecc file selector */
   GtkWidget *eccEntry;      /* ecc name entry field */
   
   GtkWidget *notebook;      /* The notebook behind our central output area */
   GtkLabel  *status;        /* The status label */

   GtkWidget *prefsButton;
   GtkWidget *helpButton;

   GtkWidget *readButton;
   GtkWidget *createButton;
   GtkWidget *scanButton;
   GtkWidget *fixButton;
   GtkWidget *testButton;

   /*** The preferences window */

   GtkWindow *prefsWindow;
   void *prefsContext;       /* local data for the preferences window */

   /*** Common stuff for drawing curves and spirals */

   GdkGC     *drawGC;
   GdkColor  *background,*grid,*black,*white;
   GdkColor  *red,*yellow,*green,*darkgreen,*blue;
   gint      lastPercent;
   gint      lastPercentPlotted;
   gint      percent; 

   /*** Widgets for the reading/scanning action */

   GtkWidget *readLinearHeadline;
   GtkWidget *readLinearDrawingArea;
   struct _Curve  *readLinearCurve;
   struct _Spiral *readLinearSpiral;
   GtkWidget *readLinearNotebook;
   GtkWidget *readLinearSpeed;
   GtkWidget *readLinearErrors;
   GtkWidget *readLinearFootline;
   GtkWidget *readLinearFootlineBox;

   /*** Widgets for the adaptive reading action */

   GtkWidget *readAdaptiveHeadline;
   GtkWidget *readAdaptiveDrawingArea;
   struct _Spiral *readAdaptiveSpiral;
   char *readAdaptiveSubtitle;
   char *readAdaptiveErrorMsg;

   /*** Widgets for compare action */

   GtkWidget *cmpHeadline;

   GtkWidget *cmpImageNotebook;
   GtkWidget *cmpImageSectors;
   GtkWidget *cmpChkSumErrors;
   GtkWidget *cmpMissingSectors;
   GtkWidget *cmpImageMd5Sum;
   GtkWidget *cmpImageResult;

   GtkWidget *cmpEccNotebook;
   GtkWidget *cmpEccEmptyMsg;
   GtkWidget *cmpEccCreatedBy;
   GtkWidget *cmpEccMethod;
   GtkWidget *cmpEccRequires;
   GtkWidget *cmpEccMediumSectors;
   GtkWidget *cmpEccImgMd5Sum;
   GtkWidget *cmpEccFingerprint;
   GtkWidget *cmpEccBlocks;
   GtkWidget *cmpEccMd5Sum;
   GtkWidget *cmpEccResult;

   GtkWidget *cmpDrawingArea;
   struct _Spiral *cmpSpiral;
   PangoLayout *cmpLayout;

} GlobalClosure;

extern GlobalClosure *Closure;  /* these should be the only global variables! */
extern int exitCode;            /* value to use on exit() */

/***
 *** 
 ***/

#define MAX_FILE_SEGMENTS 100

typedef struct _LargeFile
{  int fileSegment[MAX_FILE_SEGMENTS];
   int flags;
   mode_t mode;
   int segment;
   gint64 offset;
   char *basename;
   char *suffix;
   int namelen;
} LargeFile;

/***
 *** An info package about a medium image 
 *** (NOT part or a header of the image file!)
 */

typedef struct _ImageInfo
{  LargeFile *file;                  /* file handle for image */
   gint64 size;                      /* number of medium bytes */
   gint64 sectors;                   /* number of medium sectors */
   gint64 sectorsMissing;            /* number of missing medium sectors */
   gint64 crcErrors;                 /* sectors with CRC32 errors */
   int inLast;                       /* bytes in last sector */
   unsigned char mediumFP[16];       /* md5sum of first sector */
   unsigned char mediumSum[16];      /* complete md5sum of whole medium */
} ImageInfo;

/***
 *** An info package about a error correction file 
 *** (NOT part or a header of the ecc file!)
 */

typedef struct _EccInfo
{  FILE *file;                  /* file handle for ecc file */
   struct _EccHeader *eh;       /* the header struct below */
   gint64 sectors;              /* gint64 version of eh->sectors */ 
   struct MD5Context md5Ctxt;   /* md5 context of crc portion of file */
} EccInfo;

/***
 *** The .ecc file header
 ***/

#define FOOTPRINT_SECTOR 257   /* Sector currently used to calculate the footprint */

typedef struct _EccHeader
{  gint8 cookie[12];           /* "*dvdisaster*" */
   gint8 method[4];            /* e.g. "RS01" */
   gint8 method_flags[4];      /* for free use by the respective methods */
   guint8 mediumFP[16];        /* fingerprint of first medium sector */ 
   guint8 mediumSum[16];       /* complete md5sum of whole medium */
   guint8 eccSum[16];          /* md5sum of ecc code section of .ecc file */
   guint8 sectors[8];          /* number of sectors medium is supposed to have */
   gint32 dataBytes;           /* data bytes per ecc block */
   gint32 eccBytes;            /* ecc bytes per ecc block */
   gint32 creatorVersion;      /* which dvdisaster version created this */
   gint32 neededVersion;       /* oldest version which can decode this file */
   gint32 fpSector;            /* sector used to calculate mediumFP */
   gint8 padding[4000];        /* pad to 4096 bytes: room for future expansion */
} EccHeader;

/***
 *** dvdisaster.c
 ***/

#define OPEN_READABLE     0
#define OPEN_WRITEABLE    (1<<0)
#define OPEN_FULLSCAN     (1<<1)
#define VERBOSE_FULLSCAN  (1<<2)
#define CREATE_CRC32      (1<<3)
#define CHECK_CRC32       (1<<4)


#define READABLE_IMAGE    0
#define READABLE_ECC      0
#define WRITEABLE_IMAGE   (1<<0)
#define WRITEABLE_ECC     (1<<1)

#define PRINT_MODE        (1<<4)
#define CREATE_CRC        ((1<<1) | (1<<5))

char *ApplyAutoSuffix(char*, char*);
void PrepareDeadSector(void);

void OpenImageAndEcc(ImageInfo**, EccInfo**, int);
void FreeImageInfo(ImageInfo*);
void FreeEccInfo(EccInfo*);

void ScanImage(ImageInfo*, EccInfo*, int, GtkWidget*);

ImageInfo* XOpenImage(char*, int, FILE**);
int VerifyVersion(EccHeader*, int fatal);
void UnlinkImage(GtkWidget*);
void CreateEcc(void);
void FixEcc(void);

/***
 *** closure.c
 ***/

void InitClosure(void);
void FreeClosure(void);
void ReadDotfile(void);

/***
 *** compare.c
 ***/

#define COMPARE_IMAGE_SEGMENTS 1000

void Compare(void);
void ResetCompareWindow(void);
void CreateCompareWindow(GtkWidget*);
void AddCompareValues(int, gint64, gint64, gint64, gint64);

/***
 *** crc32.c
 ***/

guint32 Crc32(unsigned char*, int);

/***
 *** curve.c
 ***/

enum 
{  CURVE_PERCENT,
   CURVE_MEGABYTES
}; 

typedef struct _Curve
{  GtkWidget *widget;   /* drawing area which is hosting us */
   PangoLayout *layout;
   gdouble *fvalue;     /* floating point curve values */
   gint    *ivalue;     /* integer curve values */
   gint lastValueIdx;   /* end of value array */
   gint leftX,rightX;   /* Position of left and right y axis */
   gint topY,bottomY;   /* Top and bottom end of y axes */
   char *leftLabel;     /* Label of left coordinate axis */
   char *leftFormat;    /* Format of left coordinate axis numbering */
   int bottomFormat;    /* what kind of data are we displaying? */
   int margin;
   int maxX,maxY;       /* current max value at y axis */
} Curve;

#define REDRAW_ICURVE (1<<0)
#define REDRAW_FCURVE (1<<1)

Curve* CreateCurve(GtkWidget*, char*, char*, int, int);
void ZeroCurve(Curve*);
void FreeCurve(Curve*);

void UpdateCurveGeometry(Curve*, char*, int);

int  CurveX(Curve*, gdouble);
int  CurveY(Curve*, gdouble);
void RedrawAxes(Curve*);
void RedrawCurve(Curve*, int, int);

/***
 *** debug.c
 ***/

void Byteset(char*);
void Erase(char*);
void RandomError(char*);
void RandomImage(char*, char*);
void SendCDB(char*);
void ShowSector(char*);
void TruncateImage(char*);
void ZeroUnreadable(void);

/***
 *** fix-window.c
 ***/

void CreateFixWindow(GtkWidget*);

/***
 *** help-dialogs.c
 ***/

GtkWidget* ShowTextfile(char*, char*, char*, GtkScrolledWindow**, GtkTextBuffer**);
void ShowGPL();
void ShowLog();
void UpdateLog();
void AboutDialog();

/***
 *** icon-factory.c
 ***/

void CreateIconFactory();

/***
 *** large-io.c
 ***/

LargeFile *LargeOpen(const char*, int, mode_t);
int LargeSeek(LargeFile*, gint64);
int LargeRead(LargeFile*, void*, size_t);
int LargeWrite(LargeFile*, void*, size_t);
int LargeClose(LargeFile*);
int LargeTruncate(LargeFile*, gint64);
int LargeStat(char*, gint64*);
int LargeUnlink(char*);

/***
 *** main-window.c
 ***/

#define FIRST_CREATE_WINDOW 4

typedef enum 
{  ACTION_WELCOME,   /* Tab 0; not really an action   */
   ACTION_STOP,      /* ----   does not have a window */
   ACTION_READ,      /* Tab 1 (linear); Tab 2 (adaptive)*/
   ACTION_SCAN,      /* Tab 1 (linear); Tab 2 (adaptive)*/
   ACTION_COMPARE,   /* Tab 3 */
   ACTION_CREATE,    /* CREATE and FIX have separate windows assigned */
   ACTION_FIX        /* for each method. */
} MajorActions;

void CreateMainWindow(int*, char***);
void ContinueWithAction(int);

/***
 *** memtrack.c
 ***/

/*
 * Macro replacements for the glib functions.
 */

#ifdef WITH_MEMDEBUG_YES
#define g_malloc(size) malloc_ext(size,__FILE__,__LINE__)
#define g_malloc0(size) malloc_ext(size,__FILE__,__LINE__)
#define g_realloc(ptr,size) realloc_ext(ptr,size,__FILE__,__LINE__)
#define g_strdup(str) strdup_ext(str,__FILE__,__LINE__)

#define g_strdup_printf(format,args...) \
        strdup_printf_ext(format,__FILE__,__LINE__, ## args)
#define g_strdup_vprintf(format,argp) \
        strdup_vprintf_ext(format,argp,__FILE__,__LINE__)
#define g_locale_to_utf8(str,size,in,out,gerr) \
        g_locale_to_utf8_ext(str,size,in,out,gerr,__FILE__,__LINE__)
#define g_free(size) free_ext(size,__FILE__,__LINE__)

#define REMEMBER(ptr) remember(ptr, 0, __FILE__, __LINE__)
#define FORGET(ptr) forget(ptr)
#else
#define REMEMBER(ptr)
#define FORGET(ptr)
#endif

/* 
 * Protos for the replacement functions.
 */

void*	malloc_ext(int,char*,int);
void*	realloc_ext(void*, int, char*, int);
char*	strdup_ext(const char*,char*,int);
char*	strdup_printf_ext(char*, char*, int, ...);
char*	strdup_vprintf_ext(char*, va_list, char*, int);
gchar*  g_locale_to_utf8_ext(const gchar*, gssize, gsize*, gsize*, GError**, char*, int);
void	free_ext(void*,char*,int);

void    remember(void*, int, char*, int);
int     forget(void*);

void    check_memleaks(void);

/***
 *** menubar.c
 ***/

void AttachTooltip(GtkWidget*, char*, char*);
GtkWidget* CreateMenuBar(GtkWidget*);
GtkWidget* CreateToolBar(GtkWidget*);

/***
 *** method.c / method-link.c
 ***
 * method-link.c is automatically created by the configure script.
 */

typedef struct _Method
{  char name[4];                     /* Method name tag */
   char *description;                /* Fulltext description */
   void (*create)(struct _Method*);  /* Creates an error correction file */
   void (*fix)(struct _Method*);     /* Fixes a damaged image */
   void (*createCreateWindow)(struct _Method*, GtkWidget*);
   void (*createFixWindow)(struct _Method*, GtkWidget*);
   void (*createPrefsPage)(struct _Method*, GtkWidget*);
   void (*resetCreateWindow)(struct _Method*);
   void (*resetFixWindow)(struct _Method*);
   void (*destroy)(struct _Method*);
   int  tabWindowIndex;              /* our position in the (invisible) notebook */
   void *widgetList;                 /* linkage to window system */
} Method;

void BindMethods(void);        /* created by configure in method-link.c */

void CollectMethods(void);
void RegisterMethod(Method*);
void ListMethods(void);
Method* FindMethod(char*);
void CallMethodDestructors(void);

/***
 *** misc.c 
 ***/

#ifdef SYS_MINGW
#define stat _stati64
#define lseek _lseeki64

int fseeko(FILE*, gint64, int);

#define random rand
#define srandom(seed) srand(seed)
#endif

char* sgettext(char*);
char* sgettext_utf8(char*);

gint64 uchar_to_gint64(unsigned char*);
void gint64_to_uchar(unsigned char*, gint64);

void AsciiDigest(char*, unsigned char*);
void CalcSectors(gint64, gint64*, int*);

void PrintCLI(char*, ...);
void PrintLog(char*, ...);
void PrintProgress(char*, ...);
void PrintCLIorLabel(GtkLabel*, char*, ...);

void LogWarning(char*, ...);
void Stop(char*, ...);
void RegisterCleanup(char*, void (*)(gpointer), gpointer);

GThread* CreateGThread(GThreadFunc, gpointer);

void ShowWidget(GtkWidget*);
void AllowActions(gboolean);

void ShowMessage(GtkWindow*, char*, GtkMessageType);
GtkWidget* CreateMessage(char*, GtkMessageType, ...);
void SetLabelText(GtkLabel*, char*, ...);
void SetProgress(GtkWidget*, int, int);

int ModalDialog(GtkMessageType, GtkButtonsType, void (*)(GtkDialog*), char*, ...);
int ModalWarning(GtkMessageType, GtkButtonsType, void (*)(GtkDialog*), char*, ...);

void SetText(PangoLayout*, char*, int*, int*);
void SwitchAndSetFootline(GtkWidget*, int, GtkWidget*, char*, ...);

void ReverseCancelOK(GtkDialog*);
void TimedInsensitive(GtkWidget*, int);

/***
 *** preferences.c
 ***/

void CreatePreferencesWindow(void);
void HidePreferences(void);
void FreePreferences(void*);

/***
 *** print_sense.c
 ***/

void RememberSense(int, int, int);
char *GetSenseString(int, int, int, int);
char* GetLastSenseString(int);

/*** 
 *** read-linear.c
 ***/

void ReadMediumLinear(gpointer);

/***
 *** read-linear-window.c
 ***/

void ResetLinearReadWindow();
void CreateLinearReadWindow(GtkWidget*);

void SetCurveMaxValues(int, gint64, gint64, gint64);
void AddCurveValues(int, double, gint64, int);

/***
 *** random.c
 ***/

#define	MY_RAND_MAX	2147483647

gint32  Random(void);
void    SRandom(gint32);
guint32 Random32(void);

/*** 
 *** read-adaptive.c
 ***/

void ReadMediumAdaptive(gpointer);

/***
 *** read-adaptive-window.c
 ***/

#define ADAPTIVE_READ_SPIRAL_SIZE 4800

void ResetAdaptiveReadWindow();
void CreateAdaptiveReadWindow(GtkWidget*);

void ClipReadAdaptiveSpiral(int);
void SetAdaptiveReadSubtitle(char*);
void SetAdaptiveReadFootline(char*, GdkColor*);
void UpdateAdaptiveResults(gint64, gint64, gint64, int);
void ChangeSegmentColor(GdkColor*, int);
void ChangeSpiralCursor(int);
void RemoveFillMarkers();

/*** 
 *** scsi-layer.c
 ***
 * Note that there is also a scsi-layer.h with more
 * scsi wrapper dependent declarations.
 */

void OpenAspi(void);
void CloseAspi(void);
void ListAspiDrives(void);

char* DefaultDevice(void);
gint64 CurrentImageSize(void);

int SendReadCDB(char*, unsigned char*, int, int);

/***
 *** show-manual.c
 ***/

void ShowHTML(char*);

/***
 *** spiral.c
 ***/

typedef struct _Spiral
{  GdkDrawable *drawable;
   int mx, my;
   int startRadius;
   int segmentSize;
   int segmentCount;
   double *segmentPos;
   GdkColor **segmentColor;
   GdkColor *outline;
   int diameter;
   int segmentClipping;
   int cursorPos;
   GdkColor *colorUnderCursor;
} Spiral;

Spiral* CreateSpiral(GdkColor*, GdkColor*, int, int, int);
void SetSpiralWidget(Spiral*, GtkWidget*);
void FillSpiral(Spiral*, GdkColor*);
void FreeSpiral(Spiral*);
void DrawSpiral(Spiral*);
void DrawSpiralSegment(Spiral*, GdkColor*, int);
void DrawSpiralLabel(Spiral*, PangoLayout*, char*, GdkColor*, int, int);
void MoveSpiralCursor(Spiral*, int);

/***
 *** style.c
 ***/

void AdjustStyle();

/***
 *** welcome-window.c
 ***/

void CreateWelcomePage(GtkNotebook*);


#endif				/* DVDISASTER_H */
