/***********************************************************************************/
/* $Id: Uni.c,v 1.7 2003/06/03 22:51:55 sleeper Exp $ 			           */
/*										   */
/* Copyright (c) 2001, Analog Devices Inc., All Rights Reserved			   */
/*										   */
/* Uni.c									   */
/*										   */
/* UNI (User Network Interface) support module (similar to AtmUni.c from NDIS)     */
/*										   */
/* This file is part of the "ADI USB ADSL Driver for Linux".			   */
/*										   */
/* "ADI USB ADSL Driver for Linux" is free software; you can redistribute it       */
/* and/or modify it under the terms of the GNU General Public License as           */
/* published by the Free Software Foundation; either version 2 of the License,     */
/* or (at your option) any later version.					   */
/*										   */
/* "ADI USB ADSL Driver for Linux" is distributed in the hope that it will be      */
/* useful, but WITHOUT ANY WARRANTY; without even the implied warranty of	   */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the		   */
/* GNU General Public License for more details.					   */
/*										   */
/* You should have received a copy of the GNU General Public License		   */
/* along with "ADI USB ADSL Driver for Linux"; if not, write to the Free Software  */
/* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA       */
/***********************************************************************************/

#include "Adiutil.h"
#include "Uni.h"
#include "Pipes.h"
#include "Sar.h"
#include "debug.h"

extern Boolean CheckForAndProcessOAMCell(Hardware *pHw, UInt8 *pCell);

/**********************************************************************************/
/* IsPTICell									  */
/**********************************************************************************/
#define ATM_PTI_OAM_CELL	0x04
static int IsPTICell(UInt32 header) 
{
    return (((header >> 1) & ATM_PTI_OAM_CELL) == ATM_PTI_OAM_CELL);
}


/***********************************************************************************/
/* UniEstablishVc								   */
/*										   */
/* Allocates memory for and initializes our Vc (virtual channel, virtual circuit)  */
/* for this session. Real ATM allows multiple Vc's per Uni, but it seems that      */
/* there are no circumstances at this point that will require more than one        */
/* Vc in this driver. So, we don't do all the extra logic of managing multiple     */
/* Vc's.									   */
/***********************************************************************************/
void UniEstablishVc(Hardware *pHw, UInt8 *pEncapHdr, UInt32 EncapSize, UInt32 Vpi, UInt32 Vci)
{
    adi_enters (DBG_UNI);
    
    
    /* Initialize the various fields of the Vc*/
    pHw->Vc.pEncapHdr  = pEncapHdr;
    pHw->Vc.EncapSize  = EncapSize;
    pHw->Vc.Vpi        = Vpi;
    pHw->Vc.Vci        = Vci;
    pHw->Vc.VpiVci     = (pHw->Vc.Vpi<<16) | pHw->Vc.Vci;
    pHw->Vc.MaxSduSize = (OUTGOING_DATA_SIZE / ATM_CELL_SIZE) * ATM_CELL_PAYLOAD_SIZE;
    pHw->Vc.ReassemblyInProgress = FALSE;

    /* NDIS4 code always sets these to 0*/
    pHw->Vc.CpcsUu = 0;
    pHw->Vc.Cpi    = 0;

    /* This came straight from the NDIS4 code*/
    pHw->Vc.CellHeader[0] = (UInt8)( ( 0 << 4) | ((pHw->Vc.Vpi & 0xF0) >> 4));
    pHw->Vc.CellHeader[1] = (UInt8)( (pHw->Vc.Vpi << 4) | ((pHw->Vc.Vci & 0xF000) >> 12));
    pHw->Vc.CellHeader[2] = (UInt8)( (pHw->Vc.Vci & 0x0FF0) >> 4);
    pHw->Vc.CellHeader[3] = (UInt8)( (pHw->Vc.Vci & 0x000F) << 4);
    pHw->Vc.CellHeader[4] = 0;

    adi_leaves (DBG_UNI);
    
    return;
}

/***********************************************************************************/
/* UniProcessOutboundPdu							   */
/*										   */
/* Takes an ethernet or raw IP packet and turns it into ATM data. Modeled after    */
/* AtmUniSend in the NDIS code. Where we differ from the NDIS code, comments       */
/* explain why. One major difference is that we will get an entire packet in one   */
/* buffer. The NDIS code gets multiple buffers that compose the entire packet.     */
/* Also, this routine doesn't actually send the data - it just processes the	   */
/* input data into and output buffer.						   */
/*										   */
/* This routine assumes that all parameters have been verified (it doesn't	   */
/* check for NULL pointers or zero lengths).					   */
/*										   */
/* Assumes the following fields in the Vc are initialized:			   */
/*      EncapSize, pEncapHdr							   */
/*      MaxSduSize								   */
/*      CellHeader								   */
/*      CpcsUu, Cpi								   */
/***********************************************************************************/
int UniProcessOutboundPdu(Hardware *pHw, UInt8 *pPacket, UInt32 PacketSize)
{
    SarSegmenter Sar;
    UInt16 CellsInOutput;
    OUTGOING_BUFFER *pBuf;

    adi_enters (DBG_UNI);
    

    /* The NDIS code checks to make sure the VC is ACTIVE. However, since*/
    /* we don't deal with ACTIVE or INACTIVE VC's, we don't need to worry*/
    /* about that check							 */

    /* Make sure that the packet + encapsulation is within the max SDU size*/
    if ((PacketSize + pHw->Vc.EncapSize) > pHw->Vc.MaxSduSize)
    {
        adi_dbg (DBG_UNI,"UniProcessOutboundPdu: Packet+Encap size > MaxSduSize\n");
        
	pHw->Statistics[STAT_PAKTS_LOST_OSIZE]++;
	return -EMSGSIZE;
    }

    /* Prepare our segmentation state machine for segmentation by first grabbing*/
    /* a segmentation buffer, then initializing the state machine	        */
    pBuf = &pHw->SegmentationBuffer;
    if (pBuf == NULL)
    {
        adi_dbg (DBG_UNI,"UniProcessOutboundPdu: NULL buffer from IdleOutQ\n");
        
	return -ENOBUFS;
    }

    Aal5InitSegmenter(&Sar, (PacketSize+pHw->Vc.EncapSize),
                      pBuf->GB.pData, (UInt8 *)(&(pHw->Vc.CellHeader)));

    /* If there is some RFC2684 encapsulation enabled, put that in the output first*/
    if (pHw->Vc.EncapSize)
	Aal5Segment(&Sar, pHw->Vc.pEncapHdr, pHw->Vc.EncapSize);

    /* Now segment the real network packet into ATM cells*/
    Aal5Segment(&Sar, pPacket, PacketSize);

    /* Finish the CPCS PDU processing by writing the trailer and any padding*/
    CellsInOutput = Aal5WritePadAndTrailer(&Sar, pHw->Vc.CpcsUu, pHw->Vc.Cpi);
    pHw->Statistics[STAT_CELLS_TX] += CellsInOutput;
    pHw->Statistics[STAT_PAKTS_TX] ++;
    
    /* Size of output buffer is ATM cell count * size of ATM cell*/
    pBuf->TotalBytes = CellsInOutput * ATM_CELL_SIZE;
    adi_dbg (DBG_UNI,"UniProcessOutboundPdu: packet processed into ATM cells - ready to be sent!\n");
    

    adi_leaves (DBG_UNI);
    
    return 0; /* success in Linux*/
}

/***********************************************************************************/
/* UniLookupVc									   */
/*										   */
/* Finds the Vc that matches the specified ATM header VpiVci value. For now	   */
/* (and maybe forever), we only support on Vc, so this is simply a check to 	   */
/* make sure the VpiVci matches whats stored, then a return of NULL or the	   */
/* CurrentVc pointer.								   */
/***********************************************************************************/
static ATMVC *UniLookupVc(Hardware *pHw, UInt32 VpiVci)
{
    adi_enters (DBG_UNI);
    

    if (pHw->Vc.VpiVci == VpiVci)
    {
        adi_leaves (DBG_UNI);
        
	return &pHw->Vc;
    }
    else
    {
        adi_dbg (DBG_UNI,"UniLookupVc: pCurrentVc->VpiVci (%x) does not match VpiVci (%x)in cell!\n",
                 pHw->Vc.VpiVci, VpiVci);

        
        adi_leaves (DBG_UNI);
        
	return NULL;
    }
}

/***********************************************************************************/
/* UniProcessInboundData							   */
/*										   */
/* Processes the ATM data, reassembling into a CPCS PDU (which is really just	   */
/* an ethernet packet with the CPCS trailer). Once an entire Pdu has been 	   */
/* received, adds the completed ethernet packet buffer to the ReadWaitingQ 	   */
/* so the caller (the upper-level network stack components) can get the packet.    */
/***********************************************************************************/
int UniProcessInboundData(Hardware *pHw, UInt8 *pData, UInt32 DataLength) 
{
    UInt32 CellHeader, VpiVci;
    UInt16 ReassembledPduLength;
    Boolean IsLastCell;
    ATMVC *pVc;

    adi_enters (DBG_UNI);
    adi_dbg (DBG_UNI,"UniProcessInboundData: pData=%p, DataLength=%x\n",
             pData, DataLength);
    

    /* Lets first make sure there is an integral number of ATM cells in the buffer*/
    if (DataLength % ATM_CELL_SIZE)
    {
        adi_err ("UniProcessInboundData: non-integral number of cells in incoming data.\n");
        adi_err ("UniProcessInboundData: incoming data length = %x.\n", DataLength);
        
	return -EMSGSIZE;
    }

    /* Process each ATM cell in the input buffer*/
    while (DataLength)
    {
	/* The first DWORD of the ATM cell contains all but the HEC byte of the*/
	/* cell's ATM header - but we don't look at the HEC byte anyway        */
	CellHeader = be32_to_cpu(*((UInt32 *)pData));
	IsLastCell = (Boolean)(CellHeader & ATM_PT_AAL_INDICATE);
	VpiVci     = (CellHeader >> 4) & ATM_CELL_REMOVE_GFC_MASK;
	pVc        = UniLookupVc( pHw, VpiVci );
	
// FFD102802&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&

/* ZAP(DEBUG_OAM, DEBUG_INFO, (kTAB"******** OAM=%d VCI=%d VPI=%d DataMode=%d VaildVCIVPI=%d EveryTime=%d.\n",\ */
/* 							pHw->pDriverOptions[ConfigDefaultOAM].Value,\ */
/* 							pHw->pDriverOptions[ConfigVCI].Value,\ */
/* 							pHw->pDriverOptions[ConfigVPI].Value,\ */
/* 							DataMode, VaildVCIVPI, EveryTime)); */
    if (CheckForAndProcessOAMCell(pHw, pData) )
    {
        // This cell was an OAM cell, so we don't want to include it in 
        // the reassembly. Therefore, skip the cell and continue on ...
        adi_dbg (DBG_UNI,"*** FFD *** OAM cell received and processed.\n");
        
        DataLength -= ATM_CELL_SIZE;
        pData      += ATM_CELL_SIZE;
        pHw->Statistics[STAT_CELLS_OAM_RCVD]++;
        continue;
    }
// END: FFD102802&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&

	/* If we cannot find an open VC for this cell, drop the cell*/
	if (NULL == pVc)
	{
            adi_dbg (DBG_UNI,"UniProcessInboundData: no VC for cell, VpiVci=%x\n", VpiVci);
            
	    DataLength -= ATM_CELL_SIZE;
	    pData      += ATM_CELL_SIZE;
	    pHw->Statistics[STAT_CELLS_LOST_VPIVCI]++;
	    continue; /* forces code to go back to while (DataLength) line*/
	}

	if (IsPTICell(CellHeader))
	{
            adi_dbg (DBG_UNI,"UniProcessInboundData: PTI cell received\n");
            
	    DataLength -= ATM_CELL_SIZE;
	    pData      += ATM_CELL_SIZE;
	    continue; /* forces code to go back to while (DataLength) line*/
	}

	/* NDIS code now checks to make sure the VC is in the Active state    .*/
	/* Our VC is a PVC that is always active, so we don't do this right now*/
	/* Go ahead and skip over the header for this cell		      */
	DataLength -= ATM_CELL_HEADER_SIZE;
	pData      += ATM_CELL_HEADER_SIZE;
	
	/* If there is not already a reassembly happening on this VC, get the   */
	/* reassembly structure initialized so we can get started with this cell*/
	if (!pVc->ReassemblyInProgress)
	{
	    ETHERNET_PACKET_BUFFER *pReassemblyBuffer;
	    pReassemblyBuffer = &pHw->ReassemblyBuffer;
            
	    /* If no reassembly buffer is available, drop the cell*/
	    if (NULL == pReassemblyBuffer)
	    {
                adi_dbg (DBG_UNI,"UniProcessInboundData: no reassembly buffer for cell,\n");
                
		DataLength -= ATM_CELL_PAYLOAD_SIZE;
		pData      += ATM_CELL_PAYLOAD_SIZE;
		continue; /* forces code to go back to while (DataLength) line*/
	    }
	    
	    Aal5InitReassembler(&(pVc->Reassembler), pReassemblyBuffer);
	    pVc->ReassemblyInProgress = TRUE;
	}

	/********************************************************************/
	/* If there isn't sufficient space in the reassembly buffer to hold */
	/* another cell payload, drop the entire PDU being reassembled.     */
	/* (This can happen if the last cell of a PDU is lost, causing two  */
	/* PDUs to get concatenated)					   */
	/********************************************************************/
	if (pVc->Reassembler.BytesLeftInReassembly < ATM_CELL_PAYLOAD_SIZE)
	{ 
            adi_dbg (DBG_UNI,"UniProcessInboundData: no space left in buffer for cell, dropping PDU,\n");
            
	    DataLength -= ATM_CELL_PAYLOAD_SIZE;
	    pData      += ATM_CELL_PAYLOAD_SIZE;
	    pVc->ReassemblyInProgress = FALSE;
	    pHw->Statistics[STAT_CELLS_LOST_OTHER] += pVc->Reassembler.CellCount;
	    continue; /* forces code to go back to while (DataLength) line*/
	}

	/* Ok, we've eliminated all possible error conditions, we've made sure */
	/* the reassembly buffer is ready to receive this cell, so lets do it! */
	if (!IsLastCell)
	{
	    Aal5ReassembleNonFinalCell(&(pVc->Reassembler), pData);
	}
	else
	{
	    Aal5ReassembleFinalCell(&(pVc->Reassembler), pData);
	}
	
	pVc->Reassembler.CellCount++;
	
	/* Start looking at the beginning of the next cell*/
	DataLength -= ATM_CELL_PAYLOAD_SIZE;
	pData      += ATM_CELL_PAYLOAD_SIZE;

	/* If this cell was the last cell in the PDU, do PDU processing*/
	if (IsLastCell)
	{
            
            
	    /* Calculate the number of bytes we reassembled into this PDU*/
	    ReassembledPduLength = pVc->Reassembler.ReassemblyBufferSize -
		pVc->Reassembler.BytesLeftInReassembly;

            /* We can now check whether or not the message is too big */
            if ( ReassembledPduLength > pHw->mru ) 
            {
                /* Discard message */
                adi_warn("Discarding message (pdu %u > mru %u)\n",
                         ReassembledPduLength,pHw->mru);
                
                pHw->Statistics[STAT_CELLS_LOST_OTHER] += pVc->Reassembler.CellCount;
                pVc->ReassemblyInProgress = FALSE;
                continue;
            }
            

            /**************************************************************************/
	    /* The way the CRC is calculated should result in the final calculated CRC*/
	    /* of the incoming PDU being a known value (because the CRC stored in the */
	    /* trailer of the PDU is used in the calculation), so we can verify the   */
	    /* CRC by just comparing the end result with the fixed value (CRC_RESIDUE)*/
	    /**************************************************************************/
	    
	    if (pVc->Reassembler.RunningCRC != CRC_RESIDUE)
	    {
                adi_dbg (DBG_UNI,"UniProcessInboundData: CRC error, dropping PDU,\n");
                
		pVc->ReassemblyInProgress = FALSE;
		pHw->Statistics[STAT_CELLS_LOST_CRC]++;
	    }
	    else
	    {
		/* CRC is ok, lets check the packet length */
		if (ReassembledPduLength != pVc->Reassembler.PduLengthFromTrailer)
		{
                    adi_dbg (DBG_UNI,"UniProcessInboundData: PDU length error, "
                             "dropping PDU, Calculated = %x, stored = %x\n",
                             ReassembledPduLength, pVc->Reassembler.PduLengthFromTrailer);
                    
                        
		    pVc->ReassemblyInProgress = FALSE;
		    pHw->Statistics[STAT_PAKTS_LOST_OSIZE]++;
		}

		/* If we get here and ReassemblyInProgress is still true, then we can*/
		/* go ahead and mark this buffer as being ready to receive.          */
		if (pVc->ReassemblyInProgress)
		{
		    /***********************************************************/
		    /* There might be some MPOA (RFC1483/2684) processing that */
		    /* needs to be done, but we can wait until the higher level*/
		    /* code actually asks us for this packet. Reset skip size  */
		    /* for now to make sure its at a known value	       */
		    /***********************************************************/
		    
		    pHw->Statistics[STAT_CELLS_RX] += pVc->Reassembler.CellCount;
                    pHw->Statistics[STAT_PAKTS_RX] ++;
		    pVc->Reassembler.pQueueEntry->PacketSize     = ReassembledPduLength;
		    pVc->Reassembler.pQueueEntry->EncapSkipSize  = 0;
		    pVc->Reassembler.pQueueEntry->DropThisPacket = FALSE;
                    adi_dbg (DBG_UNI,"UniProcessInboundData: Pdu processed successfully,"
                             " calling ReadSendItUp,\n");
                    
                    

		    pVc->ReassemblyInProgress = FALSE;
		    ReadSendItUp(pHw, pVc->Reassembler.pQueueEntry);
		}
	    }
	}
    }

    adi_leaves (DBG_UNI);
    
    return 0; /* linux success*/
}

/******************************************************************************************************
$Log: Uni.c,v $
Revision 1.7  2003/06/03 22:51:55  sleeper
Changed ZAPS to adi_dbg/err macros

Revision 1.6  2003/04/08 23:10:53  sleeper
Add mru

Revision 1.5  2003/04/23 22:28:20  sleeper
Add stats

Revision 1.4  2003/03/31 22:03:37  sleeper
Fix a comment

Revision 1.3  2003/03/21 00:22:41  sleeper
Msg Initialization modifs from 2.0.1

Revision 1.2  2003/03/11 00:38:03  sleeper
Add OAM modifs as present in Sagemn 2.0.1 driver

Revision 1.1.1.1  2003/02/10 23:29:49  sleeper
Imported sources

Revision 1.60  2002/05/27 18:04:42  Anoosh Naderi
Clean up the code

Revision 1.5  2002/01/18 18:04:42  chris.edgington
Pull VPI and VCI from driver options if they exist.

Revision 1.4  2002/01/14 21:59:34  chris.edgington
Added GPL header.

Revision 1.3  2002/01/08 16:02:43  chris.edgington
Added some necessary comments.

Revision 1.2  2002/01/04 20:59:02  chris.edgington
Changed data processing routines to use static buffers from Hardware struct
instead of pulling buffers from queues (since we're not using queues on Linux).
Added call to ReadSendItUp when read to send data up the network stack.

Revision 1.1  2002/01/02 21:56:31  chris.edgington
First version - from MacOS9 project (with Linux mods to get to compile).


---------------------------------------------------------------------------
 Log from MacOS9 project
---------------------------------------------------------------------------

Revision 1.23 2002/07/24 11:32:39  Chinda Keodouangsy
Added support to handle PTI/OAM cells

Revision 1.22  2001/12/05 22:09:55  chris.edgington
Added Analog Devices copyright notice.

Revision 1.21  2001/11/12 14:23:25  chris.edgington
Added some comments and did some simple reformatting in preparation for code review.

Revision 1.20  2001/10/30 19:39:24  chris.edgington
Added error messages to USBExpertStatusLevel in appropriate places.

Revision 1.19  2001/10/29 23:53:14  martyn.deobald
Modified UniEstablishVC to use values read from config file

Revision 1.18  2001/10/26 18:22:45  chris.edgington
Changed CRC loss counter to packets instead of cells.

Revision 1.17  2001/10/25 19:09:33  chris.edgington
Added incs of packet and cell counters.

Revision 1.16  2001/10/17 21:08:46  chris.edgington
Removed ZAP that was reporting incorrect message.

Revision 1.15  2001/10/12 22:35:13  chris.edgington
Re-enabled read CRC checking.

Revision 1.14  2001/10/11 19:49:44  chris.edgington
Removed a DANGER that is no longer necessary.

Revision 1.13  2001/10/09 15:17:25  chris.edgington
Removed UniCreate, UniCreateQueues, UniDestroyQueues, UniEmptyQueues - no longer needed.
Updated to reflect new buffer field names.
Uni code now pulls buffers from buffer queues in Hardware.

Revision 1.12  2001/10/05 16:09:37  chris.edgington
Temporarily disabled CRC verification of incoming packets.
Changed nils to NULLs.
Added more debug output.
Removed some endian translations - we're already big-endian and that's what the network expects.

Revision 1.11  2001/10/04 15:54:21  chris.edgington
Added call to notify TCPIP stack that there is network data available. 
(Currently HACKed - find better fix when network traffic is working.)

Revision 1.10  2001/10/01 18:41:30  chris.edgington
Save the size of the reassembled PDU in the PDU holder buffer's RealDataSize field.

Revision 1.9  2001/09/28 15:22:07  chris.edgington
Since ATMUNI Uni is part of Hardware, we don't have to allocate space for it in UniCreate.
We should probably change the name of the function to UniInitialize.

Revision 1.8  2001/09/24 18:05:51  chris.edgington
Removed include of multiple files, added single include of Adiutil.h.

Revision 1.7  2001/09/11 15:35:53  chris.edgington
Added separate free queues for send and receive.
Added missing initialization to UniCreate and UniEstablishVc.

Revision 1.6  2001/09/10 18:16:00  chris.edgington
Moved header stuff into header files.
Fixed a few problems found in compile.

Revision 1.5  2001/09/07 22:25:36  chris.edgington
Wrote queue GET and PUT macros to make the code more portable.
Fixed a few problems found during code reading (typos, etc.).

Revision 1.4  2001/09/07 19:39:52  chris.edgington
Wrote UniCreate, UniCreateQueues, UniEmptyQueue, UniDestroyQueues.
Finished to 95% UniProcessInboundData.
Added ATMUNI and ATMUSB_BUFFER structs.
Not compilable - work in progress.

Revision 1.3  2001/09/06 22:30:42  chris.edgington
Changed UniProcessPdu to UniProcessOutboundPdu.
Started UniProcessInboundData.

Revision 1.2  2001/09/06 21:25:09  chris.edgington
Wrote UniProcessPdu.

Revision 1.1  2001/09/05 22:26:21  chris.edgington
Initial version.
*************************************************************************************************/
