/* easytag.c - 2000/04/28 */
/*
 *  EasyTAG - Tag editor for MP3 and Ogg Vorbis files
 *  Copyright (C) 2000-2003  Jerome Couderc <j.couderc@ifrance.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


#include <config.h> // For definition of ENABLE_OGG
#include <gtk/gtk.h>
#include <stdio.h>
#include <string.h>
#include <gdk/gdkkeysyms.h>
#include <stdlib.h>
#include <sys/stat.h>
#include <unistd.h>
#include <errno.h>
#include <signal.h>
#include <id3.h>

#include "easytag.h"
#include "browser.h"
#include "misc.h"
#include "bar.h"
#include "prefs.h"
#include "setting.h"
#include "scan.h"
#include "mpeg_header.h"
#include "id3_tag.h"
#include "ogg_tag.h"
#include "msgbox.h"
#include "et_core.h"
#include "cddb.h"
#include "i18n.h"

#include "../pixmaps/EasyTAG.xpm"


/****************
 * Declarations *
 ****************/
guint idle_handler_id;
guint progressbar_index;  /* An integer to set the value of progress bar into the recurse fonction */

GtkWidget *QuitRecursionWindow = NULL;

/* Used to force to hide the msgbox when saving tag */
gboolean SF_HideMsgbox_Write_Tag;
/* To remember which button was pressed when saving tag */
gint     SF_ButtonPressed_Write_Tag;
/* Used to force to hide the msgbox when renaming file */
gboolean SF_HideMsgbox_Rename_File;
/* To remember which button was pressed when renaming file */
gint     SF_ButtonPressed_Rename_File;
/* Used to force to hide the msgbox when deleting file */
gboolean SF_HideMsgbox_Delete_File;
/* To remember which button was pressed when deleting file */
gint     SF_ButtonPressed_Delete_File;


/**************
 * Prototypes *
 **************/
void     Handle_Crash     (gint signal_id);
gchar   *signal_to_string (gint signal);

GtkWidget *Create_Browser_Area (void);
GtkWidget *Create_File_Area    (void);
GtkWidget *Create_Tag_Area     (void);

void Menu_Mini_Button_Clicked (GtkEntry *entry);
void Mini_Button_Clicked (GtkObject *object);
void Insert_Only_Digit   (GtkEditable *editable, const gchar *text, gint length,gint *position,gpointer data);
void Disable_Command_Buttons (void);
void Clear_Tag_Entry_Fields  (void);
void Clear_File_Entry_Field  (void);
void Clear_Header_Fields     (void);

void Write_File_Tag   (ET_File *ETFile);
void Rename_File      (ET_File *ETFile);
gint Save_File        (ET_File *ETFile, gboolean multiple_files);
gint Delete_File      (ET_File *ETFile, gboolean multiple_files);
gint Save_Selected_Files_With_Answer   (void);
gint Save_All_Files_With_Answer        (void);
gint Save_List_Of_Files                (GList *etfilelist);
gint Delete_Selected_Files_With_Answer (void);

void Display_Usage (void);

void Tag_Area_Entry_Key_Press (GtkWidget *widget, GdkEvent *event);

void Init_Load_Default_Dir (void);
void EasyTAG_Exit (void);
void Quit_MainWindow_Ok_Button (void);

GList *Read_Directory_Recursively (GList *file_list, gchar *path, gint recurse);
void Open_Quit_Recursion_Function_Window    (void);
void Destroy_Quit_Recursion_Function_Window (void);
void Quit_Recursion_Function_Button_Pressed (void);
void Quit_Recursion_Window_Key_Press (GtkWidget *window, GdkEvent *event);



/********
 * Main *
 ********/
int main (int argc, char *argv[])
{
    GtkWidget *MainVBox;
    GtkWidget *HBox, *VBox;
    GdkPixmap *pixmap;
    GdkBitmap *mask;
    struct stat statbuf;

    /* Signal handling to display a message(SIGSEGV, ...) */
    signal(SIGBUS,Handle_Crash);
    signal(SIGFPE,Handle_Crash);
    signal(SIGSEGV,Handle_Crash);
    // Must handle this signal to avoid zombie of applications executed (ex: xmms)
    signal(SIGCHLD,SIG_IGN); // Fix me! : can't run nautilus 1.0.6 with "Browse Directory With"

#ifdef ENABLE_NLS
    bindtextdomain("easytag",LOCALE);
    //bind_textdomain_codeset(PACKAGE,"UTF-8"); // Needed for gtk-2.0 ?
    textdomain("easytag");
    /* Initialize i18n support */
    gtk_set_locale();
#endif

    /* Initialize GTK */
    gtk_init( &argc, &argv );

    /* Get home variable */
    HOME_VARIABLE = (gchar *)g_getenv("HOME");
    INIT_DIRECTORY = NULL;

    /* Check given arguments */
    if (argc>1)
    {
        if ( (strcmp(argv[1],"--version")==0) || (strcmp(argv[1],"-v")==0) ) // Query version
        {
            printf(_("%s %s by %s (compiled %s, %s)\n"),APPNAME,VERSION,AUTHOR,__TIME__,__DATE__);
            printf(_("E-mail: %s"),EMAIL"\n");
            printf(_("Web Page: %s"),WEBPAGE"\n");
            exit (0);
        }else if ( (strcmp(argv[1],"--help")==0) || (strcmp(argv[1],"-h")==0) ) // Query help
        {
            Display_Usage();
        }else if ( (strcmp(argv[1],".")==0) || (strcmp(argv[1],"./")==0) ) // Passed the current directory
        {
            INIT_DIRECTORY = g_get_current_dir();
        }else if (stat(argv[1],&statbuf)==0 && S_ISDIR(statbuf.st_mode)) // Passed an absolute path
        {
            INIT_DIRECTORY = g_strdup(argv[1]);
        }else if (stat(argv[1],&statbuf)==0 && S_ISREG(statbuf.st_mode)) // Passed an absolute path with filename
        {
            // When passing a file, we load only the directory
            INIT_DIRECTORY = g_dirname(argv[1]);
        }else
        {
            printf("Unknown parameter '%s'\n",argv[1]);
            Display_Usage();
        }
    }

    /* Starting message */
    g_print(_("Starting EasyTAG %s ...\n"),VERSION);
    //g_print(_("PID: #%d\n"),getpid());
    g_print(_("Currently using id3lib version %d.%d.%d ...\n"),ID3LIB_MAJOR_VERSION,
                                                               ID3LIB_MINOR_VERSION,
                                                               ID3LIB_PATCH_VERSION);

    /* Create all config files */
    Setting_Create_Files();
    /* Load Config */
    Init_Config_Variables();
    Read_Config();
    /* Display_Config(); // <- for debugging */

    /* Initialization */
    ET_Core_Create();
    Main_Stop_Button_Pressed = 0;
    Init_Mouse_Cursor();
    Init_OptionsWindow();
    Init_ScannerWindow();
    Init_CddbWindow();

    /* The main window */
    MainWindow = gtk_window_new (GTK_WINDOW_TOPLEVEL);
    gtk_window_set_title(GTK_WINDOW (MainWindow),APPNAME" "VERSION);
    gtk_window_set_policy(GTK_WINDOW(MainWindow),FALSE,TRUE,FALSE);
    if (SET_MAIN_WINDOW_SIZE)
        gtk_window_set_default_size(GTK_WINDOW(MainWindow),MAIN_WINDOW_WIDTH,MAIN_WINDOW_HEIGHT);

    gtk_signal_connect(GTK_OBJECT(MainWindow),"delete_event",GTK_SIGNAL_FUNC(Quit_MainWindow),NULL);
    gtk_signal_connect(GTK_OBJECT(MainWindow),"destroy",GTK_SIGNAL_FUNC(Quit_MainWindow),NULL);

    /* Minimised window icon  */
    gtk_widget_realize(MainWindow);
    pixmap = gdk_pixmap_create_from_xpm_d(MainWindow->window,&mask,NULL,EasyTAG_xpm);
    gdk_window_set_icon(MainWindow->window,(GdkWindow *)NULL,pixmap,mask);

    MainVBox = gtk_vbox_new(FALSE,0);
    gtk_container_add (GTK_CONTAINER(MainWindow),MainVBox);
    gtk_widget_show(MainVBox);


    /* Menu Bar */
    MenuArea = Create_Menu_Bar();
    gtk_box_pack_start(GTK_BOX(MainVBox),MenuArea,FALSE,FALSE,0);


    /* Tool Bar */
    ToolArea = Create_Tool_Bar();
    gtk_box_pack_start(GTK_BOX(MainVBox),ToolArea,FALSE,FALSE,0);


    /* The two panes: BrowserArea on the left, FileArea+TagArea on the right */
    MainWindowHPaned = gtk_hpaned_new();
    gtk_box_pack_start(GTK_BOX(MainVBox),MainWindowHPaned,TRUE,TRUE,0);
    gtk_paned_set_handle_size(GTK_PANED(MainWindowHPaned),8);
    gtk_paned_set_gutter_size(GTK_PANED(MainWindowHPaned),6);                       
    if (SET_PANE_HANDLE_POSITION1)
        gtk_paned_set_position(GTK_PANED(MainWindowHPaned),PANE_HANDLE_POSITION1);
    gtk_widget_show(MainWindowHPaned);

    /* Browser */
    BrowseArea = Create_Browser_Area();
    gtk_paned_pack1(GTK_PANED(MainWindowHPaned),BrowseArea,TRUE,TRUE);

    /* Vertical box for FileArea + TagArea */
    VBox = gtk_vbox_new(FALSE,0);
    gtk_paned_pack2(GTK_PANED(MainWindowHPaned),VBox,FALSE,FALSE);
    gtk_widget_show(VBox);

    /* File */
    FileArea = Create_File_Area();
    gtk_box_pack_start(GTK_BOX(VBox),FileArea,FALSE,FALSE,0);

    /* Tag */
    TagArea = Create_Tag_Area();
    gtk_box_pack_start(GTK_BOX(VBox),TagArea,FALSE,FALSE,0);

    /* Horizontal box for Status bar + Progress bar */
    HBox = gtk_hbox_new(FALSE,0);
    gtk_box_pack_start(GTK_BOX(MainVBox),HBox,FALSE,FALSE,0);
    gtk_widget_show(HBox);

    /* Status bar */
    StatusArea = Create_Status_Bar();
    gtk_box_pack_start(GTK_BOX(HBox),StatusArea,TRUE,TRUE,0);

    /* Progress bar */
    ProgressArea = Create_Progress_Bar();
    gtk_box_pack_end(GTK_BOX(HBox),ProgressArea,FALSE,FALSE,0);


    gtk_widget_show(MainWindow);

    /* Load the default dir when the UI is created and displayed
     * to the screen and open also the scanner window */
    idle_handler_id = gtk_idle_add((GtkFunction)Init_Load_Default_Dir,NULL);

    /* Enter the event loop */
    gtk_main ();         
    return 0;
}


GtkWidget *Create_Browser_Area (void)
{
    GtkWidget *Frame;
    GtkWidget *Tree;

    Frame = gtk_frame_new(_("Browser"));
    gtk_container_border_width(GTK_CONTAINER(Frame),2);

    Tree = Create_Browser_Items(MainWindow);
    gtk_container_add(GTK_CONTAINER(Frame),Tree);

    /* Don't load init dir here because Tag area hasn't been yet created!.
     * It will be load at the end of the main function */
    //Browser_Tree_Select_Dir(DEFAULT_PATH_TO_MP3);

    gtk_widget_show(Frame);
    return Frame;
}


#include "../pixmaps/read_only.xpm"
#include "../pixmaps/broken.xpm"
GtkWidget *Create_File_Area (void)
{
    GtkWidget *VBox, *HBox;
    GtkWidget *Separator;
    GtkTooltips *Tips;


    FileFrame = gtk_frame_new(_("File"));
    gtk_container_border_width(GTK_CONTAINER(FileFrame),2);

    VBox = gtk_vbox_new(FALSE,0);
    gtk_container_add(GTK_CONTAINER(FileFrame),VBox);
    gtk_container_border_width(GTK_CONTAINER(VBox),4);

    /* Tips */
    Tips = gtk_tooltips_new_1();

    /* HBox for FileEntry and IconBox */
    HBox = gtk_hbox_new(FALSE,2);
    gtk_box_pack_start(GTK_BOX(VBox),HBox,TRUE,TRUE,0);

    /* File index (position in list + list length) */
    FileIndex = gtk_label_new("0/0:");
    gtk_box_pack_start(GTK_BOX(HBox),FileIndex,FALSE,FALSE,0);

    /* File name */
    FileEntry = gtk_entry_new();
    gtk_box_pack_start(GTK_BOX(HBox),FileEntry,TRUE,TRUE,2);
    
    /* Access status icon */
    ReadOnlyStatusIconBox = Create_Pixmap_Icon_With_Event_Box(read_only_xpm);
    gtk_box_pack_start(GTK_BOX(HBox),ReadOnlyStatusIconBox,FALSE,FALSE,0);
    gtk_tooltips_set_tip(Tips,ReadOnlyStatusIconBox,_("Read Only File"),NULL);
    BrokenStatusIconBox = Create_Pixmap_Icon_With_Event_Box(broken_xpm);
    gtk_box_pack_start(GTK_BOX(HBox),BrokenStatusIconBox,FALSE,FALSE,0);
    gtk_tooltips_set_tip(Tips,BrokenStatusIconBox,_("File Link Broken"),NULL);

    Attach_Popup_Menu_To_Tag_Entries(GTK_ENTRY(FileEntry));
    gtk_signal_connect(GTK_OBJECT(FileEntry),"key_press_event",(GtkSignalFunc)Tag_Area_Entry_Key_Press,NULL);


    /*
     *  File Infos
     */
    HeaderInfosTable = gtk_table_new(3,5,FALSE);
    gtk_container_add(GTK_CONTAINER(VBox),HeaderInfosTable);
    gtk_container_border_width(GTK_CONTAINER(HeaderInfosTable),2);
    gtk_table_set_row_spacings(GTK_TABLE(HeaderInfosTable),1);
    gtk_table_set_col_spacings(GTK_TABLE(HeaderInfosTable),2);

    VersionLabel = gtk_label_new(_("MPEG"));
    gtk_table_attach_defaults(GTK_TABLE(HeaderInfosTable),VersionLabel,0,1,0,1);
    VersionValueLabel = gtk_label_new(_("?, Layer ?"));
    gtk_table_attach_defaults(GTK_TABLE(HeaderInfosTable),VersionValueLabel,1,2,0,1);
    gtk_misc_set_alignment(GTK_MISC(VersionLabel),1,0.5);
    gtk_misc_set_alignment(GTK_MISC(VersionValueLabel),0,0.5);

    BitrateLabel = gtk_label_new(_("Bitrate:"));
    gtk_table_attach_defaults(GTK_TABLE(HeaderInfosTable),BitrateLabel,0,1,1,2);
    BitrateValueLabel = gtk_label_new(_("? kb/s"));
    gtk_table_attach_defaults(GTK_TABLE(HeaderInfosTable),BitrateValueLabel,1,2,1,2);
    gtk_misc_set_alignment(GTK_MISC(BitrateLabel),1,0.5);
    gtk_misc_set_alignment(GTK_MISC(BitrateValueLabel),0,0.5);

    SampleRateLabel = gtk_label_new(_("Freq:"));
    gtk_table_attach_defaults(GTK_TABLE(HeaderInfosTable),SampleRateLabel,0,1,2,3);
    SampleRateValueLabel = gtk_label_new(_("? Hz"));
    gtk_table_attach_defaults(GTK_TABLE(HeaderInfosTable),SampleRateValueLabel,1,2,2,3);
    gtk_misc_set_alignment(GTK_MISC(SampleRateLabel),1,0.5);
    gtk_misc_set_alignment(GTK_MISC(SampleRateValueLabel),0,0.5);

    Separator = gtk_vseparator_new();
    gtk_table_attach(GTK_TABLE(HeaderInfosTable),Separator,2,3,0,4,GTK_FILL,GTK_FILL,0,0);

    ModeLabel = gtk_label_new(_("Mode:"));
    gtk_table_attach_defaults(GTK_TABLE(HeaderInfosTable),ModeLabel,3,4,0,1);
    ModeValueLabel = gtk_label_new(_("?"));
    gtk_table_attach_defaults(GTK_TABLE(HeaderInfosTable),ModeValueLabel,4,5,0,1);
    gtk_misc_set_alignment(GTK_MISC(ModeLabel),1,0.5);
    gtk_misc_set_alignment(GTK_MISC(ModeValueLabel),0,0.5);

    SizeLabel = gtk_label_new(_("Size:"));
    gtk_table_attach_defaults(GTK_TABLE(HeaderInfosTable),SizeLabel,3,4,1,2);
    SizeValueLabel = gtk_label_new(_("? kb"));
    gtk_table_attach_defaults(GTK_TABLE(HeaderInfosTable),SizeValueLabel,4,5,1,2);
    gtk_misc_set_alignment(GTK_MISC(SizeLabel),1,0.5);
    gtk_misc_set_alignment(GTK_MISC(SizeValueLabel),0,0.5);

    DurationLabel = gtk_label_new(_("Time:"));
    gtk_table_attach_defaults(GTK_TABLE(HeaderInfosTable),DurationLabel,3,4,2,3);
    DurationValueLabel = gtk_label_new(_("?"));
    gtk_table_attach_defaults(GTK_TABLE(HeaderInfosTable),DurationValueLabel,4,5,2,3);
    gtk_misc_set_alignment(GTK_MISC(DurationLabel),1,0.5);
    gtk_misc_set_alignment(GTK_MISC(DurationValueLabel),0,0.5);

    gtk_widget_show(FileFrame);
    gtk_widget_show(VBox);
    gtk_widget_show(HBox);
    gtk_widget_show(FileIndex);
    gtk_widget_show(FileEntry);
    if (SHOW_HEADER_INFO)
        gtk_widget_show_all(HeaderInfosTable);
    return FileFrame;
}


#include "../pixmaps/sequence_track.xpm"
GtkWidget *Create_Tag_Area (void)
{
    GtkWidget *Separator;
    GtkWidget *Table;
    GtkWidget *Label;
    GtkWidget *Icon;
    GtkTooltips *Tips;
    gint MButtonSize = 11;

    TagFrame = gtk_frame_new(_("Tag"));
    gtk_container_border_width(GTK_CONTAINER(TagFrame),2);

    Table = gtk_table_new(6,11,FALSE);
    gtk_container_add(GTK_CONTAINER(TagFrame),Table);
    gtk_container_border_width(GTK_CONTAINER(Table),5);
    gtk_table_set_row_spacings(GTK_TABLE(Table),5);
    gtk_table_set_col_spacings(GTK_TABLE(Table),5);

    /* Tips */
    Tips = gtk_tooltips_new_1();

    /* Title */
    Label = gtk_label_new(_("Title:"));
    gtk_table_attach(GTK_TABLE(Table),Label,0,1,0,1,GTK_FILL,GTK_FILL,0,0);
    gtk_misc_set_alignment(GTK_MISC(Label),1,0.5);

    TitleEntry = gtk_entry_new();
    gtk_table_attach_defaults(GTK_TABLE(Table),TitleEntry,1,10,0,1);
    gtk_widget_set_usize(TitleEntry,150,-1);

    TitleMButton = gtk_button_new();
    gtk_widget_set_usize(TitleMButton,MButtonSize,MButtonSize);
    gtk_table_attach(GTK_TABLE(Table),TitleMButton,10,11,0,1,0,0,0,0);
    gtk_signal_connect(GTK_OBJECT(TitleMButton),"clicked",GTK_SIGNAL_FUNC(Mini_Button_Clicked),NULL);
    gtk_tooltips_set_tip(Tips,TitleMButton,_("Tag selected files with this title"),NULL);

    Attach_Popup_Menu_To_Tag_Entries(GTK_ENTRY(TitleEntry));
    gtk_object_set_data(GTK_OBJECT(TitleEntry),"MButtonName",TitleMButton);
    gtk_signal_connect(GTK_OBJECT(TitleEntry),"key_press_event",(GtkSignalFunc)Tag_Area_Entry_Key_Press,NULL);



    /* Artist */
    Label = gtk_label_new(_("Artist:"));
    gtk_table_attach(GTK_TABLE(Table),Label,0,1,1,2,GTK_FILL,GTK_FILL,0,0);
    gtk_misc_set_alignment(GTK_MISC(Label),1,0.5);

    ArtistEntry = gtk_entry_new();
    gtk_table_attach_defaults(GTK_TABLE(Table),ArtistEntry,1,10,1,2);

    ArtistMButton = gtk_button_new();
    gtk_widget_set_usize(ArtistMButton,MButtonSize,MButtonSize);
    gtk_table_attach(GTK_TABLE(Table),ArtistMButton,10,11,1,2,0,0,0,0);
    gtk_signal_connect(GTK_OBJECT(ArtistMButton),"clicked",GTK_SIGNAL_FUNC(Mini_Button_Clicked),NULL);
    gtk_tooltips_set_tip(Tips,ArtistMButton,_("Tag selected files with this artist"),NULL);

    Attach_Popup_Menu_To_Tag_Entries(GTK_ENTRY(ArtistEntry));
    gtk_object_set_data(GTK_OBJECT(ArtistEntry),"MButtonName",ArtistMButton);
    gtk_signal_connect(GTK_OBJECT(ArtistEntry),"key_press_event",(GtkSignalFunc)Tag_Area_Entry_Key_Press,NULL);


    /* Album */
    Label = gtk_label_new(_("Album:"));
    gtk_table_attach(GTK_TABLE(Table),Label,0,1,2,3,GTK_FILL,GTK_FILL,0,0);
    gtk_misc_set_alignment(GTK_MISC(Label),1,0.5);

    AlbumEntry = gtk_entry_new();
    gtk_table_attach_defaults(GTK_TABLE(Table),AlbumEntry,1,10,2,3);

    AlbumMButton = gtk_button_new();
    gtk_widget_set_usize(AlbumMButton,MButtonSize,MButtonSize);
    gtk_table_attach(GTK_TABLE(Table),AlbumMButton,10,11,2,3,0,0,0,0);
    gtk_signal_connect(GTK_OBJECT(AlbumMButton),"clicked",GTK_SIGNAL_FUNC(Mini_Button_Clicked),NULL);
    gtk_tooltips_set_tip(Tips,AlbumMButton,_("Tag selected files with this album name"),NULL);

    Attach_Popup_Menu_To_Tag_Entries(GTK_ENTRY(AlbumEntry));
    gtk_object_set_data(GTK_OBJECT(AlbumEntry),"MButtonName",AlbumMButton);
    gtk_signal_connect(GTK_OBJECT(AlbumEntry),"key_press_event",(GtkSignalFunc)Tag_Area_Entry_Key_Press,NULL);


    /* Year */
    Label = gtk_label_new(_("Year:"));
    gtk_table_attach(GTK_TABLE(Table),Label,0,1,3,4,GTK_FILL,GTK_FILL,0,0);
    gtk_misc_set_alignment(GTK_MISC(Label),1,0.5);

    YearEntry = gtk_entry_new_with_max_length(4);
    gtk_table_attach_defaults(GTK_TABLE(Table),YearEntry,1,2,3,4);
    gtk_widget_set_usize(YearEntry,37,-1);
    gtk_signal_connect(GTK_OBJECT(YearEntry),"insert_text",GTK_SIGNAL_FUNC(Insert_Only_Digit),NULL);
    gtk_signal_connect(GTK_OBJECT(YearEntry),"activate",GTK_SIGNAL_FUNC(Parse_Date),NULL);
    gtk_signal_connect(GTK_OBJECT(YearEntry),"focus-out-event",GTK_SIGNAL_FUNC(Parse_Date),NULL);

    YearMButton = gtk_button_new();
    gtk_widget_set_usize(YearMButton,MButtonSize,MButtonSize);
    gtk_table_attach(GTK_TABLE(Table),YearMButton,2,3,3,4,0,0,0,0);
    gtk_signal_connect(GTK_OBJECT(YearMButton),"clicked",GTK_SIGNAL_FUNC(Mini_Button_Clicked),NULL);
    gtk_tooltips_set_tip(Tips,YearMButton,_("Tag selected files with this year"),NULL);

    gtk_object_set_data(GTK_OBJECT(YearEntry),"MButtonName",YearMButton);
    gtk_signal_connect(GTK_OBJECT(YearEntry),"key_press_event",(GtkSignalFunc)Tag_Area_Entry_Key_Press,NULL);


    /* Small vertical separator */
    Separator = gtk_vseparator_new();
    gtk_table_attach(GTK_TABLE(Table),Separator,3,4,3,4,GTK_FILL,GTK_FILL,2,2);


    /* Track */
    TrackMButtonSequence = gtk_button_new();
    gtk_widget_set_usize(TrackMButtonSequence,MButtonSize,MButtonSize);
    gtk_table_attach(GTK_TABLE(Table),TrackMButtonSequence,4,5,3,4,0,0,0,0);
    gtk_signal_connect(GTK_OBJECT(TrackMButtonSequence),"clicked",GTK_SIGNAL_FUNC(Mini_Button_Clicked),NULL);
    gtk_tooltips_set_tip(Tips,TrackMButtonSequence,_("Number selected tracks sequentially. "
                                                     "Starts at 01 in each subdirectory."), NULL);
    // Pixmap into TrackMButtonSequence button
    Icon = Create_Pixmap_Icon(sequence_track_xpm);
    gtk_container_add(GTK_CONTAINER(TrackMButtonSequence),Icon);
    gtk_misc_set_alignment(GTK_MISC(Icon),1,1); /* FIX ME : strange: values should be 0,0 !*/

    Label = gtk_label_new(_("Track #:"));
    gtk_table_attach(GTK_TABLE(Table),Label,5,6,3,4,GTK_FILL,GTK_FILL,0,0);
    gtk_misc_set_alignment(GTK_MISC(Label),1,0.5);

    TrackEntry = gtk_combo_new();
    gtk_table_attach_defaults(GTK_TABLE(Table),TrackEntry,6,7,3,4);
    gtk_widget_set_usize(TrackEntry,47,-1);
    gtk_combo_disable_activate(GTK_COMBO(TrackEntry));
    gtk_signal_connect(GTK_OBJECT(GTK_ENTRY(GTK_COMBO(TrackEntry)->entry)),"insert_text",
                GTK_SIGNAL_FUNC(Insert_Only_Digit),NULL);

    Label = gtk_label_new("/");
    gtk_table_attach(GTK_TABLE(Table),Label,7,8,3,4,GTK_FILL,GTK_FILL,0,0);
    gtk_misc_set_alignment(GTK_MISC(Label),1,0.5);

    TrackMButtonNbrFiles = gtk_button_new();
    gtk_widget_set_usize(TrackMButtonNbrFiles,MButtonSize,MButtonSize);
    gtk_table_attach(GTK_TABLE(Table),TrackMButtonNbrFiles,8,9,3,4,0,0,0,0);
    gtk_signal_connect(GTK_OBJECT(TrackMButtonNbrFiles),"clicked",GTK_SIGNAL_FUNC(Mini_Button_Clicked),NULL);
    gtk_tooltips_set_tip(Tips,TrackMButtonNbrFiles,_("Set the number of files, in the same directory of the displayed file, to the selected tracks."), NULL);
    // Pixmap into TrackMButtonNbrFiles button
    Icon = Create_Pixmap_Icon(sequence_track_xpm);
    gtk_container_add(GTK_CONTAINER(TrackMButtonNbrFiles),Icon);
    gtk_misc_set_alignment(GTK_MISC(Icon),1,1); /* FIX ME : strange: values should be 0,0 !*/

    TrackTotalEntry = gtk_entry_new();
    gtk_table_attach_defaults(GTK_TABLE(Table),TrackTotalEntry,9,10,3,4);
    gtk_widget_set_usize(TrackTotalEntry,30,-1);
    gtk_signal_connect(GTK_OBJECT(GTK_ENTRY(TrackTotalEntry)),"insert_text",
        GTK_SIGNAL_FUNC(Insert_Only_Digit),NULL);

    TrackMButton = gtk_button_new();
    gtk_widget_set_usize(TrackMButton,MButtonSize,MButtonSize);
    gtk_table_attach(GTK_TABLE(Table),TrackMButton,10,11,3,4,0,0,0,0);
    gtk_signal_connect(GTK_OBJECT(TrackMButton),"clicked",GTK_SIGNAL_FUNC(Mini_Button_Clicked),NULL);
    gtk_tooltips_set_tip(Tips,TrackMButton,_("Tag selected files with this number of tracks"),NULL);

    gtk_object_set_data(GTK_OBJECT(GTK_COMBO(TrackEntry)->entry),"MButtonName",TrackMButton);
    gtk_object_set_data(GTK_OBJECT(TrackTotalEntry),"MButtonName",TrackMButton);
    gtk_signal_connect(GTK_OBJECT(GTK_COMBO(TrackEntry)->entry),"key_press_event",(GtkSignalFunc)Tag_Area_Entry_Key_Press,NULL);
    gtk_signal_connect(GTK_OBJECT(TrackTotalEntry),"key_press_event",(GtkSignalFunc)Tag_Area_Entry_Key_Press,NULL);


    /* Genre */
    Label = gtk_label_new(_("Genre:"));
    gtk_table_attach(GTK_TABLE(Table),Label,0,1,4,5,GTK_FILL,GTK_FILL,0,0);
    gtk_misc_set_alignment(GTK_MISC(Label),1,0.5);

    GenreEntry = gtk_combo_new();
    gtk_table_attach_defaults(GTK_TABLE(Table),GenreEntry,1,10,4,5);
    gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(GenreEntry)->entry),TRUE);
    gtk_combo_set_use_arrows_always(GTK_COMBO(GenreEntry),TRUE);
    gtk_combo_set_case_sensitive(GTK_COMBO(GenreEntry),FALSE);
    gtk_combo_disable_activate(GTK_COMBO(GenreEntry)); // Avoid to open the menu when pressing the Space bar or Enter
    Load_Genres_List_To_UI();
    // Help the user to fill the genre. Important: '...after' to get the last character entered
    gtk_signal_connect_after(GTK_OBJECT(GTK_ENTRY(GTK_COMBO(GenreEntry)->entry)),"key_press_event",
                (GtkSignalFunc)Parse_Genre,GTK_COMBO(GenreEntry));

    GenreMButton = gtk_button_new();
    gtk_widget_set_usize(GenreMButton,MButtonSize,MButtonSize);
    gtk_table_attach(GTK_TABLE(Table),GenreMButton,10,11,4,5,0,0,0,0);
    gtk_signal_connect(GTK_OBJECT(GenreMButton),"clicked",GTK_SIGNAL_FUNC(Mini_Button_Clicked),NULL);
    gtk_tooltips_set_tip(Tips,GenreMButton,_("Tag selected files with this genre"),NULL);

    Attach_Popup_Menu_To_Tag_Entries(GTK_ENTRY(GTK_COMBO(GenreEntry)->entry));
    gtk_object_set_data(GTK_OBJECT(GTK_COMBO(GenreEntry)->entry),"MButtonName",GenreMButton);
    gtk_signal_connect(GTK_OBJECT(GTK_COMBO(GenreEntry)->entry),"key_press_event",(GtkSignalFunc)Tag_Area_Entry_Key_Press,NULL);


    /* Comment */
    // For translators : if the translated word is too long, please use an abbreviation to don't loose space on the interface.
    Label = gtk_label_new(_("Comnt.:"));
    gtk_table_attach(GTK_TABLE(Table),Label,0,1,5,6,GTK_FILL,GTK_FILL,0,0);
    gtk_misc_set_alignment(GTK_MISC(Label),1,0.5);

    CommentEntry = gtk_entry_new();
    gtk_table_attach_defaults(GTK_TABLE(Table),CommentEntry,1,10,5,6);

    CommentMButton = gtk_button_new();
    gtk_widget_set_usize(CommentMButton,MButtonSize,MButtonSize);
    gtk_table_attach(GTK_TABLE(Table),CommentMButton,10,11,5,6,0,0,0,0);
    gtk_signal_connect(GTK_OBJECT(CommentMButton),"clicked",GTK_SIGNAL_FUNC(Mini_Button_Clicked),NULL);
    gtk_tooltips_set_tip(Tips,CommentMButton,_("Tag selected files with this comment"),NULL);

    Attach_Popup_Menu_To_Tag_Entries(GTK_ENTRY(CommentEntry));
    gtk_object_set_data(GTK_OBJECT(CommentEntry),"MButtonName",CommentMButton);
    gtk_signal_connect(GTK_OBJECT(CommentEntry),"key_press_event",(GtkSignalFunc)Tag_Area_Entry_Key_Press,NULL);

    // Managing of entries when pressing the Enter key
    gtk_signal_connect_object(GTK_OBJECT(TitleEntry),     "activate",(GtkSignalFunc)gtk_widget_grab_focus,GTK_OBJECT(ArtistEntry));
    gtk_signal_connect_object(GTK_OBJECT(ArtistEntry),    "activate",(GtkSignalFunc)gtk_widget_grab_focus,GTK_OBJECT(AlbumEntry));
    gtk_signal_connect_object(GTK_OBJECT(AlbumEntry),     "activate",(GtkSignalFunc)gtk_widget_grab_focus,GTK_OBJECT(YearEntry));
    gtk_signal_connect_object(GTK_OBJECT(YearEntry),      "activate",(GtkSignalFunc)gtk_widget_grab_focus,GTK_OBJECT(GTK_COMBO(TrackEntry)->entry));
    gtk_signal_connect_object(GTK_OBJECT(GTK_COMBO(TrackEntry)->entry),"activate",(GtkSignalFunc)gtk_widget_grab_focus,GTK_OBJECT(TrackTotalEntry));
    gtk_signal_connect_object(GTK_OBJECT(TrackTotalEntry),"activate",(GtkSignalFunc)gtk_widget_grab_focus,GTK_OBJECT(GTK_COMBO(GenreEntry)->entry));
    gtk_signal_connect_object(GTK_OBJECT(GTK_COMBO(GenreEntry)->entry),"activate",(GtkSignalFunc)gtk_widget_grab_focus,GTK_OBJECT(CommentEntry));
    gtk_signal_connect_object(GTK_OBJECT(CommentEntry),   "activate",(GtkSignalFunc)gtk_widget_grab_focus,GTK_OBJECT(TitleEntry));

    gtk_widget_show_all(TagFrame);
    return TagFrame;
}



/*
 * Actions when mini buttons are pressed: apply the field to all others files
 */
void Menu_Mini_Button_Clicked (GtkEntry *entry)
{
    if ( gtk_object_get_data(GTK_OBJECT(entry),"MButtonName") )
        Mini_Button_Clicked((GtkObject *)gtk_object_get_data(GTK_OBJECT(entry),"MButtonName"));
}
void Mini_Button_Clicked (GtkObject *object)
{
    GList *etfilelist = NULL;
    gchar *string_to_set = NULL;
    gchar *string_to_set1 = NULL;
    gchar *msg = NULL;
    File_Tag *FileTag;


    if (!ETCore->ETFileDisplayedList) return;

    /* Save the current displayed data */
    ET_Save_File_Data_From_UI(ETCore->ETFileDisplayed);

    if (object==GTK_OBJECT(TitleMButton))
    {
        string_to_set = gtk_editable_get_chars(GTK_EDITABLE(TitleEntry),0,-1); // The string to apply to all other files
        etfilelist = ET_File_List_Get_Selection();
        while (etfilelist)
        {    
            FileTag = ET_File_Tag_Item_New();
            ET_Copy_File_Tag_Item(etfilelist->data,FileTag);
            ET_Set_Field_File_Tag_Item(&FileTag->title,string_to_set);
            ET_Manage_Changes_Of_File_Data(etfilelist->data,NULL,FileTag);
            etfilelist = g_list_next(etfilelist);
        }
        if (string_to_set != NULL && strlen(string_to_set)>0)
            msg = g_strdup_printf(_("Selected files tagged with title '%s'."),string_to_set);
        else
            msg = g_strdup(_("Removed title from selected files."));
    }
    else if (object==GTK_OBJECT(ArtistMButton))
    {
        string_to_set = gtk_editable_get_chars(GTK_EDITABLE(ArtistEntry),0,-1);
        etfilelist = ET_File_List_Get_Selection();
        while (etfilelist)
        {    
            FileTag = ET_File_Tag_Item_New();
            ET_Copy_File_Tag_Item(etfilelist->data,FileTag);
            ET_Set_Field_File_Tag_Item(&FileTag->artist,string_to_set);
            ET_Manage_Changes_Of_File_Data(etfilelist->data,NULL,FileTag);
            etfilelist = g_list_next(etfilelist);
        }
        if (string_to_set != NULL && strlen(string_to_set)>0)
            msg = g_strdup_printf(_("Selected files tagged with artist '%s'."),string_to_set);
        else
            msg = g_strdup(_("Removed artist from selected files."));
    }
    else if (object==GTK_OBJECT(AlbumMButton))
    {
        string_to_set = gtk_editable_get_chars(GTK_EDITABLE(AlbumEntry),0,-1);
        etfilelist = ET_File_List_Get_Selection();
        while (etfilelist)
        {    
            FileTag = ET_File_Tag_Item_New();
            ET_Copy_File_Tag_Item(etfilelist->data,FileTag);
            ET_Set_Field_File_Tag_Item(&FileTag->album,string_to_set);
            ET_Manage_Changes_Of_File_Data(etfilelist->data,NULL,FileTag);
            etfilelist = g_list_next(etfilelist);
        }
        if (string_to_set != NULL && strlen(string_to_set)>0)
            msg = g_strdup_printf(_("Selected files tagged with album '%s'."),string_to_set);
        else
            msg = g_strdup(_("Removed album name from selected files."));
    }
    else if (object==GTK_OBJECT(YearMButton))
    {
        string_to_set = gtk_editable_get_chars(GTK_EDITABLE(YearEntry),0,-1);
        etfilelist = ET_File_List_Get_Selection();
        while (etfilelist)
        {    
            FileTag = ET_File_Tag_Item_New();
            ET_Copy_File_Tag_Item(etfilelist->data,FileTag);
            ET_Set_Field_File_Tag_Item(&FileTag->year,string_to_set);
            ET_Manage_Changes_Of_File_Data(etfilelist->data,NULL,FileTag);
            etfilelist = g_list_next(etfilelist);
        }
        if (string_to_set != NULL && strlen(string_to_set)>0)
            msg = g_strdup_printf(_("Selected files tagged with year '%s'."),string_to_set);
        else
            msg = g_strdup(_("Removed year from selected files."));
    }
    else if (object==GTK_OBJECT(TrackMButton))
    {
        /* Used of Track and Total Track values */
        string_to_set = gtk_editable_get_chars(GTK_EDITABLE(GTK_COMBO(TrackEntry)->entry),0,-1);
        string_to_set1 = gtk_editable_get_chars(GTK_EDITABLE(TrackTotalEntry),0,-1);
        etfilelist = ET_File_List_Get_Selection();
        while (etfilelist)
        {    
            FileTag = ET_File_Tag_Item_New();
            ET_Copy_File_Tag_Item(etfilelist->data,FileTag);

            // We apply the TrackEntry field to all others files only if it is to delete
            // the field (string=""). Else we don't overwrite the track number
            if (strlen(string_to_set)==0)
                ET_Set_Field_File_Tag_Item(&FileTag->track,string_to_set);
            ET_Set_Field_File_Tag_Item(&FileTag->track_total,string_to_set1);
            ET_Manage_Changes_Of_File_Data(etfilelist->data,NULL,FileTag);
            etfilelist = g_list_next(etfilelist);
        }

        if ( string_to_set != NULL && strlen(string_to_set)>0 ) //&& atoi(string_to_set)>0 )
        {
            if ( string_to_set1 != NULL && strlen(string_to_set1)>0 ) //&& atoi(string_to_set1)>0 )
            {
                msg = g_strdup_printf(_("Selected files tagged with track like 'xx/%s'."),string_to_set1);
            }else
            {
                msg = g_strdup_printf(_("Selected files tagged with track like 'xx'."));
            }
        }else
        {
            msg = g_strdup(_("Removed track number from selected files."));
        }
    }
    /* This part doesn't set the same track number to all files, but sequence the tracks */
    else if (object==GTK_OBJECT(TrackMButtonSequence))
    {
        GList *etfilelistfull = NULL;
        gchar *path = NULL;
        gchar *path1 = NULL;
        gint i = 0;

        /* FIX ME!: see to fill also the Total Track (it's a good idea?) */
        etfilelistfull = g_list_first(ETCore->ETFileList);
        etfilelist = ET_File_List_Get_Selection();
        while (etfilelist)
        {       
            File_Name *FileNameCur = (File_Name *)((ET_File *)etfilelistfull->data)->FileNameCur->data;
            // Restart counter when entering a new directory
            if (path1) g_free(path1);
            path1 = g_dirname(FileNameCur->value);
            if ( path && path1 && strcmp(path,path1)!=0 )
                i = 0;
            if (NUMBER_TRACK_FORMATED)
                string_to_set = g_strdup_printf("%.*d",NUMBER_TRACK_FORMATED_SPIN_BUTTON,++i);
            else
                string_to_set = g_strdup_printf("%d",++i);

            // The file is in the selection?
            if ( (ET_File *)etfilelistfull->data == (ET_File *)etfilelist->data )
            {
                FileTag = ET_File_Tag_Item_New();
                ET_Copy_File_Tag_Item(etfilelist->data,FileTag);
                ET_Set_Field_File_Tag_Item(&FileTag->track,string_to_set);
                ET_Manage_Changes_Of_File_Data(etfilelist->data,NULL,FileTag);
                etfilelist = g_list_next(etfilelist);
            }
            
            if (string_to_set) {g_free(string_to_set); string_to_set = NULL;}
            if (path) g_free(path);
            path = g_strdup(path1);

            etfilelistfull = g_list_next(etfilelistfull);
        }
        if (path)  g_free(path);
        if (path1) g_free(path1);
        //msg = g_strdup_printf(_("All %d tracks numbered sequentially."), ETCore->ETFileSelectionList_Length);
        msg = g_strdup_printf(_("Selected tracks numbered sequentially."));
    }
    else if (object==GTK_OBJECT(TrackMButtonNbrFiles))
    {
        /* Used of Track and Total Track values */
        etfilelist = ET_File_List_Get_Selection();
        while (etfilelist)
        {
            gchar *path;
    
            path = g_dirname( ((File_Name *)((ET_File *)etfilelist->data)->FileNameNew->data)->value );
            if (NUMBER_TRACK_FORMATED)
                string_to_set = g_strdup_printf("%.*d",NUMBER_TRACK_FORMATED_SPIN_BUTTON,ET_Get_Number_Of_Files_In_Directory(path));
            else
                string_to_set = g_strdup_printf("%d",ET_Get_Number_Of_Files_In_Directory(path));
            g_free(path);
            if (!string_to_set1)
                string_to_set1 = g_strdup(string_to_set); // Just for the message below...
            
            FileTag = ET_File_Tag_Item_New();
            ET_Copy_File_Tag_Item(etfilelist->data,FileTag);
            ET_Set_Field_File_Tag_Item(&FileTag->track_total,string_to_set);
            ET_Manage_Changes_Of_File_Data(etfilelist->data,NULL,FileTag);
            etfilelist = g_list_next(etfilelist);
            g_free(string_to_set);
        }

        string_to_set = NULL;
        if ( string_to_set1 != NULL && strlen(string_to_set1)>0 ) //&& atoi(string_to_set1)>0 )
        {
            msg = g_strdup_printf(_("Selected files tagged with track like 'xx/%s'."),string_to_set1);
        }else
        {
            msg = g_strdup(_("Removed track number from selected files."));
        }
    }
    else if (object==GTK_OBJECT(GenreMButton))
    {
        string_to_set = gtk_editable_get_chars(GTK_EDITABLE(GTK_COMBO(GenreEntry)->entry),0,-1);
        etfilelist = ET_File_List_Get_Selection();
        while (etfilelist)
        {    
            FileTag = ET_File_Tag_Item_New();
            ET_Copy_File_Tag_Item(etfilelist->data,FileTag);
            ET_Set_Field_File_Tag_Item(&FileTag->genre,string_to_set);
            ET_Manage_Changes_Of_File_Data(etfilelist->data,NULL,FileTag);
            etfilelist = g_list_next(etfilelist);
        }
        if (string_to_set != NULL && strlen(string_to_set)>0)
            msg = g_strdup_printf(_("Selected files tagged with genre '%s'."),string_to_set);
        else
            msg = g_strdup(_("Removed genre from selected files."));
    }
    else if (object==GTK_OBJECT(CommentMButton))
    {
        string_to_set = gtk_editable_get_chars(GTK_EDITABLE(CommentEntry),0,-1);
        etfilelist = ET_File_List_Get_Selection();
        while (etfilelist)
        {    
            FileTag = ET_File_Tag_Item_New();
            ET_Copy_File_Tag_Item(etfilelist->data,FileTag);
            ET_Set_Field_File_Tag_Item(&FileTag->comment,string_to_set);
            ET_Manage_Changes_Of_File_Data(etfilelist->data,NULL,FileTag);
            etfilelist = g_list_next(etfilelist);
        }
        if (string_to_set != NULL && strlen(string_to_set)>0)
            msg = g_strdup_printf(_("Selected files tagged with comment '%s'."),string_to_set);
        else
            msg = g_strdup(_("Removed comment from selected files."));
    }

    // Refresh the whole list (faster than file by file) to show changes
    Browser_List_Refresh_Whole_List();

    /* Display the current file (Needed when sequencing tracks) */
    ET_Display_File_Data_To_UI(ETCore->ETFileDisplayed);

    if (msg)
    {
        g_print("%s\n",msg);
        Statusbar_Message(msg,TRUE);
        g_free(msg);
    }
    if (string_to_set)  g_free(string_to_set);
    if (string_to_set1) g_free(string_to_set1);

    /* To update state of Undo button */
    Update_Command_Buttons_Sensivity();
}




/*
 * Action when selecting all files
 */
void Action_Select_All_Files (void)
{
    /* Save the current displayed data */
    ET_Save_File_Data_From_UI(ETCore->ETFileDisplayed);
    
    Browser_List_Select_All_Files();
    Update_Command_Buttons_Sensivity();
}


/*
 * Action when unselecting all files
 */
void Action_Unselect_All_Files (void)
{
    /* Save the current displayed data */
    ET_Save_File_Data_From_UI(ETCore->ETFileDisplayed);
    
    Browser_List_Unselect_All_Files();
    ETCore->ETFileDisplayed = NULL;
}


/*
 * Action when inverting files selection
 */
void Action_Invert_Files_Selection (void)
{
    /* Save the current displayed data */
    ET_Save_File_Data_From_UI(ETCore->ETFileDisplayed);
    
    Browser_List_Invert_File_Selection();
    Update_Command_Buttons_Sensivity();
}



/*
 * Action when First button is selected
 */
void Action_Select_First_File (void)
{
    GList *etfilelist;

    if (!ETCore->ETFileDisplayedList || !ETCore->ETFileDisplayedList->prev)
        return;

    /* Save the current displayed data */
    ET_Save_File_Data_From_UI(ETCore->ETFileDisplayed);
    
    /* Go to the first item of the list */
    etfilelist = ET_Displayed_File_List_First();
    if (etfilelist)
    {
        Browser_List_Unselect_All_Files(); // To avoid the last line still selected
        Browser_List_Select_File((ET_File *)etfilelist->data,TRUE);
        ET_Display_File_Data_To_UI((ET_File *)etfilelist->data);
    }

    Update_Command_Buttons_Sensivity();
    Scan_Rename_File_Generate_Preview();
    Scan_Fill_Tag_Generate_Preview();
}


/*
 * Action when Prev button is selected
 */
void Action_Select_Prev_File (void)
{
    GList *etfilelist;

    if (!ETCore->ETFileDisplayedList || !ETCore->ETFileDisplayedList->prev)
        return;

    /* Save the current displayed data */
    ET_Save_File_Data_From_UI(ETCore->ETFileDisplayed);
    
    /* Go to the prev item of the list */
    etfilelist = ET_Displayed_File_List_Previous();
    if (etfilelist)
    {
        Browser_List_Unselect_All_Files();
        Browser_List_Select_File((ET_File *)etfilelist->data,TRUE);
        ET_Display_File_Data_To_UI((ET_File *)etfilelist->data);
    }

//    if (!ETFileList->prev)
//        gdk_beep(); // Warm the user

    Update_Command_Buttons_Sensivity();
    Scan_Rename_File_Generate_Preview();
    Scan_Fill_Tag_Generate_Preview();
}


/*
 * Action when Next button is selected
 */
void Action_Select_Next_File (void)
{
    GList *etfilelist;

    if (!ETCore->ETFileDisplayedList || !ETCore->ETFileDisplayedList->next)
        return;

    /* Save the current displayed data */
    ET_Save_File_Data_From_UI(ETCore->ETFileDisplayed);
    
    /* Go to the next item of the list */
    etfilelist = ET_Displayed_File_List_Next();
    if (etfilelist)
    {
        Browser_List_Unselect_All_Files();
        Browser_List_Select_File((ET_File *)etfilelist->data,TRUE);
        ET_Display_File_Data_To_UI((ET_File *)etfilelist->data);
    }

//    if (!ETFileList->next)
//        gdk_beep(); // Warm the user

    Update_Command_Buttons_Sensivity();
    Scan_Rename_File_Generate_Preview();
    Scan_Fill_Tag_Generate_Preview();
}


/*
 * Action when Last button is selected
 */
void Action_Select_Last_File (void)
{
    GList *etfilelist;

    if (!ETCore->ETFileDisplayedList || !ETCore->ETFileDisplayedList->next)
        return;

    /* Save the current displayed data */
    ET_Save_File_Data_From_UI(ETCore->ETFileDisplayed);
    
    /* Go to the last item of the list */
    etfilelist = ET_Displayed_File_List_Last();
    if (etfilelist)
    {
        Browser_List_Unselect_All_Files();
        Browser_List_Select_File((ET_File *)etfilelist->data,TRUE);
        ET_Display_File_Data_To_UI((ET_File *)etfilelist->data);
    }

    Update_Command_Buttons_Sensivity();
    Scan_Rename_File_Generate_Preview();
    Scan_Fill_Tag_Generate_Preview();
}


/*
 * Select a file in the "main list" using the ETFile adress of each item.
 */
void Action_Select_Nth_File_By_Etfile (ET_File *ETFile)
{
    if (!ETCore->ETFileDisplayedList)
        return;

    /* Save the current displayed data */
    ET_Save_File_Data_From_UI(ETCore->ETFileDisplayed);
    
    /* Display the item */
    Browser_List_Select_File(ETFile,TRUE);
    ET_Displayed_File_List_By_Etfile(ETFile); // Just to update 'ETFileDisplayedList'
    ET_Display_File_Data_To_UI(ETFile);

    Update_Command_Buttons_Sensivity();
    Scan_Rename_File_Generate_Preview();
    Scan_Fill_Tag_Generate_Preview();
}

/*
 * Select the nth file in list.
 */
void Action_Select_Nth_File_By_Position (gulong num_item) 
{
    GList *etfilelist;

    if ( !ETCore->ETFileDisplayedList || !num_item || (num_item > ETCore->ETFileDisplayedList_Length) )
        return;

    /* Save the current displayed data */
    ET_Save_File_Data_From_UI(ETCore->ETFileDisplayed);
    
    /* Display the item */
    etfilelist = ET_Displayed_File_List_By_Position(num_item);
    if (etfilelist)
    {
        Browser_List_Select_File((ET_File *)etfilelist->data,TRUE);
        ET_Display_File_Data_To_UI((ET_File *)etfilelist->data);
    }
    
    Update_Command_Buttons_Sensivity();
    Scan_Rename_File_Generate_Preview();
    Scan_Fill_Tag_Generate_Preview();
}



/*
 * Action when Scan button is pressed
 */
void Action_Scan_Selected_Files (void)
{
    gint progress_bar_index;
    GList *etfilelist = NULL;

    if (!ETCore->ETFileDisplayedList) return;

    /* Check if scanner window is opened */
    if(!ScannerWindow)
    {
        Open_ScannerWindow(SCANNER_FILL_TAG);
        Statusbar_Message(_("Select Mode and Mask, and redo the same action"),TRUE);
        return;
    }

    /* Save the current displayed data */
    ET_Save_File_Data_From_UI(ETCore->ETFileDisplayed);

    /* Initialize status bar */
    gtk_progress_configure(GTK_PROGRESS(ProgressBar),0,0,ETCore->ETFileSelectionList_Length);
    progress_bar_index = 0;

    etfilelist = ET_File_List_Get_Selection();
    while (etfilelist)
    {    
        // Run the current scanner
        Scan_Select_Mode_And_Run_Scanner((ET_File *)etfilelist->data);
        etfilelist = g_list_next(etfilelist);

        gtk_progress_set_value(GTK_PROGRESS(ProgressBar),++progress_bar_index);
        /* Needed to refresh status bar */
        while (gtk_events_pending())
            gtk_main_iteration();
    }

    // Refresh the whole list (faster than file by file) to show changes
    Browser_List_Refresh_Whole_List();

    /* Display the current file */
    ET_Display_File_Data_To_UI(ETCore->ETFileDisplayed);
    Update_Command_Buttons_Sensivity();

    gtk_progress_set_value(GTK_PROGRESS(ProgressBar),0);
    Statusbar_Message(_("All tags have been scanned"),TRUE);
}



/*
 * Action when Remove button is pressed
 */
void Action_Remove_Selected_Tags (void)
{
    GList *etfilelist = NULL;
    File_Tag *FileTag;
    gint progress_bar_index;

    if (!ETCore->ETFileDisplayedList) return;

    /* Save the current displayed data */
    ET_Save_File_Data_From_UI(ETCore->ETFileDisplayed);

    /* Initialize status bar */
    gtk_progress_configure(GTK_PROGRESS(ProgressBar),0,0,ETCore->ETFileSelectionList_Length);
    progress_bar_index = 0;

    etfilelist = ET_File_List_Get_Selection();
    while (etfilelist)
    {    
        FileTag = ET_File_Tag_Item_New();
        ET_Manage_Changes_Of_File_Data(etfilelist->data,NULL,FileTag);
        etfilelist = g_list_next(etfilelist);

        gtk_progress_set_value(GTK_PROGRESS(ProgressBar),++progress_bar_index);
        /* Needed to refresh status bar */
        while (gtk_events_pending())
            gtk_main_iteration();
    }

    // Refresh the whole list (faster than file by file) to show changes
    Browser_List_Refresh_Whole_List();

    /* Display the current file */
    ET_Display_File_Data_To_UI(ETCore->ETFileDisplayed);
    Update_Command_Buttons_Sensivity();

    gtk_progress_set_value(GTK_PROGRESS(ProgressBar),0);
    Statusbar_Message(_("All tags have been removed"),TRUE);
}



/*
 * Action when Undo button is pressed.
 * Action_Undo_Selected_Files: Undo the last changes for the selected files.
 * Action_Undo_From_History_List: Undo the changes of the last modified file of the list.
 */
gint Action_Undo_Selected_Files (void)
{
    GList *etfilelist = NULL;
    gboolean state = FALSE;
    
    if (!ETCore->ETFileDisplayedList) return FALSE;

    /* Save the current displayed data */
    ET_Save_File_Data_From_UI(ETCore->ETFileDisplayed);

    etfilelist = ET_File_List_Get_Selection();
    while (etfilelist)
    {    
        state |= ET_Undo_File_Data((ET_File *)etfilelist->data);
        etfilelist = g_list_next(etfilelist);
    }

    // Refresh the whole list (faster than file by file) to show changes
    Browser_List_Refresh_Whole_List();

    /* Display the current file */
    ET_Display_File_Data_To_UI(ETCore->ETFileDisplayed);
    Update_Command_Buttons_Sensivity();

    //ET_Debug_Print_File_List(ETCore->ETFileList,__FILE__,__LINE__,__FUNCTION__);

    return state;
}


void Action_Undo_From_History_List (void)
{
    ET_File *ETFile;
    
    if (!ETCore->ETFileList) return;

    /* Save the current displayed data */
    ET_Save_File_Data_From_UI(ETCore->ETFileDisplayed);
    
    ETFile = ET_Undo_History_File_Data();
    if (ETFile)
    {
        ET_Display_File_Data_To_UI(ETFile);
        Browser_List_Select_File(ETFile,TRUE);
        Browser_List_Refresh_File_In_List(ETFile);
    }
 
    Update_Command_Buttons_Sensivity();
}



/*
 * Action when Redo button is pressed.
 * Action_Redo_Selected_Files: Redo the last changes for the selected files.
 * Action_Redo_From_History_List: Redo the changes of the last modified file of the list.
 */
gint Action_Redo_Selected_File (void)
{
    GList *etfilelist = NULL;
    gboolean state = FALSE;
    
    if (!ETCore->ETFileDisplayedList) return FALSE;

    /* Save the current displayed data */
    ET_Save_File_Data_From_UI(ETCore->ETFileDisplayed);

    etfilelist = ET_File_List_Get_Selection();
    while (etfilelist)
    {    
        state |= ET_Redo_File_Data((ET_File *)etfilelist->data);
        etfilelist = g_list_next(etfilelist);
    }

    // Refresh the whole list (faster than file by file) to show changes
    Browser_List_Refresh_Whole_List();

    /* Display the current file */
    ET_Display_File_Data_To_UI(ETCore->ETFileDisplayed);
    Update_Command_Buttons_Sensivity();

    return state;
}


void Action_Redo_From_History_List (void)
{
    ET_File *ETFile;
    
    if (!ETCore->ETFileDisplayedList) return;

    /* Save the current displayed data */
    ET_Save_File_Data_From_UI(ETCore->ETFileDisplayed);
    
    ETFile = ET_Redo_History_File_Data();
    if (ETFile)
    {
        ET_Display_File_Data_To_UI(ETFile);
        Browser_List_Select_File(ETFile,TRUE);
        Browser_List_Refresh_File_In_List(ETFile);
    }
 
    Update_Command_Buttons_Sensivity();
}




/*
 * Action when Save button is pressed
 */
void Action_Save_Selected_Files (void)
{
    Save_Selected_Files_With_Answer();
}


gint Save_All_Files_With_Answer (void)
{
    GList *etfilelist;

    if (!ETCore || !ETCore->ETFileList) return FALSE;
    etfilelist = g_list_first(ETCore->ETFileList);
    return Save_List_Of_Files(etfilelist);
}

gint Save_Selected_Files_With_Answer (void)
{
    GList *etfilelist;

    etfilelist = ET_File_List_Get_Selection();
    return Save_List_Of_Files(etfilelist);
}

/*
 * Save_List_Of_Files: Function to save a list of files.
 */
gint Save_List_Of_Files (GList *etfilelist)
{
    gint       progress_bar_index;
    gint       saving_answer;
    gint       nb_files_to_save;
    gchar     *msg;
    GList     *etfilelist_tmp;
    ET_File   *etfile_save_position = NULL;
    File_Tag  *FileTag;
    File_Name *FileNameNew;


    if (!ETCore) return FALSE;

    /* Save the current position in the list */
    etfile_save_position = ETCore->ETFileDisplayed;

    /* Save the current displayed data */
    ET_Save_File_Data_From_UI(ETCore->ETFileDisplayed);

    /* Count the number of files to save */
    nb_files_to_save = 0;
    etfilelist_tmp = etfilelist;
    while (etfilelist_tmp)
    {
        File_Tag  *filetag  = ((ET_File *)etfilelist_tmp->data)->FileTag->data;
        File_Name *filename = ((ET_File *)etfilelist_tmp->data)->FileNameNew->data;

        if ( (filename && filename->saved==FALSE) || (filetag && filetag->saved==FALSE) )
            nb_files_to_save++;
        etfilelist_tmp = etfilelist_tmp->next;
    }

    /* Initialize status bar */
    gtk_progress_configure(GTK_PROGRESS(ProgressBar),0,0,nb_files_to_save);
    progress_bar_index = 0;
    gtk_progress_set_format_string(GTK_PROGRESS(ProgressBar),"%v/%u");
    gtk_progress_set_show_text(GTK_PROGRESS(ProgressBar),TRUE);

    /* Set to unsensitive all command buttons (except Quit button) */
    Disable_Command_Buttons();
    Browser_Area_Set_Sensitive(FALSE);
    Tag_Area_Set_Sensitive(FALSE);
    File_Area_Set_Sensitive(FALSE);

    /* Show msgbox (if needed) to ask confirmation */
    SF_HideMsgbox_Write_Tag = 0;
    SF_HideMsgbox_Rename_File = 0;
    
    Main_Stop_Button_Pressed = 0;
    gtk_widget_set_sensitive(TBStopButton,TRUE);
    
    etfilelist_tmp = etfilelist;
    while (etfilelist_tmp)
    {
        FileTag     = ((ET_File *)etfilelist_tmp->data)->FileTag->data;
        FileNameNew = ((ET_File *)etfilelist_tmp->data)->FileNameNew->data;
        
        /* We process only the files not saved */
        if ( FileTag->saved == FALSE || FileNameNew->saved == FALSE )
        {
            ET_Display_File_Data_To_UI((ET_File *)etfilelist_tmp->data);
            Browser_List_Select_File((ET_File *)etfilelist_tmp->data,FALSE);

            gtk_progress_set_value(GTK_PROGRESS(ProgressBar),++progress_bar_index);
            /* Needed to refresh status bar */
            while (gtk_events_pending())
                gtk_main_iteration();

            saving_answer = Save_File((ET_File *)etfilelist_tmp->data,(nb_files_to_save>1)?TRUE:FALSE);

            if (saving_answer == -1)
            {
                /* Stop saving files + reinit progress bar */
                gtk_progress_set_value(GTK_PROGRESS(ProgressBar),0);
                Statusbar_Message(_("All files have been saved..."),TRUE);
                /* To update state of command buttons */
                Update_Command_Buttons_Sensivity();
                Browser_Area_Set_Sensitive(TRUE);
                Tag_Area_Set_Sensitive(TRUE);
                File_Area_Set_Sensitive(TRUE);

                return -1; /* We stop all actions */
            }
        }

        etfilelist_tmp = etfilelist_tmp->next;
        if (Main_Stop_Button_Pressed == 1 )
            break;
        
    }

    if (Main_Stop_Button_Pressed)
        msg = g_strdup(_("Files have been partially saved..."));
    else
        msg = g_strdup(_("All files have been saved..."));

    Main_Stop_Button_Pressed = 0;
    gtk_widget_set_sensitive(TBStopButton,FALSE);

    /* Return to the saved position in the list */
    ET_Display_File_Data_To_UI(etfile_save_position);
    Browser_List_Select_File(etfile_save_position,TRUE);

    /* Browser is on mode : Artist + Album list */
    if ( gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(TBShowBrowserButton)) )
        Browser_Display_Tree_Or_Artist_Album_List();

    /* To update state of command buttons */
    Update_Command_Buttons_Sensivity();
    Browser_Area_Set_Sensitive(TRUE);
    Tag_Area_Set_Sensitive(TRUE);
    File_Area_Set_Sensitive(TRUE);

    gtk_progress_set_show_text(GTK_PROGRESS(ProgressBar),FALSE);
    gtk_progress_set_value(GTK_PROGRESS(ProgressBar),0);
    Statusbar_Message(msg,TRUE);
    g_free(msg);

    return TRUE;
}



/*
 * Delete a file on the hard disk
 */
void Action_Delete_Selected_Files (void)
{
    Delete_Selected_Files_With_Answer();
}


gint Delete_Selected_Files_With_Answer (void)
{
    GList *etfilelist;
    gint   progress_bar_index;
    gint   saving_answer;
    gint   nb_files_to_delete;
    gint   nb_files_deleted = 0;
    gchar *msg;


    if (!ETCore->ETFileDisplayedList) return FALSE;

    /* Save the current displayed data */
    ET_Save_File_Data_From_UI(ETCore->ETFileDisplayed);

    /* Number of files to save */
    nb_files_to_delete = ETCore->ETFileSelectionList_Length;

    /* Initialize status bar */
    gtk_progress_configure(GTK_PROGRESS(ProgressBar),0,0,nb_files_to_delete);
    progress_bar_index = 0;
    gtk_progress_set_format_string(GTK_PROGRESS(ProgressBar),"%v/%u");
    gtk_progress_set_show_text(GTK_PROGRESS(ProgressBar),TRUE);

    /* Set to unsensitive all command buttons (except Quit button) */
    Disable_Command_Buttons();
    Browser_Area_Set_Sensitive(FALSE);
    Tag_Area_Set_Sensitive(FALSE);
    File_Area_Set_Sensitive(FALSE);

    /* Show msgbox (if needed) to ask confirmation */
    SF_HideMsgbox_Delete_File = 0;
    
    etfilelist = ET_File_List_Get_Selection();
    while (etfilelist)
    {
        ET_File *ETFile;
        
        ETFile = (ET_File *)etfilelist->data;
        ET_Display_File_Data_To_UI(ETFile);
        Browser_List_Select_File(ETFile,FALSE);

        gtk_progress_set_value(GTK_PROGRESS(ProgressBar),++progress_bar_index);
        /* Needed to refresh status bar */
        while (gtk_events_pending())
            gtk_main_iteration();

        saving_answer = Delete_File(ETFile,(ETCore->ETFileSelectionList_Length>1)?TRUE:FALSE);

        switch (saving_answer)
        {
            case 1:
                nb_files_deleted += saving_answer;
                // Remove file in the browser (corresponding line in the clist)
                Browser_List_Remove_File(ETFile);
                // Remove file from file list
                ET_Remove_File_From_File_List(ETFile);
                break;
            case 0:
                break;
            case -1:
                // Stop deleting files + reinit progress bar
                gtk_progress_set_value(GTK_PROGRESS(ProgressBar),0);
                // To update state of command buttons
                Update_Command_Buttons_Sensivity();
                Browser_Area_Set_Sensitive(TRUE);
                Tag_Area_Set_Sensitive(TRUE);
                File_Area_Set_Sensitive(TRUE);
    
                return -1; // We stop all actions
                break;
        }

        etfilelist = etfilelist->next;
    }

    if (nb_files_deleted < nb_files_to_delete)
        msg = g_strdup(_("Files have been partially deleted..."));
    else
        msg = g_strdup(_("All files have been deleted..."));

    // It's important to displayed the new item, as it'll check the changes in Browser_Display_Tree_Or_Artist_Album_List
    if (ETCore->ETFileDisplayed)
        ET_Display_File_Data_To_UI(ETCore->ETFileDisplayed);
    /*else if (ET_Displayed_File_List_Current())
        ET_Display_File_Data_To_UI((ET_File *)ET_Displayed_File_List_Current()->data);*/
    
    // Load list...
    Browser_List_Load_File_List(ETCore->ETFileDisplayedList);
    // Rebuild the list...
    Browser_Display_Tree_Or_Artist_Album_List();

    /* To update state of command buttons */
    Update_Command_Buttons_Sensivity();
    Browser_Area_Set_Sensitive(TRUE);
    Tag_Area_Set_Sensitive(TRUE);
    File_Area_Set_Sensitive(TRUE);

    gtk_progress_set_show_text(GTK_PROGRESS(ProgressBar),FALSE);
    gtk_progress_set_value(GTK_PROGRESS(ProgressBar),0);
    Statusbar_Message(msg,TRUE);
    g_free(msg);

    return TRUE;
}



/*
 * Save changes of the ETFile (write tag and rename file)
 *  - multiple_files = TRUE  : when saving files, a msgbox appears with ability 
 *                             to do the same action for all files.
 *  - multiple_files = FALSE : appears only a msgbox to ask confirmation.
 */
gint Save_File (ET_File *ETFile, gboolean multiple_files)
{
    gchar *msg = NULL;
    gint stop_loop = 0;
    gchar *cur_filename = ((File_Name *)ETFile->FileNameCur->data)->value;
    File_Tag  *FileTag;
    File_Name *FileNameNew;


    if (!ETFile) return 0;

    /* Save the current displayed data */
    // Not needed, done before //ET_Save_File_Data_From_UI((ET_File *)ETFileList->data);
    FileTag     = ETFile->FileTag->data;
    FileNameNew = ETFile->FileNameNew->data;

    /*
     * First part: write tag informations (artist, title,...)
     */
    if ( FileTag->saved == FALSE ) // This tag had been already saved ?
    {
        GtkWidget *msgbox = NULL;
        gint button;

        if (CONFIRM_WRITE_TAG && !SF_HideMsgbox_Write_Tag)
        {
            ET_Display_File_Data_To_UI(ETFile);
            
            msg = g_strdup_printf(_("Do you want to write the tag of file\n'%s' ?"),g_basename(cur_filename) );
            if (multiple_files)
            {
                msgbox = msg_box_new (_("Write Tag..."),msg,MSG_QUESTION,
                    BUTTON_YES,BUTTON_NO,BUTTON_CANCEL,0);
            }else
            {
                msgbox = msg_box_new (_("Write Tag..."),msg,MSG_QUESTION,BUTTON_YES,BUTTON_NO,0);
                msg_box_hide_check_button(MSG_BOX(msgbox));
            }
            g_free(msg);

            SF_ButtonPressed_Write_Tag = button = msg_box_run(MSG_BOX(msgbox));
            /* When check button in msgbox was activated */
            if (MSG_BOX(msgbox)->check_button_state)
                SF_HideMsgbox_Write_Tag = MSG_BOX(msgbox)->check_button_state;
            gtk_widget_destroy(msgbox);
        }else
        {
            if (SF_HideMsgbox_Write_Tag)
                button = SF_ButtonPressed_Write_Tag;
            else
                button = BUTTON_YES;
        }

        switch(button)
        {
            case BUTTON_YES:
                Write_File_Tag(ETFile);
                break;
            case BUTTON_NO:
                break;
            case BUTTON_CANCEL:
            case -1:
                stop_loop = -1;
                return stop_loop;
                break;
        }
    }


    /*
     * Second part: rename the file
     */
    if ( FileNameNew->saved == FALSE ) // This filename had been already saved ?
    {
        GtkWidget *msgbox = NULL;
        gint button;

        if (CONFIRM_RENAME_FILE && !SF_HideMsgbox_Rename_File)
        {
            ET_Display_File_Data_To_UI(ETFile);

            msg = g_strdup_printf(_("Do you want to rename the file \n'%s'\nto \n'%s' ?"),
                g_basename(((File_Name *)ETFile->FileNameCur->data)->value),
                g_basename(((File_Name *)ETFile->FileNameNew->data)->value) );
            if (multiple_files)
            {
                msgbox = msg_box_new (_("Rename File..."),msg,MSG_QUESTION,
                    BUTTON_YES,BUTTON_NO,BUTTON_CANCEL,0);
            }else
            {
                msgbox = msg_box_new (_("Rename File..."),msg,MSG_QUESTION,
                    BUTTON_YES,BUTTON_NO,0);
                msg_box_hide_check_button(MSG_BOX(msgbox));
            }
            g_free(msg);
            SF_ButtonPressed_Rename_File = button = msg_box_run(MSG_BOX(msgbox));
            if (MSG_BOX(msgbox)->check_button_state)
                SF_HideMsgbox_Rename_File = MSG_BOX(msgbox)->check_button_state;
            gtk_widget_destroy(msgbox);
        }else
        {
            if (SF_HideMsgbox_Rename_File)
                button = SF_ButtonPressed_Rename_File;
            else
                button = BUTTON_YES;
        }

        switch(button)
        {
            case BUTTON_YES:
                Rename_File(ETFile);
                break;
            case BUTTON_NO:
                break;
            case BUTTON_CANCEL:
            case -1:
                stop_loop = -1;
                return stop_loop;
                break;
        }
    }

    /* Refresh file into browser list */
    Browser_List_Refresh_File_In_List(ETFile);

    return 1;
}


/*
 * Write tag of the ETFile
 */
void Write_File_Tag (ET_File *ETFile)
{
    gchar *cur_filename = ((File_Name *)ETFile->FileNameCur->data)->value;
    gchar *msg;

    msg = g_strdup_printf(_("Writing tag of '%s'"),cur_filename);
    Statusbar_Message(msg,TRUE);
    g_free(msg);

    if ( ET_Save_File_Tag_To_HD(ETFile)==FALSE )
    {
        gchar *msg;
        GtkWidget *msgbox;

        switch ( ((ET_File_Description *)ETFile->ETFileDescription)->TagType)
        {
#ifdef ENABLE_OGG
            case OGG_TAG:
                // Special for Ogg Vorbis because the error is defined into 'vcedit_error(state)'
                msg = g_strdup_printf(_("Can't write tag in file '%s'!\n(%s)"),
                                      g_basename(cur_filename),ogg_error_msg); 
                break;
#endif
            default:
                msg = g_strdup_printf(_("Can't write tag in file '%s'!\n(%s)"),
                                      g_basename(cur_filename),g_strerror(errno)); 
        }
        
        //msg = g_strdup_printf(_("Can't write tag in file '%s'!\n(%s)"),
        //                      g_basename(cur_filename),g_strerror(errno)); 
        msgbox = msg_box_new (_("Error..."),msg,MSG_ERROR,BUTTON_OK,0);
        g_free(msg);
        msg_box_hide_check_button(MSG_BOX(msgbox));
        msg_box_run(MSG_BOX(msgbox));
        gtk_widget_destroy(msgbox);
    }else
    {
        Statusbar_Message(_("Tag(s) written"),TRUE);
    }
}

/*
 * Rename the file ETFile
 */
void Rename_File (ET_File *ETFile)
{
    FILE  *file;
    gchar *tmp_filename = NULL;
    gchar *cur_filename = ((File_Name *)ETFile->FileNameCur->data)->value;
    gchar *new_filename = ((File_Name *)ETFile->FileNameNew->data)->value;
    gchar *msg;
    gint   fd_tmp;

    msg = g_strdup_printf(_("Renaming file '%s'"),cur_filename);
    Statusbar_Message(msg,TRUE);
    g_free(msg);

    /* Check if a file already exists with the new name, and detect if it's
     * only a case change (needs for vfat) */
    if ( (file=fopen(new_filename,"r"))!=NULL )
    {
        gchar *msg;
        GtkWidget *msgbox;

        fclose(file);
        if ( strcasecmp(cur_filename,new_filename)!=0 )
        {
            msg = g_strdup_printf(_("Can't rename file \n'%s'\nbecause the following "
                        "file already exists:\n'%s'"),g_basename(cur_filename),g_basename(new_filename)); 
            msgbox = msg_box_new (_("Error..."),msg,MSG_ERROR,BUTTON_OK,0);
            g_free(msg);
            msg_box_hide_check_button(MSG_BOX(msgbox));
            msg_box_run(MSG_BOX(msgbox));
            gtk_widget_destroy(msgbox);

            Statusbar_Message(_("File(s) not renamed..."),TRUE);
            return;
        }
    }

    /* We use two stages to rename file, to avoid problem with some system
     * that doesn't allow to rename the file if only the case has changed. */
    tmp_filename = g_strdup_printf("%s.XXXXXX",cur_filename);
    if ( (fd_tmp = mkstemp(tmp_filename)) >= 0 )
    {
        close(fd_tmp);
        unlink(tmp_filename);
    }

    if ( rename(cur_filename,tmp_filename)!=0 )
    {
        gchar *msg;
        GtkWidget *msgbox;

        /* Renaming file to the temporary filename has failed */
        msg = g_strdup_printf(_("Can't rename file '%s'\n to \n'%s'!\n(%s)"),
                                g_basename(cur_filename),g_basename(new_filename),g_strerror(errno)); 
        msgbox = msg_box_new (_("Error..."),msg,MSG_ERROR,BUTTON_OK,0);
        g_free(msg);
        msg_box_hide_check_button(MSG_BOX(msgbox));
        msg_box_run(MSG_BOX(msgbox));
        gtk_widget_destroy(msgbox);

        Statusbar_Message(_("File(s) not renamed..."),TRUE);
        if (tmp_filename) g_free(tmp_filename);
        return;
    }

    /* Now, we rename the file to his final name */
    if ( rename(tmp_filename,new_filename)==0 )
    {
        /* Renaming file has succeeded */
        g_print(_("Renamed file '%s' to '%s'\n"),g_basename(cur_filename),g_basename(new_filename));

        ETFile->FileNameCur = ETFile->FileNameNew;
        /* Now the file was renamed, so mark his state */
        ET_Mark_File_Name_As_Saved(ETFile);

        Statusbar_Message(_("File(s) renamed..."),TRUE);
    }else
    {
        gchar *msg;
        GtkWidget *msgbox;

        /* Renaming file has failed, but we try to set the initial name */
        rename(tmp_filename,cur_filename);

        msg = g_strdup_printf(_("Can't rename file '%s'\n to \n'%s'!\n(%s)"),
                    g_basename(cur_filename),g_basename(new_filename),g_strerror(errno)); 
        msgbox = msg_box_new (_("Error..."),msg,MSG_ERROR,BUTTON_OK,0);
        g_free(msg);
        msg_box_hide_check_button(MSG_BOX(msgbox));
        msg_box_run(MSG_BOX(msgbox));
        gtk_widget_destroy(msgbox);

        Statusbar_Message(_("File(s) not renamed..."),TRUE);
    }
    if (tmp_filename) g_free(tmp_filename);
}

/*
 * Delete the file ETFile
 */
gint Delete_File (ET_File *ETFile, gboolean multiple_files)
{
    GtkWidget *msgbox = NULL;
    gchar *cur_filename;
    gchar *msg;
    gint button;
    gint stop_loop;

    if (!ETFile) return FALSE;

    // Filename of the file to delete
    cur_filename = ((File_Name *)(ETFile->FileNameCur)->data)->value;

    
    /*
     * Remove the file
     */
    if (CONFIRM_DELETE_FILE && !SF_HideMsgbox_Delete_File)
    {
        msg = g_strdup_printf(_("Do you really want to delete definitively the file\n'%s' ?"),g_basename(cur_filename) );
        if (multiple_files)
        {
            msgbox = msg_box_new (_("Delete File..."),msg,MSG_QUESTION,
                BUTTON_YES,BUTTON_NO,BUTTON_CANCEL,0);
        }else
        {
            msgbox = msg_box_new (_("Delete File..."),msg,MSG_QUESTION,
                BUTTON_YES,BUTTON_NO,0);
            msg_box_hide_check_button(MSG_BOX(msgbox));
        }
        g_free(msg);
        SF_ButtonPressed_Delete_File = button = msg_box_run(MSG_BOX(msgbox));
        if (MSG_BOX(msgbox)->check_button_state)
            SF_HideMsgbox_Delete_File = MSG_BOX(msgbox)->check_button_state;
        gtk_widget_destroy(msgbox);
    }else
    {
        if (SF_HideMsgbox_Delete_File)
            button = SF_ButtonPressed_Delete_File;
        else
            button = BUTTON_YES;
    }

    switch(button)
    {
        case BUTTON_YES:
            if (remove(cur_filename)==0)
            {
                msg = g_strdup_printf(_("File '%s' deleted"),g_basename(cur_filename));
                Statusbar_Message(msg,FALSE);
                g_free(msg);
                return 1;
            }
            break;
        case BUTTON_NO:
            break;
        case BUTTON_CANCEL:
        case -1:
            stop_loop = -1;
            return stop_loop;
            break;
    }

    return 0;
}

void Action_Select_Browser_Style (void)
{
    if (!ETCore->ETFileDisplayedList) return;

    /* Save the current displayed data */
    ET_Save_File_Data_From_UI(ETCore->ETFileDisplayed);
    
    Browser_Display_Tree_Or_Artist_Album_List();

    Update_Command_Buttons_Sensivity();
}





/*
 * Scans the specified directory: and load files into a list.
 * If the path doesn't exist, we free the previous loaded list of files.
 */
#include <sys/types.h>
#include <dirent.h>
#include <sys/stat.h>
#include <unistd.h>
#include <string.h>
void Read_Directory (gchar *path)
{
    DIR   *dir;
    gchar *msg, *tmp;
    guint  nbrfile = 0;
    GList *FileList = NULL;
    GList *etfilelist;


    ReadingDirectory = TRUE;    /* A flag to avoid to start an other reading */

    /* Initialize file list */
    ET_Core_Free();
    ET_Core_Initialize();
    Update_Command_Buttons_Sensivity();

    /* Initialize browser clist */
    Browser_List_Clear();
    
    /* Clear entry boxes  */
    Clear_File_Entry_Field();
    Clear_Header_Fields();
    Clear_Tag_Entry_Fields();
    gtk_label_set_text(GTK_LABEL(FileIndex),"0/0:");
    
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(TBShowBrowserButton),FALSE);
    Browser_Display_Tree_Or_Artist_Album_List(); // To show the corresponding lists...

    progressbar_index = 0;


    /* Placed only here, to empty the previous list of files */
    if (path == NULL) return;
    if ((dir=opendir(path)) == NULL)
    {
        GtkWidget *msgbox;
        gchar *msg;

        msg = g_strdup_printf(_("Can't read directory :\n'%s'\n(%s)"),path,g_strerror(errno));
        msgbox = msg_box_new(_("Error..."),msg,MSG_ERROR,BUTTON_OK,0);
        g_free(msg);
        msg_box_hide_check_button(MSG_BOX(msgbox));
        msg_box_run(MSG_BOX(msgbox));
        gtk_widget_destroy(msgbox);
        ReadingDirectory = FALSE; //Allow a new reading
        return;
    }
    closedir(dir);


    /* Open the window to quit recursion (only into recursion mode) */
    Set_Busy_Cursor();
    gtk_widget_set_sensitive(TBStopButton,BROWSE_SUBDIR);
    if (BROWSE_SUBDIR)
        Open_Quit_Recursion_Function_Window();


    /* Read the directory recursively */
    msg = g_strdup_printf(_("Search in progress..."));
    Statusbar_Message(msg,FALSE);
    g_free(msg);
    // Search the supported files
    FileList = Read_Directory_Recursively(FileList,path,BROWSE_SUBDIR);
    nbrfile = g_list_length(FileList);
    gtk_progress_configure(GTK_PROGRESS(ProgressBar),0,0,nbrfile);
    gtk_progress_set_show_text(GTK_PROGRESS(ProgressBar),TRUE);
    /* Format of the string that can be displayed in the progress bar:
     * %p - percentage
     * %v - value
     * %l - lower range value
     * %u - upper range value */
    gtk_progress_set_format_string(GTK_PROGRESS(ProgressBar),"%v/%u");
    // Load the supported files (Extension recognized)
    while (FileList)
    {
        gchar *filename = FileList->data;

        msg = g_strdup_printf(_("File: '%s'"),filename);
        Statusbar_Message(msg,FALSE);
        g_free(msg);

        ET_Add_File_To_File_List(filename);
        g_free(filename);

        // Update the progress bar
        gtk_progress_set_value(GTK_PROGRESS(ProgressBar),++progressbar_index);
        while (gtk_events_pending())
            gtk_main_iteration();

        if ( !FileList->next || (Main_Stop_Button_Pressed==1) ) break;
        FileList = FileList->next;
    }
    if (FileList) g_list_free(FileList);
    gtk_progress_set_show_text(GTK_PROGRESS(ProgressBar),FALSE);


    /* Close window to quit recursion */
    Destroy_Quit_Recursion_Function_Window();
    Main_Stop_Button_Pressed = 0;
    gtk_widget_set_sensitive(TBStopButton,FALSE);

    /* Update sensitivity of buttons and menus */
    Update_Command_Buttons_Sensivity();

    //ET_Debug_Print_File_List(ETCore->ETFileList,__FILE__,__LINE__,__FUNCTION__);

    if (ETCore->ETFileList)
    {
        /* Load the list of file into the browser clist widget */
        Browser_Display_Tree_Or_Artist_Album_List();
        
        /* Load the list attached to the TrackEntry */
        Load_Track_List_To_UI();
        
        /* Display the first file */
        etfilelist = ET_Displayed_File_List_First();
        if (etfilelist)
        {
            ET_Display_File_Data_To_UI((ET_File *)etfilelist->data);
            Browser_List_Select_File((ET_File *)etfilelist->data,FALSE);
        }

        /* Prepare message for the status bar */
        if (BROWSE_SUBDIR)
            msg = g_strdup_printf(_("Found %d file(s) in this directory and subdirectories."),ETCore->ETFileDisplayedList_Length);
        else
            msg = g_strdup_printf(_("Found %d file(s) in this directory."),ETCore->ETFileDisplayedList_Length);
    }else
    {
        /* Clear entry boxes */
        Clear_File_Entry_Field();
        Clear_Header_Fields();
        Clear_Tag_Entry_Fields();

        if (FileIndex)
            gtk_label_set_text(GTK_LABEL(FileIndex),"0/0:");

        tmp = g_strdup_printf(_("%u file(s)"),0); // See in ET_Display_Filename_To_UI
        Browser_Label_Set_Text(tmp);
        g_free(tmp);

        /* Prepare message for the status bar */
        if (BROWSE_SUBDIR)
            msg = g_strdup(_("No file found in this directory and subdirectories!"));
        else
            msg = g_strdup(_("No file found in this directory!"));
    }
    
    gtk_progress_set_value(GTK_PROGRESS(ProgressBar),0);
    Statusbar_Message(msg,FALSE);
    g_free(msg);
    Set_Unbusy_Cursor();
    ReadingDirectory = FALSE;
}



/*
 * Recurse the path to create a list of files. Return the number of files found
 */
GList *Read_Directory_Recursively (GList *file_list, gchar *path, gint recurse)
{
    DIR *dir;
    struct dirent *dirent;
    struct stat statbuf;
    gchar *filename;


    if ((dir = opendir(path)) != NULL)
    {
        while ((dirent = readdir(dir)) != NULL)
        {
            if (Main_Stop_Button_Pressed == 1)
                return file_list;

            if (dirent->d_name[0] != '.')
            {
                if (path[strlen(path)-1]!='/')
                    filename = g_strconcat(path,"/",dirent->d_name,NULL);
                else
                    filename = g_strconcat(path,dirent->d_name,NULL);

                if (lstat(filename, &statbuf) == -1)
                {
                    g_free(filename);
                    continue;
                }

                if (S_ISDIR(statbuf.st_mode))
                {
                    if (recurse)
                        file_list = Read_Directory_Recursively(file_list,filename,recurse);
                }else if (S_ISREG(statbuf.st_mode) && ET_File_Is_Supported(filename))
                {
                    file_list = g_list_append(file_list,g_strdup(filename));
                }
                g_free(filename);
                // Just to not block X events
                while (gtk_events_pending())
                    gtk_main_iteration();
            }
        }
        closedir(dir);
    }
    return file_list;
}


/*
 * Window with the 'STOP' button to stop recursion when reading directories
 */
#include "../pixmaps/stop.xpm"
void Open_Quit_Recursion_Function_Window (void)
{
    GtkWidget *button;
    GtkWidget *frame;
    gint mw_x, mw_y;
    gint ba_x, ba_y;


    if (QuitRecursionWindow != NULL) 
        return;
    QuitRecursionWindow = gtk_window_new(GTK_WINDOW_TOPLEVEL);
    gtk_window_set_title(GTK_WINDOW(QuitRecursionWindow),_("Searching..."));
    gtk_window_set_policy(GTK_WINDOW(QuitRecursionWindow),FALSE,FALSE,TRUE);
    gtk_window_set_transient_for(GTK_WINDOW(QuitRecursionWindow),GTK_WINDOW(MainWindow));
    //gtk_window_set_policy(GTK_WINDOW(QuitRecursionWindow),FALSE,FALSE,TRUE);
    gtk_signal_connect(GTK_OBJECT(QuitRecursionWindow),"destroy",
        (GtkSignalFunc)Quit_Recursion_Function_Button_Pressed,NULL);
    gtk_signal_connect(GTK_OBJECT(QuitRecursionWindow),"delete_event",
        (GtkSignalFunc)Quit_Recursion_Function_Button_Pressed,NULL);
    gtk_signal_connect(GTK_OBJECT(QuitRecursionWindow),"key_press_event",
        (GtkSignalFunc)Quit_Recursion_Window_Key_Press,NULL);

    frame = gtk_frame_new(NULL);
    gtk_frame_set_shadow_type(GTK_FRAME(frame),GTK_SHADOW_IN);
    gtk_container_add(GTK_CONTAINER(QuitRecursionWindow),frame);
    gtk_container_set_border_width(GTK_CONTAINER(frame),2);

    /* Button to stop... */
    //button = gtk_button_new_with_label(_("  STOP the search...  "));
    button = Create_Button_With_Icon_And_Label(stop_xpm,_("  STOP the search...  "));
    gtk_container_set_border_width(GTK_CONTAINER(button),8);
    gtk_container_add(GTK_CONTAINER(frame),button);
    gtk_signal_connect(GTK_OBJECT(button),"clicked",
        (GtkSignalFunc)Quit_Recursion_Function_Button_Pressed,NULL);

    gtk_widget_show_all(QuitRecursionWindow);

    /* Note : the window must be shown before to move it */
    gdk_window_get_position(MainWindow->window,&mw_x,&mw_y);
    gdk_window_get_position(BrowseArea->window,&ba_x,&ba_y);
    gtk_window_reposition(GTK_WINDOW(QuitRecursionWindow),
        mw_x + ba_x + (BrowseArea->allocation.width/2)  - (QuitRecursionWindow->allocation.width)/2,
        mw_y + ba_y + (BrowseArea->allocation.height/2) - (QuitRecursionWindow->allocation.height)/2);
}
void Destroy_Quit_Recursion_Function_Window (void)
{
    if (QuitRecursionWindow)
    {
        gtk_widget_destroy(QuitRecursionWindow);
        QuitRecursionWindow = (GtkWidget *)NULL;
        /*Statusbar_Message(_("Recursive file search interrupted."),FALSE);*/
    }
}
void Quit_Recursion_Function_Button_Pressed (void)
{
    Action_Main_Stop_Button_Pressed();
    Destroy_Quit_Recursion_Function_Window();
}
void Quit_Recursion_Window_Key_Press (GtkWidget *window, GdkEvent *event ATTRIBUTE_UNUSED)
{
    GdkEventKey *kevent;

    if (event && event->type == GDK_KEY_PRESS)
    {
        kevent = (GdkEventKey *)event;
        switch(kevent->keyval)
        {
            case GDK_Escape:    
                Destroy_Quit_Recursion_Function_Window();
                break;
        }
    }
}


/*
 * To stop the recursive search within directories or saving files
 */
void Action_Main_Stop_Button_Pressed (void)
{
    Main_Stop_Button_Pressed = 1;
    gtk_widget_set_sensitive(TBStopButton,FALSE);
}


/*
 * Update_Command_Buttons_Sensivity: Set to sensitive/unsensitive the state of each button into
 * the commands area and menu items in function of state of the "main list".
 */
void item_factory_set_sensitive (guint action, gboolean sensitive)
{
    GtkWidget *widget;

    widget = gtk_item_factory_get_widget_by_action(ItemFactory,action);
    if (widget) gtk_widget_set_sensitive(widget,sensitive);
}

void Update_Command_Buttons_Sensivity (void)
{
    if (!ETCore->ETFileDisplayedList)
    {
        /* No file found */
        
        /* Tool bar buttons */
        gtk_widget_set_sensitive(TBFirstButton,FALSE);
        gtk_widget_set_sensitive(TBPrevButton,FALSE);
        gtk_widget_set_sensitive(TBNextButton,FALSE);
        gtk_widget_set_sensitive(TBLastButton,FALSE);
        gtk_widget_set_sensitive(TBScanButton,FALSE);
        gtk_widget_set_sensitive(TBRemoveButton,FALSE);
        gtk_widget_set_sensitive(TBUndoButton,FALSE);
        gtk_widget_set_sensitive(TBRedoButton,FALSE);
        gtk_widget_set_sensitive(TBSaveButton,FALSE);
        //gtk_widget_set_sensitive(TBShowBrowserButton,FALSE);
        gtk_widget_set_sensitive(TBSelectAllButton,FALSE);
        gtk_widget_set_sensitive(TBInvertSelectionButton,FALSE);
        gtk_widget_set_sensitive(TBStopButton,FALSE);
        /* Scanner Window */
        if (SWScanButton)    gtk_widget_set_sensitive(SWScanButton,FALSE);
        /* Commands into menu */
        item_factory_set_sensitive(AM_BROWSE_FILE_WITH,FALSE);
        item_factory_set_sensitive(AM_SELECT_ALL_FILES,FALSE);
        item_factory_set_sensitive(AM_UNSELECT_ALL_FILES,FALSE);
        item_factory_set_sensitive(AM_INVERT_SELECTION,FALSE);
        item_factory_set_sensitive(AM_DELETE_FILE,FALSE);
        item_factory_set_sensitive(AM_SORT_ASCENDING_FILENAME,FALSE);
        item_factory_set_sensitive(AM_SORT_DESCENDING_FILENAME,FALSE);
        item_factory_set_sensitive(AM_SORT_ASCENDING_CREATION_DATE,FALSE);
        item_factory_set_sensitive(AM_SORT_DESCENDING_CREATION_DATE,FALSE);
        item_factory_set_sensitive(AM_SORT_ASCENDING_TRACK_NUMBER,FALSE);
        item_factory_set_sensitive(AM_SORT_DESCENDING_TRACK_NUMBER,FALSE);
        item_factory_set_sensitive(AM_SORT_ASCENDING_TITLE,FALSE);
        item_factory_set_sensitive(AM_SORT_DESCENDING_TITLE,FALSE);
        item_factory_set_sensitive(AM_SORT_ASCENDING_ARTIST,FALSE);
        item_factory_set_sensitive(AM_SORT_DESCENDING_ARTIST,FALSE);
        item_factory_set_sensitive(AM_SORT_ASCENDING_ALBUM,FALSE);
        item_factory_set_sensitive(AM_SORT_DESCENDING_ALBUM,FALSE);
        item_factory_set_sensitive(AM_SORT_ASCENDING_YEAR,FALSE);
        item_factory_set_sensitive(AM_SORT_DESCENDING_YEAR,FALSE);
        item_factory_set_sensitive(AM_SORT_ASCENDING_GENRE,FALSE);
        item_factory_set_sensitive(AM_SORT_DESCENDING_GENRE,FALSE);
        item_factory_set_sensitive(AM_SORT_ASCENDING_COMMENT,FALSE);
        item_factory_set_sensitive(AM_SORT_DESCENDING_COMMENT,FALSE);
        item_factory_set_sensitive(AM_SORT_ASCENDING_FILE_TYPE,FALSE);
        item_factory_set_sensitive(AM_SORT_DESCENDING_FILE_TYPE,FALSE);
        item_factory_set_sensitive(AM_SORT_ASCENDING_FILE_SIZE,FALSE);
        item_factory_set_sensitive(AM_SORT_DESCENDING_FILE_SIZE,FALSE);
        item_factory_set_sensitive(AM_SORT_ASCENDING_FILE_DURATION,FALSE);
        item_factory_set_sensitive(AM_SORT_DESCENDING_FILE_DURATION,FALSE);
        item_factory_set_sensitive(AM_SORT_ASCENDING_FILE_BITRATE,FALSE);
        item_factory_set_sensitive(AM_SORT_DESCENDING_FILE_BITRATE,FALSE);
        item_factory_set_sensitive(AM_SORT_ASCENDING_FILE_SAMPLERATE,FALSE);
        item_factory_set_sensitive(AM_SORT_DESCENDING_FILE_SAMPLERATE,FALSE);
        item_factory_set_sensitive(AM_PREV,FALSE);
        item_factory_set_sensitive(AM_NEXT,FALSE);
        item_factory_set_sensitive(AM_FIRST,FALSE);
        item_factory_set_sensitive(AM_LAST,FALSE);
        item_factory_set_sensitive(AM_SCAN,FALSE);
        item_factory_set_sensitive(AM_REMOVE,FALSE);
        item_factory_set_sensitive(AM_UNDO,FALSE);
        item_factory_set_sensitive(AM_REDO,FALSE);
        item_factory_set_sensitive(AM_SAVE,FALSE);
        item_factory_set_sensitive(AM_UNDO_HISTORY,FALSE);
        item_factory_set_sensitive(AM_REDO_HISTORY,FALSE);
        item_factory_set_sensitive(AM_SEARCH_FILE,FALSE);
        item_factory_set_sensitive(AM_FILENAME_FROM_TXT,FALSE);
        item_factory_set_sensitive(AM_WRITE_PLAYLIST,FALSE);
        item_factory_set_sensitive(AM_RUN_AUDIO_PLAYER,FALSE);
        /* File and Tag frames */
        gtk_widget_set_sensitive(GTK_BIN(FileArea)->child,FALSE);
        gtk_widget_set_sensitive(GTK_BIN(TagArea)->child,FALSE);
        return;
    }else
    {
        GList *etfilelist = NULL;
        gboolean has_undo = FALSE;
        gboolean has_redo = FALSE;
        //gboolean has_to_save = FALSE;
        
        /* File and Tag frames */
        gtk_widget_set_sensitive(GTK_BIN(FileArea)->child,TRUE);
        gtk_widget_set_sensitive(GTK_BIN(TagArea)->child,TRUE);
        /* Tool bar buttons */
        gtk_widget_set_sensitive(TBScanButton,TRUE);
        gtk_widget_set_sensitive(TBRemoveButton,TRUE);
        //gtk_widget_set_sensitive(TBShowBrowserButton,TRUE);
        gtk_widget_set_sensitive(TBSelectAllButton,TRUE);
        gtk_widget_set_sensitive(TBInvertSelectionButton,TRUE);
        gtk_widget_set_sensitive(TBStopButton,FALSE);
        /* Scanner Window */
        if (SWScanButton)    gtk_widget_set_sensitive(SWScanButton,TRUE);
        /* Commands into menu */
        item_factory_set_sensitive(AM_BROWSE_FILE_WITH,TRUE);
        item_factory_set_sensitive(AM_SELECT_ALL_FILES,TRUE);
        item_factory_set_sensitive(AM_UNSELECT_ALL_FILES,TRUE);
        item_factory_set_sensitive(AM_INVERT_SELECTION,TRUE);
        item_factory_set_sensitive(AM_DELETE_FILE,TRUE);
        item_factory_set_sensitive(AM_SORT_ASCENDING_FILENAME,TRUE);
        item_factory_set_sensitive(AM_SORT_DESCENDING_FILENAME,TRUE);
        item_factory_set_sensitive(AM_SORT_ASCENDING_CREATION_DATE,TRUE);
        item_factory_set_sensitive(AM_SORT_DESCENDING_CREATION_DATE,TRUE);
        item_factory_set_sensitive(AM_SORT_ASCENDING_TRACK_NUMBER,TRUE);
        item_factory_set_sensitive(AM_SORT_DESCENDING_TRACK_NUMBER,TRUE);
        item_factory_set_sensitive(AM_SORT_ASCENDING_TITLE,TRUE);
        item_factory_set_sensitive(AM_SORT_DESCENDING_TITLE,TRUE);
        item_factory_set_sensitive(AM_SORT_ASCENDING_ARTIST,TRUE);
        item_factory_set_sensitive(AM_SORT_DESCENDING_ARTIST,TRUE);
        item_factory_set_sensitive(AM_SORT_ASCENDING_ALBUM,TRUE);
        item_factory_set_sensitive(AM_SORT_DESCENDING_ALBUM,TRUE);
        item_factory_set_sensitive(AM_SORT_ASCENDING_YEAR,TRUE);
        item_factory_set_sensitive(AM_SORT_DESCENDING_YEAR,TRUE);
        item_factory_set_sensitive(AM_SORT_ASCENDING_GENRE,TRUE);
        item_factory_set_sensitive(AM_SORT_DESCENDING_GENRE,TRUE);
        item_factory_set_sensitive(AM_SORT_ASCENDING_COMMENT,TRUE);
        item_factory_set_sensitive(AM_SORT_DESCENDING_COMMENT,TRUE);
        item_factory_set_sensitive(AM_SORT_ASCENDING_FILE_TYPE,TRUE);
        item_factory_set_sensitive(AM_SORT_DESCENDING_FILE_TYPE,TRUE);
        item_factory_set_sensitive(AM_SORT_ASCENDING_FILE_SIZE,TRUE);
        item_factory_set_sensitive(AM_SORT_DESCENDING_FILE_SIZE,TRUE);
        item_factory_set_sensitive(AM_SORT_ASCENDING_FILE_DURATION,TRUE);
        item_factory_set_sensitive(AM_SORT_DESCENDING_FILE_DURATION,TRUE);
        item_factory_set_sensitive(AM_SORT_ASCENDING_FILE_BITRATE,TRUE);
        item_factory_set_sensitive(AM_SORT_DESCENDING_FILE_BITRATE,TRUE);
        item_factory_set_sensitive(AM_SORT_ASCENDING_FILE_SAMPLERATE,TRUE);
        item_factory_set_sensitive(AM_SORT_DESCENDING_FILE_SAMPLERATE,TRUE);
        item_factory_set_sensitive(AM_SCAN,TRUE);
        item_factory_set_sensitive(AM_REMOVE,TRUE);
        item_factory_set_sensitive(AM_SEARCH_FILE,TRUE);
        item_factory_set_sensitive(AM_FILENAME_FROM_TXT,TRUE);
        item_factory_set_sensitive(AM_WRITE_PLAYLIST,TRUE);
        item_factory_set_sensitive(AM_RUN_AUDIO_PLAYER,TRUE);

        /* Check if one of the selected files has undo or redo data */
        etfilelist = ET_File_List_Get_Selection();
        while (etfilelist)
        {
            has_undo    |= ET_File_Data_Has_Undo_Data((ET_File *)etfilelist->data);
            has_redo    |= ET_File_Data_Has_Redo_Data((ET_File *)etfilelist->data);
            //has_to_save |= ET_Check_If_File_Is_Saved((ET_File *)etfilelist->data);
            if (has_undo && has_redo) // Useless to check the other files
                break;
            etfilelist = g_list_next(etfilelist);
        }
        /* Enable undo commands if there are undo data */
        if (has_undo)
        {
            gtk_widget_set_sensitive(TBUndoButton,TRUE);
            item_factory_set_sensitive(AM_UNDO,TRUE);
        }else
        {
            gtk_widget_set_sensitive(TBUndoButton,FALSE);
            item_factory_set_sensitive(AM_UNDO,FALSE);
        }
        /* Enable redo commands if there are redo data */
        if (has_redo)
        {
            gtk_widget_set_sensitive(TBRedoButton,TRUE);
            item_factory_set_sensitive(AM_REDO,TRUE);
        }else
        {
            gtk_widget_set_sensitive(TBRedoButton,FALSE);
            item_factory_set_sensitive(AM_REDO,FALSE);
        }

        /* Enable save file command if file has been changed */
        // Desactivated because problem with only one file in the list
        //if (has_to_save)
        //{
        //    gtk_widget_set_sensitive(TBSaveButton,FALSE);
        //    item_factory_set_sensitive(AM_SAVE,FALSE);
        //}else
        //{
            gtk_widget_set_sensitive(TBSaveButton,TRUE);
            item_factory_set_sensitive(AM_SAVE,TRUE);
        //}

        /* Enable undo command if there are data into main undo list (history list) */
        if (ET_History_File_List_Has_Undo_Data())
            item_factory_set_sensitive(AM_UNDO_HISTORY,TRUE);
        else
            item_factory_set_sensitive(AM_UNDO_HISTORY,FALSE);
        /* Enable redo commands if there are data into main redo list (history list) */
        if (ET_History_File_List_Has_Redo_Data())
            item_factory_set_sensitive(AM_REDO_HISTORY,TRUE);
        else
            item_factory_set_sensitive(AM_REDO_HISTORY,FALSE);
    }

    if (!ETCore->ETFileDisplayedList->prev)    /* Is it the 1rst item ? */
    {
        gtk_widget_set_sensitive(TBFirstButton,FALSE);
        gtk_widget_set_sensitive(TBPrevButton,FALSE);
        item_factory_set_sensitive(AM_PREV,FALSE);
        item_factory_set_sensitive(AM_FIRST,FALSE);
    }else
    {
        gtk_widget_set_sensitive(TBFirstButton,TRUE);
        gtk_widget_set_sensitive(TBPrevButton,TRUE);
        item_factory_set_sensitive(AM_PREV,TRUE);
        item_factory_set_sensitive(AM_FIRST,TRUE);
    }
    if (!ETCore->ETFileDisplayedList->next)    /* Is it the last item ? */
    {
        gtk_widget_set_sensitive(TBNextButton,FALSE);
        gtk_widget_set_sensitive(TBLastButton,FALSE);
        item_factory_set_sensitive(AM_NEXT,FALSE);
        item_factory_set_sensitive(AM_LAST,FALSE);
    }else
    {
        gtk_widget_set_sensitive(TBNextButton,TRUE);
        gtk_widget_set_sensitive(TBLastButton,TRUE);
        item_factory_set_sensitive(AM_NEXT,TRUE);
        item_factory_set_sensitive(AM_LAST,TRUE);
    }
}

/*
 * Just to disable buttons when we are saving files (do not disable Quit button)
 */
void Disable_Command_Buttons (void)
{
    /* Scanner Window */
    if (SWScanButton)    gtk_widget_set_sensitive(SWScanButton,FALSE);
    /* Tool bar buttons */
    gtk_widget_set_sensitive(TBFirstButton,FALSE);
    gtk_widget_set_sensitive(TBPrevButton,FALSE);
    gtk_widget_set_sensitive(TBNextButton,FALSE);
    gtk_widget_set_sensitive(TBLastButton,FALSE);
    gtk_widget_set_sensitive(TBScanButton,FALSE);
    gtk_widget_set_sensitive(TBRemoveButton,FALSE);
    gtk_widget_set_sensitive(TBUndoButton,FALSE);
    gtk_widget_set_sensitive(TBRedoButton,FALSE);
    gtk_widget_set_sensitive(TBSaveButton,FALSE);
    //gtk_widget_set_sensitive(TBShowBrowserButton,FALSE);
    gtk_widget_set_sensitive(TBSelectAllButton,FALSE);
    gtk_widget_set_sensitive(TBInvertSelectionButton,FALSE);
    /* Commands into menu */
    item_factory_set_sensitive(AM_BROWSE_FILE_WITH,FALSE);
    item_factory_set_sensitive(AM_SELECT_ALL_FILES,FALSE);
    item_factory_set_sensitive(AM_UNSELECT_ALL_FILES,FALSE);
    item_factory_set_sensitive(AM_INVERT_SELECTION,FALSE);
    item_factory_set_sensitive(AM_DELETE_FILE,FALSE);
    item_factory_set_sensitive(AM_FIRST,FALSE);
    item_factory_set_sensitive(AM_PREV,FALSE);
    item_factory_set_sensitive(AM_NEXT,FALSE);
    item_factory_set_sensitive(AM_LAST,FALSE);
    item_factory_set_sensitive(AM_SCAN,FALSE);
    item_factory_set_sensitive(AM_REMOVE,FALSE);
    item_factory_set_sensitive(AM_UNDO,FALSE);
    item_factory_set_sensitive(AM_REDO,FALSE);
    item_factory_set_sensitive(AM_SAVE,FALSE);
    item_factory_set_sensitive(AM_UNDO_HISTORY,FALSE);
    item_factory_set_sensitive(AM_REDO_HISTORY,FALSE);
}

/*
 * Disable (FALSE) / Enable (TRUE) all user widgets in the tag area
 */
void Tag_Area_Set_Sensitive (gboolean activate)
{
    /* TAG Area */
    gtk_widget_set_sensitive(GTK_WIDGET(TitleEntry),            activate);
    gtk_widget_set_sensitive(GTK_WIDGET(ArtistEntry),           activate);
    gtk_widget_set_sensitive(GTK_WIDGET(AlbumEntry),            activate);
    gtk_widget_set_sensitive(GTK_WIDGET(YearEntry),             activate);
    gtk_widget_set_sensitive(GTK_WIDGET(TrackEntry),            activate);
    gtk_widget_set_sensitive(GTK_WIDGET(TrackTotalEntry),       activate);
    gtk_widget_set_sensitive(GTK_WIDGET(CommentEntry),          activate);
    gtk_widget_set_sensitive(GTK_WIDGET(GenreEntry),            activate);
    /* Mini buttons */
    gtk_widget_set_sensitive(GTK_WIDGET(TitleMButton),          activate);
    gtk_widget_set_sensitive(GTK_WIDGET(ArtistMButton),         activate);
    gtk_widget_set_sensitive(GTK_WIDGET(AlbumMButton),          activate);
    gtk_widget_set_sensitive(GTK_WIDGET(YearMButton),           activate);
    gtk_widget_set_sensitive(GTK_WIDGET(TrackMButton),          activate);
    gtk_widget_set_sensitive(GTK_WIDGET(TrackMButtonSequence),  activate);
    gtk_widget_set_sensitive(GTK_WIDGET(TrackMButtonNbrFiles),  activate);
    gtk_widget_set_sensitive(GTK_WIDGET(CommentMButton),        activate);
    gtk_widget_set_sensitive(GTK_WIDGET(GenreMButton),          activate);
}

/*
 * Disable (FALSE) / Enable (TRUE) all user widgets in the file area
 */
void File_Area_Set_Sensitive (gboolean activate)
{
    gtk_widget_set_sensitive(GTK_WIDGET(FileEntry),activate);
}

/*
 * Clear the entries of tag area
 */
void Clear_Tag_Entry_Fields (void)
{
    if (!TitleEntry) return;

    gtk_entry_set_text(GTK_ENTRY(TitleEntry),"");
    gtk_entry_set_text(GTK_ENTRY(ArtistEntry),"");
    gtk_entry_set_text(GTK_ENTRY(AlbumEntry),"");
    gtk_entry_set_text(GTK_ENTRY(YearEntry),"");
    gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(TrackEntry)->entry),"");
    gtk_entry_set_text(GTK_ENTRY(TrackTotalEntry),"");
    gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(GenreEntry)->entry),"");
    gtk_entry_set_text(GTK_ENTRY(CommentEntry),"");
}


/*
 * Clear the entry of file area
 */
void Clear_File_Entry_Field (void)
{
    if (!FileEntry) return;

    gtk_entry_set_text(GTK_ENTRY(FileEntry),"");
}


/*
 * Clear the header informations
 */
void Clear_Header_Fields (void)
{
    if (!VersionValueLabel) return;

    /* Default values are MPs data */
    gtk_label_set_text(GTK_LABEL(VersionLabel),        _("MPEG"));
    gtk_label_set_text(GTK_LABEL(VersionValueLabel),   _("?, Layer ?"));
    gtk_label_set_text(GTK_LABEL(BitrateValueLabel),   _("? kb/s"));
    gtk_label_set_text(GTK_LABEL(SampleRateValueLabel),_("? Hz"));
    gtk_label_set_text(GTK_LABEL(ModeLabel),           _("Mode:"));
    gtk_label_set_text(GTK_LABEL(ModeValueLabel),      _("?"));
    gtk_label_set_text(GTK_LABEL(SizeValueLabel),      _("?"));
    gtk_label_set_text(GTK_LABEL(DurationValueLabel),  _("?"));
}




/*
 * Load the default directory when the user interface is completely displayed
 * to avoid bad visualization effect at startup.
 */
void Init_Load_Default_Dir (void)
{
    //ETCore->ETFileList = (GList *)NULL;
    ET_Core_Free();
    ET_Core_Initialize();

    // Open the scanner window
    if (OPEN_SCANNER_WINDOW_ON_STARTUP)
        Open_ScannerWindow(SCANNER_FILL_TAG);

    Statusbar_Message(_("Select a directory to browse!"),FALSE);
    if (INIT_DIRECTORY)
    {
        Browser_Tree_Select_Dir(INIT_DIRECTORY);
    }else
    {
        Browser_Load_Default_Directory();
    }

    gtk_idle_remove(idle_handler_id);
}


/*
 * Attach event to the entry to select previous or next file when pression PageUp and PageDown keys.
 */
void Tag_Area_Entry_Key_Press (GtkWidget *widget, GdkEvent *event)
{
    GdkEventKey *kevent;

    if (event && event->type == GDK_KEY_PRESS)
    {
        kevent = (GdkEventKey *)event;
        switch(kevent->keyval)
        {
            case GDK_Page_Up:    
                Action_Select_Prev_File();
                break;

            case GDK_Page_Down:    
                Action_Select_Next_File();
                break;
        }
    }
}




void Convert_P20_And_Undescore_Into_Spaces (GtkWidget *entry)
{
    gchar *text = gtk_entry_get_text_1(entry);
    Scan_Convert_Underscore_Into_Space(text);
    Scan_Convert_P20_Into_Space(text);
    gtk_entry_set_text_1(entry,text);
}
void Convert_Space_Into_Undescore (GtkWidget *entry)
{
    gchar *text = gtk_entry_get_text_1(entry);
    Scan_Convert_Space_Into_Undescore(text);
    gtk_entry_set_text_1(entry,text);
}
void Convert_All_Uppercase (GtkWidget *entry)
{
    gchar *text = gtk_entry_get_text_1(entry);
    //g_print("s: %d\n",GTK_EDITABLE(GTK_ENTRY(entry))->selection_start_pos);
    //g_print("e: %d\n",GTK_EDITABLE(GTK_ENTRY(entry))->selection_end_pos);
    Scan_Process_Fields_All_Uppercase(text);
    gtk_entry_set_text_1(entry,text);
}
void Convert_All_Downcase (GtkWidget *entry)
{
    gchar *text = gtk_entry_get_text_1(entry);
    Scan_Process_Fields_All_Downcase(text);
    gtk_entry_set_text_1(entry,text);
}
void Convert_Letter_Uppercase (GtkWidget *entry)
{
    gchar *text = gtk_entry_get_text_1(entry);
    Scan_Process_Fields_Letter_Uppercase(text);
    gtk_entry_set_text_1(entry,text);
}
void Convert_First_Letters_Uppercase (GtkWidget *entry)
{
    gchar *text = gtk_entry_get_text_1(entry);
    Scan_Process_Fields_First_Letters_Uppercase(text);
    gtk_entry_set_text_1(entry,text);
}
void Convert_Remove_Space (GtkWidget *entry)
{
    gchar *text = gtk_entry_get_text_1(entry);
    Scan_Process_Fields_Remove_Space(text);
    gtk_entry_set_text_1(entry,text);
}
void Convert_Insert_Space (GtkWidget *entry)
{
    /* Memory must be allocated cause size of string may increase */
    guint string_len = 2 * strlen(gtk_entry_get_text_1(entry));
    gchar *text = g_malloc(string_len+1);
    strncpy(text,gtk_entry_get_text_1(entry),string_len);
    Scan_Process_Fields_Insert_Space(text);
    gtk_entry_set_text_1(entry,text);
    g_free(text);
}
void Convert_Only_One_Space (GtkWidget *entry)
{
    gchar *text = gtk_entry_get_text_1(entry);
    Scan_Process_Fields_Keep_One_Space(text);
    gtk_entry_set_text_1(entry,text);
}

/*
 * Entry_Popup_Menu_Handler: show the popup menu when the third mouse button is pressed.
 */
gboolean Entry_Popup_Menu_Handler (GtkMenu *menu, GdkEventButton *event)
{
    if (event && (event->type==GDK_BUTTON_PRESS) && (event->button==3))
    {
        /* FIX ME : this is not very clean, but if we use 'event->button' (contains value of
         * the 3rd button) instead of '1', we need to click two times the left mouse button
         * to activate an item of the opened popup menu (when menu is attached to an entry). */
        //gtk_menu_popup(menu,NULL,NULL,NULL,NULL,event->button,event->time);
        gtk_menu_popup(menu,NULL,NULL,NULL,NULL,1,event->time);
        return TRUE;
    }
    return FALSE;
}

/*
 * Popup menu attached to all entries of tag + filename + rename combobox.
 * Displayed when pressing the right mouse button and contains functions to process ths strings.
 */
void Attach_Popup_Menu_To_Tag_Entries (GtkEntry *entry)
{
    GtkWidget *PopupMenu;
    GtkWidget *MenuItem;


    PopupMenu = gtk_menu_new();
    gtk_signal_connect_object(GTK_OBJECT(entry),"button_press_event",
        (GtkSignalFunc)Entry_Popup_Menu_Handler,GTK_OBJECT(PopupMenu));

    /* Menu items */
    MenuItem = gtk_menu_item_new_with_label(_("Tag selected files with this field"));
    gtk_menu_append(GTK_MENU(PopupMenu),MenuItem);
    gtk_signal_connect_object(GTK_OBJECT(MenuItem),"activate",
        (GtkSignalFunc)Menu_Mini_Button_Clicked,GTK_OBJECT(entry));

    /* Separator */
    MenuItem = gtk_menu_item_new();
    gtk_menu_append(GTK_MENU(PopupMenu),MenuItem);

    MenuItem = gtk_menu_item_new_with_label(_("Convert '_' and '%20' to spaces"));
    gtk_menu_append(GTK_MENU(PopupMenu),MenuItem);
    gtk_signal_connect_object(GTK_OBJECT(MenuItem),"activate",
        (GtkSignalFunc)Convert_P20_And_Undescore_Into_Spaces,GTK_OBJECT(entry));

    MenuItem = gtk_menu_item_new_with_label(_("Convert ' ' to '_'"));
    gtk_menu_append(GTK_MENU(PopupMenu),MenuItem);
    gtk_signal_connect_object(GTK_OBJECT(MenuItem),"activate",
        (GtkSignalFunc)Convert_Space_Into_Undescore,GTK_OBJECT(entry));

    /* Separator */
    MenuItem = gtk_menu_item_new();
    gtk_menu_append(GTK_MENU(PopupMenu),MenuItem);

    MenuItem = gtk_menu_item_new_with_label(_("All uppercase"));
    gtk_menu_append(GTK_MENU(PopupMenu),MenuItem);
    gtk_signal_connect_object(GTK_OBJECT(MenuItem),"activate",
        (GtkSignalFunc)Convert_All_Uppercase,GTK_OBJECT(entry));

    MenuItem = gtk_menu_item_new_with_label(_("All downcase"));
    gtk_menu_append(GTK_MENU(PopupMenu),MenuItem);
    gtk_signal_connect_object(GTK_OBJECT(MenuItem),"activate",
        (GtkSignalFunc)Convert_All_Downcase,GTK_OBJECT(entry));

    MenuItem = gtk_menu_item_new_with_label(_("First letter uppercase"));
    gtk_menu_append(GTK_MENU(PopupMenu),MenuItem);
    gtk_signal_connect_object(GTK_OBJECT(MenuItem),"activate",
        (GtkSignalFunc)Convert_Letter_Uppercase,GTK_OBJECT(entry));

    MenuItem = gtk_menu_item_new_with_label(_("First letter uppercase of each word"));
    gtk_menu_append(GTK_MENU(PopupMenu),MenuItem);
    gtk_signal_connect_object(GTK_OBJECT(MenuItem),"activate",
        (GtkSignalFunc)Convert_First_Letters_Uppercase,GTK_OBJECT(entry));

    /* Separator */
    MenuItem = gtk_menu_item_new();
    gtk_menu_append(GTK_MENU(PopupMenu),MenuItem);

    MenuItem = gtk_menu_item_new_with_label(_("Remove spaces"));
    gtk_menu_append(GTK_MENU(PopupMenu),MenuItem);
    gtk_signal_connect_object(GTK_OBJECT(MenuItem),"activate",
        (GtkSignalFunc)Convert_Remove_Space,GTK_OBJECT(entry));

    MenuItem = gtk_menu_item_new_with_label(_("Insert space before uppercase letter"));
    gtk_menu_append(GTK_MENU(PopupMenu),MenuItem);
    gtk_signal_connect_object(GTK_OBJECT(MenuItem),"activate",
        (GtkSignalFunc)Convert_Insert_Space,GTK_OBJECT(entry));

    MenuItem = gtk_menu_item_new_with_label(_("Remove duplicate spaces or underscores"));
    gtk_menu_append(GTK_MENU(PopupMenu),MenuItem);
    gtk_signal_connect_object(GTK_OBJECT(MenuItem),"activate",
        (GtkSignalFunc)Convert_Only_One_Space,GTK_OBJECT(entry));

    gtk_widget_show_all(PopupMenu);
}



/*
 * Function to manage the received signals (specially for segfaults)
 * Handle crashs
 */
void Handle_Crash (gint signal_id)
{
    //gchar commmand[256];

    g_print(_("EasyTAG: Abnormal exit!.\n"));
    g_print(_("Received signal %s (%d)\a\n"),signal_to_string(signal_id),signal_id);

    g_print(_("You have probably found a bug in EasyTAG. Please, send a bug report "
              "with a gdb backtrace ('gdb easytag core' then 'bt') to j.couderc@ifrance.com\n"));

    signal(signal_id,SIG_DFL); // Let the OS handle recursive seg faults
    //signal(SIGTSTP, exit);
    //snprintf(commmand,sizeof(commmand),"gdb -x /root/core.txt easytag %d", getpid());
    //system(commmand);
}

gchar *signal_to_string (gint signal)
{
#ifdef SIGHUP
    if (signal == SIGHUP)     return ("SIGHUP");
#endif
#ifdef SIGINT
    if (signal == SIGINT)     return ("SIGINT");
#endif
#ifdef SIGQUIT
    if (signal == SIGQUIT)    return ("SIGQUIT");
#endif
#ifdef SIGILL
    if (signal == SIGILL)     return ("SIGILL");
#endif
#ifdef SIGTRAP
    if (signal == SIGTRAP)    return ("SIGTRAP");
#endif
#ifdef SIGABRT
    if (signal == SIGABRT)    return ("SIGABRT");
#endif
#ifdef SIGIOT
    if (signal == SIGIOT)     return ("SIGIOT");
#endif
#ifdef SIGEMT
    if (signal == SIGEMT)     return ("SIGEMT");
#endif
#ifdef SIGFPE
    if (signal == SIGFPE)     return ("SIGFPE");
#endif
#ifdef SIGKILL
    if (signal == SIGKILL)    return ("SIGKILL");
#endif
#ifdef SIGBUS
    if (signal == SIGBUS)     return ("SIGBUS");
#endif
#ifdef SIGSEGV
    if (signal == SIGSEGV)    return ("SIGSEGV");
#endif
#ifdef SIGSYS
    if (signal == SIGSYS)     return ("SIGSYS");
#endif
#ifdef SIGPIPE
    if (signal == SIGPIPE)    return ("SIGPIPE");
#endif
#ifdef SIGALRM
    if (signal == SIGALRM)    return ("SIGALRM");
#endif
#ifdef SIGTERM
    if (signal == SIGTERM)    return ("SIGTERM");
#endif
#ifdef SIGUSR1
    if (signal == SIGUSR1)    return ("SIGUSR1");
#endif
#ifdef SIGUSR2
    if (signal == SIGUSR2)    return ("SIGUSR2");
#endif
#ifdef SIGCHLD
    if (signal == SIGCHLD)    return ("SIGCHLD");
#endif
#ifdef SIGCLD
    if (signal == SIGCLD)     return ("SIGCLD");
#endif
#ifdef SIGPWR
    if (signal == SIGPWR)     return ("SIGPWR");
#endif
#ifdef SIGVTALRM
    if (signal == SIGVTALRM)  return ("SIGVTALRM");
#endif
#ifdef SIGPROF
    if (signal == SIGPROF)    return ("SIGPROF");
#endif
#ifdef SIGIO
    if (signal == SIGIO)      return ("SIGIO");
#endif
#ifdef SIGPOLL
    if (signal == SIGPOLL)    return ("SIGPOLL");
#endif
#ifdef SIGWINCH
    if (signal == SIGWINCH)   return ("SIGWINCH");
#endif
#ifdef SIGWINDOW
    if (signal == SIGWINDOW)  return ("SIGWINDOW");
#endif
#ifdef SIGSTOP
    if (signal == SIGSTOP)    return ("SIGSTOP");
#endif
#ifdef SIGTSTP
    if (signal == SIGTSTP)    return ("SIGTSTP");
#endif
#ifdef SIGCONT
    if (signal == SIGCONT)    return ("SIGCONT");
#endif
#ifdef SIGTTIN
    if (signal == SIGTTIN)    return ("SIGTTIN");
#endif
#ifdef SIGTTOU
    if (signal == SIGTTOU)    return ("SIGTTOU");
#endif
#ifdef SIGURG
    if (signal == SIGURG)     return ("SIGURG");
#endif
#ifdef SIGLOST
    if (signal == SIGLOST)    return ("SIGLOST");
#endif
#ifdef SIGRESERVE
    if (signal == SIGRESERVE) return ("SIGRESERVE");
#endif
#ifdef SIGDIL
    if (signal == SIGDIL)     return ("SIGDIL");
#endif
#ifdef SIGXCPU
    if (signal == SIGXCPU)    return ("SIGXCPU");
#endif
#ifdef SIGXFSZ
    if (signal == SIGXFSZ)    return ("SIGXFSZ");
#endif
    return (_("Unknown signal"));
}


/*
 * Display usage informations
 */
void Display_Usage (void)
{
    fprintf(stderr,
        _("\nUsage: easytag [option] "
          "\n   or: easytag [directory]\n"
          "\n"
          "Option:\n"
          "-------\n"
          "-h, --help        Display this text and exit.\n"
          "-v, --version     Print basic informations and exit.\n"
          "\n"
          "Directory:\n"
          "----------\n"
          "/path_to/files    Use an absolute path to load,\n"
          "., ./             Load the current directory.\n"
          "\n"));
    exit(0);
}



/*
 * Exit the program
 */
void EasyTAG_Exit (void)
{
    ET_Core_Destroy();
    g_print(_("EasyTAG: Normal exit.\n"));
    gtk_main_quit();
    exit(0);
}

void Quit_MainWindow_Confirmed (void)
{
    /* Save config if wanted */
    if (SAVE_CONFIG_BEFORE_EXIT)
    {
        Apply_Changes(); /* Can be replaced by Save_Changes() */
        Save_Config_To_File();
    }
    EasyTAG_Exit();
}

void Quit_MainWindow_Save_And_Quit (void)
{
    /* Save modified tags */
    if (Save_All_Files_With_Answer() == -1) return;
    Quit_MainWindow_Confirmed();
}

void Quit_MainWindow (void)
{
    GtkWidget *msgbox;
    gint button;
    GList *list;


    /* If you change the displayed data and quit immediately */
    if (ETCore->ETFileList)
        ET_Save_File_Data_From_UI(ETCore->ETFileDisplayed); // To detect change before exiting

    /* Save combobox history lists before exit */
    list = gtk_object_get_data(GTK_OBJECT(BrowserEntry),"History");
    Save_Path_Entry_List(list);

    /* Exit ? */
    if(CONFIRM_BEFORE_EXIT) 
    {
        if (ET_Check_If_All_Files_Are_Saved() != TRUE)
        {
            /* Some files haven't been saved */
            msgbox = msg_box_new (_("Confirm..."),_("Some files have been modified but not "
                "saved...\nDo you want to save them before exiting the program?"),
                MSG_QUESTION,BUTTON_YES,BUTTON_NO,BUTTON_CANCEL,0);
            msg_box_hide_check_button(MSG_BOX(msgbox));
            button = msg_box_run(MSG_BOX(msgbox));
            gtk_widget_destroy(msgbox);
            switch(button)
            {
                case BUTTON_YES:
                    Quit_MainWindow_Save_And_Quit();
                    break;
                case BUTTON_NO:
                    Quit_MainWindow_Confirmed();
                    break;
                case BUTTON_CANCEL:
                case -1:
                    return;
            }
        } else
        {
            msgbox = msg_box_new (_("Confirm..."),_(" Do you really want to exit the program? "),
                MSG_QUESTION,BUTTON_YES,BUTTON_NO,0);
            msg_box_hide_check_button(MSG_BOX(msgbox));
            button = msg_box_run(MSG_BOX(msgbox));
            gtk_widget_destroy(msgbox);
            switch(button)
            {
                case BUTTON_YES:
                    Quit_MainWindow_Confirmed();
                    break;
                case BUTTON_NO:
                case -1:
                    return;
                    break;
            }
        }
    }else if (ET_Check_If_All_Files_Are_Saved() != TRUE)
    {
        /* Some files aren't saved */
        msgbox = msg_box_new (_("Confirm..."),_("Some files have been modified but not "
            "saved...\nDo you want to save them before exiting the program?"),
            MSG_QUESTION,BUTTON_YES,BUTTON_NO,BUTTON_CANCEL,0);
        msg_box_hide_check_button(MSG_BOX(msgbox));
        button = msg_box_run(MSG_BOX(msgbox));
        gtk_widget_destroy(msgbox);
        switch(button)
        {
            case BUTTON_YES:
                Quit_MainWindow_Save_And_Quit();
                break;
            case BUTTON_NO:
                Quit_MainWindow_Confirmed();
                break;
            case BUTTON_CANCEL:
            case -1:
                return;
        }
    }else
    {
        Quit_MainWindow_Confirmed();
    }
}
