/* scan.c - 2000/06/16 */
/*
 *  EasyTAG - Tag editor for MP3 and Ogg Vorbis files
 *  Copyright (C) 2000-2003  Jerome Couderc <j.couderc@ifrance.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


#include <gtk/gtk.h>
#include <ctype.h>
#include <string.h>
#include <stdlib.h>
#include <gdk/gdkkeysyms.h>

#include "scan.h"
#include "easytag.h"
#include "prefs.h"
#include "setting.h"
#include "id3_tag.h"
#include "bar.h"
#include "misc.h"
#include "et_core.h"
#include "crc32.h"
#include "i18n.h"


/****************
 * Declarations *
 ****************/
GtkWidget *DummyEntry = NULL;    /* Used to simulate a gtkentry widget for mask code '%i' */
GtkWidget *ScannerScanTagMaskCombo    = NULL;
GtkWidget *ScannerRenameFileMaskCombo = NULL;
GtkWidget *ScannerOptionMenu          = NULL;

GtkWidget *ScanTagFrame;
GtkWidget *RenameFileFrame;
GtkWidget *ProcessFieldsFrame;
GtkWidget *FillTagPreviewLabel        = NULL;
GtkWidget *RenameFilePreviewLabel     = NULL;

GtkWidget *ProcessFileNameField;
GtkWidget *ProcessTitleField;
GtkWidget *ProcessArtistField;
GtkWidget *ProcessAlbumField;
GtkWidget *ProcessGenreField;
GtkWidget *ProcessCommentField;
GtkWidget *ProcessFieldsConvertIntoSpace      = NULL;
GtkWidget *ProcessFieldsConvertSpace          = NULL;
GtkWidget *ProcessFieldsConvert               = NULL;
GtkWidget *ProcessFieldsConvertTo             = NULL;
GtkWidget *ProcessFieldsConvertFrom           = NULL;
GtkWidget *ProcessFieldsAllUppercase          = NULL;
GtkWidget *ProcessFieldsAllDowncase           = NULL;
GtkWidget *ProcessFieldsFirstLetterUppercase  = NULL;
GtkWidget *ProcessFieldsFirstLettersUppercase = NULL;
GtkWidget *ProcessFieldsRemoveSpace           = NULL;
GtkWidget *ProcessFieldsInsertSpace           = NULL;
GtkWidget *ProcessFieldsOnlyOneSpace          = NULL;

GtkWidget *LegendFrame    = NULL;
GtkWidget *LegendButton   = NULL;

GtkWidget *MaskEditorButton = NULL;
GtkWidget *MaskEditorFrame  = NULL;
GtkWidget *MaskEditorVBox;
GtkWidget *MaskEditorHBox;
GtkWidget *MaskEditorScrollWindow;
GtkWidget *MaskEditorClist;
GtkWidget *MaskEditorEntry;
GtkWidget *MaskEditorNewButton;
GtkWidget *MaskEditorCopyButton;
GtkWidget *MaskEditorAddButton;
GtkWidget *MaskEditorRemoveButton;
GtkWidget *MaskEditorUpButton;
GtkWidget *MaskEditorDownButton;
GtkWidget *MaskEditorSaveButton;

/* Lists */
GList *MasksList           = NULL;
GList *RenameFileMasksList = NULL;

/* Some predefined masks */
gchar *Scan_Masks [] = 
{
    "%a - %b/%n - %t",
    "%a_-_%b/%n_-_%t",
    "%a - %b/%n. %t",
    "%a_-_%b/%n._%t",
    "%a-%b/%n-%t",
    "%b/%n. %a - %t",
    "%b/%n._%a_-_%t",
    "%b/%n - %a - %t",
    "%b/%n_-_%a_-_%t",
    "%b/%n-%a-%t",
    "%a-%b/%n-%t",
    "%a/%b/%n. %t",
    "%g/%a/%b/%t",
    "%a_-_%b-%n-%t-%y",
    "%a - %b/%n. %t(%c)",
    "%t",
    "Track%n",
    "Track%i %n"
};

gchar *Rename_File_Masks [] = 
{
    "%n - %a - %t",
    "%n_-_%a_-_%t",
    "%n. %a - %t",
    "%n._%a_-_%t",
    "%n - %t",
    "%n_-_%t",
    "%n. %t",
    "%n._%t",
    "%n - %a - %b - %t",
    "%n_-_%a_-_%b_-_%t",
    "%a - %b - %t",
    "%a_-_%b_-_%t",
    "%a - %b - %n - %t",
    "%a_-_%b_-_%n_-_%t",
    "%a - %t",
    "%a_-_%t",
    "Track %n"
};

gchar *Scanner_Option_Menu_Items [] = 
{
    N_("Fill Tag"),
    N_("Rename File"),
    N_("Process Fields")
};
/* Must have the same order than Scanner_Option_Menu_Items */
/*typedef enum
{
    SCANNER_FILL_TAG = 0,
    SCANNER_RENAME_FILE,
    SCANNER_PROCESS_FIELDS
} Scanner_Option_Menu_Items_Value;
*/
typedef enum
{
    LEADING_SEPARATOR = 0, /* characters before the first code */
    TRAILING_SEPARATOR,    /* characters after the last code */
    SEPARATOR,             /* item is a separator between two codes */
    FIELD,                 /* item contains text (not empty) of entry */
    EMPTY_FIELD            /* item when entry contains no text */
} Mask_Item_Type;

/*
 * Used into Rename File Scanner
 */
typedef struct _File_Mask_Item File_Mask_Item;
struct _File_Mask_Item
{
    Mask_Item_Type  type;
    gchar          *string;
};

/*
 * Used into Scan Tag Scanner
 */
typedef struct _Scan_Mask_Item Scan_Mask_Item;
struct _Scan_Mask_Item
{
    gchar  code;   // The code of the mask without % (ex: %a => a)
    gchar *string; // The string found by the scanner for the code defined the line above
};



/**************
 * Prototypes *
 **************/
void     ScannerWindow_Quit (void);
void     ScannerWindow_Key_Press (GtkWidget *window, GdkEvent *event);
void     Scan_Toggle_Legend_Button (void);
void     Scan_Toggle_Mask_Editor_Button (void);
gchar   *Scan_Replace_String (gchar *string, gchar *last, gchar *new);
void     Scan_Option_Button (void);
gboolean Scan_Check_Scan_Tag_Mask    (GtkObject *widget_to_show_hide, GtkEntry *widget_source);
gboolean Scan_Check_Rename_File_Mask (GtkObject *widget_to_show_hide, GtkEntry *widget_source);
gboolean Scan_Check_Editor_Mask      (GtkObject *widget_to_show_hide, GtkEntry *widget_source);

gchar   *Scan_Generate_New_Filename_From_Mask (ET_File *ETFile, gchar *mask);
GList   *Scan_Generate_New_Tag_From_Mask      (ET_File *ETFile, gchar *mask);
void     Scan_Rename_File_Generate_Preview    (void);
void     Scan_Fill_Tag_Generate_Preview       (void);
void     Scan_Free_File_Rename_List           (GList *list);
void     Scan_Free_File_Fill_Tag_List         (GList *list);

gchar  **Scan_Return_File_Tag_Field_From_Mask_Code (File_Tag *FileTag, gchar code);
void     Scan_Process_Fields_Functions (gchar *string);

void Process_Fields_Check_Button_Toggled (GtkObject *object, GList *list);
void Process_Fields_Convert_Check_Button_Toggled (GtkObject *object);
void Select_Fields_Invert_Selection    (void);
void Select_Fields_Select_Unselect_All (void);
void Select_Fields_Set_Sensitive       (void);

void Mask_Editor_Clist_Select_Row   (GtkCList *clist,gint row,gint column,GdkEventButton *event,gpointer data);
void Mask_Editor_Clist_Unselect_Row (void);
void Mask_Editor_Clist_New          (void);
void Mask_Editor_Clist_Copy         (void);
void Mask_Editor_Clist_Add          (void);
void Mask_Editor_Clist_Remove       (void);
void Mask_Editor_Clist_Move_Up      (void);
void Mask_Editor_Clist_Move_Down    (void);
void Mask_Editor_Clist_Save_Button  (void);
void Mask_Editor_Entry_Changed      (void);
void Mask_Editor_Clist_Key_Press    (GtkWidget *widget, GdkEvent *event);

void Mask_Editor_Create_Masks_List_From_Clist (void);
void Mask_Editor_Clist_Load_Mask_List (GList *list);

void Scanner_Option_Menu_Activate_Item (GtkWidget *widget, Scanner_Option_Menu_Items_Value item);



/*************
 *           * 
 * Functions *
 *           * 
 *************/

void Init_ScannerWindow (void)
{
    ScannerWindow     = (GtkWidget *)NULL;
    ScannerOptionMenu = (GtkWidget *)NULL;
    SWScanButton      = (GtkWidget *)NULL;
}


/*******************
 * Scanner for Tag *
 *******************/
/*
 * Uses the filename and path to fill tag informations
 * Note: mask and source are read from the right to the left
 */
void Scan_Tag_With_Mask (ET_File *ETFile)
{
    GList *fill_tag_list = NULL;
    gchar **dest = NULL;
    gchar *mask = NULL; // The 'mask' in the entry
    File_Tag *FileTag;


    if (!ScannerWindow || !ScannerScanTagMaskCombo || !ETFile) return;

    mask = g_strdup(gtk_entry_get_text_1(ScannerScanTagMaskCombo));
    if (!mask) return;

    // Create a new File_Tag item
    FileTag = ET_File_Tag_Item_New();
    ET_Copy_File_Tag_Item(ETFile,FileTag);

    
    // Process this mask with file
    fill_tag_list = Scan_Generate_New_Tag_From_Mask(ETFile,mask);
    while (fill_tag_list)
    {
        Scan_Mask_Item *mask_item = fill_tag_list->data;
        
        // Get the target entry for this code
        dest = Scan_Return_File_Tag_Field_From_Mask_Code(FileTag,mask_item->code);
        
        // We display the text affected to the code
        if ( dest && ( OVERWRITE_TAG_FIELD || *dest==NULL || strlen(*dest)==0 ) )
            ET_Set_Field_File_Tag_Item(dest,mask_item->string);

        if (!fill_tag_list->next) break;
            fill_tag_list = fill_tag_list->next;
    }
    Scan_Free_File_Fill_Tag_List(fill_tag_list);

    
    // Set the default text to comment
    if (SET_DEFAULT_COMMENT && (OVERWRITE_TAG_FIELD || FileTag->comment==NULL || strlen(FileTag->comment)==0 ) )
        ET_Set_Field_File_Tag_Item((void *)&FileTag->comment,DEFAULT_COMMENT);

    // Set CRC-32 value as default comment (for files with ID3 tag only ;-)
    if (SET_CRC32_COMMENT && (OVERWRITE_TAG_FIELD || FileTag->comment==NULL || strlen(FileTag->comment)==0 ) )
    {
        gulong crc32_value = 0;
        gchar *buffer;
        ET_File_Description *ETFileDescription;
        
        ETFileDescription = ETFile->ETFileDescription;
        switch (ETFileDescription->TagType)
        {
            case ID3_TAG:
                crc32_file_with_ID3_tag( ((File_Name *)((GList *)ETFile->FileNameNew)->data)->value, &crc32_value);

            if (crc32_value > 0)
                {
                    buffer = g_strdup_printf("%.8lx",crc32_value);
                    ET_Set_Field_File_Tag_Item((void *)&FileTag->comment,buffer);
                    g_free(buffer);
                }
                break;
            default:
                break;
        }
    }


    // Save changes of the 'File_Tag' item
    ET_Manage_Changes_Of_File_Data(ETFile,NULL,FileTag);

    g_free(mask);

    Statusbar_Message(_("Tag successfully scanned..."),TRUE);
    g_print(_("Tag successfully scanned...(%s)\n"),g_basename( ((File_Name *)ETFile->FileNameNew->data)->value ) );
}

GList *Scan_Generate_New_Tag_From_Mask (ET_File *ETFile, gchar *mask)
{
    GList *fill_tag_list = NULL;
    gchar *file = NULL;
    gchar *tmp;
    gchar buf[256];
    gchar separator[256];
    gchar string[1024];
    gint  len, i, loop=0;
    gchar **mask_splitted;
    gchar **file_splitted;
    guint mask_splitted_number;
    guint file_splitted_number;
    guint mask_splitted_index;
    guint file_splitted_index;
    Scan_Mask_Item *mask_item;

    if (!ETFile || !mask) return NULL;
        
    file = g_strdup(((File_Name *)((GList *)ETFile->FileNameNew)->data)->value);
    if (!file) return NULL;

    // Remove extension of file (if found)
    tmp = strrchr(file,'.');
    for (i=0; i<=(gint)ET_FILE_DESCRIPTION_SIZE; i++)
    {
        if ( strcasecmp(tmp,ETFileDescription[i].Extension)==0 )
        {
            *tmp = 0; //strrchr(source,'.') = 0;
            break;
        }
    }
    if (i==ET_FILE_DESCRIPTION_SIZE)
        g_print("Tag scanner: strange..., the extension '%s' was not found into filename '%s'!\n",tmp,g_basename(file));


    // Replace characters into mask and filename before parsing
    if (FTS_CONVERT_UNDERSCORE_AND_P20_INTO_SPACE)
    {
        Scan_Convert_Underscore_Into_Space(mask);
        Scan_Convert_Underscore_Into_Space(file);
        Scan_Convert_P20_Into_Space(mask);
        Scan_Convert_P20_Into_Space(file);
    }
    if (FTS_CONVERT_SPACE_INTO_UNDERSCORE)
    {
        Scan_Convert_Space_Into_Undescore(mask);
        Scan_Convert_Space_Into_Undescore(file);
    }


    // Split the Scanner mask
    mask_splitted = g_strsplit(mask,"/",0);
    // Get number of arguments into 'mask_splitted'
    for (mask_splitted_number=0;mask_splitted[mask_splitted_number];mask_splitted_number++);

    // Split the File Path
    file_splitted = g_strsplit(file,"/",0);
    // Get number of arguments into 'file_splitted'
    for (file_splitted_number=0;file_splitted[file_splitted_number];file_splitted_number++);

    // Set the starting position for each tab
    if (mask_splitted_number <= file_splitted_number)
    {
        mask_splitted_index = 0;
        file_splitted_index = file_splitted_number - mask_splitted_number;
    }else
    {
        mask_splitted_index = mask_splitted_number - file_splitted_number;
        file_splitted_index = 0;
    }

    loop = 0;
    while ( mask_splitted[mask_splitted_index]!= NULL && file_splitted[file_splitted_index]!=NULL )
    {
        gchar *mask_seq = mask_splitted[mask_splitted_index];
        gchar *file_seq = file_splitted[file_splitted_index];
        
        //g_print(">%d> seq '%s' '%s'\n",loop,mask_seq,file_seq);
        while ( mask_seq && strlen(mask_seq)>0 )
        {

            /*
             * Determine (first) code and destination
             */
            if ( (tmp=strchr(mask_seq,'%')) == NULL || strlen(tmp) < 2 )
            {
                break;
            }

            /*
             * Allocate a new iten for the fill_tag_list
             */
            mask_item = g_malloc0(sizeof(Scan_Mask_Item));

            // Get the code (used to determine the corresponding target entry)
            mask_item->code = tmp[1];
            
            /*
             * Delete text before the code
             */
            if ( (len = strlen(mask_seq) - strlen(tmp)) > 0 )
            {
                // Get this text in 'mask_seq'
                strncpy(buf,mask_seq,len);
                buf[len] = 0;
                // We remove it in 'mask_seq'
                mask_seq = mask_seq + len;
                // Find the same text at the begining of 'file_seq' ?
                if ( (strstr(file_seq,buf)) == file_seq )
                {
                    file_seq = file_seq + len; // We remove it
                }else
                {
                    g_print(_("Scan Error: can't find separator '%s' within '%s'\n"),buf,file_seq);
                }
            }

            // Remove the current code into 'mask_seq'
            mask_seq = mask_seq + 2;

            /*
             * Determine separator between two code or trailing text (after code)
             */
            if ( mask_seq && strlen(mask_seq)>0 )
            {
                if ( (tmp=strchr(mask_seq,'%')) == NULL || strlen(tmp) < 2 )
                {
                    // No more code found
                    strcpy(separator,mask_seq);
                    separator[strlen(mask_seq)] = 0;
                }else
                {
                    len = strlen(mask_seq) - strlen(tmp);
                    strncpy(separator,mask_seq,len);
                    separator[len] = 0;
                }

                // Remove the current separator in 'mask_seq'
                mask_seq = mask_seq + strlen(separator);

                // Try to find the separator in 'file_seq'
                if ( (tmp=strstr(file_seq,separator)) == NULL )
                {
                    g_print(_("Scan Error: can't find separator '%s' within '%s'\n"),separator,file_seq);
                    separator[0] = 0; // Needed to avoid error when calculting 'len' below
                }

                // Get the string affected to the code (or the corresponding entry field)
                len = strlen(file_seq) - (tmp!=NULL?strlen(tmp):0);
                strncpy(string,file_seq,len);
                string[len] = 0;

                // Remove the current separator in 'file_seq'
                file_seq = file_seq + strlen(string) + strlen(separator);

                // We get the text affected to the code
                mask_item->string = g_strdup(string);
            }else
            {
                // We display the remaining text, affected to the code (no more data in 'mask_seq')
                mask_item->string = g_strdup(file_seq);
            }

            // Add the filled mask_iten to the list
            fill_tag_list = g_list_append(fill_tag_list,mask_item);
        }
        
        // Next sequences
        mask_splitted_index++;
        file_splitted_index++;
        loop++;
    }

    g_free(file);
    g_strfreev(mask_splitted);
    g_strfreev(file_splitted);

    // The 'fill_tag_list' must be freed after use
    return fill_tag_list;
}

void Scan_Fill_Tag_Generate_Preview (void)
{
    gchar *mask = NULL;
    gchar *preview_text = NULL;
    GList *fill_tag_list = NULL;

    if (!ETCore->ETFileDisplayedList || !ScannerWindow || !ScannerRenameFileMaskCombo || !FillTagPreviewLabel) return;
    
    mask = g_strdup(gtk_entry_get_text_1(ScannerScanTagMaskCombo));
    if (!mask) return;

    preview_text = g_strdup("");
    fill_tag_list = Scan_Generate_New_Tag_From_Mask(ETCore->ETFileDisplayed,mask);
    while (fill_tag_list)
    {
        Scan_Mask_Item *mask_item = fill_tag_list->data;
        gchar *tmp_code = g_strdup_printf("%c",mask_item->code);
        gchar *tmp_preview_text = preview_text;
        
        preview_text = g_strconcat(tmp_preview_text,"%",tmp_code," = ",mask_item->string,NULL);
        g_free(tmp_code);
        g_free(tmp_preview_text);
        
        if (!fill_tag_list->next) break;
        fill_tag_list = fill_tag_list->next;
        
        tmp_preview_text = preview_text;
        preview_text = g_strconcat(tmp_preview_text,"  ||  ",NULL);
        g_free(tmp_preview_text);
    }
    Scan_Free_File_Fill_Tag_List(fill_tag_list);

    if (GTK_IS_LABEL(FillTagPreviewLabel))
    {
        if (preview_text)
            gtk_label_set_text(GTK_LABEL(FillTagPreviewLabel),preview_text);
        else
            gtk_label_set_text(GTK_LABEL(FillTagPreviewLabel),"");
    }

    if (mask) g_free(mask);
    if (preview_text) g_free(preview_text);
}

void Scan_Free_File_Fill_Tag_List (GList *list)
{
    // Free the list
    list = g_list_first(list);
    while (list)
    {
        if (list->data)
        {
            if ( ((Scan_Mask_Item *)list->data)->string )
                g_free(((Scan_Mask_Item *)list->data)->string);
            g_free( (Scan_Mask_Item *)list->data );
        }
        if (!list->next) break;
        list = list->next;
    }
    g_list_free(list);
    list = (GList *)NULL;
}



/**************************
 * Scanner To Rename File *
 **************************/
/*
 * Uses tag informations (displayed into tag entries) to rename file
 * Note: mask and source are read from the right to the left.
 * Note1: a mask code may be used severals times...
 */
void Scan_Rename_File_With_Mask (ET_File *ETFile)
{
    gchar *filename_generated = NULL;
    gchar *filename_new = NULL;
    gchar *mask = NULL;
    File_Name *FileName;


    if (!ScannerWindow || !ScannerRenameFileMaskCombo || !ETFile) return;

    mask = g_strdup(gtk_entry_get_text_1(ScannerRenameFileMaskCombo));
    filename_generated = Scan_Generate_New_Filename_From_Mask(ETFile,mask);
    ET_File_Name_Convert_Character(filename_generated); // Replace invalid characters
    if (mask) g_free(mask);

    if (!filename_generated)
        return;
    if (strlen(filename_generated)<1)
    {
        g_free(filename_generated);
        return;
    }


    /* Build the filename with the path */
    filename_new = ET_Generate_File_Name(ETFile,filename_generated);
    g_free(filename_generated);


    /* Set the new filename */
    // Create a new 'File_Name' item
    FileName = ET_File_Name_Item_New();
    // Save changes of the 'File_Name' item
    ET_Set_Field_File_Name_Item(&FileName->value,filename_new);
    ET_Manage_Changes_Of_File_Data(ETFile,FileName,NULL);

    if (filename_new) g_free(filename_new);

    Statusbar_Message(_("New file name successfully scanned..."),TRUE);
    g_print(_("New file name successfully scanned...(%s)\n"),g_basename( ((File_Name *)ETFile->FileNameNew->data)->value ) );

    return;
}

gchar *Scan_Generate_New_Filename_From_Mask (ET_File *ETFile, gchar *mask)
{
    gchar *tmp;
    gchar **source = NULL;
    gchar *filename_new = NULL;
    gchar *filename_tmp = NULL;
    GList *rename_file_list = NULL;
    File_Mask_Item *mask_item;
    File_Mask_Item *mask_item_prev;
    File_Mask_Item *mask_item_next;
    gint counter = 0;


    if (!ETFile || !mask) return NULL;


    /*
     * Parse the codes to generate a list (1rst item = 1rst code)
     */
    while ( mask!=NULL && (tmp=strrchr(mask,'%'))!=NULL && strlen(tmp)>1 )
    {
        // Mask contains some characters after the code ('%b__')
        if (strlen(tmp)>2)
        {
            mask_item = g_malloc0(sizeof(File_Mask_Item));
            if (counter) mask_item->type = SEPARATOR;
            else         mask_item->type = TRAILING_SEPARATOR;
            mask_item->string = g_strdup(tmp+2);
            rename_file_list = g_list_prepend(rename_file_list,mask_item);
        }

        // Now, parses the code
        source = Scan_Return_File_Tag_Field_From_Mask_Code((File_Tag *)ETFile->FileTag->data,tmp[1]);
        mask_item = g_malloc0(sizeof(File_Mask_Item));
        if (source && *source && strlen(*source)>0)
        {
            mask_item->type = FIELD;
            mask_item->string = g_strdup(*source);
        }else
        {
            mask_item->type = EMPTY_FIELD;
            mask_item->string = NULL;
        }
        rename_file_list = g_list_prepend(rename_file_list,mask_item);
        *tmp = '\0'; // Cut parsed data of mask
        counter++; // To indicate that we made at least one loop to identifiate 'separator' or 'trailing_separator'
    }
    
    // It may have some characters before the last remaining code ('__%a')
    if (mask!=NULL && strlen(mask)>0)
    {
        mask_item = g_malloc0(sizeof(File_Mask_Item));
        mask_item->type = LEADING_SEPARATOR;
        mask_item->string = g_strdup(mask);
        rename_file_list = g_list_prepend(rename_file_list,mask_item);
    }
    
    if (!rename_file_list) return NULL;


    /*
     * Build the new filename with items placed into the list
     * (read the list from the end to the beginning)
     */
    rename_file_list = g_list_last(rename_file_list);
    filename_new = g_strdup("");

    while (rename_file_list)
    {
        File_Mask_Item *mask_item = rename_file_list->data;

        if ( mask_item->type==TRAILING_SEPARATOR ) // Trailing characters of mask
        {
            // Doesn't write it if previous field is empty
            if (rename_file_list->prev && ((File_Mask_Item *)rename_file_list->prev->data)->type!=EMPTY_FIELD)
            {
                filename_tmp = filename_new;
                filename_new = g_strconcat(mask_item->string,filename_new,NULL);
                g_free(filename_tmp);
            }
        }else
        if ( mask_item->type==LEADING_SEPARATOR ) // Leading characters of mask
        {
            filename_tmp = filename_new;
            filename_new = g_strconcat(mask_item->string,filename_new,NULL);
            g_free(filename_tmp);
        }else
        if ( mask_item->type==EMPTY_FIELD )
        // We don't concatenate the field value (empty) and the previous separator to the filename
        // If field is the 'first', we don't concatenate it and the next separator too.
        {
            if (rename_file_list->prev) // If previous string is a separator, we don't use it
            {
                mask_item_prev = rename_file_list->prev->data;
                if ( mask_item_prev->type==SEPARATOR )
                    rename_file_list = rename_file_list->prev;
                if (rename_file_list->next && (mask_item_next=rename_file_list->next->data)
                &&  mask_item_prev->type==LEADING_SEPARATOR && mask_item_next->type==SEPARATOR)
                {
                    if ( filename_new && mask_item_next->string && (strlen(filename_new)>=strlen(mask_item_next->string)) ) // To avoid crash if filename_new is 'empty'
                    {
                        filename_tmp = filename_new;
                        filename_new = g_strdup(filename_new+strlen(mask_item_next->string));
                        g_free(filename_tmp);
                     }
                }
            }else
            if (rename_file_list->next && (mask_item_next=rename_file_list->next->data)
            &&  mask_item_next->type==SEPARATOR) // We are at the 'beginning' of the mask
            {
                if ( filename_new && mask_item_next->string && (strlen(filename_new)>=strlen(mask_item_next->string)) ) // To avoid crash if filename_new is 'empty'
                {
                    filename_tmp = filename_new;
                    filename_new = g_strdup(filename_new+strlen(mask_item_next->string));
                    g_free(filename_tmp);
                 }
            }

        }else // SEPARATOR, FIELD
        {
            filename_tmp = filename_new;
            filename_new = g_strconcat(mask_item->string,filename_new,NULL);
            g_free(filename_tmp);
        }
        
        if (!rename_file_list->prev) break;
        rename_file_list = rename_file_list->prev;
    }

    // Free the list
    Scan_Free_File_Rename_List(rename_file_list);

    /* Replace characters */
    if (RFS_CONVERT_UNDERSCORE_AND_P20_INTO_SPACE)
    {
        Scan_Convert_Underscore_Into_Space(filename_new);
        Scan_Convert_P20_Into_Space(filename_new);
    }
    if (RFS_CONVERT_SPACE_INTO_UNDERSCORE)
        Scan_Convert_Space_Into_Undescore(filename_new);

    return filename_new;
}

void Scan_Rename_File_Generate_Preview (void)
{
    gchar *preview_text = NULL;
    gchar *mask = NULL;


    if (!ETCore->ETFileDisplayedList || !ScannerWindow || !ScannerRenameFileMaskCombo || !RenameFilePreviewLabel) return;

    mask = g_strdup(gtk_entry_get_text_1(ScannerRenameFileMaskCombo));
    preview_text = Scan_Generate_New_Filename_From_Mask(ETCore->ETFileDisplayed,mask);

    if (GTK_IS_LABEL(RenameFilePreviewLabel))
    {
        if (preview_text)
            gtk_label_set_text(GTK_LABEL(RenameFilePreviewLabel),preview_text);
        else
            gtk_label_set_text(GTK_LABEL(RenameFilePreviewLabel),"");
    }

    if (mask) g_free(mask);
    if (preview_text) g_free(preview_text);
}


void Scan_Free_File_Rename_List (GList *list)
{
    // Free the list
    list = g_list_last(list);
    while (list)
    {
        if (list->data)
        {
            if ( ((File_Mask_Item *)list->data)->string )
                g_free(((File_Mask_Item *)list->data)->string);
            g_free( (File_Mask_Item *)list->data );
        }
        if (!list->prev) break;
        list = list->prev;
    }
    g_list_free(list);
    list = (GList *)NULL;
}


/*****************************
 * Scanner To Process Fields *
 *****************************/
void Scan_Process_Fields (ET_File *ETFile)
{
    File_Name *FileName = NULL;
    File_Tag  *FileTag  = NULL;
    File_Name *st_filename;
    File_Tag  *st_filetag;
    //GString *string2process;
    guint  string_length;
    gchar *string;


    if (!ScannerWindow || !ETFile) return;

    st_filename = (File_Name *)ETFile->FileNameNew->data;
    st_filetag  = (File_Tag  *)ETFile->FileTag->data;
    //string2process = g_string_sized_new(512);

    /* Process the filename */
    if (st_filename != NULL)
    {
        if (st_filename->value && GTK_TOGGLE_BUTTON(ProcessFileNameField)->active) // File name field
        {
            gchar *string1;
            gchar *pos;

            if (!FileName)
                FileName = ET_File_Name_Item_New();

            // FIX ME : we suppose that it will not grow more than 2 times its size...
            string_length = 2 * strlen(g_basename(st_filename->value));
            string        = g_malloc(string_length+1);
            strncpy(string,g_basename(st_filename->value),string_length);
            string[string_length]='\0';
            // Remove the extension to set it to lower case (to avoid problem with undo)
            if ((pos=strrchr(string,'.'))!=NULL) *pos = 0;

            Scan_Process_Fields_Functions(string);

            string1 = ET_Generate_File_Name(ETFile,string);
            ET_Set_Field_File_Name_Item(&FileName->value,string1);

            if (string1) g_free(string1);
            if (string)  g_free(string);
        }
    }

    /* Process data of the tag */
    if (st_filetag != NULL)
    {
        if (st_filetag->title && GTK_TOGGLE_BUTTON(ProcessTitleField)->active) // Title field
        {
            if (!FileTag)
            {
                FileTag = ET_File_Tag_Item_New();
                ET_Copy_File_Tag_Item(ETFile,FileTag);
            }

            // FIX ME : we suppose that it will not grow more than 2 times its size...
            string_length = 2 * strlen(st_filetag->title);
            string        = g_malloc(string_length+1);
            strncpy(string,st_filetag->title,string_length);
            string[string_length]='\0';

            Scan_Process_Fields_Functions(string);

            ET_Set_Field_File_Tag_Item(&FileTag->title,string);

            if (string)  g_free(string);
        }

        if (st_filetag->artist && GTK_TOGGLE_BUTTON(ProcessArtistField)->active) // Artist field
        {
            if (!FileTag)
            {
                FileTag = ET_File_Tag_Item_New();
                ET_Copy_File_Tag_Item(ETFile,FileTag);
            }
        
            // FIX ME : we suppose that it will not grow more than 2 times its size...
            string_length = 2 * strlen(st_filetag->artist);
            string        = g_malloc(string_length+1);
            strncpy(string,st_filetag->artist,string_length);
            string[string_length]='\0';

            Scan_Process_Fields_Functions(string);

            ET_Set_Field_File_Tag_Item(&FileTag->artist,string);

            if (string)  g_free(string);
        }

        if (st_filetag->album && GTK_TOGGLE_BUTTON(ProcessAlbumField)->active) // Album field
        {
            if (!FileTag)
            {
                FileTag = ET_File_Tag_Item_New();
                ET_Copy_File_Tag_Item(ETFile,FileTag);
            }
        
            // FIX ME : we suppose that it will not grow more than 2 times its size...
            string_length = 2 * strlen(st_filetag->album);
            string        = g_malloc(string_length+1);
            strncpy(string,st_filetag->album,string_length);
            string[string_length]='\0';

            Scan_Process_Fields_Functions(string);

            ET_Set_Field_File_Tag_Item(&FileTag->album,string);

            if (string)  g_free(string);
        }

        if (st_filetag->genre && GTK_TOGGLE_BUTTON(ProcessGenreField)->active) // Genre field
        {
            if (!FileTag)
            {
                FileTag = ET_File_Tag_Item_New();
                ET_Copy_File_Tag_Item(ETFile,FileTag);
            }
        
            // FIX ME : we suppose that it will not grow more than 2 times its size...
            string_length = 2 * strlen(st_filetag->genre);
            string        = g_malloc(string_length+1);
            strncpy(string,st_filetag->genre,string_length);
            string[string_length]='\0';

            Scan_Process_Fields_Functions(string);

            ET_Set_Field_File_Tag_Item(&FileTag->genre,string);

            if (string)  g_free(string);
        }

        if (st_filetag->comment && GTK_TOGGLE_BUTTON(ProcessCommentField)->active) // Comment field
        {
            if (!FileTag)
            {
                FileTag = ET_File_Tag_Item_New();
                ET_Copy_File_Tag_Item(ETFile,FileTag);
            }
        
            // FIX ME : we suppose that it will not grow more than 2 times its size...
            string_length = 2 * strlen(st_filetag->comment);
            string        = g_malloc(string_length+1);
            strncpy(string,st_filetag->comment,string_length);
            string[string_length]='\0';

            Scan_Process_Fields_Functions(string);

            ET_Set_Field_File_Tag_Item(&FileTag->comment,string);

            if (string)  g_free(string);
        }
    }

    if (FileName && FileTag)
    {
        // Synchronize undo key of the both structures (used for the
        // undo functions, as they are generated as the same time)
        FileName->key = FileTag->key;
    }
    ET_Manage_Changes_Of_File_Data(ETFile,FileName,FileTag);

}


void Scan_Process_Fields_Functions (gchar *string)
{

    if (GTK_TOGGLE_BUTTON(ProcessFieldsConvertIntoSpace)->active)
    {
        Scan_Convert_Underscore_Into_Space(string);
        Scan_Convert_P20_Into_Space(string);
    }

    if (GTK_TOGGLE_BUTTON(ProcessFieldsConvertSpace)->active)
        Scan_Convert_Space_Into_Undescore(string);

    if (GTK_TOGGLE_BUTTON(ProcessFieldsConvert)->active)
        Scan_Convert(string);

    if (GTK_TOGGLE_BUTTON(ProcessFieldsAllUppercase)->active)
        Scan_Process_Fields_All_Uppercase(string);

    if (GTK_TOGGLE_BUTTON(ProcessFieldsAllDowncase)->active)
        Scan_Process_Fields_All_Downcase(string);

    if (GTK_TOGGLE_BUTTON(ProcessFieldsFirstLetterUppercase)->active)
         Scan_Process_Fields_Letter_Uppercase(string);

    if (GTK_TOGGLE_BUTTON(ProcessFieldsFirstLettersUppercase)->active)
        Scan_Process_Fields_First_Letters_Uppercase(string);

    if (GTK_TOGGLE_BUTTON(ProcessFieldsRemoveSpace)->active)
        Scan_Process_Fields_Remove_Space(string);

    if (GTK_TOGGLE_BUTTON(ProcessFieldsInsertSpace)->active)
        Scan_Process_Fields_Insert_Space(string);

    if (GTK_TOGGLE_BUTTON(ProcessFieldsOnlyOneSpace)->active)
        Scan_Process_Fields_Keep_One_Space(string);
}


void Scan_Process_Fields_All_Uppercase (gchar *text)
{
    gint j;
    gint ascii_caps_interval = 'A' - 'a';

    for (j=0; j<(gint)strlen(text); j++)
        if ( (text[j]>='a' && text[j]<='z') || (text[j]>='' && text[j]<='') )
            text[j] = text[j] + ascii_caps_interval;
}
void Scan_Process_Fields_All_Downcase (gchar *text)
{
    gint j;
    gint ascii_caps_interval = 'A' - 'a';

    for (j=0; j<(gint)strlen(text); j++)
        if ( (text[j]>='A' && text[j]<='Z') || (text[j]>='' && text[j]<='') )
            text[j] = text[j] - ascii_caps_interval;
}
void Scan_Process_Fields_Letter_Uppercase (gchar *text)
{
    gint j;
    gint set_to_upper_case = 1;
    gint ascii_caps_interval = 'A' - 'a';

    for (j=0; j<(gint)strlen(text); j++)
    {
        if ( ((text[j]>='a' && text[j]<='z') || (text[j]>='' && text[j]<='')) && set_to_upper_case )
            text[j] = text[j] + ascii_caps_interval;
        else if ( ((text[j]>='A' && text[j]<='Z') || (text[j]>='' && text[j]<='')) && !set_to_upper_case )
            text[j] = text[j] - ascii_caps_interval;
        set_to_upper_case = 0;
    }
}
void Scan_Process_Fields_First_Letters_Uppercase (gchar *text)
{
    gint j;
    gint set_to_upper_case = 1;    /* if 1 => capitalize */
    gint ascii_caps_interval = 'A' - 'a';

    for (j=0; j<(gint)strlen(text); j++)
    {
        /* It's an alphabetic character? (including '', '', ...) */
        if ( (text[j]>='A' && text[j]<='Z') || (text[j]>='a' && text[j]<='z')
          || (text[j]>='' && text[j]<='') || (text[j]>='' && text[j]<='') )
        {
            if (set_to_upper_case)
            {
                if ( (text[j]>='a' && text[j]<='z') || (text[j]>='' && text[j]<='') )
                    text[j] = text[j] + ascii_caps_interval;
                /* The next character musn't be capitalized */
                set_to_upper_case = 0;
            }else
            {
                if ( (text[j]>='A' && text[j]<='Z') || (text[j]>='' && text[j]<='') )
                    text[j] = text[j] - ascii_caps_interval;
            }
        }else
        {
            /* If character isn't alphabetic => unlock => next alphabetic character to uppercase
             * If a point found => probably an extension => doesn't uppercase next chararacter
             * If an apostrophe found (code 39) => ex: "...I'll..." => doesn't uppercase 
             * next chararacter */
            if (text[j]!='.' && text[j]!=39)
                set_to_upper_case = 1;
        }
    }
}
void Scan_Process_Fields_Remove_Space (gchar *text)
{
    gchar *tmp, *tmp1;

    while ((tmp=strchr(text,' '))!=NULL)
    {
        tmp1 = tmp + 1;
        while (*tmp1)
            *(tmp++) = *(tmp1++);
        *tmp = '\0';
    }
}
void Scan_Process_Fields_Insert_Space (gchar *text)
{
    gint i,j;
    
    /* FIXME: try to use realloc */
    for (i=1; i<(gint)strlen(text); i++)    // i=1 to not consider first "uppercase" letter
    {
        if (isupper(text[i]))
        {
            for (j=(gint)strlen(text); j>=i; j--)
            {
                text[j+1] = text[j];
            }
            text[i] = ' ';
            i++;
        }
    }
}
void Scan_Process_Fields_Keep_One_Space (gchar *text)
{
    gchar *tmp, *tmp1;

    /* Remove multiple consecutive spaces
     * Note: if exists severals spaces => exist at least 2 consecutive spaces */
    while ((tmp=strstr(text,"  "))!=NULL) // Search 2 spaces
    {
        tmp1 = tmp + 1;
        while (*tmp1)
            *(tmp++) = *(tmp1++);
        *tmp = '\0';
    }

    /* Remove multiple consecutive underscores */
    while ((tmp=strstr(text,"__"))!=NULL)
    {
        tmp1 = tmp + 1;
        while (*tmp1)
            *(tmp++) = *(tmp1++);
        *tmp = '\0';
    }
}

/*
 * Function to replace underscore '_' by a space
 */
void Scan_Convert_Underscore_Into_Space (gchar *string)
{
    gchar *tmp;

    while ((tmp=strchr(string,'_'))!=NULL)
        *tmp = ' ';
}

/*
 * Function to replace %20 by a space
 */
void Scan_Convert_P20_Into_Space (gchar *string)
{
    gchar *tmp, *tmp1;

    while ((tmp=strstr(string,"%20"))!=NULL)
    {
        tmp1 = tmp + 3;
        *(tmp++) = ' ';
        while (*tmp1)
            *(tmp++) = *(tmp1++);
        *tmp = '\0';
    }
}

/*
 * Function to replace space by '_'
 */
void Scan_Convert_Space_Into_Undescore (gchar *string)
{
    gchar *tmp;

    while ((tmp=strchr(string,' '))!=NULL)
        *tmp = '_';
}

/*
 * Replace something with something else ;)
 * Currently this only works with one character for each
 */
void Scan_Convert (gchar *string)
{
    gchar *tmp;
    gchar *from = gtk_editable_get_chars(GTK_EDITABLE(ProcessFieldsConvertFrom),0,-1 );
    gchar *to   = gtk_editable_get_chars(GTK_EDITABLE(ProcessFieldsConvertTo),0,-1 );
    
    if ( from && to && strlen(from)>0 && strlen(to)>0 )
        while ((tmp=strchr(string,*from))!=NULL)
            *tmp = *to;
}



/*
 * Return the field of a 'File_Tag' structure correxponding to the mask code
 */
gchar **Scan_Return_File_Tag_Field_From_Mask_Code (File_Tag *FileTag, gchar code)
{
    switch (code)
    {
        case 't':    /* Title */
            return &FileTag->title;
        case 'a':    /* Artist */
            return &FileTag->artist;
        case 'b':    /* Album */
            return &FileTag->album;
        case 'y':    /* Year */
            return &FileTag->year;
        case 'n':    /* Track */
            return &FileTag->track;
        case 'l':    /* Track Total */
            return &FileTag->track_total;
        case 'g':    /* Genre */
            return &FileTag->genre;
        case 'c':    /* Comment */
            return &FileTag->comment;
        case 'i':    /* Ignored */
            return NULL;
        default:
            g_print("Scanner: Invalid code '%%%c' found!\n",code);
            return NULL;
    }
}



/******************
 * Scanner Window *
 ******************/
#include "../pixmaps/scan.xpm"
#include "../pixmaps/help.xpm"
#include "../pixmaps/exit.xpm"
#include "../pixmaps/setting.xpm"
#include "../pixmaps/mask.xpm"
#include "../pixmaps/save.xpm"
#include "../pixmaps/up.xpm"
#include "../pixmaps/down.xpm"
#include "../pixmaps/new.xpm"
#include "../pixmaps/copy.xpm"
#include "../pixmaps/trash.xpm"
#include "../pixmaps/add.xpm"
#include "../pixmaps/blackwhite.xpm"
#include "../pixmaps/forbidden.xpm"

void Open_ScannerWindow (Scanner_Option_Menu_Items_Value scanner_type)
{
    GtkWidget *ScanVBox;
    GtkWidget *HBox1, *HBox2, *HBox4, *VBox, *hbox, *vbox;
    GtkWidget *Table;
    GtkWidget *Label;
    GtkWidget *Button;
    GtkWidget *Separator;
    GtkWidget *Icon;
    GtkWidget *EventBox;
    GtkTooltips *Tips;
    GtkWidget *Menu, *MenuItem;
    GtkWidget *MaskStatusIconBox;
    gint i;
    GList *pf_cb_group1 = NULL;
    GList *pf_cb_group2 = NULL;
    GList *pf_cb_group3 = NULL;


    /* Check if already opened */
    if (ScannerWindow)
    {
        //gdk_window_show(ScannerWindow->window);
        gdk_window_raise(ScannerWindow->window);
        if (ScannerOptionMenu)
        {
            gtk_option_menu_set_history(GTK_OPTION_MENU(ScannerOptionMenu),scanner_type);
            Scanner_Option_Menu_Activate_Item(NULL,scanner_type);
        }
        return;
    }

    if (  scanner_type < SCANNER_FILL_TAG
       || scanner_type > SCANNER_PROCESS_FIELDS)
          scanner_type = SCANNER_FILL_TAG;

    /* The window */
    ScannerWindow = gtk_window_new(GTK_WINDOW_TOPLEVEL);
    /* Config */
    gtk_container_border_width(GTK_CONTAINER(ScannerWindow),5);
    gtk_window_set_policy(GTK_WINDOW(ScannerWindow),FALSE,FALSE,TRUE);
    gtk_window_set_wmclass(GTK_WINDOW(ScannerWindow),"EasyTag_Scanner","easytag"); // Patch from Nikolai Prokoschenko (Debian)
    if (SCANNER_WINDOW_ON_TOP)
        gtk_window_set_transient_for(GTK_WINDOW(ScannerWindow),GTK_WINDOW(MainWindow));

    /* The init position is define below, cause the scanner window must be showed before
     * to be able to move it. */

    /* Title */
    gtk_window_set_title(GTK_WINDOW(ScannerWindow),_("Tag and File Name scan"));

    /* Signals connection */
    gtk_signal_connect(GTK_OBJECT(ScannerWindow),"destroy",(GtkSignalFunc)ScannerWindow_Quit,NULL);
    gtk_signal_connect(GTK_OBJECT(ScannerWindow),"delete_event",(GtkSignalFunc)ScannerWindow_Quit,NULL);
    gtk_signal_connect(GTK_OBJECT(ScannerWindow),"key_press_event",(GtkSignalFunc)ScannerWindow_Key_Press,NULL);

    /* The tooltips */
    Tips = gtk_tooltips_new_1();

    /* The main vbox */
    ScanVBox = gtk_vbox_new(FALSE,2);
    gtk_container_add(GTK_CONTAINER(ScannerWindow),ScanVBox);


    /*
     * The hbox for mode buttons + buttons + what to scan
     */
    HBox1 = gtk_hbox_new(FALSE,0);
    gtk_box_pack_start(GTK_BOX(ScanVBox),HBox1,FALSE,FALSE,0);

    /* Option Menu */
    Label = gtk_label_new(_("Scanner :"));
    gtk_box_pack_start(GTK_BOX(HBox1),Label,FALSE,FALSE,0);
    ScannerOptionMenu = gtk_option_menu_new();
    gtk_box_pack_start(GTK_BOX(HBox1),ScannerOptionMenu,TRUE,TRUE,2);
    gtk_widget_set_usize(ScannerOptionMenu,140,-1);
    /* The menu */
    Menu = gtk_menu_new();
    /* Option for Tag */
    MenuItem = gtk_menu_item_new_with_label(_(Scanner_Option_Menu_Items[SCANNER_FILL_TAG]));
    gtk_menu_append(GTK_MENU(Menu),MenuItem);
    gtk_signal_connect(GTK_OBJECT(MenuItem),"activate",(GtkSignalFunc)Scanner_Option_Menu_Activate_Item,GINT_TO_POINTER(SCANNER_FILL_TAG));
    /* Option for FileName */
    MenuItem = gtk_menu_item_new_with_label(_(Scanner_Option_Menu_Items[SCANNER_RENAME_FILE]));
    gtk_menu_append(GTK_MENU(Menu),MenuItem);
    gtk_signal_connect(GTK_OBJECT(MenuItem),"activate",(GtkSignalFunc)Scanner_Option_Menu_Activate_Item,GINT_TO_POINTER(SCANNER_RENAME_FILE));
    /* Option for ProcessFields */
    MenuItem = gtk_menu_item_new_with_label(_(Scanner_Option_Menu_Items[SCANNER_PROCESS_FIELDS]));
    gtk_menu_append(GTK_MENU(Menu),MenuItem);
    gtk_signal_connect(GTK_OBJECT(MenuItem),"activate",(GtkSignalFunc)Scanner_Option_Menu_Activate_Item,GINT_TO_POINTER(SCANNER_PROCESS_FIELDS));

    gtk_option_menu_set_menu(GTK_OPTION_MENU(ScannerOptionMenu),Menu);
    // Selection of the item made at the end of the function
    gtk_option_menu_set_history(GTK_OPTION_MENU(ScannerOptionMenu),scanner_type);
    gtk_tooltips_set_tip(Tips,ScannerOptionMenu,_("Select the type of scanner to use"),NULL);


    /* 'Scan selected files' button */
    SWScanButton = gtk_button_new();
    Icon = Create_Pixmap_Icon(scan_xpm);
    gtk_container_add(GTK_CONTAINER(SWScanButton),Icon);
    gtk_box_pack_start(GTK_BOX(HBox1),SWScanButton,FALSE,FALSE,0);
    gtk_button_set_relief(GTK_BUTTON(SWScanButton),GTK_RELIEF_NONE);
    gtk_tooltips_set_tip(Tips,SWScanButton,_("Open scanner window / Scan selected files"),NULL);
    gtk_signal_connect(GTK_OBJECT(SWScanButton),"clicked",(GtkSignalFunc)Action_Scan_Selected_Files,NULL);

    /* Separator line */
    Separator = gtk_vseparator_new();
    gtk_box_pack_start(GTK_BOX(HBox1),Separator,FALSE,FALSE,2);

    /* Options button */
    Button = gtk_button_new();
    Icon = Create_Pixmap_Icon(setting_xpm);
    gtk_container_add(GTK_CONTAINER(Button),Icon);
    gtk_box_pack_start(GTK_BOX(HBox1),Button,FALSE,FALSE,0);
    gtk_button_set_relief(GTK_BUTTON(Button),GTK_RELIEF_NONE);
    gtk_tooltips_set_tip(Tips,Button,_("Scanner Options"),NULL);
    gtk_signal_connect(GTK_OBJECT(Button),"clicked",(GtkSignalFunc)Scan_Option_Button,NULL);

    /* Mask Editor button */
    MaskEditorButton = gtk_toggle_button_new();
    Icon = Create_Pixmap_Icon(mask_xpm);
    gtk_container_add(GTK_CONTAINER(MaskEditorButton),Icon);
    gtk_box_pack_start(GTK_BOX(HBox1),MaskEditorButton,FALSE,FALSE,0);
    gtk_button_set_relief(GTK_BUTTON(MaskEditorButton),GTK_RELIEF_NONE);
    gtk_tooltips_set_tip(Tips,MaskEditorButton,_("Show / Hide Masks Editor"),NULL);
    gtk_signal_connect(GTK_OBJECT(MaskEditorButton),"toggled",(GtkSignalFunc)Scan_Toggle_Mask_Editor_Button,NULL);

    /* Legend button */
    LegendButton = gtk_toggle_button_new();
    Icon = Create_Pixmap_Icon(help_xpm);
    gtk_container_add(GTK_CONTAINER(LegendButton),Icon);
    gtk_box_pack_start(GTK_BOX(HBox1),LegendButton,FALSE,FALSE,0);
    gtk_button_set_relief(GTK_BUTTON(LegendButton),GTK_RELIEF_NONE);
    gtk_tooltips_set_tip(Tips,LegendButton,_("Show / Hide Legend"),NULL);
    gtk_signal_connect(GTK_OBJECT(LegendButton),"toggled",(GtkSignalFunc)Scan_Toggle_Legend_Button,NULL);

    /* Close button */
    Button = gtk_button_new();
    Icon = Create_Pixmap_Icon(exit_xpm);
    gtk_container_add(GTK_CONTAINER(Button),Icon);
    gtk_box_pack_start(GTK_BOX(HBox1),Button,FALSE,FALSE,0);
    gtk_button_set_relief(GTK_BUTTON(Button),GTK_RELIEF_NONE);
    gtk_tooltips_set_tip(Tips,Button,_("Close this window"),NULL);
    gtk_signal_connect(GTK_OBJECT(Button),"clicked",(GtkSignalFunc)ScannerWindow_Quit,NULL);

    /*
     * Frame for Scan Tag
     */
    ScanTagFrame = gtk_frame_new (_("Scan Tag"));
    gtk_box_pack_start(GTK_BOX(ScanVBox),ScanTagFrame,FALSE,FALSE,0);

    vbox = gtk_vbox_new(FALSE,4);
    gtk_container_add(GTK_CONTAINER(ScanTagFrame),vbox);
    gtk_container_border_width(GTK_CONTAINER(vbox),4);
    gtk_widget_show(vbox);

    /* The combo box + Status icon */
    HBox2 = gtk_hbox_new(FALSE,2);
    gtk_box_pack_start(GTK_BOX(vbox),HBox2,TRUE,TRUE,0);
    // The combo box to select the mask to apply
    ScannerScanTagMaskCombo = gtk_combo_new();
    gtk_box_pack_start(GTK_BOX(HBox2),ScannerScanTagMaskCombo,TRUE,TRUE,2);
    gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(ScannerScanTagMaskCombo)->entry),TRUE);
    gtk_combo_set_use_arrows_always(GTK_COMBO(ScannerScanTagMaskCombo),TRUE);
    gtk_tooltips_set_tip(Tips,GTK_COMBO(ScannerScanTagMaskCombo)->entry,_("Select or type in a mask "
        "using codes (see Legend) to parse file name and path. Used to fill in tag fields."),NULL);
    // Signal to generate preview (preview of the new filename)
    gtk_signal_connect_object(GTK_OBJECT(GTK_COMBO(ScannerScanTagMaskCombo)->entry),"changed",
        GTK_SIGNAL_FUNC(Scan_Fill_Tag_Generate_Preview),NULL);
    // Load masks into the combobox from a file
    MasksList = Load_Scan_Tag_Masks_List();
    // No mask read => load default masks
    if (MasksList==NULL)
    {
        g_print(_("Loading default 'Fill Tag' masks...\n"));
        for (i=0;i<(gint)(sizeof(Scan_Masks)/sizeof(Scan_Masks[0]));i++)
            MasksList = g_list_append(MasksList,Scan_Masks[i]);
    }
    gtk_combo_set_popdown_strings(GTK_COMBO(ScannerScanTagMaskCombo),MasksList);
    // Mask status icon
    MaskStatusIconBox = Create_Pixmap_Icon_With_Event_Box(forbidden_xpm);
    gtk_box_pack_start(GTK_BOX(HBox2),MaskStatusIconBox,FALSE,FALSE,0);
    gtk_tooltips_set_tip(Tips,MaskStatusIconBox,_("Invalid Scanner Mask"),NULL);
    // Signal connection to check if mask is correct into the mask entry
    gtk_signal_connect_object(GTK_OBJECT(GTK_COMBO(ScannerScanTagMaskCombo)->entry),"changed",
        GTK_SIGNAL_FUNC(Scan_Check_Scan_Tag_Mask),GTK_OBJECT(MaskStatusIconBox));

    /* Preview label */
    FillTagPreviewLabel = gtk_label_new(_("Fill tag preview..."));
    gtk_label_set_line_wrap(GTK_LABEL(FillTagPreviewLabel),TRUE);
    gtk_widget_show(FillTagPreviewLabel);
    gtk_box_pack_start(GTK_BOX(vbox),FillTagPreviewLabel,TRUE,TRUE,0);

    

    /*
     * Frame for Rename File
     */
    RenameFileFrame = gtk_frame_new (_("Scan File Name"));
    gtk_box_pack_start(GTK_BOX(ScanVBox),RenameFileFrame,FALSE,FALSE,0);

    vbox = gtk_vbox_new(FALSE,4);
    gtk_container_add(GTK_CONTAINER(RenameFileFrame),vbox);
    gtk_container_border_width(GTK_CONTAINER(vbox),4);
    gtk_widget_show(vbox);

    /* The combo box + Status icon */
    HBox4 = gtk_hbox_new(FALSE,2);
    gtk_box_pack_start(GTK_BOX(vbox),HBox4,TRUE,TRUE,0);
    // The combo box to select the mask to apply
    ScannerRenameFileMaskCombo = gtk_combo_new();
    gtk_box_pack_start(GTK_BOX(HBox4),ScannerRenameFileMaskCombo,TRUE,TRUE,2);
    gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(ScannerRenameFileMaskCombo)->entry),TRUE);
    gtk_combo_set_use_arrows_always(GTK_COMBO(ScannerRenameFileMaskCombo),TRUE);
    gtk_container_border_width(GTK_CONTAINER(GTK_BIN(RenameFileFrame)->child),4);
    gtk_tooltips_set_tip(Tips,GTK_COMBO(ScannerRenameFileMaskCombo)->entry,_("Select or type in "
        "a mask using codes (see Legend) to parse tag fields. Used to rename the file."),NULL);
    // Signal to generate preview (preview of the new filename)
    gtk_signal_connect_object(GTK_OBJECT(GTK_COMBO(ScannerRenameFileMaskCombo)->entry),"changed",
        GTK_SIGNAL_FUNC(Scan_Rename_File_Generate_Preview),NULL);
    // Load masks into the combobox
    RenameFileMasksList = Load_Rename_File_Masks_List();
    // Load default masks
    if (RenameFileMasksList==NULL)
    {
        g_print(_("Loading default 'Rename File' masks...\n"));
        for (i=0;i<(gint)(sizeof(Rename_File_Masks)/sizeof(Rename_File_Masks[0]));i++)
            RenameFileMasksList = g_list_append(RenameFileMasksList,Rename_File_Masks[i]);
    }
    gtk_combo_set_popdown_strings(GTK_COMBO(ScannerRenameFileMaskCombo),RenameFileMasksList);
    // Mask status icon
    MaskStatusIconBox = Create_Pixmap_Icon_With_Event_Box(forbidden_xpm);
    gtk_box_pack_start(GTK_BOX(HBox4),MaskStatusIconBox,FALSE,FALSE,0);
    gtk_tooltips_set_tip(Tips,MaskStatusIconBox,_("Invalid Scanner Mask"),NULL);
    // Signal connection to check if mask is correct into the mask entry
    gtk_signal_connect_object(GTK_OBJECT(GTK_COMBO(ScannerRenameFileMaskCombo)->entry),"changed",
        GTK_SIGNAL_FUNC(Scan_Check_Rename_File_Mask),GTK_OBJECT(MaskStatusIconBox));

    /* Preview label */
    RenameFilePreviewLabel = gtk_label_new(_("Rename file preview..."));
    gtk_label_set_line_wrap(GTK_LABEL(RenameFilePreviewLabel),TRUE);
    gtk_widget_show(RenameFilePreviewLabel);
    gtk_box_pack_start(GTK_BOX(vbox),RenameFilePreviewLabel,TRUE,TRUE,0);



    /*
     * Frame for Processing Fields
     */
    ProcessFieldsFrame = gtk_frame_new (_("Process Fields"));
    gtk_box_pack_start(GTK_BOX(ScanVBox),ProcessFieldsFrame,FALSE,FALSE,0);

    VBox = gtk_vbox_new(FALSE,0);
    gtk_container_add(GTK_CONTAINER(ProcessFieldsFrame),VBox);
    gtk_container_border_width(GTK_CONTAINER(VBox),4);
    gtk_widget_show(VBox);

    /* Group: select entry fields to process */
    hbox = gtk_hbox_new(FALSE,0);
    gtk_box_pack_start(GTK_BOX(VBox),hbox,FALSE,FALSE,2);
    EventBox = gtk_event_box_new();
    Label = gtk_label_new(_("Select fields:"));
    gtk_box_pack_start(GTK_BOX(hbox),EventBox,FALSE,FALSE,2);
    gtk_container_add(GTK_CONTAINER(EventBox),Label);
    gtk_tooltips_set_tip(Tips,EventBox,_("The buttons on the right represent the fields which can "
        "be processed. Select those who interest you."),NULL);
    ProcessFileNameField = gtk_toggle_button_new_with_label(_("F"));    
    gtk_tooltips_set_tip(Tips,ProcessFileNameField,         _("Process file name field"),NULL);
    ProcessTitleField = gtk_toggle_button_new_with_label(   _("T"));    
    gtk_tooltips_set_tip(Tips,ProcessTitleField,            _("Process title field"),NULL);
    ProcessArtistField = gtk_toggle_button_new_with_label(  _("Ar"));    
    gtk_tooltips_set_tip(Tips,ProcessArtistField,           _("Process file artist field"),NULL);
    ProcessAlbumField = gtk_toggle_button_new_with_label(   _("Al"));    
    gtk_tooltips_set_tip(Tips,ProcessAlbumField,            _("Process album field"),NULL);
    ProcessGenreField = gtk_toggle_button_new_with_label(   _("G"));    
    gtk_tooltips_set_tip(Tips,ProcessGenreField,            _("Process genre field"),NULL);
    ProcessCommentField = gtk_toggle_button_new_with_label( _("C"));    
    gtk_tooltips_set_tip(Tips,ProcessCommentField,          _("Process comment field"),NULL);
    gtk_box_pack_start(GTK_BOX(hbox),ProcessFileNameField,TRUE,TRUE,2);
    gtk_box_pack_start(GTK_BOX(hbox),ProcessTitleField,   TRUE,TRUE,2);
    gtk_box_pack_start(GTK_BOX(hbox),ProcessArtistField,  TRUE,TRUE,2);
    gtk_box_pack_start(GTK_BOX(hbox),ProcessAlbumField,   TRUE,TRUE,2);
    gtk_box_pack_start(GTK_BOX(hbox),ProcessGenreField,   TRUE,TRUE,2);
    gtk_box_pack_start(GTK_BOX(hbox),ProcessCommentField, TRUE,TRUE,2);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(ProcessFileNameField),PROCESS_FILENAME_FIELD);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(ProcessTitleField),   PROCESS_TITLE_FIELD);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(ProcessArtistField),  PROCESS_ARTIST_FIELD);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(ProcessAlbumField),   PROCESS_ALBUM_FIELD);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(ProcessGenreField),   PROCESS_GENRE_FIELD);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(ProcessCommentField), PROCESS_COMMENT_FIELD);
    gtk_signal_connect(GTK_OBJECT(ProcessFileNameField),"toggled",GTK_SIGNAL_FUNC(Select_Fields_Set_Sensitive),NULL);
    gtk_signal_connect(GTK_OBJECT(ProcessTitleField),   "toggled",GTK_SIGNAL_FUNC(Select_Fields_Set_Sensitive),NULL);
    gtk_signal_connect(GTK_OBJECT(ProcessArtistField),  "toggled",GTK_SIGNAL_FUNC(Select_Fields_Set_Sensitive),NULL);
    gtk_signal_connect(GTK_OBJECT(ProcessAlbumField),   "toggled",GTK_SIGNAL_FUNC(Select_Fields_Set_Sensitive),NULL);
    gtk_signal_connect(GTK_OBJECT(ProcessGenreField),   "toggled",GTK_SIGNAL_FUNC(Select_Fields_Set_Sensitive),NULL);
    gtk_signal_connect(GTK_OBJECT(ProcessCommentField), "toggled",GTK_SIGNAL_FUNC(Select_Fields_Set_Sensitive),NULL);
    /* The small buttons */
    vbox = gtk_vbox_new(FALSE,2);
    gtk_box_pack_start(GTK_BOX(hbox),vbox,FALSE,FALSE,0);
    Button = gtk_button_new();
    gtk_signal_connect(GTK_OBJECT(Button),"clicked",GTK_SIGNAL_FUNC(Select_Fields_Invert_Selection),NULL);
    gtk_box_pack_start(GTK_BOX(vbox),Button,FALSE,FALSE,0);
    gtk_widget_set_usize(Button,10,10);
    Icon = Create_Pixmap_Icon(blackwhite_xpm);
    gtk_container_add(GTK_CONTAINER(Button),Icon);
    gtk_tooltips_set_tip(Tips,Button,_("Invert Selection"),NULL);
    Button = gtk_button_new();
    gtk_signal_connect(GTK_OBJECT(Button),"clicked",GTK_SIGNAL_FUNC(Select_Fields_Select_Unselect_All),NULL);
    gtk_box_pack_start(GTK_BOX(vbox),Button,FALSE,FALSE,0);
    gtk_widget_set_usize(Button,10,10);
    gtk_tooltips_set_tip(Tips,Button,_("Select/Unselect All."),NULL);
     
    /* Separator line */
    Separator = gtk_hseparator_new();
    gtk_box_pack_start(GTK_BOX(VBox),Separator,FALSE,FALSE,0);

    /* Group: character conversion */
    ProcessFieldsConvertIntoSpace = gtk_check_button_new_with_label(_("Convert '_' and '%20' to ' '"));
    ProcessFieldsConvertSpace     = gtk_check_button_new_with_label(_("Convert ' ' to '_'"));
    gtk_box_pack_start(GTK_BOX(VBox),ProcessFieldsConvertIntoSpace,FALSE,FALSE,0);
    gtk_box_pack_start(GTK_BOX(VBox),ProcessFieldsConvertSpace,    FALSE,FALSE,0);
    hbox = gtk_hbox_new(FALSE,2);
    ProcessFieldsConvert          = gtk_check_button_new_with_label(_("Convert :"));  // Patch from Ben Hearsum, Oct. 3, 2003
    ProcessFieldsConvertTo        = gtk_entry_new_with_max_length(1);
    Label                         = gtk_label_new(_("to : ")); // A "space" at the end to allow an other traduction for "to :" (needed in French!)
    ProcessFieldsConvertFrom      = gtk_entry_new_with_max_length(1);
    gtk_widget_set_usize(ProcessFieldsConvertTo,40,-1);
    gtk_widget_set_usize(ProcessFieldsConvertFrom,40,-1);
    gtk_box_pack_start(GTK_BOX(hbox),ProcessFieldsConvert,    FALSE,FALSE,0);
    gtk_box_pack_start(GTK_BOX(hbox),ProcessFieldsConvertFrom,FALSE,FALSE,0);
    gtk_box_pack_start(GTK_BOX(hbox),Label,                   FALSE,FALSE,4);
    gtk_box_pack_start(GTK_BOX(hbox),ProcessFieldsConvertTo,  FALSE,FALSE,0);
    gtk_box_pack_start(GTK_BOX(VBox),hbox,FALSE,FALSE,0);
    /* List creation for check buttons in group */
    pf_cb_group1 = g_list_append (pf_cb_group1,ProcessFieldsConvertIntoSpace);
    pf_cb_group1 = g_list_append (pf_cb_group1,ProcessFieldsConvertSpace);
    pf_cb_group1 = g_list_append (pf_cb_group1,ProcessFieldsConvert);
    /* Toggled signals */
    gtk_signal_connect(GTK_OBJECT(ProcessFieldsConvertIntoSpace),"toggled",Process_Fields_Check_Button_Toggled,pf_cb_group1);
    gtk_signal_connect(GTK_OBJECT(ProcessFieldsConvertSpace),    "toggled",Process_Fields_Check_Button_Toggled,pf_cb_group1);
    gtk_signal_connect(GTK_OBJECT(ProcessFieldsConvert),         "toggled",Process_Fields_Check_Button_Toggled,pf_cb_group1);
    gtk_signal_connect(GTK_OBJECT(ProcessFieldsConvert),         "toggled",Process_Fields_Convert_Check_Button_Toggled,NULL);
    /* Set check buttons to init value */
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(ProcessFieldsConvertIntoSpace),PF_CONVERT_INTO_SPACE);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(ProcessFieldsConvertSpace),PF_CONVERT_SPACE);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(ProcessFieldsConvert),PF_CONVERT);
    /* Tooltips */
    gtk_tooltips_set_tip(Tips,ProcessFieldsConvertIntoSpace,
        _("The underscore character or the string '%20' are replaced by one space. "
          "Example, before: 'Text%20In%20An_Entry', after: 'Text In An Entry'."),NULL);
    gtk_tooltips_set_tip(Tips,ProcessFieldsConvertSpace,
        _("The space character is replaced by one underscore character. "
          "Example, before: 'Text In An Entry', after: 'Text_In_An_Entry'."),NULL);
    gtk_tooltips_set_tip(Tips,ProcessFieldsConvert,_("Replace a character by an other one."),NULL);

    /* Separator line */
    Separator = gtk_hseparator_new();
    gtk_box_pack_start(GTK_BOX(VBox),Separator,FALSE,FALSE,0);

    /* Group: capitalize, ... */
    ProcessFieldsAllUppercase = gtk_check_button_new_with_label         (_("All uppercase"));
    ProcessFieldsAllDowncase  = gtk_check_button_new_with_label         (_("All downcase"));
    ProcessFieldsFirstLetterUppercase  = gtk_check_button_new_with_label(_("First letter uppercase"));
    ProcessFieldsFirstLettersUppercase = gtk_check_button_new_with_label(_("First letter uppercase of each word"));
    gtk_box_pack_start(GTK_BOX(VBox),ProcessFieldsAllUppercase,         FALSE,FALSE,0);
    gtk_box_pack_start(GTK_BOX(VBox),ProcessFieldsAllDowncase,          FALSE,FALSE,0);
    gtk_box_pack_start(GTK_BOX(VBox),ProcessFieldsFirstLetterUppercase, FALSE,FALSE,0);
    gtk_box_pack_start(GTK_BOX(VBox),ProcessFieldsFirstLettersUppercase,FALSE,FALSE,0);
    /* List creation for check buttons in group */
    pf_cb_group2 = g_list_append(pf_cb_group2,ProcessFieldsAllUppercase);
    pf_cb_group2 = g_list_append(pf_cb_group2,ProcessFieldsAllDowncase);
    pf_cb_group2 = g_list_append(pf_cb_group2,ProcessFieldsFirstLetterUppercase);
    pf_cb_group2 = g_list_append(pf_cb_group2,ProcessFieldsFirstLettersUppercase);
    /* Toggled signals */
    gtk_signal_connect(GTK_OBJECT(ProcessFieldsAllUppercase),"toggled",Process_Fields_Check_Button_Toggled,pf_cb_group2);
    gtk_signal_connect(GTK_OBJECT(ProcessFieldsAllDowncase), "toggled",Process_Fields_Check_Button_Toggled,pf_cb_group2);
    gtk_signal_connect(GTK_OBJECT(ProcessFieldsFirstLetterUppercase),"toggled",Process_Fields_Check_Button_Toggled,pf_cb_group2);
    gtk_signal_connect(GTK_OBJECT(ProcessFieldsFirstLettersUppercase),"toggled",Process_Fields_Check_Button_Toggled,pf_cb_group2);
    /* Set check buttons to init value */
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(ProcessFieldsAllUppercase),PF_CONVERT_ALL_UPPERCASE);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(ProcessFieldsAllDowncase),PF_CONVERT_ALL_DOWNCASE);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(ProcessFieldsFirstLetterUppercase),PF_CONVERT_FIRST_LETTER_UPPERCASE);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(ProcessFieldsFirstLettersUppercase),PF_CONVERT_FIRST_LETTERS_UPPERCASE);
    /* Tooltips */
    gtk_tooltips_set_tip(Tips,ProcessFieldsAllUppercase,
        _("Convert all words in all fields to upper case. "
          "Example, before: 'Text IN AN entry', after: 'TEXT IN AN ENTRY'."),NULL);
    gtk_tooltips_set_tip(Tips,ProcessFieldsAllDowncase,
        _("Convert all words in all fields to lower case. "
          "Example, before: 'TEXT IN an entry', after: 'text in an entry'."),NULL);
    gtk_tooltips_set_tip(Tips,ProcessFieldsFirstLetterUppercase,
        _("Convert the initial of the first word in all fields to upper case. "
          "Example, before: 'text IN An ENTRY', after: 'Text in an entry'."),NULL);
    gtk_tooltips_set_tip(Tips,ProcessFieldsFirstLettersUppercase,
        _("Convert the initial of each word in all fields to upper case. "
          "Example, before: 'Text in an ENTRY', after: 'Text In An Entry'."),NULL);

    /* Separator line */
    Separator = gtk_hseparator_new();
    gtk_box_pack_start(GTK_BOX(VBox),Separator,FALSE,FALSE,0);

    /* Group: insert/remove spaces */
    ProcessFieldsRemoveSpace = gtk_check_button_new_with_label(_("Remove spaces"));
    ProcessFieldsInsertSpace = gtk_check_button_new_with_label(_("Insert a space before an uppercase letter"));
    ProcessFieldsOnlyOneSpace = gtk_check_button_new_with_label(_("Remove duplicates of space or underscore"));
    gtk_box_pack_start(GTK_BOX(VBox),ProcessFieldsRemoveSpace,FALSE,FALSE,0);
    gtk_box_pack_start(GTK_BOX(VBox),ProcessFieldsInsertSpace,FALSE,FALSE,0);
    gtk_box_pack_start(GTK_BOX(VBox),ProcessFieldsOnlyOneSpace,FALSE,FALSE,0);
    /* List creation for check buttons in group */
    pf_cb_group3 = g_list_append (pf_cb_group3,ProcessFieldsRemoveSpace);
    pf_cb_group3 = g_list_append (pf_cb_group3,ProcessFieldsInsertSpace);
    pf_cb_group3 = g_list_append (pf_cb_group3,ProcessFieldsOnlyOneSpace);
    /* Toggled signals */
    gtk_signal_connect(GTK_OBJECT(ProcessFieldsRemoveSpace), "toggled",Process_Fields_Check_Button_Toggled,pf_cb_group3);
    gtk_signal_connect(GTK_OBJECT(ProcessFieldsInsertSpace), "toggled",Process_Fields_Check_Button_Toggled,pf_cb_group3);
    gtk_signal_connect(GTK_OBJECT(ProcessFieldsOnlyOneSpace),"toggled",Process_Fields_Check_Button_Toggled,pf_cb_group3);
    /* Set check buttons to init value */
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(ProcessFieldsRemoveSpace),PF_REMOVE_SPACE);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(ProcessFieldsInsertSpace),PF_INSERT_SPACE);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(ProcessFieldsOnlyOneSpace),PF_ONLY_ONE_SPACE);
    /* Tooltips */
    gtk_tooltips_set_tip(Tips,ProcessFieldsRemoveSpace,
        _("All spaces between words are removed. "
          "Example, before: 'Text In An Entry', after: 'TextInAnEntry'."),NULL);
    gtk_tooltips_set_tip(Tips,ProcessFieldsInsertSpace,
        _("A space is inserted before each upper case letter. "
          "Example, before: 'TextInAnEntry', after: 'Text In An Entry'."),NULL);
    gtk_tooltips_set_tip(Tips,ProcessFieldsOnlyOneSpace,
        _("Duplicated spaces or underscores are removed. "
          "Example, before: 'Text__In__An   Entry', after: 'Text_In_An Entry'."),NULL);



    /*
     * Frame to display codes legend
     */
    LegendFrame = gtk_frame_new (_("Legend"));
    gtk_box_pack_start(GTK_BOX(ScanVBox),LegendFrame,FALSE,FALSE,0);
    /* Legend labels */
    Table = gtk_table_new(3,3,FALSE);
    gtk_container_add(GTK_CONTAINER(LegendFrame),Table);
    gtk_container_border_width(GTK_CONTAINER(Table),4);
    Label = gtk_label_new(_("%a : artist"));
    gtk_table_attach_defaults(GTK_TABLE(Table),Label,0,1,0,1);
    gtk_misc_set_alignment(GTK_MISC(Label),0,0.5);
    Label = gtk_label_new(_("%b : album"));
    gtk_table_attach_defaults(GTK_TABLE(Table),Label,0,1,1,2);
    gtk_misc_set_alignment(GTK_MISC(Label),0,0.5);
    Label = gtk_label_new(_("%c : comment"));
    gtk_table_attach_defaults(GTK_TABLE(Table),Label,0,1,2,3);
    gtk_misc_set_alignment(GTK_MISC(Label),0,0.5);
    Label = gtk_label_new(_("%g : genre"));
    gtk_table_attach_defaults(GTK_TABLE(Table),Label,1,2,0,1);
    gtk_misc_set_alignment(GTK_MISC(Label),0,0.5);
    Label = gtk_label_new(_("%i : ignored"));
    gtk_table_attach_defaults(GTK_TABLE(Table),Label,1,2,1,2);
    gtk_misc_set_alignment(GTK_MISC(Label),0,0.5);
    Label = gtk_label_new(_("%l : number of tracks"));
    gtk_table_attach_defaults(GTK_TABLE(Table),Label,1,2,2,3);
    gtk_misc_set_alignment(GTK_MISC(Label),0,0.5);
    Label = gtk_label_new(_("%n : track"));
    gtk_table_attach_defaults(GTK_TABLE(Table),Label,2,3,0,1);
    gtk_misc_set_alignment(GTK_MISC(Label),0,0.5);
    Label = gtk_label_new(_("%t : title"));
    gtk_table_attach_defaults(GTK_TABLE(Table),Label,2,3,1,2);
    gtk_misc_set_alignment(GTK_MISC(Label),0,0.5);
    Label = gtk_label_new(_("%y : year"));
    gtk_table_attach_defaults(GTK_TABLE(Table),Label,2,3,2,3);
    gtk_misc_set_alignment(GTK_MISC(Label),0,0.5);


    /*
     * Masks Editor
     */
    MaskEditorFrame = gtk_frame_new (_("Mask Editor"));
    gtk_box_pack_start(GTK_BOX(ScanVBox),MaskEditorFrame,FALSE,FALSE,0);
    MaskEditorHBox = gtk_hbox_new(FALSE,4);
    gtk_container_add(GTK_CONTAINER(MaskEditorFrame),MaskEditorHBox);
    gtk_container_border_width(GTK_CONTAINER(MaskEditorHBox),4);

    /* The editor part */
    MaskEditorVBox = gtk_vbox_new(FALSE,2);
    gtk_box_pack_start(GTK_BOX(MaskEditorHBox),MaskEditorVBox,TRUE,TRUE,0);
    MaskEditorScrollWindow = gtk_scrolled_window_new(NULL,NULL);
    gtk_box_pack_start(GTK_BOX(MaskEditorVBox),MaskEditorScrollWindow,TRUE,TRUE,0);
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(MaskEditorScrollWindow),
                                   GTK_POLICY_AUTOMATIC,GTK_POLICY_AUTOMATIC);
    gtk_widget_set_usize(GTK_WIDGET(MaskEditorScrollWindow),-1,101);

    /* The Clist */
    MaskEditorClist = gtk_clist_new(1);
    gtk_container_add(GTK_CONTAINER(MaskEditorScrollWindow),MaskEditorClist);
    gtk_clist_set_reorderable(GTK_CLIST(MaskEditorClist),TRUE);
    gtk_clist_set_column_auto_resize(GTK_CLIST(MaskEditorClist),0,TRUE);
    gtk_clist_set_selection_mode(GTK_CLIST(MaskEditorClist),GTK_SELECTION_EXTENDED);
    gtk_signal_connect_after(GTK_OBJECT(MaskEditorClist),"select-row",
        (GtkSignalFunc)Mask_Editor_Clist_Select_Row,NULL);
    gtk_signal_connect(GTK_OBJECT(MaskEditorClist),"key-press-event",
        (GtkSignalFunc)Mask_Editor_Clist_Key_Press,NULL);

    /* The entry */
    hbox = gtk_hbox_new(FALSE,2);
    gtk_box_pack_start(GTK_BOX(MaskEditorVBox),hbox,FALSE,FALSE,0);
    MaskEditorEntry = gtk_entry_new();
    gtk_box_pack_start(GTK_BOX(hbox),MaskEditorEntry,TRUE,TRUE,2);
    gtk_signal_connect(GTK_OBJECT(MaskEditorEntry),"changed",
        (GtkSignalFunc)Mask_Editor_Entry_Changed,NULL);
    // Mask status icon
    MaskStatusIconBox = Create_Pixmap_Icon_With_Event_Box(forbidden_xpm);
    gtk_box_pack_start(GTK_BOX(hbox),MaskStatusIconBox,FALSE,FALSE,0);
    gtk_tooltips_set_tip(Tips,MaskStatusIconBox,_("Invalid Scanner Mask"),NULL);
    // Signal connection to check if mask is correct into the mask entry
    gtk_signal_connect_object(GTK_OBJECT(MaskEditorEntry),"changed",
        GTK_SIGNAL_FUNC(Scan_Check_Editor_Mask),GTK_OBJECT(MaskStatusIconBox));

    /* The buttons part */
    MaskEditorVBox = gtk_vbox_new(FALSE,0);
    gtk_box_pack_start(GTK_BOX(MaskEditorHBox),MaskEditorVBox,FALSE,FALSE,0);

    /* New mask button */
    MaskEditorNewButton = gtk_button_new();
    Icon = Create_Pixmap_Icon(new_xpm);
    gtk_container_add(GTK_CONTAINER(MaskEditorNewButton),Icon);
    gtk_box_pack_start(GTK_BOX(MaskEditorVBox),MaskEditorNewButton,FALSE,FALSE,0);
    gtk_button_set_relief(GTK_BUTTON(MaskEditorNewButton),GTK_RELIEF_NONE);
    gtk_tooltips_set_tip(Tips,MaskEditorNewButton,_("Create New Mask"),NULL);
    gtk_signal_connect(GTK_OBJECT(MaskEditorNewButton),"clicked",
        (GtkSignalFunc)Mask_Editor_Clist_New,NULL);

    /* Move up mask button */
    MaskEditorUpButton = gtk_button_new();
    Icon = Create_Pixmap_Icon(up_xpm);
    gtk_container_add(GTK_CONTAINER(MaskEditorUpButton),Icon);
    gtk_box_pack_start(GTK_BOX(MaskEditorVBox),MaskEditorUpButton,FALSE,FALSE,0);
    gtk_button_set_relief(GTK_BUTTON(MaskEditorUpButton),GTK_RELIEF_NONE);
    gtk_tooltips_set_tip(Tips,MaskEditorUpButton,_("Move Up this Mask"),NULL);
    gtk_signal_connect(GTK_OBJECT(MaskEditorUpButton),"clicked",
        (GtkSignalFunc)Mask_Editor_Clist_Move_Up,NULL);

    /* Move down mask button */
    MaskEditorDownButton = gtk_button_new();
    Icon = Create_Pixmap_Icon(down_xpm);
    gtk_container_add(GTK_CONTAINER(MaskEditorDownButton),Icon);
    gtk_box_pack_start(GTK_BOX(MaskEditorVBox),MaskEditorDownButton,FALSE,FALSE,0);
    gtk_button_set_relief(GTK_BUTTON(MaskEditorDownButton),GTK_RELIEF_NONE);
    gtk_tooltips_set_tip(Tips,MaskEditorDownButton,_("Move Down this Mask"),NULL);
    gtk_signal_connect(GTK_OBJECT(MaskEditorDownButton),"clicked",
        (GtkSignalFunc)Mask_Editor_Clist_Move_Down,NULL);

    /* Copy mask button */
    MaskEditorCopyButton = gtk_button_new();
    Icon = Create_Pixmap_Icon(copy_xpm);
    gtk_container_add(GTK_CONTAINER(MaskEditorCopyButton),Icon);
    gtk_box_pack_start(GTK_BOX(MaskEditorVBox),MaskEditorCopyButton,FALSE,FALSE,0);
    gtk_button_set_relief(GTK_BUTTON(MaskEditorCopyButton),GTK_RELIEF_NONE);
    gtk_tooltips_set_tip(Tips,MaskEditorCopyButton,_("Duplicate Mask"),NULL);
    gtk_signal_connect(GTK_OBJECT(MaskEditorCopyButton),"clicked",
        (GtkSignalFunc)Mask_Editor_Clist_Copy,NULL);

    /* Add mask button */
    MaskEditorAddButton = gtk_button_new();
    Icon = Create_Pixmap_Icon(add_xpm);
    gtk_container_add(GTK_CONTAINER(MaskEditorAddButton),Icon);
    gtk_box_pack_start(GTK_BOX(MaskEditorVBox),MaskEditorAddButton,FALSE,FALSE,0);
    gtk_button_set_relief(GTK_BUTTON(MaskEditorAddButton),GTK_RELIEF_NONE);
    gtk_tooltips_set_tip(Tips,MaskEditorAddButton,_("Add Default Masks"),NULL);
    gtk_signal_connect(GTK_OBJECT(MaskEditorAddButton),"clicked",
        (GtkSignalFunc)Mask_Editor_Clist_Add,NULL);

    /* Remove mask button */
    MaskEditorRemoveButton = gtk_button_new();
    Icon = Create_Pixmap_Icon(trash_xpm);
    gtk_container_add(GTK_CONTAINER(MaskEditorRemoveButton),Icon);
    gtk_box_pack_start(GTK_BOX(MaskEditorVBox),MaskEditorRemoveButton,FALSE,FALSE,0);
    gtk_button_set_relief(GTK_BUTTON(MaskEditorRemoveButton),GTK_RELIEF_NONE);
    gtk_tooltips_set_tip(Tips,MaskEditorRemoveButton,_("Remove Mask"),NULL);
    gtk_signal_connect(GTK_OBJECT(MaskEditorRemoveButton),"clicked",
        (GtkSignalFunc)Mask_Editor_Clist_Remove,NULL);

    /* Save mask button */
    MaskEditorSaveButton = gtk_button_new();
    Icon = Create_Pixmap_Icon(save_xpm);
    gtk_container_add(GTK_CONTAINER(MaskEditorSaveButton),Icon);
    gtk_box_pack_end(GTK_BOX(MaskEditorVBox),MaskEditorSaveButton,FALSE,FALSE,0);
    gtk_button_set_relief(GTK_BUTTON(MaskEditorSaveButton),GTK_RELIEF_NONE);
    gtk_tooltips_set_tip(Tips,MaskEditorSaveButton,_("Save Masks"),NULL);
    gtk_signal_connect(GTK_OBJECT(MaskEditorSaveButton),"clicked",
        (GtkSignalFunc)Mask_Editor_Clist_Save_Button,NULL);

    /* Load masks list into clist */
    Mask_Editor_Clist_Load_Mask_List(MasksList);

    gtk_widget_show(ScanVBox);
    gtk_widget_show_all(HBox1);
    gtk_widget_show_all(HBox2);
    gtk_widget_show_all(HBox4);
    gtk_widget_show(ScannerWindow);

    /* Init position of the scanner window */
    Scan_Set_Scanner_Window_Init_Position();

    /* To initialize the mask status icon and visibility */
    gtk_signal_emit_by_name(GTK_OBJECT(GTK_COMBO(ScannerScanTagMaskCombo)->entry),"changed");
    gtk_signal_emit_by_name(GTK_OBJECT(GTK_COMBO(ScannerRenameFileMaskCombo)->entry),"changed");
    gtk_signal_emit_by_name(GTK_OBJECT(MaskEditorEntry),"changed");
    gtk_signal_emit_by_name(GTK_OBJECT(LegendButton),"toggled");        /* To hide legend frame */
    gtk_signal_emit_by_name(GTK_OBJECT(MaskEditorButton),"toggled");    /* To hide mask editor frame */
    gtk_signal_emit_by_name(GTK_OBJECT(ProcessFieldsConvert),"toggled");    /* To enable / disable entries */

    // Activate the current menu in the option menu
    Scanner_Option_Menu_Activate_Item(NULL,scanner_type);
}
void ScannerWindow_Key_Press (GtkWidget *window, GdkEvent *event)
{
    GdkEventKey *kevent;

    if (event && event->type == GDK_KEY_PRESS)
    {
        kevent = (GdkEventKey *)event;
        switch(kevent->keyval)
        {
            case GDK_Escape:
                ScannerWindow_Quit();
                break;
        }
    }
}



/*
 * Select the scanner to run for the current ETFile
 */
void Scan_Select_Mode_And_Run_Scanner (ET_File *ETFile)
{
    if (!ScannerWindow && !ETFile) return;

    if (Scanner_Option_Menu_Get_Active_Menu_Item(SCANNER_FILL_TAG))
    {
        /* Run Scanner Type: Scan Tag */
        Scan_Tag_With_Mask(ETFile);
    }else if (Scanner_Option_Menu_Get_Active_Menu_Item(SCANNER_RENAME_FILE))
    {
        /* Run Scanner Type: Rename File */
        Scan_Rename_File_With_Mask(ETFile);
    }else if (Scanner_Option_Menu_Get_Active_Menu_Item(SCANNER_PROCESS_FIELDS))
    {
        /* Run Scanner Type: Process Fields */
        Scan_Process_Fields(ETFile);
    }
}

void Scan_Use_Fill_Tag_Scanner (void)
{
    if (!ScannerWindow || !Scanner_Option_Menu_Get_Active_Menu_Item(SCANNER_FILL_TAG))
        Open_ScannerWindow(SCANNER_FILL_TAG);
    else
        Action_Scan_Selected_Files();
}


void Scan_Use_Rename_File_Scanner (void)
{
    if (!ScannerWindow || !Scanner_Option_Menu_Get_Active_Menu_Item(SCANNER_RENAME_FILE))
        Open_ScannerWindow(SCANNER_RENAME_FILE);
    else
        Action_Scan_Selected_Files();
}

void Scan_Use_Process_Fields_Scanner (void)
{
    if (!ScannerWindow || !Scanner_Option_Menu_Get_Active_Menu_Item(SCANNER_PROCESS_FIELDS))
        Open_ScannerWindow(SCANNER_PROCESS_FIELDS);
    else
        Action_Scan_Selected_Files();
}


/* Callback from Open_ScannerWindow */
void ScannerWindow_Quit (void)
{
    if (ScannerWindow)
    {
        Scan_Apply_Changes();

        gtk_widget_destroy(ScannerWindow);
        ScannerWindow     = (GtkWidget *)NULL;
        ScannerOptionMenu = (GtkWidget *)NULL;
        SWScanButton      = (GtkWidget *)NULL;

        // To avoid crashs after tests
        ScannerScanTagMaskCombo       = (GtkWidget *)NULL;
        ScannerRenameFileMaskCombo    = (GtkWidget *)NULL;
        MaskEditorEntry               = (GtkWidget *)NULL;
        LegendFrame                   = (GtkWidget *)NULL;
        ProcessFieldsConvertIntoSpace = (GtkWidget *)NULL;
        ProcessFieldsConvertSpace     = (GtkWidget *)NULL;
        FillTagPreviewLabel           = (GtkWidget *)NULL;
        RenameFilePreviewLabel        = (GtkWidget *)NULL;
    }
}


/* Callback from Option button */
void Scan_Option_Button (void)
{
    Open_OptionsWindow();
    gtk_notebook_set_page(GTK_NOTEBOOK(OptionsNoteBook),OptionsNoteBook_Scanner_Page_Num);
}



/*
 * Check if mask of the "Scan Tag" is valid. Return TRUE if valid, else FALSE.
 */
gboolean Scan_Check_Scan_Tag_Mask (GtkObject *widget_to_show_hide, GtkEntry *widget_source)
{
    gchar *tmp  = NULL;
    gchar *mask = NULL;
    gint loop = 0;


    if (!widget_to_show_hide || !widget_source)
        goto Bad_Mask;

    mask = g_strdup(gtk_entry_get_text(GTK_ENTRY(widget_source)));
    if (!mask || strlen(mask)<1)
        goto Bad_Mask;

    while (mask)
    {
        if ( (tmp=strrchr(mask,'%'))==NULL )
        {
            if (loop==0)
                /* There is no code the first time => not accepted */
                goto Bad_Mask;
            else
                /* There is no more code => accepted */
                goto Good_Mask;
        }
        if (strlen(tmp)>1 && (tmp[1]=='t' || tmp[1]=='a' || tmp[1]=='b' || tmp[1]=='y' ||
                              tmp[1]=='g' || tmp[1]=='n' || tmp[1]=='l' || tmp[1]=='c' || tmp[1]=='i'))
        {
            /* Code is correct */
            *(mask+strlen(mask)-strlen(tmp)) = '\0';
        }else
        {
            goto Bad_Mask;
        }

        /* Check the following code and separator */
        if ( (tmp=strrchr(mask,'%'))==NULL )
            /* There is no more code => accepted */
            goto Good_Mask;

        if (strlen(tmp)>2 && (tmp[1]=='t' || tmp[1]=='a' || tmp[1]=='b' || tmp[1]=='y' ||
                              tmp[1]=='g' || tmp[1]=='n' || tmp[1]=='l' || tmp[1]=='c' || tmp[1]=='i'))
        {
            /* There is a separator and code is correct */
            *(mask+strlen(mask)-strlen(tmp)) = '\0';
        }else
        {
            goto Bad_Mask;
        }
        loop++;
    }

    Bad_Mask:
        if (mask) g_free(mask);
        gtk_widget_show(GTK_WIDGET(widget_to_show_hide));
        return FALSE;

    Good_Mask:
        if (mask) g_free(mask);
        gtk_widget_hide(GTK_WIDGET(widget_to_show_hide));
        return TRUE;
}
/*
 * Check if mask of the "Rename File" is valid. Return TRUE if valid, else FALSE.
 */
gboolean Scan_Check_Rename_File_Mask (GtkObject *widget_to_show_hide, GtkEntry *widget_source)
{
    gchar *tmp = NULL;
    gchar *mask = NULL;


    if (!widget_to_show_hide || !widget_source)
        goto Bad_Mask;

    mask = g_strdup(gtk_entry_get_text(GTK_ENTRY(widget_source)));
    if (!mask || strlen(mask)<1)
        goto Bad_Mask;

    if ( strchr(mask,'/')!=NULL ) // Renaming directory is not yet available
        goto Bad_Mask;

    while (mask)
    {
        if ( (tmp=strrchr(mask,'%'))==NULL )
        {
            /* There is no more code. */
            /* No code in mask is accepted. */
            goto Good_Mask;
        }
        if (strlen(tmp)>1 && (tmp[1]=='t' || tmp[1]=='a' || tmp[1]=='b' || tmp[1]=='y' ||
                              tmp[1]=='g' || tmp[1]=='n' || tmp[1]=='l' || tmp[1]=='c' || tmp[1]=='i'))
        {
            /* The code is valid. */
            /* No separator is accepted. */
            *(mask+strlen(mask)-strlen(tmp)) = '\0';
        }else
        {
            goto Bad_Mask;
        }
    }

    Bad_Mask:
        if (mask) g_free(mask);
        gtk_widget_show(GTK_WIDGET(widget_to_show_hide));
        return FALSE;

    Good_Mask:
        if (mask) g_free(mask);
        gtk_widget_hide(GTK_WIDGET(widget_to_show_hide));
        return TRUE;
}


/*
 * Check if the selected mask in the Mask Editor is valid, else display the mask status icon.
 */
gboolean Scan_Check_Editor_Mask (GtkObject *widget_to_show_hide, GtkEntry *widget_source)
{
    /* Select and get result of check scanner */
    if (Scanner_Option_Menu_Get_Active_Menu_Item(SCANNER_FILL_TAG))
    {
        return Scan_Check_Scan_Tag_Mask(widget_to_show_hide,widget_source);
    }else if (Scanner_Option_Menu_Get_Active_Menu_Item(SCANNER_RENAME_FILE))
    {
        return Scan_Check_Rename_File_Mask(widget_to_show_hide,widget_source);
    }else
        return FALSE;
}


void Scan_Toggle_Legend_Button (void)
{
    if (!LegendButton && !LegendFrame) return;

    if ( gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(LegendButton)) )
        gtk_widget_show_all(LegendFrame);
    else
        gtk_widget_hide(LegendFrame);
}


void Scan_Toggle_Mask_Editor_Button (void)
{
    if (!MaskEditorButton && !MaskEditorFrame) return;

    if ( gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(MaskEditorButton)) )
    {
        gtk_widget_show_all(MaskEditorFrame);
        // Update status of the icon box cause prev instruction show it for all cases
        gtk_signal_emit_by_name(GTK_OBJECT(MaskEditorEntry),"changed");
    }else
    {
        gtk_widget_hide(MaskEditorFrame);
    }
}



/*
 * Manage/Toggle check buttons into 'Process Fields' frame
 */
void Process_Fields_Check_Button_Toggled (GtkObject *object, GList *list)
{
    gint i = 0;
    
    if ( GTK_TOGGLE_BUTTON(object)->active )
    {
        while (list)
        {
            if ( list->data!=NULL && GTK_OBJECT(list->data)!=object )
                gtk_toggle_button_set_active((GtkToggleButton *)list->data,FALSE);
            i++;
            if (!list->next) break;
            list = list->next;
        }
    }
}


void Process_Fields_Convert_Check_Button_Toggled (GtkObject *object)
{
    gtk_widget_set_sensitive(GTK_WIDGET(ProcessFieldsConvertTo),GTK_TOGGLE_BUTTON(object)->active);
    gtk_widget_set_sensitive(GTK_WIDGET(ProcessFieldsConvertFrom),GTK_TOGGLE_BUTTON(object)->active);
}


/*
 * Small buttons of Process Fields scanner
 */
void Select_Fields_Invert_Selection (void)
{
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(ProcessFileNameField),
                                !GTK_TOGGLE_BUTTON(ProcessFileNameField)->active);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(ProcessTitleField),
                                !GTK_TOGGLE_BUTTON(ProcessTitleField)->active);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(ProcessArtistField),
                                !GTK_TOGGLE_BUTTON(ProcessArtistField)->active);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(ProcessAlbumField),
                                !GTK_TOGGLE_BUTTON(ProcessAlbumField)->active);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(ProcessGenreField),
                                !GTK_TOGGLE_BUTTON(ProcessGenreField)->active);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(ProcessCommentField),
                                !GTK_TOGGLE_BUTTON(ProcessCommentField)->active);
}
void Select_Fields_Select_Unselect_All (void)
{
    static gboolean state = 1;
    
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(ProcessFileNameField),state);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(ProcessTitleField),   state);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(ProcessArtistField),  state);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(ProcessAlbumField),   state);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(ProcessGenreField),   state);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(ProcessCommentField),  state);
    state = !state;
}

/*
 * Set sensitive state of the processing check boxes : if no one is selected => all disabled
 */
void Select_Fields_Set_Sensitive (void)
{
    if (GTK_TOGGLE_BUTTON(ProcessFileNameField)->active
    ||  GTK_TOGGLE_BUTTON(ProcessTitleField)->active
    ||  GTK_TOGGLE_BUTTON(ProcessArtistField)->active
    ||  GTK_TOGGLE_BUTTON(ProcessAlbumField)->active
    ||  GTK_TOGGLE_BUTTON(ProcessGenreField)->active
    ||  GTK_TOGGLE_BUTTON(ProcessCommentField)->active)
    {
        gtk_widget_set_sensitive(GTK_WIDGET(ProcessFieldsConvertIntoSpace),     TRUE);
        gtk_widget_set_sensitive(GTK_WIDGET(ProcessFieldsConvertSpace),         TRUE);
        gtk_widget_set_sensitive(GTK_WIDGET(ProcessFieldsAllUppercase),         TRUE);
        gtk_widget_set_sensitive(GTK_WIDGET(ProcessFieldsAllDowncase),          TRUE);
        gtk_widget_set_sensitive(GTK_WIDGET(ProcessFieldsFirstLetterUppercase), TRUE);
        gtk_widget_set_sensitive(GTK_WIDGET(ProcessFieldsFirstLettersUppercase),TRUE);
        gtk_widget_set_sensitive(GTK_WIDGET(ProcessFieldsRemoveSpace),          TRUE);
        gtk_widget_set_sensitive(GTK_WIDGET(ProcessFieldsInsertSpace),          TRUE);
        gtk_widget_set_sensitive(GTK_WIDGET(ProcessFieldsOnlyOneSpace),         TRUE);
    }else
    {
        gtk_widget_set_sensitive(GTK_WIDGET(ProcessFieldsConvertIntoSpace),     FALSE);
        gtk_widget_set_sensitive(GTK_WIDGET(ProcessFieldsConvertSpace),         FALSE);
        gtk_widget_set_sensitive(GTK_WIDGET(ProcessFieldsAllUppercase),         FALSE);
        gtk_widget_set_sensitive(GTK_WIDGET(ProcessFieldsAllDowncase),          FALSE);
        gtk_widget_set_sensitive(GTK_WIDGET(ProcessFieldsFirstLetterUppercase), FALSE);
        gtk_widget_set_sensitive(GTK_WIDGET(ProcessFieldsFirstLettersUppercase),FALSE);
        gtk_widget_set_sensitive(GTK_WIDGET(ProcessFieldsRemoveSpace),          FALSE);
        gtk_widget_set_sensitive(GTK_WIDGET(ProcessFieldsInsertSpace),          FALSE);
        gtk_widget_set_sensitive(GTK_WIDGET(ProcessFieldsOnlyOneSpace),         FALSE);
    }
}


/*
 *  Load a masks list into the scan editor clist
 */
void Mask_Editor_Clist_Load_Mask_List (GList *list)
{
    GList *tmplist;

    /* Clear the clist */
    gtk_clist_clear(GTK_CLIST(MaskEditorClist));

    tmplist = g_list_first(list);
    while (tmplist)
    {
        gchar *tmp;
        tmp = (gchar*)tmplist->data;
        gtk_clist_append(GTK_CLIST(MaskEditorClist),&tmp);
        tmplist = g_list_next(tmplist);
    }
    /* Select the first row */
    gtk_clist_select_row(GTK_CLIST(MaskEditorClist),0,0);
}


/*
 * Callbacks from Mask Editor buttons
 */
void Mask_Editor_Clist_Select_Row (GtkCList *clist,gint row,gint column,GdkEventButton *event,gpointer data)
{
    GList *SelectedRow = NULL;
    gchar *text = NULL;

    /* We must block the function, else the previous selected row will be modified */
    gtk_signal_handler_block_by_func(GTK_OBJECT(MaskEditorEntry),
        (GtkSignalFunc)Mask_Editor_Entry_Changed,NULL);

    /* Get the text of the last selected row */
    SelectedRow = ((GtkCList*)MaskEditorClist)->selection;
    SelectedRow = g_list_last(SelectedRow);
    if (SelectedRow)
    {
        gtk_clist_get_text(GTK_CLIST(MaskEditorClist),GPOINTER_TO_INT(SelectedRow->data),0,&text);
        if (text)
            gtk_entry_set_text(GTK_ENTRY(MaskEditorEntry),text);
    }

    gtk_signal_handler_unblock_by_func(GTK_OBJECT(MaskEditorEntry),
        (GtkSignalFunc)Mask_Editor_Entry_Changed,NULL);
}

void Mask_Editor_Clist_Unselect_Row (void)
{
/*    gtk_signal_handler_block_by_func(GTK_OBJECT(MaskEditorEntry),
 *        (GtkSignalFunc)Mask_Editor_Entry_Changed,NULL);
 */
    gtk_entry_set_text(GTK_ENTRY(MaskEditorEntry),"");

/*    gtk_signal_handler_unblock_by_func(GTK_OBJECT(MaskEditorEntry),
 *        (GtkSignalFunc)Mask_Editor_Entry_Changed,NULL);
 */
}

gint Sort_List_Compare(gconstpointer a,gconstpointer b)
{
    if (a==NULL || b==NULL) return 0; /* Be carefull: a or b may have the value 0 */

    /* Sort from the bigger to the lower... */
    if ( a < b ) return 1;
    if ( a > b ) return -1;
    return 0;
}

void Mask_Editor_Clist_New (void)
{
    gchar *text = _("New_mask");

    gtk_clist_insert(GTK_CLIST(MaskEditorClist),0,&text);
}

void Mask_Editor_Clist_Copy (void)
{
    GList *SelectedRow = NULL;
    GList *SelectedRow_Sorted = NULL;
    gchar *text = NULL;

    SelectedRow = ((GtkCList*)MaskEditorClist)->selection;
    if (SelectedRow==NULL)
    {
        g_print(_("Copy: No row selected!\n"));
        return;
    }

    /* Duplicate and sort (numbers of the rows) the list.
     * The list is sorted to insert news masks into the same order of the clist */
    SelectedRow_Sorted = g_list_copy(SelectedRow);
    SelectedRow_Sorted = g_list_first(SelectedRow_Sorted);
    SelectedRow_Sorted = g_list_sort(SelectedRow_Sorted,Sort_List_Compare);

    /* Replace number of rows into list by text */
    while(SelectedRow_Sorted)
    {
        gtk_clist_get_text(GTK_CLIST(MaskEditorClist),GPOINTER_TO_INT(SelectedRow_Sorted->data),0,&text);
        SelectedRow_Sorted->data = g_strdup(text);
        if (!SelectedRow_Sorted->next) break;
        SelectedRow_Sorted = SelectedRow_Sorted->next;
    }

    /* Insert the new row(s) */
    SelectedRow_Sorted = g_list_first(SelectedRow_Sorted);
    while(SelectedRow_Sorted)
    {
        gchar *tmp = g_strdup(SelectedRow_Sorted->data);
        gtk_clist_insert(GTK_CLIST(MaskEditorClist),0,&tmp);
        g_free(tmp);
        if (!SelectedRow_Sorted->next) break;
        SelectedRow_Sorted = SelectedRow_Sorted->next;
    }

    /* Free data */
    SelectedRow_Sorted = g_list_first(SelectedRow_Sorted);
    while(SelectedRow_Sorted)
    {
        g_free(SelectedRow_Sorted->data);
        if (!SelectedRow_Sorted->next) break;
        SelectedRow_Sorted = SelectedRow_Sorted->next;
    }
    g_list_free(SelectedRow_Sorted);
}

void Mask_Editor_Clist_Add (void)
{
    gint i;
    gchar *text = NULL;


    if (Scanner_Option_Menu_Get_Active_Menu_Item(SCANNER_FILL_TAG))
    {
        for (i=0;i<(gint)(sizeof(Scan_Masks)/sizeof(Scan_Masks[0]));i++)
        {
            text = g_strdup(Scan_Masks[i]);
            gtk_clist_append(GTK_CLIST(MaskEditorClist),&text);
            g_free(text);
        }
    }

    if (Scanner_Option_Menu_Get_Active_Menu_Item(SCANNER_RENAME_FILE))
    {
        for (i=0;i<(gint)(sizeof(Rename_File_Masks)/sizeof(Rename_File_Masks[0]));i++)
        {
            text = g_strdup(Rename_File_Masks[i]);
            gtk_clist_append(GTK_CLIST(MaskEditorClist),&text);
            g_free(text);
        }
    }
}

void Mask_Editor_Clist_Remove (void)
{
    GList *SelectedRow = NULL;
    GList *SelectedRow_Sorted = NULL;

    SelectedRow = ((GtkCList*)MaskEditorClist)->selection;
    if (SelectedRow==NULL)
    {
        g_print(_("Remove: No row selected!\n"));
        return;
    }

    /* Duplicate and sort (numbers of the rows) the list */
    SelectedRow_Sorted = g_list_copy(SelectedRow);
    SelectedRow_Sorted = g_list_first(SelectedRow_Sorted);
    SelectedRow_Sorted = g_list_sort(SelectedRow_Sorted,Sort_List_Compare);

    while(SelectedRow_Sorted)
    {
        gtk_clist_remove(GTK_CLIST(MaskEditorClist),GPOINTER_TO_INT(SelectedRow_Sorted->data));
        if (!SelectedRow_Sorted->next) break;
        SelectedRow_Sorted = SelectedRow_Sorted->next;
    }
    if ( GPOINTER_TO_INT(SelectedRow_Sorted->data) < ((GtkCList *)MaskEditorClist)->rows )
        gtk_clist_select_row(GTK_CLIST(MaskEditorClist),GPOINTER_TO_INT(SelectedRow_Sorted->data),0);
    else
        gtk_clist_select_row(GTK_CLIST(MaskEditorClist),((GtkCList *)MaskEditorClist)->rows-1,0);
    g_list_free(SelectedRow_Sorted);
}

void Mask_Editor_Clist_Move_Up (void)
{
    GList *SelectedRow = NULL;
    gint row;

    SelectedRow = ((GtkCList*)MaskEditorClist)->selection;
    if (SelectedRow==NULL)
    {
        g_print(_("Move Up: No row selected!\n"));
        return;
    }

    while(SelectedRow)
    {
        row = GPOINTER_TO_INT(SelectedRow->data);
        gtk_clist_row_move(GTK_CLIST(MaskEditorClist),row,row-1);
        if (!SelectedRow->next) break;
        SelectedRow = SelectedRow->next;
    }
}

void Mask_Editor_Clist_Move_Down (void)
{
    GList *SelectedRow = NULL;
    gint row;

    SelectedRow = ((GtkCList*)MaskEditorClist)->selection;
    if (SelectedRow==NULL)
    {
        g_print(_("Move Down: No row selected!\n"));
        return;
    }

    while(SelectedRow)
    {
        row = GPOINTER_TO_INT(SelectedRow->data);
        gtk_clist_row_move(GTK_CLIST(MaskEditorClist),row,row+1);
        if (!SelectedRow->next) break;
        SelectedRow = SelectedRow->next;
    }
}

void Mask_Editor_Clist_Save_Button (void)
{
    if (Scanner_Option_Menu_Get_Active_Menu_Item(SCANNER_FILL_TAG))
    {
        Mask_Editor_Create_Masks_List_From_Clist();
        Save_Scan_Tag_Masks_List(MasksList);
        /* Attach the new list to the combo box */
        if (!MasksList)
            MasksList = g_list_append(MasksList,"");
        gtk_combo_set_popdown_strings(GTK_COMBO(ScannerScanTagMaskCombo),MasksList);
    }else if (Scanner_Option_Menu_Get_Active_Menu_Item(SCANNER_RENAME_FILE))
    {
        Mask_Editor_Create_Masks_List_From_Clist();
        Save_Rename_File_Masks_List(RenameFileMasksList);
        /* Attach the new list to the combo box */
        if (!RenameFileMasksList)
            RenameFileMasksList = g_list_append(RenameFileMasksList,"");
        gtk_combo_set_popdown_strings(GTK_COMBO(ScannerRenameFileMaskCombo),RenameFileMasksList);
    }
}


/*
 * Read rows of clist to recompose the masks list
 */
void Mask_Editor_Create_Masks_List_From_Clist (void)
{
/*    GList *tmplist; */
    gchar *text = NULL;
    gchar *text1 = NULL;
    gint row;
    gint row1;

    /* Free data of the list */
/*    MasksList = g_list_first(MasksList);
    tmplist = MasksList;
    g_print("len1:%d\n"),g_list_lenght(MasksList));
    while (tmplist)
    {
        g_free(tmplist->data);
        tmplist = g_list_next(tmplist);
    }
*/

    /* Suppress blank and duplicate row(s) */
    for (row=0;row<((GtkCList *)MaskEditorClist)->rows;row++)
    {
        gtk_clist_get_text(GTK_CLIST(MaskEditorClist),row,0,&text);
        if (text!=NULL && strlen(text)==0)
        {
            gtk_clist_remove(GTK_CLIST(MaskEditorClist),row--);
            continue;
        }
        for (row1=row+1;row1<((GtkCList *)MaskEditorClist)->rows;row1++)
        {
            gtk_clist_get_text(GTK_CLIST(MaskEditorClist),row1,0,&text1);
            if (text1!=NULL && strcmp(text,text1)==0)
                gtk_clist_remove(GTK_CLIST(MaskEditorClist),row1--);

        }
    }


    /* Recreate list */
    if (Scanner_Option_Menu_Get_Active_Menu_Item(SCANNER_FILL_TAG))
    {
        MasksList = NULL;
        for (row=0;row<((GtkCList *)MaskEditorClist)->rows;row++)
        {
            gtk_clist_get_text(GTK_CLIST(MaskEditorClist),row,0,&text);
            if (text!=NULL)
                MasksList = g_list_append(MasksList,g_strdup(text));
        }
    }else if (Scanner_Option_Menu_Get_Active_Menu_Item(SCANNER_RENAME_FILE))
    {
        RenameFileMasksList = NULL;
        for (row=0;row<((GtkCList *)MaskEditorClist)->rows;row++)
        {
            gtk_clist_get_text(GTK_CLIST(MaskEditorClist),row,0,&text);
            if (text!=NULL)
                RenameFileMasksList = g_list_append(RenameFileMasksList,g_strdup(text));
        }
    }
}


void Mask_Editor_Entry_Changed (void)
{
    GList *SelectedRow = NULL;
    gint row;

    SelectedRow = ((GtkCList*)MaskEditorClist)->selection;
    if (SelectedRow==NULL)
    {
        g_print(_("No row selected!\n"));
        return;
    }
    row = GPOINTER_TO_INT(SelectedRow->data);
    gtk_clist_set_text(GTK_CLIST(MaskEditorClist),row,0,gtk_entry_get_text_1(MaskEditorEntry));
}


/*
 * Actions when the a key is pressed into the masks editor clist
 */
void Mask_Editor_Clist_Key_Press (GtkWidget *widget, GdkEvent *event)
{
    if (event && event->type == GDK_KEY_PRESS)
    {
        GdkEventKey *kevent = (GdkEventKey *)event;

        switch(kevent->keyval)
        {
            case GDK_Delete:
                Mask_Editor_Clist_Remove();
                break;
/*            case GDK_Up:
                Mask_Editor_Clist_Move_Up();
                break;
            case GDK_Down:
                Mask_Editor_Clist_Move_Down();
                break;
*/        }
    }
}




/*
 * For the configuration file...
 */
void Scan_Apply_Changes (void)
{
    /* Then this variables must be set to NULL into function 'ScannerWindow_Quit' */

    if (ScannerWindow)
    {
        /* Group: select entries to process */
        PROCESS_FILENAME_FIELD = GTK_TOGGLE_BUTTON(ProcessFileNameField)->active;
        PROCESS_TITLE_FIELD    = GTK_TOGGLE_BUTTON(ProcessTitleField)->active;
        PROCESS_ARTIST_FIELD   = GTK_TOGGLE_BUTTON(ProcessArtistField)->active;
        PROCESS_ALBUM_FIELD    = GTK_TOGGLE_BUTTON(ProcessAlbumField)->active;
        PROCESS_GENRE_FIELD    = GTK_TOGGLE_BUTTON(ProcessGenreField)->active;
        PROCESS_COMMENT_FIELD  = GTK_TOGGLE_BUTTON(ProcessCommentField)->active;
        /* Group: convert one character */
        PF_CONVERT_INTO_SPACE = GTK_TOGGLE_BUTTON(ProcessFieldsConvertIntoSpace)->active;
        PF_CONVERT_SPACE      = GTK_TOGGLE_BUTTON(ProcessFieldsConvertSpace)->active;
        PF_CONVERT            = GTK_TOGGLE_BUTTON(ProcessFieldsConvert)->active;
        /* Group: capitalize */
        PF_CONVERT_ALL_UPPERCASE = GTK_TOGGLE_BUTTON(ProcessFieldsAllUppercase)->active;
        PF_CONVERT_ALL_DOWNCASE  = GTK_TOGGLE_BUTTON(ProcessFieldsAllDowncase)->active;
        PF_CONVERT_FIRST_LETTER_UPPERCASE  = GTK_TOGGLE_BUTTON(ProcessFieldsFirstLetterUppercase)->active;
        PF_CONVERT_FIRST_LETTERS_UPPERCASE = GTK_TOGGLE_BUTTON(ProcessFieldsFirstLettersUppercase)->active;
        /* Group: remove/insert space */
        PF_REMOVE_SPACE   = GTK_TOGGLE_BUTTON(ProcessFieldsConvertIntoSpace)->active;
        PF_INSERT_SPACE   = GTK_TOGGLE_BUTTON(ProcessFieldsConvertSpace)->active;
        PF_ONLY_ONE_SPACE = GTK_TOGGLE_BUTTON(ProcessFieldsOnlyOneSpace)->active;
    }
}


/*
 * Send a message to the statubar to display status of scanner
 */
void Scan_Display_Info (void)
{
    if (!ScannerWindow)
        Statusbar_Message(_("The Scanner window isn't opened."),TRUE);
    else
        Statusbar_Message(_("Press the Scan button again to run the scanner."),TRUE);
}


/*
 * Returns 1 if item match with the current active option menu item, else returns 0.
 */
gboolean Scanner_Option_Menu_Get_Active_Menu_Item (Scanner_Option_Menu_Items_Value item)
{
    GtkWidget *child;
    gchar *text;

    if (ScannerOptionMenu && GTK_BIN(ScannerOptionMenu)->child)
    {
        child = GTK_BIN(ScannerOptionMenu)->child;
        if (GTK_IS_LABEL(child))
            gtk_label_get(GTK_LABEL(child), &text);
        else
            return 0;
    }else
    {
        return 0;
    }

    if ( item >= 0 && item < (sizeof(Scanner_Option_Menu_Items)/sizeof(Scanner_Option_Menu_Items[0])) )
    {
        /* Do not forget gettext codes to enable the comparison */
        if (strcmp(text,_(Scanner_Option_Menu_Items[item]))==0)
            return 1;
        else
            return 0;
    }else
        return 0;
}


/*
 * Function when you select an item of the option menu
 */
void Scanner_Option_Menu_Activate_Item (GtkWidget *widget, Scanner_Option_Menu_Items_Value item)
{
    switch (item)
    {
        case SCANNER_FILL_TAG:
            gtk_widget_show(MaskEditorButton);
            gtk_widget_show(LegendButton);
            gtk_widget_show(ScanTagFrame);
            gtk_widget_hide(RenameFileFrame);
            gtk_widget_hide(ProcessFieldsFrame);
            Mask_Editor_Clist_Load_Mask_List(MasksList);
            Scan_Fill_Tag_Generate_Preview();
            break;
        case SCANNER_RENAME_FILE:
            gtk_widget_show(MaskEditorButton);
            gtk_widget_show(LegendButton);
            gtk_widget_hide(ScanTagFrame);
            gtk_widget_show(RenameFileFrame);
            gtk_widget_hide(ProcessFieldsFrame);
            Mask_Editor_Clist_Load_Mask_List(RenameFileMasksList);
            Scan_Rename_File_Generate_Preview();
            break;
        case SCANNER_PROCESS_FIELDS:
            gtk_widget_hide(MaskEditorButton);
            gtk_widget_hide(LegendButton);
            gtk_widget_hide(ScanTagFrame);
            gtk_widget_hide(RenameFileFrame);
            gtk_widget_show_all(ProcessFieldsFrame);
            gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(MaskEditorButton),FALSE);
            gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(LegendButton),FALSE);
            Mask_Editor_Clist_Load_Mask_List(NULL);
            break;
    }
}


/*
 * Init the position of the scanner window
 */
void Scan_Set_Scanner_Window_Init_Position (void)
{
    gint x,y;
    gint width,height;
    gint MIN_X, MAX_X,MIN_Y,MAX_Y;


    if (!MainWindow && !ScannerWindow) return;

    if (SET_SCANNER_WINDOW_POSITION)
    {
        gtk_widget_realize(ScannerWindow);
        gdk_window_get_position(MainWindow->window,&x,&y);
        /* Calculate init position */
        x = x + SCANNER_WINDOW_X;
        y = y + SCANNER_WINDOW_Y;

        /* Check if init position doesn't exceed screen boundary */
        gdk_window_get_size(ScannerWindow->window,&width,&height);
        /* Boundaries */
        MIN_X = 0;
        MAX_X = gdk_screen_width() - width;
        MIN_Y = 20; /* "~20" for WM border size */
        MAX_Y = gdk_screen_height() - height;
        if (x<=MIN_X) x = MIN_X;
        if (x>=MAX_X) x = MAX_X;
        if (y<=MIN_Y) y = MIN_Y;
        if (y>=MAX_Y) y = MAX_Y;
        /* Now, move the window */
        gdk_window_move(ScannerWindow->window,x,y);
    }
}





/*
 * For DEBUGGING: Display data (text) of a masks list
 */
void Scan_Display_Data_List (GList *list)
{
    GList *list_save;
    guint item = 0;

    if (!list) g_print("list is NULL\n");
    g_print("\n");

    list_save = list;
    list = g_list_first(list);
    while (list)
    {
        g_print("(%2d) text:'%s'\n",item,(gchar *)list->data);
        list = g_list_next(list);
        item++;
    }
    list = list_save;
}
void Scan_Display_Data_List_Masks_List (void)
{
    Scan_Display_Data_List(MasksList);
}
