# Copyright (C) 2005 Warp Networks S.L., DBS Servicios Informaticos S.L.
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License, version 2, as
# published by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

package EBox::Logs;

use strict;
use warnings;

#FIXME: readd EBox::LogObserver to have logadmin working
#use base qw(EBox::GConfModule EBox::LogObserver);
use base qw(EBox::GConfModule);

use EBox::Global;
use EBox::Gettext;
use EBox::Loggerd;
use EBox::Config;
use EBox::Sudo qw( :all );
use EBox::Summary::Module;
use EBox::Exceptions::InvalidData;
use EBox::Exceptions::External;
use EBox::Exceptions::Internal;
use EBox::DBEngineFactory;
use EBox::Logs::Model::ConfigureLogDataTable;
use POSIX qw(ceil);

use constant IMAGEPATH => EBox::Config::tmp . '/varimages';
use constant PIDPATH => EBox::Config::tmp . '/pids/';
use constant ENABLED_LOG_CONF_DIR => EBox::Config::conf  . '/logs';;
use constant ENABLED_LOG_CONF_FILE => ENABLED_LOG_CONF_DIR . '/enabled.conf';


#	EBox::GConfModule interface
#

sub _create 
{
	my $class = shift;
	my $self = $class->SUPER::_create(name => 'logs',
					  domain => 'ebox');

	bless($self, $class);
	return $self;
}

sub _regenConfig
{
	my ($self) = @_;

	$self->_saveEnabledLogs();
	_stopService();
	root(EBox::Config::libexec . 'ebox-loggerd');
}

sub _stopService
{
	if (-f PIDPATH . "loggerd.pid") {
        	root(EBox::Config::libexec . 'ebox-kill-pid loggerd');
	}
}

sub cleanup 
{
	my $self = shift;
	$self->SUPER::revokeConfig();
}

#	Module API	

# Method: configureLogModel 
#
#   This function returns the model for the configure log data table
#
# Returns:
#
#   An object of class <EBox::Logs::Model::ConfigureLogDataTable>
#
sub configureLogModel 
{
    my ($self) = @_; 

    unless (exists $self->{'configureLogModel'}) {
        $self->{'configureLogModel'} =
			new EBox::Logs::Model::ConfigureLogDataTable(
				'gconfmodule' => $self,
        			'directory' => 'configureLogTable');
    }   
        
    return $self->{'configureLogModel'};
}

# Method: allLogDomains
#
#	This function fetchs all the log domains available throughout 
#	ebox. *(Deprecated)*
#
# Returns:
#
#       An array ref containing  hash references holding:
#
#               logdomain - log domain name
#               text - log domain name i18n
#
sub allLogDomains
{
	my $self = shift;

	my $global = EBox::Global->getInstance();

	my @domains;
	my @mods = @{$global->modInstancesOfType('EBox::LogObserver')};
	foreach my $mod (@mods) {
		my $dm = $mod->logDomain();
		next unless defined($dm);
		push @domains, $dm;
	}
	return \@domains;
}

# Method: allEnabledLogHelpers
#
#	This function fetchs all the classes implemeting the interface
#	<EBox::LogHelper> which have been enabled for the user.
#
#	If the user has not configured anything yet, all are enabled
#	by default.
#
# Returns:
#
#	Array ref of objects <EBox::LogObserver>
#
sub allEnabledLogHelpers
{
	my ($self) = @_;

	my $global = EBox::Global->getInstance();

	my $enabledLogs = $self->_restoreEnabledLogs();	
	
	# If there's no configuration stored it means the user
	# has not configured them yet. So by default, we enable all them
	unless (defined($enabledLogs)) {
		return $self->allLogHelpers();
	}
	
	my @enabledObjects;
	my @mods = @{$global->modInstancesOfType('EBox::LogObserver')};
	foreach my $object (@mods) {
		my $domain = $object->tableInfo()->{'index'};
		if (exists $enabledLogs->{$domain}) {
			push (@enabledObjects, $object->logHelper());
		}
	}

	return \@enabledObjects;
}

# Method: allLogHelpers 
#
#	This function fetchs all the classes implemeting the interface
#	<EBox::LogHelper> and its associated fifo. 
#
# Returns:
#
#       An array ref containing  hash references holding:
#
#               fifo - path to log file
#               classname - class name of the <EBox::LogHelper> implementation
#
sub allLogHelpers
{
	my $self = shift;

	my $global = EBox::Global->getInstance();

	my @objects;
	my @mods = @{$global->modInstancesOfType('EBox::LogObserver')};
	foreach my $mod (@mods) {
		my $obj = $mod->logHelper();
		next unless defined($obj);
		push @objects, $obj;
	}
	return \@objects;
}
sub getLogsModules
{

	my $global = EBox::Global->getInstance();

	return $global->modInstancesOfType('EBox::LogObserver');
}

sub getAllTables
{
	my ($self) = @_;
	my $global = EBox::Global->getInstance();	
	my $tables;
	
	return $self->{tables} if ($self->{tables});
	
	foreach my $mod (@{getLogsModules()}) {
		my $comp = $mod->tableInfo();
		next unless ($comp);
		$tables->{$comp->{'index'}} = $comp;
	}

	$self->{tables} = $tables;
	return $tables;
}


sub  getTableInfo
{
	my ($self, $index) = @_;

	my $tables = $self->getAllTables();
	return $tables->{$index};
}

sub getLogDomains 
{
	my ($self) = @_;
	
	my $tables = $self->getAllTables();
	my %logdomains = map { $_ => $tables->{$_}->{'name'} } keys %{$tables};
	return \%logdomains;
}

sub extendedBackup
{
  my ($self, %params) = @_;
  my $dir    = $params{dir};
  
  my $dbengine = EBox::DBEngineFactory::DBEngine();
  my $dumpFile = "$dir/eboxlogs.dump";

  $dbengine->dumpDB($dumpFile);
}


sub extendedRestore
{
  my ($self, %params) = @_;
  my $dir    = $params{dir};

  my $dbengine = EBox::DBEngineFactory::DBEngine();
  my $dumpFile = "$dir/eboxlogs.dump";

  $dbengine->restoreDB($dumpFile);
}

sub _checkValidDate # (date)
{
	my $datestr = shift;

	my ($date, $time) = split (/ /, $datestr); 
	my ($year, $month, $day) = split (/-/, $date);
	my ($hour, $min, $sec) = split (/:/, $time);

	unless (defined($year) and defined($month) and defined($day)) {
		return undef;
	}
	return undef unless ($year =~ /\d\d\d\d/ );
	return undef unless ($month =~ /\d+/ and $month < 13 and $month > 0);
	return undef unless ($day =~ /\d+/ and $day < 32 and $day > 0);
	return undef unless ($hour =~ /\d+/ and $hour < 24 and $hour > -1);
	return undef unless ($min =~ /\d+/ and $min < 60 and $min > -1);
	return undef unless ($sec =~ /\d+/ and $sec < 60 and $sec > -1);

	return 1;
}

sub search {
	my ($self, $from, $to, $index, 
	    $pagesize, $page, $timecol, $filters) = @_;

	my $dbengine = EBox::DBEngineFactory::DBEngine();
	
	my $tables = $self->getAllTables();
	my $tableinfo = $tables->{$index};
	my $table = $tableinfo->{'tablename'};
	
	unless (defined $tableinfo) {
		   throw  EBox::Exceptions::External( __x(
		   'Table {table} does not exist', 'table' => $table));
	}
	
	
	$self->_addTableName($table);
	if (_checkValidDate($from)) {
		$self->_addDateFilter($timecol, $from, '>');
	}
	if (_checkValidDate($to)) {
		$self->_addDateFilter($timecol, $to, '<');
	}
	if ($filters) {
		foreach my $field (keys %{$filters}) {
			unless (exists $tableinfo->{'titles'}->{$field}) {
			   throw  EBox::Exceptions::External( __x(
			   "Field {field} does not exist", 'field' => $field));
			}
			if ($field eq 'event') {
				$self->_addFilter($field, $filters->{$field});
			} else {
				$self->_addRegExp($field, $filters->{$field});
			}
		}
	}
	
	$self->_addSelect('COUNT(*)');
	my @count = @{$dbengine->query($self->_sqlStmnt())};
	my $tcount = $count[0]{'count'};
	my $tpages = ceil($tcount / $pagesize) - 1;
	
	if ($page < 0) { $page = 0; }
	if ($page > $tpages) { $page = $tpages; }
	
	my $offset = $page * $pagesize;
	$self->_addPager($offset, $pagesize);
	$self->_addSelect('*');
	my @ret = @{$dbengine->query($self->_sqlStmnt())};
	
	$self->{'sqlselect'} = undef;

	my $hashret = {
		'totalret' => $tcount,
		'arrayret' => \@ret
	};
	
	
	return $hashret;
}



sub totalRecords {
	my ($self, $table) = @_;
	my $dbengine = EBox::DBEngineFactory::DBEngine();

	my $sql = "SELECT COUNT(*) FROM $table";
	my @tarray = @{$dbengine->query($sql)};
	my $tcount = $tarray[0]{'count'};
	
	return $tcount;
}

sub _addRegExp
{
	my ($self, $field, $regexp) = @_;
	return unless (defined($field) and defined($regexp) 
			and length($regexp) > 0);
	$self->{'sqlselect'}->{'regexp'}->{$field} = $regexp;
}

sub _addFilter
{
	my ($self, $field, $filter) = @_;
	return unless (defined($field) and defined($filter) 
			and length($filter) > 0);
	$self->{'sqlselect'}->{'filter'}->{$field} = $filter;
}

sub _addDateFilter
{
	my ($self, $field, $date, $operator) = @_;
	return unless (defined($date) and defined($operator));
	$self->{'sqlselect'}->{'date'}->{$operator}->{'date'} = $date;
	$self->{'sqlselect'}->{'date'}->{$operator}->{'field'} = $field;
}

sub _addPager
{

	my ($self, $offset, $limit) = @_;
	$self->{'sqlselect'}->{'offset'} = $offset;
	$self->{'sqlselect'}->{'limit'} = $limit;
}

sub _addTableName
{
	my ($self, $table) = @_;
	$self->{'sqlselect'}->{'table'} = $table;
}

sub _addSelect
{
	my ($self, $select) = @_;
	$self->{'sqlselect'}->{'select'} = $select;
}

sub _sqlStmnt {
	my $self = shift;

	my @params;
	my $sql = $self->{'sqlselect'};
	my $stmt = "SELECT $sql->{'select'} FROM  $sql->{'table'} ";
	
	if ($sql->{'regexp'} or $sql->{'date'}) {
		$stmt .= 'WHERE ';
	}
	
	my $and = '';
	if ($sql->{'date'}) {
		foreach my $op (keys %{$sql->{'date'}}) {
			$stmt .= "$and $sql->{'date'}->{$op}->{'field'} $op ? ";
			$and = 'AND';
			push @params, $sql->{'date'}->{$op}->{'date'};
		}
	}
	
	if ($sql->{'regexp'}) {
		foreach my $field (keys %{$sql->{'regexp'}}) {
			$stmt .= "$and $field ~ ? ";
			$and = 'AND';
			push @params, $sql->{'regexp'}->{$field};
		}
	}
	if ($sql->{'filter'}) {
		foreach my $field (keys %{$sql->{'filter'}}) {
			$stmt .= "$and $field = ? ";
			$and = 'AND';
			push @params, $sql->{'filter'}->{$field};
		}
	}

	$stmt .= "OFFSET ? LIMIT ?";
	push @params, $sql->{'offset'}, $sql->{'limit'};

	return $stmt, @params;
}

# Implement GConfModule interface

# Method: menu 
#
#       Overrides EBox::Module method.
#   
#
sub menu
{
	my ($self, $root) = @_;
        my $folder = new EBox::Menu::Folder('name' => 'Logs',
                                            'text' => __('Logs'));

        $folder->add(new EBox::Menu::Item('url' => 'Logs/Index',
                                          'text' => __('Query logs')));
        $folder->add(new EBox::Menu::Item('url' =>
					  'Logs/View/ConfigureLogDataTable',
                                          'text' => __('Configure logs')));
 
	$root->add($folder);
}


# Implement LogObserver interface

sub tableInfo {
	my $self = shift;

	my $titles = { 'timestamp' => __('Date'),
		'clientaddress' => __('Client Address'),
		'module' => __('Module'),
		'action' => __('Action'),
		'params' => __('Params'),
		'committed' => __('Committed')
	};
	my @order = ('timestamp', 'clientaddress', 'module',
		'action', 'params', 'committed');
	return {
		'name' => __('Admin'),
		'index' => 'admin',
		'titles' => $titles,
		'order' => \@order,
		'tablename' => 'admin',
		'timecol' => 'timestamp',
		'filter' => ['clientaddress', 'module']
	};
}

# Helper functions

# Method: _saveEnabledLogs 
#	
#	(Private)
#	
#	This function saves the enabled logs in a file.
#	We have to do this beacuse the logger daemon will request this
#	configuration as root user.
#
#	Anotther approach could be creating a separated script to
#	query ebox conf.
#
sub _saveEnabledLogs
{
	my ($self) = @_;

	my $enabledLogs = $self->configureLogModel()->enabledLogs();
	
	unless (-d ENABLED_LOG_CONF_DIR) {
		mkdir (ENABLED_LOG_CONF_DIR);
	}

	# Create a string of domains separated by comas
	my $enabledLogsString = join (',', keys %{$enabledLogs});

	my $file;
	unless (open($file, '>' . ENABLED_LOG_CONF_FILE)) {
		throw EBox::Exceptions::Internal(
				'Cannot open ' . ENABLED_LOG_CONF_FILE);
	}

	print $file "$enabledLogsString";
	close($file);
}

# Method: _restoreEnabledLogs 
#	
#	(Private)
#	
#	This function restores the  enabled logs saved in a file by
#	<EBox::Logs::_saveEnabledLogs>
#	We have to do this beacuse the logger daemon will request this
#	configuration as root user.
#
#	Anotther approach could be creating a separated script to
#	query ebox conf.
#
# Returns:
#
# 	undef  - if there's no enabled logs stored yet
# 	hash ref containing the enabled logs
sub _restoreEnabledLogs
{
	my ($self) = @_;
	
	my $file;
	unless (open($file, ENABLED_LOG_CONF_FILE)) {
		return undef;	
	}

	my $string = <$file>;
	close($file);
	
	my %enabled;
	foreach my $domain (split(/,/, $string)) {
		$enabled{$domain} = 1;
	}
	
	return \%enabled;
}
1;
