#ifndef _ECS_ENTMAIL_PCP_H_
#define _ECS_ENTMAIL_PCP_H_

/*============================================================================
 *  Prototypes des fonctions de base
 *   associes  la structure `ecs_entmail_t' dcrivant une entit de maillage
 *  Ces fonctions concernent les entits de maillage PRINCIPALES
 *   par opposition avec les sous-entits de maillage
 *============================================================================*/

/*
  This file is part of the Code_Saturne Preprocessor, element of the
  Code_Saturne CFD tool.

  Copyright (C) 1999-2007 EDF S.A., France

  contact: saturne-support@edf.fr

  The Code_Saturne Preprocessor is free software; you can redistribute it
  and/or modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either version 2 of
  the License, or (at your option) any later version.

  The Code_Saturne Preprocessor is distributed in the hope that it will be
  useful, but WITHOUT ANY WARRANTY; without even the implied warranty
  of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with the Code_Saturne Preprocessor; if not, write to the
  Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor,
  Boston, MA  02110-1301  USA
*/


/*============================================================================
 *                                 Visibilit
 *============================================================================*/


/*----------------------------------------------------------------------------
 *  Fichiers `include' librairie standard C
 *----------------------------------------------------------------------------*/


/*----------------------------------------------------------------------------
 *  Fichiers `include' publics  du  paquetage global "Utilitaire"
 *----------------------------------------------------------------------------*/

#include "ecs_param_perio_glob.h"
#include "ecs_param_rc_glob.h"
#include "ecs_def.h"
#include "ecs_tab_glob.h"


/*----------------------------------------------------------------------------
 *  Fichiers `include' visibles du  paquetage global "Post-Traitement"
 *----------------------------------------------------------------------------*/

#include "ecs_post_glob.h"


/*----------------------------------------------------------------------------
 *  Fichiers `include' publics  des paquetages visibles
 *----------------------------------------------------------------------------*/

#include "ecs_famille_publ.h"


/*----------------------------------------------------------------------------
 *  Fichiers `include' publics  du  paquetage courant
 *----------------------------------------------------------------------------*/

#include "ecs_entmail_publ.h"


/*============================================================================
 *                       Prototypes de fonctions publiques
 *============================================================================*/

/*----------------------------------------------------------------------------
 *  Fonction ralisant la transformation d'une entit principale donne
 *    partir du contenu d'une autre entit de maillage principale donne
 *----------------------------------------------------------------------------*/

void ecs_entmail_pcp__cree_indice
(
 ecs_entmail_t * this_entmail,      /* <-> Entit de maillage  transformer   */
 ecs_entmail_t * entmail_dep        /* --> Entit de maillage dont dpend     */
                                    /*     l'entit de maillage  transformer */
) ;


/*----------------------------------------------------------------------------
 * Fonction qui dcompose les cellules d'une entit de maillage principale
 *  en ses faces
 *
 * Renvoie l'entit de maillage  obtenue par dcomposition
 *----------------------------------------------------------------------------*/

ecs_entmail_t * ecs_entmail_pcp__decompose_cel
(
 ecs_entmail_t * entmail_cel          /* <-> Entite de maillage a dcomposer  */
) ;


/*----------------------------------------------------------------------------
 * Fonction qui dcompose les faces d'une entit de maillage principale
 *  en ses artes
 *
 * Renvoie l'entit de maillage  obtenue par dcomposition
 *----------------------------------------------------------------------------*/

ecs_entmail_t * ecs_entmail_pcp__decompose_fac
(
 ecs_entmail_t * entmail_fac          /* <-> Entit de maillage a decomposer  */
) ;


/*----------------------------------------------------------------------------
 *  Fonction ralisant la concatnation de deux entits de maillage principales
 *   Les 2 entits doivent tre de mme dimension
 *    c'est-a-dire 2 ECS_ENTMAIL_CEL, ou 2 ECS_ENTMAIL_FAC, etc.)
 *
 *  La 2nde entit de maillage `entmail_concat' est concatne  la suite
 *   de l'entit de maillage rceptrice `this_entmail'
 *
 *  Le 3me argument est un tableau sur les entits de maillage
 *   dont les lments non NULL
 *   sont les entits de maillage pour lesquelles le champ "dfinition"
 *   sera modifie pour prendre en compte
 *   le dcalage des rfrences des lments
 *   de l'entit de maillage  concatner
 *   (dcalage correspondant au nombre d'lments de l'entit rceptrice)
 *----------------------------------------------------------------------------*/

void ecs_entmail_pcp__concatene
(
 ecs_entmail_t *      this_entmail,
 ecs_entmail_t *      entmail_concat,
 ecs_entmail_t *const liste_entmail_connect[]
) ;


/*----------------------------------------------------------------------------
 *  Fonction ralisant la fusion des lments topologiquement identiques
 *   d'une entit de maillage principale donne
 *----------------------------------------------------------------------------*/

void ecs_entmail_pcp__fusionne
(
 ecs_entmail_t *      entmail,
 ecs_entmail_t *const liste_entmail_connect[]
) ;


/*----------------------------------------------------------------------------
 *  Fusion des sommets confondus et suppression des lments dgnrs
 *
 *  On ne met pas  jour les champs de type attribut non hritable,
 *  de type ECS_CHAMP_STATUT_REF_ELT (ex : champ_fac_perio), ces champs
 *  pouvant tre construits aprs l'appel de cette fonction.
 *----------------------------------------------------------------------------*/

void ecs_entmail_pcp__nettoie_descend
(
       ecs_entmail_t  *const *const vect_entmail
) ;


/*----------------------------------------------------------------------------
 *  Suppression des sommets ne participant pas  la connectivit
 *   et fusion des lments surfaciques confondus ventuels
 *----------------------------------------------------------------------------*/

void ecs_entmail_pcp__nettoie_nodal
(
       ecs_entmail_t  *const *const vect_entmail
) ;


/*----------------------------------------------------------------------------
 *  Correction si ncessaire de l'orientation des lments en
 *   connectivit nodale.
 *----------------------------------------------------------------------------*/

void ecs_entmail_pcp__orient_nodal
(
       ecs_entmail_t  *const *const vect_entmail,   /*  -> Df. maillage      */
       ecs_tab_int_t  *const        liste_cel_err,  /* <-  Liste cels. erreur
                                                     *     (optionnelle)      */
       ecs_tab_int_t  *const        liste_cel_cor,  /* <-  Liste cels. corr.
                                                     *     (optionnelle)      */
 const ecs_bool_t                   correc_orient   /*  -> Correction ou non  */
) ;


/*----------------------------------------------------------------------------
 * Fonction qui renvoie le nombre de faces de type "polygone"
 *  de l'entit de maillage de type "faces" donne
 *  (faces dfinies par plus de 4 artes)
 *----------------------------------------------------------------------------*/

size_t  ecs_entmail_pcp__nbr_fac_poly
(
 ecs_entmail_t * entmail_fac
) ;


/*----------------------------------------------------------------------------
 *  Fonction qui ralise le tri des lments
 *   de l'entit de maillage principale donne
 *   suivant leur type gomtrique
 *  La fonction affiche le nombre d'lments par type gomtriques
 *----------------------------------------------------------------------------*/

void ecs_entmail_pcp__trie_typ_geo
(
 ecs_entmail_t  * entmail,
 ECS_ENTMAIL_E    typ_entmail
) ;


/*----------------------------------------------------------------------------
 *  Fonction ralisant le dcoupage des faces en triangles
 *   de l'entit de maillage principale des faces
 *
 *  Le maillage doit tre en connectivit nodale (faces -> sommets)
 *----------------------------------------------------------------------------*/

void ecs_entmail_pcp__dec_poly_tria
(
 ecs_entmail_t  *const *const vect_entmail
) ;


/*----------------------------------------------------------------------------
 *  Fonction qui cre la liste des faces intrieures  un maillage.
 *  On ne compte pas ici les faces priodiques parmi les faces intrieures,
 *  cette fonction tant destine  filtrer le post-triatment, et les
 *  faces priodiques tant dj affichables par ailleurs.
 *----------------------------------------------------------------------------*/

ecs_tab_int_t ecs_entmail_pcp__liste_fac_int
(
 ecs_entmail_t  *const   vect_entmail[]
) ;


/*----------------------------------------------------------------------------
 *  Fonction qui compte les faces intrieures et de bord et cre une liste
 *  correspondant  chacun de ces deux ensembles. On peut aussi renvoyer
 *  d'ventuelles listes de faces isoles ou de faces appartenant  plus de
 *  deux cellules (erreur de connectivit en cas de cellules partiellement
 *  superposes).
 *
 *  Dans chaque cas, si l'argument pointeur vers une liste donne est
 *   NULL, cette liste n'est pas cre.
 *----------------------------------------------------------------------------*/

void ecs_entmail_pcp__compte_typ_fac
(
 ecs_entmail_t  *const   vect_entmail[],
 ecs_tab_int_t  *const   liste_fac_erreur,
 ecs_tab_int_t  *const   liste_fac_interne,
 ecs_tab_int_t  *const   liste_fac_de_bord,
 ecs_tab_int_t  *const   liste_fac_isolee
) ;


/*----------------------------------------------------------------------------
 *  Fonction qui construit la liste des cellules attaches  une liste
 *  de faces fournie en argument.
 *----------------------------------------------------------------------------*/

ecs_tab_int_t ecs_entmail_pcp__liste_cel_fac
(
       ecs_entmail_t  *const vect_entmail[],
 const ecs_tab_int_t         liste_fac
) ;


/*----------------------------------------------------------------------------
 *  Fonction qui dfinit de nouvelles entits de maillage principales
 *   par extraction d'une partie des lments
 *   d'une entit de maillage principale donne
 *  Les lments  extraire sont ceux qui ont un boolen  `ECS_TRUE'
 *
 *  On construit automatiquement une filiation.
 *----------------------------------------------------------------------------*/

ecs_entmail_t * * ecs_entmail_pcp__extrait
(
       ecs_entmail_t    *const vect_entmail[],
       ECS_ENTMAIL_E           entmail_e,
 const ecs_tab_bool_t          bool_elt_select[],
 const ecs_bool_t              herite_attributs
) ;


/*----------------------------------------------------------------------------
 *  Fonction qui dfinit de nouvelles entits de maillage principales
 *   en connectivit nodale par extraction d'une partie des lments
 *   d'une entit de maillage principale donne
 *  Les lments  extraire sont ceux qui ont un booleen  `ECS_TRUE'
 *----------------------------------------------------------------------------*/

ecs_entmail_t * * ecs_entmail_pcp__extrait_nodal
(
       ecs_entmail_t    *const vect_entmail[],
       ECS_ENTMAIL_E           entmail_e,
 const ecs_tab_bool_t          bool_elt_select[]
) ;


/*----------------------------------------------------------------------------
 *  Fonction qui slectionne les lments des diffrentes entits principales
 *   ayant une des couleurs de la liste des couleurs
 *   ou appartenant  un des groupes de la liste des groupes
 *  Seules sont concernes les entits de type `entmail_sel_e'
 *----------------------------------------------------------------------------*/

ecs_tab_bool_t * ecs_entmail_pcp__selectionne
(
       ecs_entmail_t        *const *const vect_entmail,
       ECS_ENTMAIL_E                      entmail_sel_e,
 const ecs_tab_int_t        *const        liste_filtre,
 const ecs_tab_int_t                      liste_couleur,
 const ecs_tab_char_t                     liste_groupe,
       ecs_bool_t                         inv_selection
) ;


/*----------------------------------------------------------------------------
 * Fonction qui construit la liste des lments de l'entit principale
 *  slectionne portant des numros de familles marqus dans le tableau
 *  indic_famille
 *----------------------------------------------------------------------------*/

ecs_tab_int_t  ecs_entmail_pcp__liste_ent_fam
(
       ecs_entmail_t        *const *const vect_entmail,
       ECS_ENTMAIL_E                      entmail_sel_e,
 const ecs_tab_int_t        *const        liste_filtre,
 const ecs_tab_bool_t       *const        indic_famille
) ;


/*----------------------------------------------------------------------------
 *  Fonction qui transforme
 *      la  dfinition des faces en artes
 *   en une dfinition des faces en sommets
 *----------------------------------------------------------------------------*/

void ecs_entmail_pcp__def_fac_som
(
       ecs_entmail_t *const entmail_fac,
 const ecs_entmail_t *const entmail_are
) ;


/*----------------------------------------------------------------------------
 *  Fonction qui calcule les coordonnes min et max du domaine
 *----------------------------------------------------------------------------*/

void ecs_entmail_pcp__calc_coo_ext
(
 const ecs_entmail_t *const entmail_som
) ;


/*----------------------------------------------------------------------------
 *  Fonction qui modifie les coordonnes du maillage
 *----------------------------------------------------------------------------*/

void ecs_entmail_pcp__transf_coo
(
 const ecs_entmail_t  *const entmail_som,
 const double                matrice[3][4]
) ;


/*----------------------------------------------------------------------------
 *  Fonction qui recolle les faces non conformes
 *
 *  Les listes des faces nouvelles ou modifies sont construites (et alloues)
 *  ici ; les structures liste_fac_new et liste_fac_mod correspondantes sont
 *  donc vides en entre ; idem pour liste_fac_err qui indiquera les indices
 *  des faces pour lesquelles le dcoupage en sous-faces a chou
 *
 *  On prend en entre soit le nom d'un champ dfinissant une "visibilit"
 *  entre faces  recoller (par exemple une filiation), ou alors un
 *  tableau indicateur de faces slectionnes.
 *  L'un de nom_champ_vis_vis et tab_bool_fac_select doit donc
 *  tre  NULL, et l'autre non.
 *
 *  On renvoie 1 si l'on a effectivement appell le recollement, 0 sinon
 *  (i.e. si la slection de faces  recoller est vide ou le champ de type
 *  "visibilit" est introuvable.
 *----------------------------------------------------------------------------*/

ecs_int_t ecs_entmail_pcp__recolle
(
       ecs_entmail_t    *const *const vect_entmail,
 const char                    *const nom_champ_vis,
       ecs_tab_bool_t          *const tab_bool_fac_select,
       ecs_tab_int_t           *const liste_fac_new,
       ecs_tab_int_t           *const liste_fac_mod,
       ecs_tab_int_t           *const liste_fac_err,
 const ecs_param_rc_t                 param_rc
) ;


/*----------------------------------------------------------------------------
 *  Fonction qui supprime les faces isoles.
 *----------------------------------------------------------------------------*/

void ecs_entmail_pcp__compacte_faces
(
 ecs_entmail_t  *const *const vect_entmail,
 ecs_tab_int_t         *const liste_fac_isolees
) ;


/*----------------------------------------------------------------------------
 *  Fonction ne conservant d'une entit de maillage que le champ
 *  'filiation' (le seul ncessaire sur les faces d'un maillage destin
 *   une coupe aprs la sortie du maillage pour le post traitement des
 *  variables affectes aux faces)
 *----------------------------------------------------------------------------*/

void ecs_entmail_pcp__epure_coupe
(
 ecs_entmail_t   *const entmail_coupe
) ;


/*----------------------------------------------------------------------------
 *  Fonction qui vrifie la cohrence d'un maillage.
 *----------------------------------------------------------------------------*/

ecs_bool_t ecs_entmail_pcp__verif
(
 ecs_entmail_t     *const *const vect_entmail,
 ecs_post_t               *const cas_post
) ;


/*----------------------------------------------------------------------------
 *  Fonction qui construit les familles
 *----------------------------------------------------------------------------*/

ecs_famille_t * * ecs_entmail_pcp__cree_famille
(
 ecs_entmail_t  *const vect_entmail[]
) ;


/*----------------------------------------------------------------------------
 *  Fonction qui dtruit le champ attribut dtermin par son nom
 *----------------------------------------------------------------------------*/

void ecs_entmail_pcp__detruit_att_nom
(
       ecs_entmail_t  *const this_entmail,
 const char           *const nom_champ
) ;


/*----------------------------------------------------------------------------
 *  Fonction qui construit les attributs "groupe" et "couleur"
 *    partir des familles
 *----------------------------------------------------------------------------*/

void ecs_entmail_pcp__cree_attribut
(
 ecs_entmail_t *const this_entmail,
 ecs_famille_t *const famille
) ;


/*----------------------------------------------------------------------------
 * Fonction qui supprime les attributs "groupe" et "couleur"
 *----------------------------------------------------------------------------*/

void ecs_entmail_pcp__suppr_attribut
(
 ecs_entmail_t *const this_entmail
) ;


#endif /* _ECS_ENTMAIL_PCP_H_ */
