/*============================================================================
 *  Dfinitions des fonctions
 *   associes  la structure `ecs_champ_t' dcrivant un champ
 *   et lies  la structure `ecs_vec_int_t'
 *============================================================================*/

/*
  This file is part of the Code_Saturne Preprocessor, element of the
  Code_Saturne CFD tool.

  Copyright (C) 1999-2007 EDF S.A., France

  contact: saturne-support@edf.fr

  The Code_Saturne Preprocessor is free software; you can redistribute it
  and/or modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either version 2 of
  the License, or (at your option) any later version.

  The Code_Saturne Preprocessor is distributed in the hope that it will be
  useful, but WITHOUT ANY WARRANTY; without even the implied warranty
  of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with the Code_Saturne Preprocessor; if not, write to the
  Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor,
  Boston, MA  02110-1301  USA
*/


/*----------------------------------------------------------------------------
 *  Fichiers `include' librairie standard C ou BFT
 *----------------------------------------------------------------------------*/

#include <assert.h>
#include <string.h> /* strlen(), strcpy() */


#include <bft_mem.h>


/*----------------------------------------------------------------------------
 *  Fichiers `include' visibles du  paquetage global "Utilitaire"
 *----------------------------------------------------------------------------*/

#include "ecs_chaine_glob.h"
#include "ecs_def.h"
#include "ecs_tab.h"


/*----------------------------------------------------------------------------
 *  Fichiers `include' visibles des paquetages visibles
 *----------------------------------------------------------------------------*/

#include "ecs_vec_int.h"
#include "ecs_vec_int_tri.h"


/*----------------------------------------------------------------------------
 *  Fichiers `include' visibles du  paquetage courant
 *----------------------------------------------------------------------------*/

#include "ecs_champ.h"

/*----------------------------------------------------------------------------
 *  Fichier  `include' du  paquetage courant associ au fichier courant
 *----------------------------------------------------------------------------*/

#include "ecs_champ_vec_int.h"


/*----------------------------------------------------------------------------
 *  Fichiers `include' privs   du  paquetage courant
 *----------------------------------------------------------------------------*/

#include "ecs_champ_priv.h"



/*============================================================================
 *                       Prototypes de fonctions prives
 *============================================================================*/

/*----------------------------------------------------------------------------
 *  Fonction qui ralise la transformation
 *   d'une table REGLEE dont les valeurs sont des REFERENCES  une autre table
 *  en une table REGLEE dont les valeurs sont des INDICES de cette autre table
 *
 *  Cette fonction retourne :
 *
 *   - `ECS_TRUE'  si la table des rfrences   est     une REGLE
 *                 (    la transformation   a t realise)
 *   - `ECS_FALSE' si la table des rfrences n'est pas une REGLE
 *                 (aucune transformation n'a t ralise)
 *----------------------------------------------------------------------------*/

ecs_bool_t ecs_loc_champ_vec_int__ref_en_ind
(
 ecs_champ_t  *this_champ_transf,
 ecs_champ_t  *     champ_ref
) ;


/*============================================================================
 *                             Fonctions publiques
 *============================================================================*/

/*----------------------------------------------------------------------------
 *  Fonction construisant un vecteur `ecs_vec_int_t' associ au champ donn
 *
 *  Si les tables sont REGLEES, les tableaux sont construits
 *
 *  Cette fonction doit tre utilise conjointement avec :
 *  - soit `ecs_champ__transfere_vec_int()'
 *                               si le contenu du vecteur   a chang
 *  - soit `ecs_champ__libere_vec_int()'
 *                               si le contenu du vecteur n'a pas t modifi
 *  Ceci afin de reconstuire, si ncessaire, les tables REGLEES
 *----------------------------------------------------------------------------*/

ecs_vec_int_t * ecs_champ__initialise_vec_int
(
 ecs_champ_t *this_champ
)
{

  ecs_vec_int_t *vec_int_loc ;

  size_t       pos_nbr ;  /* Nombre   de positions */
  ecs_size_t  *pos_tab ;  /* Tableau des positions */
  ecs_int_t   *val_tab ;  /* Tableau des valeurs */


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  assert(this_champ != NULL) ;


  assert(ecs_champ__ret_val_typ(this_champ) == ECS_TYPE_ecs_int_t) ;


  /* Rcupration des rfrences des tableaux positions et valeurs   */
  /*  des tables contenues par le champ                              */
  /* (pour une table REGLEE, le tableau correspondant est developp) */
  /*-----------------------------------------------------------------*/

  pos_nbr =              this_champ->nbr_elt + 1 ;
  pos_tab =              ecs_champ__ret_pos_tab(this_champ) ;
  val_tab = (ecs_int_t *)(this_champ->val_tab) ;


  /* Affectation des valeurs recupres au vecteur `ecs_vec_int_t' cre */
  /*--------------------------------------------------------------------*/

  vec_int_loc = ecs_vec_int__initialise(pos_nbr,
                                        pos_tab,
                                        val_tab) ;

  return vec_int_loc ;


}


/*----------------------------------------------------------------------------
 *  Fonction qui transfre le contenu d'un vecteur `ecs_vec_int_t'
 *   dans les tables d'un champ donn
 *
 *  Les tables sont transformes en REGLES si possible
 *
 *  La structure `ecs_vec_int_t' est libre
 *----------------------------------------------------------------------------*/

void ecs_champ__transfere_vec_int
(
 ecs_champ_t    *this_champ,  /* <-> Champ dans lequel on transfre `vec_int' */
 ecs_vec_int_t  *vec_int      /* <-> Vecteur `ecs_vec_int_t'  transfrer     */
)
{

  size_t       pos_nbr ;  /* Nombre   de positions */
  ecs_size_t  *pos_tab ;  /* Tableau des positions */
  size_t       val_nbr ;  /* Nombre   de valeurs */
  ecs_int_t   *val_tab ;  /* Tableau des valeurs */


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  assert(this_champ->typ_val == ECS_TYPE_ecs_int_t) ;


  pos_nbr = ecs_vec_int__ret_pos_nbr(vec_int) ;
  pos_tab = ecs_vec_int__ret_pos_tab(vec_int) ;
  val_nbr = ecs_vec_int__ret_val_nbr(vec_int) ;
  val_tab = ecs_vec_int__ret_val_tab(vec_int) ;


  if (pos_tab == NULL) {
    /* Les positions forment une REGLE de pas unit */
    this_champ->pos_pas = 1 ;
  }
  else {
    this_champ->pos_pas = 0 ;
  }

  this_champ->nbr_elt = pos_nbr - 1 ;

  this_champ->pos_tab = pos_tab ;
  ecs_champ__pos_en_regle(this_champ) ;

  this_champ->val_tab = val_tab ;

  ecs_vec_int__libere(vec_int) ;

}


/*----------------------------------------------------------------------------
 *  Fonction qui libre le tableau d'une table REGLEE qui a t
 *   prcdemment developp par l'appel de `ecs_champ__initialise_vec_int()'
 *  et qui dtruit le vecteur `ecs_vec_int_t' qui a t
 *   prcdemment cre      par l'appel de `ecs_champ__initialise_vec_int()'
 *----------------------------------------------------------------------------*/

void ecs_champ__libere_vec_int
(
 ecs_champ_t    * this_champ,                 /* <-> Champ associ au vecteur */
 ecs_vec_int_t  * vec_int                     /* <-> Vecteur  dtruire       */
)
{

  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  ecs_champ__libere_pos_tab(this_champ,
                            ecs_vec_int__ret_pos_tab(vec_int)) ;

  ecs_vec_int__libere(vec_int) ;

}


/*----------------------------------------------------------------------------
 *  Fonction qui cre un champ  partir
 *   du contenu d'un vecteur `ecs_vec_int_t'
 *
 *  Seuls le nom du champ et les tables du champ sont remplis
 *   (les tables sont transformes en REGLES si possible)
 *
 *  Les autres membres de la structure prennent leur valeur d'initialisation
 *
 *  La structure `ecs_vec_int_t' est libre
 *----------------------------------------------------------------------------*/

ecs_champ_t * ecs_champ__init_avec_vec_int
(
       ecs_vec_int_t  *      vec_int,            /* --> Vecteur  transfrer */
 const char           *const nom
)
{

  ecs_champ_t  *this_champ ;

  size_t        pos_nbr ;  /* Nombre   de positions */
  ecs_size_t   *pos_tab ;  /* Tableau des positions */
  ecs_int_t    *val_tab ;  /* Tableau des valeurs */


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  /* Rcupration des valeurs du vecteur `ecs_vec_int_t' */

  pos_nbr = ecs_vec_int__ret_pos_nbr(vec_int) ;
  pos_tab = ecs_vec_int__ret_pos_tab(vec_int) ;
  val_tab = ecs_vec_int__ret_val_tab(vec_int) ;


  /* Libration du vecteur index */
  /*------------------------------*/

  ecs_vec_int__libere(vec_int) ;


  /*------------------------------------------------*/
  /* Cration de la structure de type `ecs_champ_t' */
  /*------------------------------------------------*/

  BFT_MALLOC(this_champ, 1, ecs_champ_t);


  this_champ->nbr_elt = pos_nbr - 1 ;

  this_champ->typ_val = ECS_TYPE_ecs_int_t ;


  /* Allocation et affectation du nom du champ */
  /*-------------------------------------------*/

  if (nom != NULL) {
    BFT_MALLOC(this_champ->nom, strlen(nom) + 1, char) ;
    strcpy(this_champ->nom, nom) ;
  }
  else {
    this_champ->nom = NULL ;
  }


  /* Initialisation de la rfrence au descripteur de champ */
  /*--------------------------------------------------------*/

  this_champ->descr       = NULL ;


  /* Initialisation du lien pour les listes chanes */
  /* des champs auxiliaires                          */
  /*-------------------------------------------------*/

  this_champ->l_champ_sui = NULL ;


  /* Statut du champ dans une transformation */
  /*  de type dcoupage ou recollement       */
  /*-----------------------------------------*/

  this_champ->statut_e    = ECS_CHAMP_STATUT_INDEFINI ;


  /* Affectation de la table des positions des valeurs */
  /*---------------------------------------------------*/

  if (pos_tab == NULL) {
    /* Les positions forment une REGLE de pas unit */
    this_champ->pos_pas = 1 ;
  }
  else {
    this_champ->pos_pas = 0 ;
  }

  this_champ->pos_tab = pos_tab ;
  ecs_champ__pos_en_regle(this_champ) ;


  /* Affectation de la table des valeurs */
  /*-------------------------------------*/

  this_champ->val_tab = val_tab ;


  return this_champ ;

}


/*----------------------------------------------------------------------------
 *  Fonction ralisant la transformation du contenu d'un champ donn,
 *   dpendant des rfrences  un autre champ,
 *   en un contenu qui dpend des indices de cet autre champ
 *----------------------------------------------------------------------------*/

void ecs_champ__reference_en_indice
(
 ecs_champ_t  *this_champ_transf,      /* <-> Champ des valeurs  transformer */
 ecs_champ_t  *     champ_ref          /* --> Champ des valeurs de rfrence  */
)
{

  ecs_bool_t   bool_regle ;  /* Drapeau indiquant si la table des rfrences  */
                             /*                            est une REGLE      */


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/



  assert(this_champ_transf != NULL && champ_ref != NULL) ;


  /* Si la table des rfrences est une REGLE          */
  /* Il y a une relation simple                        */
  /* entre les valeurs des rfrences et leurs indices */
  /*---------------------------------------------------*/


  bool_regle = ecs_loc_champ_vec_int__ref_en_ind(this_champ_transf,
                                                 champ_ref) ;

  if (bool_regle == ECS_FALSE) {


    /* La table des rfrences n'est pas une REGLE */
    /*---------------------------------------------*/

    /* Il faut rechercher les indices  partir de la table des rfrences */


    ecs_vec_int_t *vec_int_transf     ;
    ecs_vec_int_t *vec_int_ref        ;

    ecs_tab_int_t  tab_int_ref        ;
    ecs_tab_int_t  tab_int_transf     ;

    size_t       nbr_rec ;
    size_t       nbr_ord ;


    vec_int_transf = ecs_champ__initialise_vec_int(this_champ_transf) ;
    vec_int_ref    = ecs_champ__initialise_vec_int(champ_ref) ;

    nbr_rec = ecs_vec_int__ret_val_nbr(vec_int_transf) ;
    nbr_ord = ecs_vec_int__ret_val_nbr(vec_int_ref) ;


    tab_int_ref.val = ecs_vec_int__ret_val_tab(vec_int_ref) ;
    tab_int_ref.nbr = nbr_ord ;

    tab_int_transf.val = ecs_vec_int__ret_val_tab(vec_int_transf) ;
    tab_int_transf.nbr = nbr_rec ;


    tab_int_transf = ecs_tab_int__ref_en_indice(tab_int_transf,
                                                tab_int_ref,
                                                ECS_FALSE) ;

    /* Les indices trouvs par la recherche doivent tre incrments */
    /*  de `1' car ce sont des positions                             */

    ecs_vec_int__incremente_val(vec_int_transf, 1) ;


    /* Reconstruction ventuelle des REGLES */

    ecs_champ__transfere_vec_int(this_champ_transf,
                                 vec_int_transf) ;

    ecs_champ__libere_vec_int(champ_ref,
                              vec_int_ref) ;

  }

}


/*----------------------------------------------------------------------------
 *  Fonction qui incrmente les valeurs d'un champ donn
 *   d'une constante donne
 *----------------------------------------------------------------------------*/

void ecs_champ__incremente_val
(
       ecs_champ_t  * this_champ,
 const ecs_int_t      increment
)
{

  ecs_vec_int_t *vec_int ;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  vec_int = ecs_champ__initialise_vec_int(this_champ) ;

  ecs_vec_int__incremente_val_sgn(vec_int,
                                  increment) ;

  ecs_champ__transfere_vec_int(this_champ,
                               vec_int) ;


}


/*----------------------------------------------------------------------------
 *  Fonction ralisant la transformation d'un champ
 *   en appliquant directement le vecteur de transformation donn
 *   sur ses valeurs
 *----------------------------------------------------------------------------*/

void ecs_champ__transforme_val
(
       ecs_champ_t    *const this_champ,
       size_t                nbr_elt_new,
 const ecs_tab_int_t         vect_transf
)
{

  ecs_vec_int_t *vec_int ;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  vec_int = ecs_champ__initialise_vec_int(this_champ) ;


  ecs_vec_int__transforme_val(vec_int,
                              nbr_elt_new,
                              vect_transf) ;

  ecs_champ__transfere_vec_int(this_champ,
                               vec_int) ;


}


/*----------------------------------------------------------------------------
 *  Fonction ralisant la transformation d'un champ
 *   en appliquant directement le vecteur de transformation donn
 *   sur ses positions
 *----------------------------------------------------------------------------*/

void ecs_champ__transforme_pos
(
       ecs_champ_t    *const this_champ,
       size_t                nbr_elt_ref,
 const ecs_tab_int_t         vect_transf
)
{

  ecs_vec_int_t *vec_int ;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  vec_int = ecs_champ__initialise_vec_int(this_champ) ;


  ecs_vec_int__transforme_pos(vec_int,
                              nbr_elt_ref,
                              vect_transf) ;

  ecs_champ__transfere_vec_int(this_champ,
                               vec_int) ;


}


/*----------------------------------------------------------------------------
 *  Fonction ralisant la transformation d'un champ
 *   en appliquant directement le vecteur de transformation donn
 *   sur les valeurs associes  ses lments
 *----------------------------------------------------------------------------*/

void ecs_champ__renumerote
(
       ecs_champ_t    *const this_champ,
 const ecs_tab_int_t         vect_transf
)
{

  ecs_vec_int_t *vec_int ;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  vec_int = ecs_champ__initialise_vec_int(this_champ) ;


  ecs_vec_int__renumerote(vec_int,
                          vect_transf) ;

  ecs_champ__transfere_vec_int(this_champ,
                               vec_int) ;


}


void ecs_champ__renumerote_et_sgn
(
       ecs_champ_t    *const this_champ,
 const ecs_tab_int_t         vect_transf,
 const ecs_tab_int_t         signe_elt
)
{

  ecs_vec_int_t *vec_int ;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  vec_int = ecs_champ__initialise_vec_int(this_champ) ;


  ecs_vec_int__renumerote_et_sgn(vec_int,
                                 vect_transf,
                                 signe_elt) ;

  ecs_champ__transfere_vec_int(this_champ,
                               vec_int) ;


}


/*----------------------------------------------------------------------------
 *  Fonction qui retourne le tableau des valeurs du champ donn,
 *   dimensionn au nombre de rfrences distinctes du champ,
 *   (valeurs qui correspondent  des rfrences)
 *----------------------------------------------------------------------------*/

ecs_tab_int_t ecs_champ__ret_reference
(
 ecs_champ_t  *const this_champ
)
{

  ecs_tab_int_t   tab_ref ;

  ecs_vec_int_t * vec_int ;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  vec_int = ecs_champ__initialise_vec_int(this_champ) ;


  tab_ref = ecs_vec_int__ret_reference(vec_int) ;


  ecs_champ__libere_vec_int(this_champ,
                            vec_int) ;


  return tab_ref ;


}


/*----------------------------------------------------------------------------
 *  Fonction qui remplace les rfrences  des lments
 *  en des rfrences  d'autres lments lis aux premiers
 *  par un champ de type "dfinition"
 *----------------------------------------------------------------------------*/

void ecs_champ__remplace_ref
(
 ecs_champ_t *const champ_rep,
 ecs_champ_t *const champ_def
)
{

  ecs_vec_int_t * vec_rep ;
  ecs_vec_int_t * vec_def ;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  vec_rep = ecs_champ__initialise_vec_int(champ_rep) ;
  vec_def = ecs_champ__initialise_vec_int(champ_def) ;


  ecs_vec_int__remplace_ref(vec_rep,
                            vec_def) ;


  ecs_champ__libere_vec_int(champ_def,
                            vec_def) ;

  ecs_champ__transfere_vec_int(champ_rep,
                               vec_rep) ;


}


/*----------------------------------------------------------------------------
 *  Fonction qui construit la  table de connectivit inverse
 *  "sous-lments -> lments"
 *----------------------------------------------------------------------------*/

ecs_champ_t * ecs_champ__inverse
(
 ecs_champ_t  *const champ_def_elt,
 size_t              nbr_sselt
)
{

  ecs_champ_t   * champ_sselt_def_elt ;

  ecs_vec_int_t * vec_def_elt ;
  ecs_vec_int_t * vec_sselt_def_elt ;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  assert(champ_def_elt != NULL) ;


  vec_def_elt = ecs_champ__initialise_vec_int(champ_def_elt) ;


  vec_sselt_def_elt = ecs_vec_int__inverse(vec_def_elt,
                                           nbr_sselt) ;


  ecs_champ__libere_vec_int(champ_def_elt,
                            vec_def_elt) ;

  champ_sselt_def_elt = ecs_champ__init_avec_vec_int(vec_sselt_def_elt,
                                                     ECS_CHAMP_NOM_DEFINIT) ;

  champ_sselt_def_elt->statut_e = ECS_CHAMP_STATUT_INDEFINI ;


  return  champ_sselt_def_elt ;


}


/*============================================================================
 *                              Fonctions prives
 *============================================================================*/

/*----------------------------------------------------------------------------
 *  Fonction qui ralise la transformation
 *   d'une table REGLEE dont les valeurs sont des REFERENCES  une autre table
 *  en une table REGLEE dont les valeurs sont des INDICES de cette autre table
 *
 *  Cette fonction retourne :
 *
 *   - `ECS_TRUE'  si la table des rfrences   est     une REGLE
 *                 (    la transformation   a t realise)
 *   - `ECS_FALSE' si la table des rfrences n'est pas une REGLE
 *                 (aucune transformation n'a t ralise)
 *----------------------------------------------------------------------------*/

ecs_bool_t ecs_loc_champ_vec_int__ref_en_ind
(
 ecs_champ_t  *this_champ_transf,
 ecs_champ_t  *     champ_ref
)
{

  ecs_bool_t  bool_regle ; /* Drapeau indiquant si la table des references    */
                           /*                                est une REGLE    */

  size_t      nbr_val ;
  size_t      itab ;
  size_t      ival ;

  ecs_bool_t  val_ref_regle ;
  ecs_int_t   val_ref_deb ;
  ecs_int_t   val_ref_pas ;
  ecs_int_t  *val_ref_tab ;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  nbr_val = ecs_champ__ret_val_nbr(champ_ref) ;

  if (nbr_val == 0)
    return ECS_TRUE ;


  val_ref_tab = (ecs_int_t *)(champ_ref->val_tab) ;


  /* On verifie si les valeurs correspondent a une REGLE */

  val_ref_regle = ECS_TRUE ;

  val_ref_deb = val_ref_tab[0] ;

  if (nbr_val > 1) {

    val_ref_pas = val_ref_tab[1] - val_ref_tab[0] ;

    for (ival = 2 ; ival < nbr_val ; ival++) {

      if (   (val_ref_tab[ival] - val_ref_tab[ival - 1])
          != val_ref_pas) {
        val_ref_regle = ECS_FALSE ;
        break ;
      }

    }

  }
  else /* if nbr_val == 1 */
    val_ref_pas = 1 ;


  if (   val_ref_regle == ECS_TRUE
      && val_ref_pas > 0) {

    /* Pour la transformation des references en indices */
    /* le pas de la REGLE doit etre non nul             */


    /* La table des references est une REGLE */
    /*---------------------------------------*/

    /* Il y a donc une relation simple                   */
    /* entre les valeurs des references et leurs indices */


    if (   val_ref_pas != 1
        || val_ref_deb != 1) {

      size_t     transf_nbr = ecs_champ__ret_val_nbr(this_champ_transf) ;
      ecs_int_t *transf_tab = (ecs_int_t *)(this_champ_transf->val_tab) ;

      for (itab = 0 ; itab < transf_nbr ; itab++)
        transf_tab[itab] =
          (transf_tab[itab] - val_ref_deb) / val_ref_pas + 1 ;
      /*                division entiere --^    position --^  */

    }
    /* else : rien a faire : les references sont egales aux indices */

    bool_regle = ECS_TRUE ;

  }
  else {

    /* La table des references n'est pas une REGLE */
    /*---------------------------------------------*/

    /* Il faut rechercher les indices a partir de la table des references */


    bool_regle = ECS_FALSE ;

  }


  return bool_regle ;

}

