/*============================================================================
 *  Dfinitions des fonctions de base
 *   associes  la structure `ecs_famille_t' dcrivant une famille
 *============================================================================*/

/*
  This file is part of the Code_Saturne Preprocessor, element of the
  Code_Saturne CFD tool.

  Copyright (C) 1999-2007 EDF S.A., France

  contact: saturne-support@edf.fr

  The Code_Saturne Preprocessor is free software; you can redistribute it
  and/or modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either version 2 of
  the License, or (at your option) any later version.

  The Code_Saturne Preprocessor is distributed in the hope that it will be
  useful, but WITHOUT ANY WARRANTY; without even the implied warranty
  of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with the Code_Saturne Preprocessor; if not, write to the
  Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor,
  Boston, MA  02110-1301  USA
*/


/*============================================================================
 *                                 Visibilit
 *============================================================================*/

/*----------------------------------------------------------------------------
 *  Fichiers `include' librairie standard C ou BFT
 *----------------------------------------------------------------------------*/

#include <assert.h>
#include <string.h>  /* memcmp() */

#include <bft_file.h>
#include <bft_mem.h>
#include <bft_printf.h>


/*----------------------------------------------------------------------------
 *  Fichiers `include' systme
 *----------------------------------------------------------------------------*/


/*----------------------------------------------------------------------------
 *  Fichiers `include' visibles du  paquetage global "Utilitaire"
 *----------------------------------------------------------------------------*/

#include "ecs_def.h"
#include "ecs_fic.h"


/*----------------------------------------------------------------------------
  *  Fichiers `include' visibles des paquetages visibles
 *----------------------------------------------------------------------------*/



/*----------------------------------------------------------------------------
 *  Fichiers `include' visibles du  paquetage courant
 *----------------------------------------------------------------------------*/

#include "ecs_descr_chaine.h"


/*----------------------------------------------------------------------------
 *  Fichier  `include' du  paquetage courant associe au fichier courant
 *----------------------------------------------------------------------------*/

#include "ecs_famille.h"


/*----------------------------------------------------------------------------
 *  Fichiers `include' prives   du  paquetage courant
 *----------------------------------------------------------------------------*/

#include "ecs_famille_priv.h"


/*============================================================================
 *                       Prototypes de fonctions prives
 *============================================================================*/


/*============================================================================
 *                             Fonctions publiques
 *============================================================================*/

/*----------------------------------------------------------------------------
 *    Fonction de cration d'une structure de famille `ecs_famille_t'
 *----------------------------------------------------------------------------*/

ecs_famille_t * ecs_famille__cree
(
 ecs_int_t          num        ,    /* --> Numero de la famille               */
 ecs_descr_t *const descr_tete      /* --> Tete de la liste chainee           */
                                    /*      de descripteurs d'attributs       */
)
{

  ecs_famille_t * fam_loc ;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  /* Allocation et initialisation de la structure */
  /*==============================================*/

  /* Allocation de la structure globale de la famille */
  /*--------------------------------------------------*/

  BFT_MALLOC(fam_loc, 1, ecs_famille_t) ;


  /* Initialisation de la structure */
  /*--------------------------------*/

  fam_loc->num           = num ;
  fam_loc->descr         = descr_tete ;
  fam_loc->l_famille_sui = NULL ;


  return fam_loc ;


}


/*----------------------------------------------------------------------------
 *  Fonction librant la structure `ecs_famille_t' donne en argument.
 *  Elle renvoie un pointeur NULL
 *----------------------------------------------------------------------------*/

ecs_famille_t * ecs_famille__detruit
(
 ecs_famille_t * this_fam
)
{

  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  assert(this_fam != NULL) ;


  ecs_descr_chaine__detruit(&this_fam->descr) ;


  BFT_FREE(this_fam) ;


  assert(this_fam == NULL) ;


  return this_fam ;


}


/*----------------------------------------------------------------------------
 *  Fonction imprimant le contenu d'une structure `ecs_famille_t' donne
 *   sur le flux dcrit par la structure `bft_file_t'
 *----------------------------------------------------------------------------*/

void ecs_famille__imprime
(
 const ecs_famille_t *const this_fam ,          /* --> Structure  imprimer   */
       ecs_int_t            imp_col  ,
       bft_file_t    *const fic_imp             /* --> Descripteur du fichier */
)
{


#define ECS_FCT_IMP_FAMILLE_NUM           "numero"
#define ECS_FCT_IMP_FAMILLE_DESCR         "descr"
#define ECS_FCT_IMP_FAMILLE_L_FAMILLE     "l_famille_sui"


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  assert(this_fam != NULL) ;


  imp_col++ ;


  /* Impression du numero de famille */

  ecs_fic__imprime_val(fic_imp, imp_col, ECS_FCT_IMP_FAMILLE_NUM,
                       ECS_TYPE_ecs_int_t, &(this_fam->num));


  /* Impression de la liste chainee des descripteurs composant la famille */
  /*----------------------------------------------------------------------*/

  /* Impression du pointeur sur le descripteur de tete */

  ecs_fic__imprime_val(fic_imp, imp_col, ECS_FCT_IMP_FAMILLE_DESCR,
                       ECS_TYPE_void, this_fam->descr);


  /* Appel  la fonction d'impression d'une chaine de descripteurs de champ */

  ecs_descr_chaine__imprime(this_fam->descr,
                            imp_col + 1,
                            fic_imp) ;

  /* Impression du lien sur une eventuelle famille suivante */
  /*--------------------------------------------------------*/

  ecs_fic__imprime_val(fic_imp, imp_col, ECS_FCT_IMP_FAMILLE_L_FAMILLE,
                       ECS_TYPE_void, this_fam->l_famille_sui);


}


/*----------------------------------------------------------------------------
 *  Fonction qui renvoie la taille en octets d'une structure `ecs_famille_t'
 *----------------------------------------------------------------------------*/

float ecs_famille__ret_taille
(
  const ecs_famille_t *const this_fam
)
{

  float         taille ;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  assert(this_fam != NULL) ;


  taille = sizeof(*this_fam) ;

  taille += ecs_descr_chaine__ret_taille(this_fam->descr) ;


  return taille ;


}


/*----------------------------------------------------------------------------
 *  Fonction qui alloue une structure `ecs_famille_t' et qui remplit
 *   son contenu en copiant le contenu de la structure donne en argument
 *----------------------------------------------------------------------------*/

ecs_famille_t * ecs_famille__copie
(                                 /* <-- Renvoie la structure copiee          */
 ecs_famille_t * this_famille     /* --> Structure  copier                   */
)
{

  ecs_descr_t     * descr ;

  ecs_famille_t   * famille_loc ;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  assert(this_famille != NULL) ;

  descr = ecs_descr_chaine__copie(this_famille->descr) ;

  famille_loc = ecs_famille__cree(this_famille->num, descr) ;


  return famille_loc ;


}


/*----------------------------------------------------------------------------
 *  Fonction qui affiche la dfinition de la famille
 *----------------------------------------------------------------------------*/

void ecs_famille__affiche
(
 const ecs_famille_t *const this_fam
)
{

  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  assert(this_fam != NULL) ;


  bft_printf("  %s %" ECS_FORMAT_ecs_int_t "\n",
             _("Family"), this_fam->num) ;

  ecs_descr_chaine__affiche(this_fam->descr,
                            (int)strlen(_("Family")) + 1) ;


}


/*----------------------------------------------------------------------------
 *  Fonction qui renvoie le nombre et la liste des identificateurs des
 *   descripteurs de la famille donne en argument
 *----------------------------------------------------------------------------*/

ecs_tab_int_t ecs_famille__ret_ide
(
 const ecs_famille_t *const this_fam
)
{

  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/

  assert(this_fam != NULL) ;

  return ecs_descr_chaine__ret_ide(this_fam->descr) ;

}


/*----------------------------------------------------------------------------
 *  Fonction qui renvoie le nombre et la liste des des pointeurs sur les noms
 *   des descripteurs de la famille donne en argument
 *----------------------------------------------------------------------------*/

ecs_tab_char_t ecs_famille__ret_nom
(
 const ecs_famille_t *const this_fam
)
{

  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/

  assert(this_fam != NULL) ;

  return ecs_descr_chaine__ret_nom(this_fam->descr) ;

}


/*----------------------------------------------------------------------------
 *  Fonction qui construit les 2 listes chanees de descripteurs
 *   de type "couleur" et "groupe"  partir de la famille donne
 *----------------------------------------------------------------------------*/

void ecs_famille__cree_descr
(
 ecs_famille_t   *const this_fam           ,
 ecs_descr_t   * *const descr_tete_couleur ,
 ecs_descr_t   * *const descr_tete_groupe
)
{

  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  assert(this_fam != NULL) ;


  *descr_tete_couleur
    = ecs_descr_chaine__ret_descr_typ(this_fam->descr,
                                      ECS_DESCR_TYP_COULEUR) ;

  *descr_tete_groupe
    = ecs_descr_chaine__ret_descr_typ(this_fam->descr,
                                      ECS_DESCR_TYP_GROUPE) ;


}


/*============================================================================
 *                              Fonctions prives
 *============================================================================*/
