/*============================================================================
 *  Dfinition de la fonction de base
 *   de remplissage de la structure de maillage  partir des donnes lues
 *============================================================================*/

/*
  This file is part of the Code_Saturne Preprocessor, element of the
  Code_Saturne CFD tool.

  Copyright (C) 1999-2007 EDF S.A., France

  contact: saturne-support@edf.fr

  The Code_Saturne Preprocessor is free software; you can redistribute it
  and/or modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either version 2 of
  the License, or (at your option) any later version.

  The Code_Saturne Preprocessor is distributed in the hope that it will be
  useful, but WITHOUT ANY WARRANTY; without even the implied warranty
  of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with the Code_Saturne Preprocessor; if not, write to the
  Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor,
  Boston, MA  02110-1301  USA
*/


/*============================================================================
 *                                 Visibilit
 *============================================================================*/

/*----------------------------------------------------------------------------
 *  Fichiers `include' librairie standard C ou BFT
 *----------------------------------------------------------------------------*/

#include <assert.h>
#include <stdlib.h>
#include <string.h>

#include <bft_mem.h>
#include <bft_printf.h>
#include <bft_version.h>


/*----------------------------------------------------------------------------
 *  Fichiers `include' visibles du  paquetage global "Utilitaire"
 *----------------------------------------------------------------------------*/

#include "ecs_chaine_glob.h"
#include "ecs_def.h"


/*----------------------------------------------------------------------------
 *  Fichiers `include' visibles des paquetages visibles
 *----------------------------------------------------------------------------*/

#include "ecs_descr.h"
#include "ecs_descr_chaine.h"

#include "ecs_champ_chaine.h"
#include "ecs_champ.h"

#include "ecs_entmail.h"


/*----------------------------------------------------------------------------
 *  Fichiers `include' visibles du  paquetage courant
 *----------------------------------------------------------------------------*/


/*----------------------------------------------------------------------------
 *  Fichier  `include' du  paquetage courant associe au fichier courant
 *----------------------------------------------------------------------------*/

#include "ecs_entmail_pre.h"


/*----------------------------------------------------------------------------
 *  Fichiers `include' privs   du  paquetage courant
 *----------------------------------------------------------------------------*/


/*============================================================================
 *                       Prototypes de fonctions prives
 *============================================================================*/

/*----------------------------------------------------------------------------
 *  Fonction qui affiche des informations sur l'entit de maillage
 *    partir du maillage Ideas lu
 *----------------------------------------------------------------------------*/

static void _ecs_entmail_pre__aff_info
(
       ECS_ENTMAIL_E      ent_e,
 const ecs_int_t          nbr_elt,
 const ecs_int_t          nbr_coul,
 const ecs_int_t   *const coul_val,
 const ecs_size_t  *const nbr_elt_coul
) ;


/*----------------------------------------------------------------------------
 *  Fonction qui affiche des informations sur l'entit de maillage
 *    partir du maillage lu contenant des familles
 *
 *  Les informations affiches sont :
 *  - le nombre d'lments de l'entit
 *  - pour chaque famille le nombre d'lments de cette famille
 *----------------------------------------------------------------------------*/

static void _ecs_entmail_pre__aff_info_fam
(
       ECS_ENTMAIL_E         ent_e,
 const ecs_int_t             nbr_elt,
 const ecs_int_t      *const fam_val
) ;


/*============================================================================
 *                             Fonctions publiques
 *============================================================================*/

/*----------------------------------------------------------------------------
 *  Fonction qui renvoit une structure "entit de maillage" pour les sommets,
 *   construite  partir de tableaux remplis lors de la lecture des donnes
 *----------------------------------------------------------------------------*/

ecs_entmail_t * ecs_entmail_pre__cree_som
(
 ecs_dim_t     dim_e,
 size_t        nbr_som,
 ecs_real_t  * som_val_coord,
 ecs_int_t   * som_val_label
)
{

  ecs_champ_t   * champ_definit ;
  ecs_champ_t   * champ_label ;
  ecs_entmail_t * entmail_som ;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  assert(nbr_som != 0) ;


  /* Initialisation et allocation de la structure de l'entit `sommet' */
  /*-------------------------------------------------------------------*/

  entmail_som = ecs_entmail__cree() ;


  /* Affichage de la dimension de l'espace */

  bft_printf("  ") ;
  ecs_print_padded_str(_("Dimension"), ECS_LNG_AFF_STR) ;
  bft_printf(" : %*d\n", ECS_LNG_AFF_ENT, dim_e) ;


  /* Coordonnes des sommets */
  /*-------------------------*/

  assert(som_val_coord != NULL) ;


  champ_definit
    = ecs_champ__cree(nbr_som,
                      ECS_DIM_3,
                      NULL,
                      som_val_coord,
                      ECS_TYPE_ecs_real_t,
                      ECS_CHAMP_NOM_DEFINIT,
                      NULL,                    /* Aucun descripteur de champ */
                      ECS_CHAMP_STATUT_HERITABLE) ;


  ecs_entmail__ajoute_champ(entmail_som,
                            champ_definit,
                            ECS_CHAMP_DEF  ) ;


  /* Etiquettes des sommets */
  /*------------------------*/

  if (som_val_label != NULL) {


    champ_label
      = ecs_champ__cree(nbr_som,
                        ECS_PAS_UNITE,
                        NULL,
                        som_val_label,
                        ECS_TYPE_ecs_int_t,
                        ECS_CHAMP_NOM_LABEL,
                        NULL,
                        ECS_CHAMP_STATUT_HERITABLE) ;


    ecs_entmail__ajoute_champ(entmail_som,
                              champ_label,
                              ECS_CHAMP_ATT) ;


  }


  /* Impressions */
  /*-------------*/

  _ecs_entmail_pre__aff_info(ECS_ENTMAIL_SOM,
                               nbr_som,
                               0,
                               NULL,
                               NULL) ;


  return entmail_som ;

}


/*----------------------------------------------------------------------------
 *  Fonction qui renvoie un vecteur de structures "entit de maillage"
 *   pour les lments (aretes, faces et cellules),
 *   construites  partir de tableaux remplis lors de la lecture des donnes
 *----------------------------------------------------------------------------*/

ecs_entmail_t * * ecs_entmail_pre__cree_elt
(
 const size_t       nbr_elt_ent        [],
       ecs_size_t * elt_pos_som_ent    [],
       ecs_int_t  * elt_val_som_ent    [],
       ecs_int_t  * elt_val_label_ent  [],
       ecs_int_t  * elt_val_famille_ent[],
       ecs_int_t  * elt_val_couleur_ent[],
 const ecs_int_t    nbr_coul_ent       [],
       ecs_int_t  * val_coul_ent       [],
       ecs_size_t * nbr_elt_coul_ent   []
)
{


  ecs_int_t       icoul ;
  size_t          ielt ;
  ecs_int_t       ient ;

  ecs_bool_t      bool_famille_ent ;

  ecs_champ_t     * champ_definit ;
  ecs_champ_t     * champ_label ;
  ecs_champ_t     * champ_famille ;
  ecs_champ_t     * champ_couleur ;
  ecs_descr_t     * descr_couleur_tete ;
  ecs_descr_t     * descr_couleur ;
  ecs_entmail_t * * vect_entmail ;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  BFT_MALLOC(vect_entmail, ECS_ENTMAIL_FIN, ecs_entmail_t *) ;

  for (ient = ECS_ENTMAIL_ARE ; ient < ECS_ENTMAIL_FIN ; ient++)
    vect_entmail[ient] = NULL ;


  /* Boucle de remplissage des entits du maillage */
  /*===============================================*/


  for (ient = ECS_ENTMAIL_ARE ; ient < ECS_ENTMAIL_FIN ; ient++) {


    /*----------------------------------------------------------*/
    /* S'il y a au moins un lment de cette entit de maillage */
    /*----------------------------------------------------------*/


    if (nbr_elt_ent[ient] != 0) {


      /* Initialisation et allocation des structures des entits de maillage */
      /*---------------------------------------------------------------------*/

      vect_entmail[ient] = ecs_entmail__cree() ;


      /* lments dfinis en fonction des sommets */
      /*------------------------------------------*/


      assert(elt_val_som_ent[ient] != NULL) ;


      champ_definit
        = ecs_champ__cree(nbr_elt_ent[ient],
                          ECS_PAS_NUL,
                          elt_pos_som_ent[ient],
                          elt_val_som_ent[ient],
                          ECS_TYPE_ecs_int_t,
                          ECS_CHAMP_NOM_DEFINIT,
                          NULL,
                          ECS_CHAMP_STATUT_HERITABLE) ;


      ecs_entmail__ajoute_champ(vect_entmail[ient],
                                champ_definit,
                                ECS_CHAMP_DEF ) ;


      /* tiquettes des lments */
      /*-------------------------*/

      if (elt_val_label_ent != NULL) {

        if (elt_val_label_ent[ient] != NULL) {


          champ_label
            = ecs_champ__cree(nbr_elt_ent[ient],
                              ECS_PAS_UNITE,
                              NULL,
                              elt_val_label_ent[ient],
                              ECS_TYPE_ecs_int_t,
                              ECS_CHAMP_NOM_LABEL,
                              NULL,
                              ECS_CHAMP_STATUT_HERITABLE) ;


          ecs_entmail__ajoute_champ(vect_entmail[ient],
                                    champ_label,
                                    ECS_CHAMP_ATT ) ;


        }

      }


      /* Familles ou couleurs des lments */
      /*-----------------------------------*/

      bool_famille_ent = ECS_FALSE ;

      if (elt_val_famille_ent != NULL) {

        if (elt_val_famille_ent[ient] != NULL) {

          bool_famille_ent = ECS_TRUE ;


          /* Impressions */
          /*-------------*/

          _ecs_entmail_pre__aff_info_fam((ECS_ENTMAIL_E)ient,
                                           nbr_elt_ent[ient],
                                           elt_val_famille_ent[ient]) ;

          /* Si les numros de famille sont tous gaux */
          /*  au numro de la famille par defaut (`0') */
          /*  on ne cre pas le champ famille          */

          ielt = 0 ;
          while (ielt < nbr_elt_ent[ient] &&
                 elt_val_famille_ent[ient][ielt] == 0)
            ielt++ ;

          if (ielt != nbr_elt_ent[ient]) {

            /* Tous les numros de famille ne sont pas gaux   */
            /*  au numro de la famille par defaut (`0')     : */
            /*  on cre le champ famille                       */

            /* Familles des lments */
            /*-----------------------*/

            champ_famille
              = ecs_champ__cree(nbr_elt_ent[ient],
                                ECS_PAS_UNITE,
                                NULL,
                                elt_val_famille_ent[ient],
                                ECS_TYPE_ecs_int_t,
                                ECS_CHAMP_NOM_FAMILLE,
                                NULL,
                                ECS_CHAMP_STATUT_HERITABLE) ;


            ecs_entmail__ajoute_champ(vect_entmail[ient],
                                      champ_famille,
                                      ECS_CHAMP_ATT ) ;

          }
          else {

            BFT_FREE(elt_val_famille_ent[ient]) ;

          }

        }

      }


      assert(bool_famille_ent == ECS_FALSE || nbr_coul_ent[ient] == 0) ;

      if (bool_famille_ent == ECS_FALSE) {

        /* Couleurs des lments */
        /*-----------------------*/

        /* Cration des descripteurs de champ "couleur" */

        descr_couleur_tete = NULL ;

        for (icoul = 0 ; icoul < nbr_coul_ent[ient] ; icoul++) {

          descr_couleur = ecs_descr__cree(ECS_DESCR_TYP_COULEUR,
                                          val_coul_ent[ient][icoul],
                                          NULL) ;

          ecs_descr_chaine__ajoute(&descr_couleur_tete,
                                   descr_couleur) ;

        }

        if (elt_val_couleur_ent[ient] != NULL) {

          champ_couleur
            = ecs_champ__cree(nbr_elt_ent[ient],
                              ECS_PAS_UNITE,
                              NULL,
                              elt_val_couleur_ent[ient],
                              ECS_TYPE_ecs_int_t,
                              ECS_CHAMP_NOM_COULEUR,
                              descr_couleur_tete,
                              ECS_CHAMP_STATUT_HERITABLE) ;


          ecs_entmail__ajoute_champ(vect_entmail[ient],
                                    champ_couleur,
                                    ECS_CHAMP_ATT ) ;

        }

        /* Impressions */
        /*-------------*/

        _ecs_entmail_pre__aff_info((ECS_ENTMAIL_E)ient,
                                     nbr_elt_ent[ient],
                                     nbr_coul_ent[ient],
                                     val_coul_ent[ient],
                                     nbr_elt_coul_ent[ient]) ;


      }


    } /* Fin : s'il y a au moins un lment pour l'entit courante */


    if (nbr_coul_ent[ient] != 0) {
      BFT_FREE(val_coul_ent[ient]);
      BFT_FREE(nbr_elt_coul_ent[ient]);
    }


  } /* Fin : boucle sur les entits */


  return vect_entmail ;


}


/*----------------------------------------------------------------------------
 *  Fonction qui renvoie l'identificateur de l'entit
 *   auquel appartient le type gomtrique donn
 *--------------------------------------------------------------------------- */

ECS_ENTMAIL_E ecs_entmail_pre__ret_typ_geo
(
 ecs_int_t typ_geo
)
{

  ECS_ENTMAIL_E     entmail_e ;
  ecs_elt_typ_t     typ_geo_e ;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  typ_geo_e = (ecs_elt_typ_t)typ_geo ;

  entmail_e = ECS_ENTMAIL_FIN ;


  /* Identification de l'entit concerne */
  /*--------------------------------------*/

  switch (typ_geo_e) {

  case ECS_ELT_TYP_SOM:

    entmail_e = ECS_ENTMAIL_SOM ;
    break ;

  case ECS_ELT_TYP_ARE:

    entmail_e = ECS_ENTMAIL_ARE ;
    break ;

  case ECS_ELT_TYP_FAC_TRIA:
  case ECS_ELT_TYP_FAC_QUAD:
  case ECS_ELT_TYP_FAC_POLY:

    entmail_e = ECS_ENTMAIL_FAC ;
    break ;

  case ECS_ELT_TYP_CEL_TETRA:
  case ECS_ELT_TYP_CEL_PRISM:
  case ECS_ELT_TYP_CEL_PYRAM:
  case ECS_ELT_TYP_CEL_HEXA:
  case ECS_ELT_TYP_CEL_POLY:

    entmail_e = ECS_ENTMAIL_CEL ;
    break ;

  default:

    assert(typ_geo_e == ECS_ELT_TYP_SOM       ||
           typ_geo_e == ECS_ELT_TYP_ARE       ||
           typ_geo_e == ECS_ELT_TYP_FAC_TRIA  ||
           typ_geo_e == ECS_ELT_TYP_FAC_QUAD  ||
           typ_geo_e == ECS_ELT_TYP_FAC_POLY  ||
           typ_geo_e == ECS_ELT_TYP_CEL_TETRA ||
           typ_geo_e == ECS_ELT_TYP_CEL_PRISM ||
           typ_geo_e == ECS_ELT_TYP_CEL_PYRAM ||
           typ_geo_e == ECS_ELT_TYP_CEL_HEXA  ||
           typ_geo_e == ECS_ELT_TYP_CEL_POLY    );

  }


  return entmail_e ;

}


/*============================================================================
 *                              Fonctions prives
 *============================================================================*/

/*----------------------------------------------------------------------------
 *  Fonction qui affiche des informations sur l'entit de maillage
 *    partir du maillage Ideas lu
 *
 *  Les informations affiches sont :
 *  - le nombre d'lments de l'entit
 *  - les couleurs dfinies pour cette entit
 *    et pour chaque couleur le nombre d'lments ayant cette couleur
 *----------------------------------------------------------------------------*/

static void _ecs_entmail_pre__aff_info
(
       ECS_ENTMAIL_E         ent_e,
 const ecs_int_t             nbr_elt,
 const ecs_int_t             nbr_coul,
 const ecs_int_t      *const coul_val,
 const ecs_size_t     *const nbr_elt_coul
)
{

  ecs_int_t         max_coul_val ;
  ecs_int_t         max_lng_str_coul ;
  ecs_int_t         icoul ;
  char              str_num_coul[ECS_STR_SIZE] ;
  char       *      str_coul ;

  const char *const ent_msg_info_var_nbr_c[ECS_ENTMAIL_FIN]
    = { N_("Number of vertices"),
        N_("Number of edges"),
        N_("Number of faces"),
        N_("Number of cells")  } ;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  bft_printf("  ") ;
  ecs_print_padded_str(_(ent_msg_info_var_nbr_c[ent_e]), ECS_LNG_AFF_STR) ;
  bft_printf(" : %*ld\n", ECS_LNG_AFF_ENT, (long)nbr_elt) ;


  if (nbr_coul != 0) {

    max_coul_val = coul_val[0] ;

    for (icoul = 1 ; icoul < nbr_coul ; icoul++)
      max_coul_val = ECS_MAX(max_coul_val, coul_val[icoul]) ;

    /* On affiche au minimum sur 2 champs le numro de couleur */
    if (max_coul_val < 10)
      max_coul_val = 10 ;

    sprintf(str_num_coul, "%" ECS_FORMAT_ecs_int_t, max_coul_val) ;


    max_lng_str_coul
      = (ecs_int_t)strlen(_("Color"))
      + (ecs_int_t)strlen(str_num_coul)
      + 2 ; /* Pour ` \0' */

    BFT_MALLOC(str_coul, max_lng_str_coul, char) ;

    for (icoul = 0 ; icoul < nbr_coul ; icoul++) {

      sprintf(str_coul, "%s %" ECS_FORMAT_ecs_int_t,
              _("Color"), coul_val[icoul]) ;

      ecs_print_padded_str(NULL, ECS_LNG_AFF_STR - max_lng_str_coul + 2) ;
      ecs_print_padded_str(str_coul, max_lng_str_coul) ;
      bft_printf(" : %*ld\n", ECS_LNG_AFF_ENT, (long)(nbr_elt_coul[icoul])) ;

    }

    BFT_FREE(str_coul) ;

  }


}


/*----------------------------------------------------------------------------
 *  Fonction qui affiche des informations sur l'entit de maillage
 *    partir du maillage lu contenant des familles
 *
 *  Les informations affiches sont :
 *  - le nombre d'lments de l'entit
 *  - pour chaque famille le nombre d'lments de cette famille
 *----------------------------------------------------------------------------*/

static void _ecs_entmail_pre__aff_info_fam
(
       ECS_ENTMAIL_E         ent_e,
 const ecs_int_t             nbr_elt,
 const ecs_int_t      *const fam_val
)
{

  ecs_int_t         max_fam_val ;
  ecs_int_t         min_fam_val ;
  ecs_int_t         max_lng_str_fam ;

  ecs_int_t         ient ;
  ecs_int_t         ifam ;
  ecs_int_t         nbr_fam_loc ;
  ecs_int_t  *      nbr_elt_fam ;

  char              str_num_fam [ECS_STR_SIZE] ;
  char       *      str_fam ;

  const char *const ent_msg_info_var_nbr_c[ECS_ENTMAIL_FIN]
    = { N_("Number of vertices"),
        N_("Number of edges"),
        N_("Number of faces"),
        N_("Number of cells")  } ;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  bft_printf("  ") ;
  ecs_print_padded_str(_(ent_msg_info_var_nbr_c[ent_e]), ECS_LNG_AFF_STR) ;
  bft_printf(" : %*ld\n", ECS_LNG_AFF_ENT, (long)nbr_elt) ;


  max_fam_val = fam_val[0] ;
  min_fam_val = fam_val[0] ;

  /* Comptage nombre de familles max */

  for (ient = 1 ; ient < nbr_elt ; ient++) {

    if (fam_val[ient] < min_fam_val)
      min_fam_val = fam_val[ient] ;
    else if (fam_val[ient] > max_fam_val)
      max_fam_val = fam_val[ient] ;

  }

  nbr_fam_loc = max_fam_val - min_fam_val + 1 ;

  BFT_MALLOC(nbr_elt_fam, nbr_fam_loc, ecs_int_t) ;

  for (ifam = 0 ; ifam < nbr_fam_loc ; ifam++)
    nbr_elt_fam[ifam] = 0 ;


  /* Comptage nombre de familles par lment */

  for (ient = 0 ; ient < nbr_elt ; ient++)
    nbr_elt_fam[fam_val[ient] - min_fam_val] += 1 ;


  /* Affichage */

  if (nbr_fam_loc != 0) {

    /* On affiche au minimum sur 2 champs le numro de famille */

    sprintf(str_num_fam, "%" ECS_FORMAT_ecs_int_t, ECS_MAX(10, max_fam_val)) ;


    max_lng_str_fam = (ecs_int_t)(  strlen(_("Family"))
                                  + strlen(str_num_fam)
                                  + 2) ; /* Pour ` \0' */

    BFT_MALLOC(str_fam, max_lng_str_fam, char) ;

    for (ifam = 0 ; ifam < nbr_fam_loc ; ifam++) {

      if (nbr_elt_fam[ifam] > 0) {

        sprintf(str_fam, "%s %" ECS_FORMAT_ecs_int_t,
                _("Family"), min_fam_val + ifam) ;

        ecs_print_padded_str(NULL, ECS_LNG_AFF_STR - max_lng_str_fam + 2) ;
        ecs_print_padded_str(str_fam, max_lng_str_fam) ;
        bft_printf(" : %*ld\n", ECS_LNG_AFF_ENT, (long)(nbr_elt_fam[ifam])) ;

      }

    }

    BFT_FREE(str_fam) ;

  }

  BFT_FREE(nbr_elt_fam) ;

}

