/*============================================================================
 *  Définitions des fonctions
 *   associées à la structure `ecs_champ_t' décrivant un champ
 *   et liées à la structure `ecs_vec_real_t'
 *============================================================================*/

/*
  This file is part of the Code_Saturne Preprocessor, element of the
  Code_Saturne CFD tool.

  Copyright (C) 1999-2007 EDF S.A., France

  contact: saturne-support@edf.fr

  The Code_Saturne Preprocessor is free software; you can redistribute it
  and/or modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either version 2 of
  the License, or (at your option) any later version.

  The Code_Saturne Preprocessor is distributed in the hope that it will be
  useful, but WITHOUT ANY WARRANTY; without even the implied warranty
  of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with the Code_Saturne Preprocessor; if not, write to the
  Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor,
  Boston, MA  02110-1301  USA
*/


/*----------------------------------------------------------------------------
 *  Fichiers `include' librairie standard C
 *----------------------------------------------------------------------------*/

#include <assert.h>
#include <string.h> /* strlen() , strcpy() */


/*----------------------------------------------------------------------------
 *  Fichiers `include' système ou BFT
 *----------------------------------------------------------------------------*/

#include <bft_mem.h>


/*----------------------------------------------------------------------------
 *  Fichiers `include' visibles du  paquetage global "Utilitaire"
 *----------------------------------------------------------------------------*/

#include "ecs_def.h"


/*----------------------------------------------------------------------------
 *  Fichiers `include' visibles des paquetages visibles
 *----------------------------------------------------------------------------*/

#include "ecs_vec_real.h"


/*----------------------------------------------------------------------------
 *  Fichiers `include' visibles du  paquetage courant
 *----------------------------------------------------------------------------*/

#include "ecs_champ.h"


/*----------------------------------------------------------------------------
 *  Fichier  `include' du  paquetage courant associé au fichier courant
 *----------------------------------------------------------------------------*/

#include "ecs_champ_vec_real.h"


/*----------------------------------------------------------------------------
 *  Fichiers `include' prives   du  paquetage courant
 *----------------------------------------------------------------------------*/

#include "ecs_champ_priv.h"


/*============================================================================
 *                       Prototypes de fonctions privées
 *============================================================================*/


/*============================================================================
 *                             Fonctions publiques
 *============================================================================*/

/*----------------------------------------------------------------------------
 *  Fonction construisant un vecteur `ecs_vec_real_t' associé au champ donné
 *
 *  Si les tables sont REGLEES, les tableaux sont construits
 *
 *  Cette fonction doit être utilisée conjointement avec :
 *  - soit `ecs_champ__transfere_vec_real()'
 *                               si le contenu du vecteur   a changé
 *  - soit `ecs_champ__libere_vec_real()'
 *                               si le contenu du vecteur n'a pas été modifié
 *  Ceci afin de reconstuire, si nécessaire, les tables REGLEES
 *----------------------------------------------------------------------------*/

ecs_vec_real_t * ecs_champ__initialise_vec_real
(
 ecs_champ_t *this_champ
)
{

  ecs_vec_real_t *vec_real_loc ;

  size_t        pos_nbr    ;  /* Nombre   de positions                        */
  size_t        pos_pas    ;  /* Pas des positions                            */
  ecs_real_t   *val_tab    ;  /* Tableau des valeurs                          */


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  assert(this_champ          != NULL &&
         this_champ->pos_tab == NULL &&
         this_champ->val_tab != NULL    ) ;


  assert(ecs_champ__ret_val_typ(this_champ) == ECS_TYPE_ecs_real_t) ;


  /* Récupération des références des tableaux positions et valeurs   */
  /*  des tables contenues par le champ                              */
  /* (pour une table REGLEE, le tableau correspondant est développé) */
  /*-----------------------------------------------------------------*/

  pos_nbr =               this_champ->nbr_elt + 1 ;
  pos_pas =               this_champ->pos_pas ;
  val_tab = (ecs_real_t *)this_champ->val_tab ;


  /* Affectation des valeurs récupérées au vecteur `ecs_vec_real_t' créé */
  /*---------------------------------------------------------------------*/

  vec_real_loc = ecs_vec_real__initialise(pos_nbr,
                                          pos_pas,
                                          val_tab) ;

  return vec_real_loc ;


}


/*----------------------------------------------------------------------------
 *  Fonction qui transfère le contenu d'un vecteur `ecs_vec_real_t'
 *   dans les tables d'un champ donné
 *
 *  Les tables sont transformées en REGLES si possible
 *
 *  La structure `ecs_vec_real_t' est libérée
 *----------------------------------------------------------------------------*/

void ecs_champ__transfere_vec_real
(
 ecs_champ_t    *this_champ , /* <-> Champ pour lequel transfert est effectue */
 ecs_vec_real_t *vec_real     /* <-> Vecteur `ecs_vec_real_t' à transferer    */
)
{

  size_t       pos_nbr     ;  /* Nombre   de positions                        */
  size_t       pos_pas     ;  /* Pas des positions                            */
  size_t       val_nbr     ;  /* Nombre   de valeurs                          */
  ecs_real_t  *val_tab     ;  /* Tableau des valeurs                          */


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  assert(this_champ->typ_val == ECS_TYPE_ecs_real_t) ;


  pos_nbr = ecs_vec_real__ret_pos_nbr(vec_real) ;
  pos_pas = ecs_vec_real__ret_pos_pas(vec_real) ;
  val_nbr = ecs_vec_real__ret_val_nbr(vec_real) ;
  val_tab = ecs_vec_real__ret_val_tab(vec_real) ;


  this_champ->nbr_elt = pos_nbr - 1 ;


  /* Les positions forment une REGLE de pas unité */

  this_champ->pos_pas = pos_pas ;
  this_champ->pos_tab = NULL ;


  this_champ->val_tab = val_tab ;


  ecs_vec_real__libere(vec_real) ;

}


/*----------------------------------------------------------------------------
 *  Fonction qui libère le tableau d'une table REGLEE qui a été
 *   précédemment développé par l'appel de `ecs_champ__initialise_vec_real()'
 *  et qui détruit le vecteur `ecs_vec_real_t' qui a été
 *   précédemment créé      par l'appel de `ecs_champ__initialise_vec_real()'
 *----------------------------------------------------------------------------*/

void ecs_champ__libere_vec_real
(
 ecs_champ_t     *this_champ ,/* <-> Champ associé au vecteur                 */
 ecs_vec_real_t  *vec_real    /* <-> Vecteur `ecs_vec_real_t' à detruire      */
)
{

  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  assert(this_champ->val_tab == ecs_vec_real__ret_val_tab(vec_real)) ;

  ecs_vec_real__libere(vec_real) ;

}


/*----------------------------------------------------------------------------
 *  Fonction qui crée un champ à partir
 *   du contenu d'un vecteur `ecs_vec_real_t'
 *
 *  Seules les tables sont remplies
 *   (elles sont transformées en REGLES si possible)
 *
 *  Les autres membres de la structure prennent leur valeur d'initialisation
 *
 *  La structure `ecs_vec_real_t' est libérée
 *----------------------------------------------------------------------------*/

ecs_champ_t * ecs_champ__init_avec_vec_real
(
       ecs_vec_real_t *      vec_real ,           /* --> Vecteur à transférer */
 const char           *const nom
)
{

  ecs_champ_t   *this_champ  ;

  size_t       pos_nbr   ;  /* Nombre   de positions                          */
  size_t       val_nbr   ;  /* Nombre   de valeurs                            */
  ecs_real_t  *val_tab   ;  /* Tableau des valeurs                            */

  size_t       pos_pas   ;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  /* Récupération des valeurs du vecteur `ecs_vec_real_t' */

  pos_nbr = ecs_vec_real__ret_pos_nbr(vec_real) ;
  pos_pas = ecs_vec_real__ret_pos_pas(vec_real) ;
  val_nbr = ecs_vec_real__ret_val_nbr(vec_real) ;
  val_tab = ecs_vec_real__ret_val_tab(vec_real) ;


  /* Libération du vecteur indexe */
  /*------------------------------*/

  ecs_vec_real__libere(vec_real) ;


  /*--------------------------------------------*/
  /* Création de la structure de type `ecs_champ_t' */
  /*--------------------------------------------*/

  BFT_MALLOC(this_champ, 1, ecs_champ_t);


  this_champ->nbr_elt = pos_nbr - 1 ;

  this_champ->typ_val = ECS_TYPE_ecs_real_t ;


  /* Allocation et affectation du nom du champ */
  /*-------------------------------------------*/

  if (nom != NULL) {
    BFT_MALLOC(this_champ->nom, strlen(nom) + 1, char) ;
    strcpy(this_champ->nom, nom) ;
  }
  else {
    this_champ->nom = NULL ;
  }


  /* Initialisation de la référence au descripteur de champ */
  /*--------------------------------------------------------*/

  this_champ->descr       = NULL ;


  /* Initialisation du lien pour les listes chaînées */
  /* des champs auxiliaires                          */
  /*-------------------------------------------------*/

  this_champ->l_champ_sui = NULL ;


  /* Statut du champ dans une transformation */
  /*  de type découpage ou recollement       */
  /*-----------------------------------------*/

  this_champ->statut_e    = ECS_CHAMP_STATUT_INDEFINI ;


  /* Affectation de la table des positions des valeurs */
  /*---------------------------------------------------*/

  this_champ->pos_pas = pos_pas ;
  this_champ->pos_tab = NULL ;


  /* Affectation de la table des valeurs */
  /*-------------------------------------*/

  this_champ->val_tab = val_tab ;


  return this_champ ;

}


/*============================================================================
 *                              Fonctions privées
 *============================================================================*/

