/* This file is part of EdiTeX, an editor of mathematical
 * expressions based on TeX syntax.
 * 
 * Copyright (C) 2002-2003 Luca Padovani <lpadovan@cs.unibo.it>,
 *                    2003 Paolo Marinelli <pmarinel@cs.unibo.it>.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * For more information, please visit the project's home page
 * http://helm.cs.unibo.it/editex/
 * or send an email to <lpadovan@cs.unibo.it>
 */

#include <GdomeSmartDOMXSLT.hh>
#include <cassert>

#include "ALogger.hh"
#include "TDictionary.hh"
#include "CMathMLFactoryXSLT.hh"
#include "TPushLexer.hh"
#include "LPushLexer.hh"
#include "TPushParser.hh"

class CCallbackLogger : public ALogger
{
public:
  CCallbackLogger(void (*)(int, const char*, void*), void*);
  virtual ~CCallbackLogger() { };

protected:
  virtual void message(Level, const std::string&);

private:
  void (*callback)(int, const char*, void*);
  void* user_data;
};

CCallbackLogger::CCallbackLogger(void (*cb)(int, const char*, void*), void* data) : callback(cb), user_data(data)
{
  assert(callback);
}

void
CCallbackLogger::message(Level l, const std::string& s)
{
  assert(callback);
  callback(l, s.c_str(), user_data);
}

struct Editor
{
  Editor(const char*, const char*, const char*, void (*)(int, const char*, void*), void*, bool);
  ~Editor();

  ALogger*        logger;
  TDictionary*    dictionary;
  DOMX::XSLTStylesheet* tml_mml;
  DOMX::XSLTStylesheet* tml_tex;
  AMathMLFactory* factory;
  TPushParser*    parser;
  APushLexer*     lexer;
};

Editor::Editor(const char* dict_uri, const char* mml_uri, const char* tex_uri,
	       void (*cb)(int, const char*, void*), void* data, bool alt)
{
  assert(dict_uri);
  assert(mml_uri);
  assert(tex_uri);
  assert(cb);
  logger = new CCallbackLogger(cb, data);
  dictionary = new TDictionary(*logger);
  dictionary->load(dict_uri);
  DOM::DOMImplementation di;
  DOM::Document mml = di.createDocumentFromURI(mml_uri);
  DOM::Document tex = di.createDocumentFromURI(tex_uri);
  tml_mml = new DOMX::XSLTStylesheet(mml);
  tml_tex = new DOMX::XSLTStylesheet(tex);
  factory = new CMathMLFactoryXSLT(*logger, *tml_mml);
  parser = new TPushParser(*logger, *factory, *dictionary);
  if (alt) lexer = new LPushLexer(*logger, *parser);
  else lexer = new TPushLexer(*logger, *parser);
}

Editor::~Editor()
{
  delete lexer;
  delete parser;
  delete factory;
  delete tml_tex;
  delete tml_mml;
  delete dictionary;
  delete logger;
}

extern "C" const char*
c_mathml_editor_get_default_dictionary_path()
{
  return TDictionary::getDefaultDictionaryPath().c_str();
}

extern "C" const char*
c_mathml_editor_get_default_mathml_stylesheet_path()
{
  return AMathMLFactory::getDefaultMathMLStylesheetPath().c_str();
}

extern "C" const char*
c_mathml_editor_get_default_tex_stylesheet_path()
{
  return AMathMLFactory::getDefaultTeXStylesheetPath().c_str();
}

extern "C" Editor*
c_mathml_editor_new(bool alt,
		    const char* dictionary_uri,
		    const char* tml_mml_uri,
		    const char* tml_tex_uri,
		    void (*log_message_cb)(int, const char*, void*),
		    void* user_data)
{
  return new Editor(dictionary_uri, tml_mml_uri, tml_tex_uri, log_message_cb, user_data, alt);
}

extern "C" void
c_mathml_editor_destroy(Editor* editor)
{
  assert(editor);
  delete editor;
}

extern "C" int
c_mathml_editor_freeze(Editor* editor)
{
  assert(editor);
  return editor->parser->freeze();
}

extern "C" int
c_mathml_editor_thaw(Editor* editor)
{
  assert(editor);
  return editor->parser->thaw();
}

extern "C" void
c_mathml_editor_push(Editor* editor, char ch)
{
  assert(editor);
  editor->lexer->push(ch);
}

extern "C" void
c_mathml_editor_drop(Editor* editor, int alt)
{
  assert(editor);
  editor->lexer->drop(alt != 0);
}

extern "C" int
c_mathml_editor_cursor_hide(Editor* editor)
{
  assert(editor);
  return editor->parser->hideCursor();
}

extern "C" int
c_mathml_editor_cursor_show(Editor* editor)
{
  assert(editor);
  return editor->parser->showCursor();
}

extern "C" char*
c_mathml_editor_get_tex(const Editor* editor)
{
  assert(editor);
  DOM::Document res = editor->tml_tex->apply(editor->parser->document());
  assert(res);
  res.normalize();
  assert(res.get_firstChild() && res.get_firstChild().get_nodeName() == "#text");
  return strdup(std::string(res.get_firstChild().get_nodeValue()).c_str());
}

extern "C" void
c_mathml_editor_reset(Editor* editor)
{
  assert(editor);
  editor->lexer->reset();
  editor->parser->reset();
}

extern "C" GdomeDocument*
c_mathml_editor_get_tml(const Editor* editor)
{
  assert(editor);
  GdomeNode* n = editor->parser->document().cloneNode(true).gdome_object();
  GdomeDocument* doc = gdome_cast_doc(n);
  assert(n && doc);
  return doc;
}

extern "C" GdomeDocument*
c_mathml_editor_get_mml(const Editor* editor)
{
  assert(editor);
  GdomeNode* n = editor->factory->document().gdome_object();
  GdomeDocument* doc = gdome_cast_doc(n);
  assert(n && doc);
  return doc;
}

