/* Copyright (C) 2005 Chris Vine

This program is distributed under the General Public Licence, version 2.
For particulars of this and relevant disclaimers see the file
COPYING distributed with the source files.

*/

#include <utility>

#include <glib/gmain.h>

#include "notifier.h"

int Notifier::count = -1;
InstanceMap Notifier::instance_map;
PipeFifo Notifier::pipe;
guint Notifier::iowatch_tag;

Notifier::Notifier(void) {

  count++;

  if (count < 0) {
    write_error("More than INT_MAX Notifier objects created: this one will not be active\n");
    id = -1;
  }

  else id = make_id();

  if (id != -1) {
    instance_map.insert(std::pair<int, Notifier*>(id, this));

    if (!count) {

      if (pipe.open(PipeFifo::block) == -1) {
	write_error("Cannot open pipe in Notifier::Notifier()\n");
      }

      else {
	iowatch_tag = start_iowatch(pipe.get_read_fd(),
				    sigc::mem_fun(*this, &Notifier::read_pipe_slot),
				    G_IO_IN);
      }
    }
  }
}

Notifier::~Notifier(void) {

  count--;
  if (count < 0) {
    g_source_remove(iowatch_tag);
    pipe.close();
  }
  if (id != -1) instance_map.erase(id);
}

sigc::connection Notifier::connect(const sigc::slot<void>& cb) {

  return sig.connect(cb);
}

void Notifier::emit(void) {

  if (id != -1) pipe.write(reinterpret_cast<char*>(&id), sizeof(int));
}

int Notifier::make_id(void) {

  // valid values of id are 0 to INT_MAX
  // as an approximation, start with the current count for new id
  int new_id = count;

  int iteration_count = -1;

  InstanceMap::iterator map_iter;
  do {
    iteration_count++;
    map_iter = instance_map.find(new_id);

    // map_iter == instance_map.end() if the id is not yet in use
    if (map_iter != instance_map.end()) {
      if (new_id < 0) new_id = 0;
      else new_id++;
    }
  } while (map_iter != instance_map.end() && iteration_count >= 0);
  
  if (iteration_count < 0) {
    write_error("Mapping count error in Notifier::make_id()\n");
    new_id = -1;
  }
  
  return new_id;
}

bool Notifier::read_pipe_slot(void) {
  int id;
  int remaining = sizeof(int);
  ssize_t result;
  char* temp_p = reinterpret_cast<char*>(&id);
  do {
    result = Notifier::pipe.read(temp_p, remaining);
    if (result > 0) {
      temp_p += result;
      remaining -= result;
    }
  } while (remaining             // more to come
	   && result             // not end of file
	   && result != -1);     // no error

  if (result > 0) {
    // extract the character from the pipe and iterators for the slots to which it refers
    InstanceMap::iterator map_iter = instance_map.find(id);
    if (map_iter == instance_map.end()) {
      write_error("Yikes! There is a mapping error in Notifier::read_pipe_slot()\n");
    }

    else {
      // signal to the connected slot(s)
      map_iter->second->sig();
    }
  }
  else {  // end of file or error on pipe
    write_error("IO error in Notifier::read_pipe_slot()\n");
  }
  return true; // multishot
}
