/* Copyright (C) 2005 Chris Vine

This program is distributed under the General Public Licence, version 2.
For particulars of this and relevant disclaimers see the file
COPYING distributed with the source files.

*/


#include "thread.h"
#include "sem_sync.h"

struct ThreadArgs {
  const sigc::slot<void>* slot_p;
  SemSync* sem_p;
};

namespace {
void* thread_func(void* arg) {

  // this is executing in the new thread
  ThreadArgs* thread_args_p = static_cast<ThreadArgs*>(arg);
  sigc::slot<void> callback = *(thread_args_p->slot_p);
  thread_args_p->sem_p->post();
  // at this point the pointers in pthread_args will cease to be valid
  // but that's OK as we now hold our own local copy of the slot and
  // we have finished with the semaphore
  callback();
  return 0;
}
} // anonymous namespace

namespace Thread {

std::auto_ptr<Thread> Thread::start(const sigc::slot<void>& cb, bool joinable) {

  // set joinable attribute
  int detach_state;
  if (joinable) detach_state = PTHREAD_CREATE_JOINABLE;
  else detach_state = PTHREAD_CREATE_DETACHED;
  pthread_attr_t attr;
  pthread_attr_init(&attr);
  pthread_attr_setdetachstate(&attr, detach_state);

  Thread* instance_p = new Thread;

  ThreadArgs thread_args;
  SemSync sem;
  thread_args.slot_p = &cb;
  thread_args.sem_p = &sem;

  std::auto_ptr<Thread> return_val(instance_p);
  pthread_t thread;
  if (!pthread_create(&thread, &attr, ::thread_func, &thread_args)) {
    instance_p->thread = thread;
    sem.wait();
  }
  else {
    return_val.reset();
  }
  return return_val;
}

CancelBlock::CancelBlock(bool blocking) {
  // the default value of blocking is true
  if (blocking) block();
  else unblock();
}

} // namespace Thread
