/* Copyright (C) 2005 Chris Vine

The library comprised in this file or of which this file is part is
distributed by Chris Vine under the GNU Lesser General Public
License as follows:

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public License
   as published by the Free Software Foundation; either version 2.1 of
   the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License, version 2.1, for more details.

   You should have received a copy of the GNU Lesser General Public
   License, version 2.1, along with this library (see the file LGPL.TXT
   which came with this source code package in the src/utils sub-directory);
   if not, write to the Free Software Foundation, Inc.,
   59 Temple Place - Suite 330, Boston, MA, 02111-1307, USA.

However, it is not intended that the object code of a program whose
source code instantiates a template from this file should by reason
only of that instantiation be subject to the restrictions of use in
the GNU Lesser General Public License.  With that in mind, the words
"and instantiations of templates (of any length)" shall be treated as
inserted in the fourth paragraph of section 5 of that licence after
the words "and small inline functions (ten lines or less in length)".
This does not affect any other reason why object code may be subject
to the restrictions in that licence (nor for the avoidance of doubt
does it affect the application of section 2 of that licence to
modifications of the source code in this file).

*/

#ifndef GOBJ_HANDLE_H
#define GOBJ_HANDLE_H

/*
   This is a class which manages the reference count of g_objects.  It
   does not maintain its own reference count, but interfaces with that
   kept by the g_object system.  To that extent it is analogous to an
   intrusive smart pointer or handle.

   It can also be used to reference GTK+ widgets derived from
   GtkObject not embedded in a container (or from which ownership is
   to be claimed from a container), but if the object concerned has
   never been embedded in a container, g_object_ref() and
   gtk_object_sink() must be called on it (in that order) before the
   GobjHandle object managing it goes out of scope.

   The class has operator*() and operator->() dereferencing operators,
   so has normal smart pointer functionality, but as it is intended
   for use with the normal C g_object/pango/GTK+ interfaces, ordinary
   use would involve passing the handle to a function taking a pointer
   to the contained object by means of the operatorT*() type
   conversion operator (which returns the underlying pointer), or by
   explicitly calling the get() method to obtain the underlying
   pointer.
*/

#include <glib-object.h>

template <class T> class GobjHandle {

  T* obj_p;

  void unreference(void) {
    if (obj_p) g_object_unref(G_OBJECT(obj_p));
  }

  void reference(void) {
    if (obj_p) g_object_ref(G_OBJECT(obj_p));
  }

public:
  // constructor of first GobjHandle holding the referenced object
  // this method doesn't throw
  explicit GobjHandle(T* ptr = 0) {
    obj_p = ptr;
  }

  // copy constructor
  // this method doesn't throw
  GobjHandle(const GobjHandle& gobj_ptr) {
    obj_p = gobj_ptr.obj_p;
    reference();
  }

  // copy assignment
  // this method doesn't throw
  GobjHandle& operator=(const GobjHandle& gobj_ptr) {

    // check whether we are already referencing this object -
    // if so make this a null op.  This will also deal with
    // self-assignment
    if (obj_p != gobj_ptr.obj_p) {

      // first unreference any object referenced by this handle
      unreference();

      // now inherit the GObject from the assigning handle
      // and reference it
      obj_p = gobj_ptr.obj_p;
      reference();
    }
    return *this;
  }

  // these accessor methods don't throw
  T* get(void) const {return obj_p;}
  T& operator*(void) const {return *obj_p;}
  T* operator->(void) const {return obj_p;}
  operator T*(void) const {return obj_p;}

  // destructor
  // this method doesn't throw
  ~GobjHandle(void) {unreference();}
};

#endif
