/* Copyright (C) 2005 Chris Vine

The library comprised in this file or of which this file is part is
distributed by Chris Vine under the GNU Lesser General Public
License as follows:

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public License
   as published by the Free Software Foundation; either version 2.1 of
   the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License, version 2.1, for more details.

   You should have received a copy of the GNU Lesser General Public
   License, version 2.1, along with this library (see the file LGPL.TXT
   which came with this source code package in the src/utils sub-directory);
   if not, write to the Free Software Foundation, Inc.,
   59 Temple Place - Suite 330, Boston, MA, 02111-1307, USA.

*/

#ifndef NOTIFIER_H
#define NOTIFIER_H

/* This class provides thread-safe signalling between two threads. It
   does this through a pipe, to which an iowatch (WatchSource) object
   is attached to connect it to the Glib program event loop.  It is
   similar in effect to, though a little different in implementation
   from, a Glib::Dispatcher object from glibmm.  A slot is connected
   to the notifier, which is called in the receiving thread via the
   program event loop when operator() (or emit()) is called on the
   Notifier object by the signalling thread.  It is therefore similar
   in effect and syntax to a (not-thread safe) signal of type
   sigc::signal0<void>.

   It is a requirement that a Notifier object be constructed in the
   thread in which the default main Glib event loop (GMainContext)
   executes (which is the one with which the program first starts),
   and the connected slot will execute in that thread.  Its main use
   is for a worker thread to signal an event to the main thread in
   which GTK+ is executing, which implies that GTK+ should also be
   executing in the default Glib program event loop (as will almost
   always be the case, and is the case with efax-gtk).

   For a program with two GMainContext program event loops (not a
   usual case), it would be possible for a Notifier-like object to be
   constructed in the non-default GMainContext thread, and execute in
   that thread, by passing that other GMainContext object as the last
   argument when calling start_iowatch() in Notifier::Notifier().
   However, to conserve file descriptors all Notifier objects share a
   common pipe and iowatch event watch, which implies that all
   Notifier objects would also need to be constructed and execute in
   that other thread.  To get around this it would be possible to
   templatize Notifier with tag types for different GMainContexts (so
   that there would be a different static pipe/iowatch object for each
   GMainContext), but an easier solution for one-off cases would be to
   have a version of Notifier which does not use static (shared)
   PipeFifo and iowatch objects, which would also save having to keep
   individual Notifier object ids internally and sending them over the
   pipe at the expense of greater use of file descriptor resources.
   None of this is relevant to efax-gtk.

   Such a special Notifier object could also be used to signal from a
   Unix (asynchronous) signal/interrupt handler, but in that case the
   write file descriptor of the pipe should be set non-blocking to
   prevent the very unlikely but theoretically possible case (in a
   program executing in a system under extreme load) of the pipe
   filling up before being emptied by the Notifier::read_pipe_slot()
   callback function executing in the main program and so blocking in
   the handler, thus deadlocking the program.
*/


#include "prog_defs.h"      // for sigc::slot, ssize_t and write_error()

#include <string>
#include <map>

#include <sigc++/sigc++.h>

#include "pipes.h"
#include "io_watch.h"

// uncomment the following if the specialist write_error() function in
// prog_defs.h is not available in the application in question
/*
#include <iostream>
inline void write_error(const char* message) {
  std::cerr << message;
}
*/

class Notifier;

typedef std::map<int, Notifier*> InstanceMap;

namespace Thread {
  class Mutex;
}

struct NotifierError {
  std::string error_message;
  NotifierError(const char* msg): error_message(msg) {}
};

class Notifier {

  static int count;
  static InstanceMap instance_map;
  static PipeFifo pipe;
  static Thread::Mutex* write_mutex_p;
  static guint iowatch_tag;
  static bool read_pipe_slot(void);
  static int make_id(void);
 
  sigc::signal0<void> sig;
  int id;

  // Notifier objects cannot be copied
  Notifier(const Notifier&);
  Notifier& operator=(const Notifier&);
public:
  void emit(void);
  void operator()(void) {emit();}
  sigc::connection connect(const sigc::slot<void>&);

  // the constructor can throw exception PipeError or exception NotifierError
  Notifier(void);
  ~Notifier(void);
};


#endif
