/* Copyright (C) 2005 to 2010 Chris Vine

The library comprised in this file or of which this file is part is
distributed by Chris Vine under the GNU Lesser General Public
License as follows:

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public License
   as published by the Free Software Foundation; either version 2.1 of
   the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License, version 2.1, for more details.

   You should have received a copy of the GNU Lesser General Public
   License, version 2.1, along with this library (see the file LGPL.TXT
   which came with this source code package in the c++-gtk-utils
   sub-directory); if not, write to the Free Software Foundation, Inc.,
   59 Temple Place - Suite 330, Boston, MA, 02111-1307, USA.

*/

#ifndef CGU_THREAD_H
#define CGU_THREAD_H

#include <memory>

#include <pthread.h>

#include <c++-gtk-utils/callback.h>
#include <c++-gtk-utils/mutex.h>
#include <c++-gtk-utils/cgu_config.h>

namespace Cgu {

namespace Thread {

/**
 * @class Cgu::Thread::Thread thread.h c++-gtk-utils/thread.h
 * @brief A class representing a pthread thread.
 * @sa Thread::Mutex Thread::Mutex::Lock Thread::Cond Thread::Future Thread::JoinableHandle
 *
 * The Thread class encapsulates a pthread thread.  It can start, join
 * and cancel a thread.
 * @note The Thread class, and the associated CancelBlock class, can
 * be used interchangibly with (and mixed with) GThread objects and
 * functions, and GMutex, GStaticMutex, GStaticRecMutex and GCond, as
 * they all use pthreads underneath on POSIX and other unix-like OSes.
 * The main feature available with this thread implementation and not
 * GThreads is thread cancellation, the mutex scoped locking classes
 * Thread::Mutex::Lock and Thread::RecMutex::Lock, the joinable thread
 * scoped management class Thread::JoinableHandle and the
 * Thread::Future class (abstracting thread functions which provide a
 * result).  There is no need from the perspective of this class to
 * call g_thread_init() before Thread::Thread::start() is called, but
 * glib itself is not thread-safe without g_thread_init(), so where
 * this class is used with glib g_thread_init() should be called at
 * program initialization.
 */


class Thread {
  pthread_t thread;
  // private constructor - this class can only be created with Thread::start
  Thread() {}
  // and it cannot be copied - this class has single ownership semantics
  Thread(const Thread&);
  Thread& operator=(const Thread&);
public:

/**
 * Cancels the thread represented by this Thread object.  It can be
 * called by any thread.  The effect is undefined if the thread
 * represented by this Thread object has both (a) already terminated
 * and (b) been detached or had a call to join() made for it.
 * Accordingly, if the user is not able to establish from the program
 * logic whether the thread has terminated, the thread must be created
 * as joinable and cancel() must not be called after a call to
 * detach() has been made or a call to join() has returned.  A
 * Thread::JoinableHandle object can used to ensure this.  It does not
 * throw.
 * @note Use this method with great care - sometimes its use is
 * unavoidable but destructors for local objects may not be called if
 * a thread exits by virtue of a call to cancel() (that depends on the
 * implementation).  Most modern commercial unixes, and recent
 * linux/BSD distributions based on NPTL, will unwind the stack and
 * call destructors on thread cancellation, but older distributions
 * relying on the former linuxthreads implementation will not.
 * Therefore for maximum portability only use it if there are plain
 * data structures/built-in types in existence in local scope when it
 * is called and if there is anything in free store to be released
 * implement some clean-ups with
 * pthread_cleanup_push()/pthread_cleanup_pop().  This should be
 * controlled with pthread_setcancelstate() and/or the CancelBlock
 * class to choose the cancellation point.
 * @sa Cgu::Thread::Exit
 */
  void cancel() {pthread_cancel(thread);}

/**
 * Joins the thread represented by this Thread object (that is, waits
 * for it to terminate).  It can be called by any thread other than
 * the one represented by this Thread object.  The result is undefined
 * if the thread is or was detached or join() has already been called
 * for the thread (a Thread::JoinableHandle object will however give a
 * defined result in such cases for threads originally started as
 * joinable).  It does not throw.
 */
  void join() {pthread_join(thread, 0);}

/**
 * Detaches the thread represented by this Thread object where it is
 * joinable, so as to make it unjoinable.  The effect is unspecified
 * if the thread is already unjoinable (a Thread::JoinableHandle
 * object will however give a defined result in such cases for threads
 * originally started as joinable).  It does not throw.
 */
  void detach() {pthread_detach(thread);}

/**
 * Specifies whether the calling thread is the same thread as is
 * represented by this Thread object.  The effect is undefined if the
 * thread represented by this Thread object has both (a) already
 * terminated and (b) been detached or had a call to join() made for
 * it.  Accordingly, if the user is not able to establish from the
 * program logic whether the thread has terminated, the thread must be
 * created as joinable and is_caller() must not be called after a call
 * to detach() has been made or a call to join() has returned.  A
 * Thread::JoinableHandle object can used to ensure this.  This method
 * does not throw.
 * @return Returns true if the caller is in the thread represented by
 * this Thread object.
 */
  bool is_caller() {return pthread_equal(thread, pthread_self());}

/**
 * Starts a new thread.  It can be called by any thread.
 * @param cb A callback object (created by Callback::make())
 * encapsulating the function to be executed by the new thread.  The
 * Thread object returned by this function will take ownership of the
 * callback: it will automatically be deleted either by the new thread
 * when it has finished with it, or by this method in the calling
 * thread if the attempt to start a new thread fails (including if
 * std::bad_alloc is thrown).
 * @param joinable Whether the join() method may be called in relation
 * to the new thread.
 * @return A Thread object representing the new thread which has been
 * started, held by a std::auto_ptr object as it has single ownership
 * semantics.  The std::auto_ptr object will be empty (that is
 * std::auto_ptr<Cgu::Thread::Thread>::get() will return 0) if the
 * thread did not start correctly.
 * @exception std::bad_alloc This method might throw std::bad_alloc if
 * memory is exhausted and the system throws in that case.  (This
 * exception will not be thrown if the library has been installed
 * using the --with-glib-memory-slices-no-compat configuration option:
 * instead glib will terminate the program if it is unable to obtain
 * memory from the operating system.)
 * @note 1. The thread will keep running even if the return value of
 * start() goes out of scope (but it will no longer be possible to
 * call any of the methods in this class for it, which is fine if the
 * thread is not started as joinable and it is not intended to cancel
 * it).
 * @note 2. If the thread is started with the joinable attribute, the
 * user must subsequently either call the join() or the detach()
 * method, as otherwise a resource leak may occur (the destructor of
 * this class does not call detach() automatically).  Alternatively,
 * the return value of this method can be passed to a
 * Thread::JoinableHandle object which will do this automatically in
 * the Thread::JoinableHandle object's destructor.
 * @note 3. Any Thread::Exit exception thrown from the function
 * executed by the new thread will be caught and consumed.  The thread
 * will safely terminate and unwind the stack in so doing.
 * @note 4. If any uncaught exception other than Thread::Exit is
 * allowed to propagate from that function, the exception is not
 * consumed (NPTL's forced stack unwinding on cancellation does not
 * permit catching with an ellipsis argument without rethrowing, and
 * even if it did permit it, the result would be an unreported error).
 * Neither POSIX nor the current C++98/03 standard specifies what is
 * to happen in that case.  In practice, most implementations will
 * call std::terminate() and so terminate the entire program.
 * Accordingly, a user must make sure that no exceptions, other than
 * Thread::Exit or any cancellation pseudo-exception, can propagate
 * from the function executed by the new thread.
 */
  static std::auto_ptr<Cgu::Thread::Thread> start(const Cgu::Callback::Callback* cb,
						  bool joinable);

#ifdef CGU_USE_GLIB_MEMORY_SLICES_NO_COMPAT
  CGU_GLIB_MEMORY_SLICES_FUNCS
#endif
};

/**
 * @class Cgu::Thread::JoinableHandle thread.h c++-gtk-utils/thread.h
 * @brief A class wrapping a Thread::Thread object representing a
 * joinable thread.
 * @sa Thread::Thread Thread::Future
 *
 * This class enables a joinable thread to be made more easily
 * exception safe.  It can also be used to provide that a joinable
 * thread is not detached or joined while other methods dependent on
 * that might still be called, and to provide a defined result where
 * there are multiple calls to join() and/or detach().  When it is
 * destroyed, it will either detach or join the thread represented by
 * the wrapped Thread::Thread object unless it has previously been
 * detached or joined using the detach() or join() methods, so
 * avoiding thread resource leaks.  Whether it will detach() or join()
 * on destruction depends on the Thread::JoinableHandle::Action
 * argument passed to the
 * Thread::JoinableHandle::JoinableHandle(std::auto_ptr<Thread::Thread>&,
 * Action) constructor.
 * 
 * Passing Thread::JoinableHandle::detach_on_exit to that argument is
 * not always the correct choice where the thread callback has been
 * bound to a reference argument in local scope and an exception might
 * be thrown, because the thread will keep running after the
 * Thread::JoinableHandle object and other local variables have
 * (because of the exception) gone out of scope.  Consider the
 * following trivial parallelized calculation example:
 *
 * @code
 * std::vector<int> get_readings();
 * void get_mean(const std::vector<int>& v, int& result);
 * void get_std_deviation(const std::vector<int>& v, int& result); // might throw
 * void show_result(int mean, int deviation);
 *
 * using namespace Cgu;
 * void do_calc() {
 *   int i, j;
 *   std::vector<int> v = get_readings();
 *   // with bound reference arguments, Callback::make() requires explicit type instantation
 *   std::auto_ptr<Thread::Thread> t =
 *     Thread::Thread::start(Callback::make<const std::vector<int>&, int&>(&get_mean, v, i), true);
 *   if (t.get()) {  // checks whether thread started correctly
 *     get_std_deviation(v, j);
 *     t->join();
 *     show_result(i, j);
 *   }
 * }
 * @endcode
 *
 * If get_std_deviation() throws, as well as there being a potential
 * thread resource leak by virtue of no join being made, the thread
 * executing get_mean() will continue running and attempt to access
 * variable v, and put its result in variable i, which may by then
 * both be out of scope.  To deal with such a case, the thread could
 * be wrapped in a Thread::JoinableHandle object which joins on exit
 * rather than detaches, for example:
 *
 * @code
 * ...
 * using namespace Cgu;
 * void do_calc() {
 *   int i, j;
 *   std::vector<int> v = get_readings();
 *   // with reference arguments, Callback::make() requires explicit type instantation
 *   Thread::JoinableHandle t(Thread::Thread::start(Callback::make<const std::vector<int>&, int&>(&get_mean, v, i), true),
 *                            Thread::JoinableHandle::join_on_exit);
 *   if (t.is_managing()) {  // checks whether thread started correctly
 *     get_std_deviation(v, j);
 *     t.join();
 *     show_result(i, j);
 *   }
 * }
 * @endcode
 *
 * Better still, however, would be to use Cgu::Thread::Future in this
 * kind of usage, namely a usage where a worker thread is intended to
 * provide a result for inspection.
 *
 * @note These examples assume that the std::vector library
 * implementation permits concurrent reads of a vector object by
 * different threads.  Whether that is the case depends on the
 * documentation of the library concerned (if designed for a
 * multi-threaded environment, most will permit this).
 */
class JoinableHandle {
public:
  enum Action {detach_on_exit, join_on_exit};

private:
  Mutex mutex; // make this the first member so the constructors are strongly exception safe
  Action action;
  bool detached;
  std::auto_ptr<Cgu::Thread::Thread> thread_a;

public:
/**
 * Cancels the thread represented by the wrapped Thread::Thread
 * object.  It can be called by any thread.  The effect is undefined
 * if when called the thread represented by the wrapped Thread::Thread
 * object has both (a) already terminated and (b) had a call to join()
 * or detach() made for it.  Accordingly, if the user is not able to
 * establish from the program logic whether the thread has terminated,
 * cancel() must not be called after a call to detach() has been made
 * or a call to join() has returned: this can be ensured by only
 * detaching or joining via this object's destructor (that is, by not
 * using the explicit detach() and join() methods).  This method does
 * not throw.
 * @note Use this method with great care - see Thread::cancel() for
 * further information.
 *
 * Since 1.0.2
 */
  void cancel();

/**
 * Joins the thread represented by the wrapped Thread::Thread object
 * (that is, waits for it to terminate), unless the detach() or join()
 * method has previously been called in which case this call does
 * nothing.  It can be called by any thread other than the one
 * represented by the wrapped Thread::Thread object, but only one
 * thread can wait on it: if one thread (thread A) calls it while
 * another thread (thread B) is already blocking on it, thread A's
 * call to this method will return immediately and return false.  It
 * does not throw.
 * @return true if a successful join() has been accomplished (that is,
 * detach() or join() have not previously been called), otherwise
 * false.
 *
 * Since 1.0.2
 */
  bool join();

/**
 * Detaches the thread represented by this Thread::Thread object, so
 * as to make it unjoinable, unless the detach() or join() method has
 * previously been called in which case this call does nothing.  It
 * does not throw.
 *
 * Since 1.0.2
 */
  void detach();

/**
 * Specifies whether the calling thread is the same thread as is
 * represented by the wrapped Thread::Thread object.  It can be called
 * by any thread.  The effect is undefined if the thread represented
 * by the wrapped Thread::Thread object has both (a) already
 * terminated and (b) had a call to join() or detach() made for it.
 * Accordingly, if the user is not able to establish from the program
 * logic whether the thread has terminated, is_caller() must not be
 * called after a call to detach() has been made or a call to join()
 * has returned: this can be ensured by only detaching or joining via
 * this object's destructor (that is, by not using the explicit
 * detach() and join() methods).  This method does not throw.
 * @return Returns true if the caller is in the thread represented by
 * the wrapped Thread::Thread object.  If not, or this JoinableHandle
 * does not wrap any Thread object, then returns false.
 *
 * Since 1.0.2
 */
  bool is_caller();

/**
 * Specifies whether this JoinableHandle object has been initialized
 * with a Thread::Thread object representing a correctly started
 * thread in respect of which neither JoinableHandle::detach() nor
 * JoinableHandle::join() has been called.  It can be called by any
 * thread.  It is principally intended to enable the constructor
 * taking a std::auto_ptr<Cgu::Thread::Thread> object to be directly
 * initialized by a call to Thread::Thread::start(), by providing a
 * means for the thread calling Thread::Thread::start() to check
 * afterwards that the new thread did, in fact, start correctly.  Note
 * that this method will return true even after the thread has
 * finished, provided neither the join() nor detach() method has been
 * called.
 * @return Returns true if this object has been initialized by a
 * Thread::Thread object representing a correctly started thread in
 * respect of which neither JoinableHandle::detach() nor
 * JoinableHandle::join() has been called, otherwise false.
 *
 * Since 1.2.0
 */
  bool is_managing();

/**
 * Moves one JoinableHandle object to another JoinableHandle object.
 * This is a move operation which transfers ownership to the assignee,
 * as the handles store their Thread::Thread object by
 * std::auto_ptr<>.  Any existing thread managed by the assignee prior
 * to the move will be detached if it has not already been detached or
 * joined.  This method is thread safe.  It will not throw.
 * @param h The assignor/movant, which will cease to hold a valid
 * Thread::Thread object after the move has taken place.
 * @return A reference to the assignee JoinableHandle object after
 * assignment.
 *
 * Since 1.0.2
 */
  JoinableHandle& operator=(JoinableHandle& h);

/**
 * This constructor initializes a new JoinableHandle object with a
 * std::auto_ptr<Thread::Thread> object, as provided by
 * Thread::Thread::start().  This is a move operation which transfers
 * ownership to the new object.
 * @param thread The initializing Thread::Thread object (which must
 * have been created as joinable) passed by a std::auto_ptr smart
 * pointer.  The std::auto_ptr smart pointer will cease to hold a
 * valid Thread::Thread object after the initialization has taken
 * place.
 * @param act Either Thread::JoinableHandle::detach_on_exit (which
 * will cause the destructor to detach the thread if it has not
 * previously been detached or joined) or
 * Thread::JoinableHandle::join_on_exit (which will cause the
 * destructor to join the thread if it has not previously been
 * detached or joined).
 * @exception Cgu::Thread::MutexError Throws this exception if
 * initialization of the internal mutex fails.  The constructor is
 * strongly exception safe: if Cgu::Thread::MutexError is thrown, the
 * initializing std::auto_ptr<Cgu::Thread::Thread> object will be left
 * unchanged.  (It is often not worth checking for this exception, as
 * it means either memory is exhausted or pthread has run out of other
 * resources to create new mutexes.)
 * @note It is not necessary to check that the thread parameter
 * represents a correctly started thread (that is, that thread.get()
 * does not return 0) before this constructor is invoked, because that
 * can be done after construction by calling
 * JoinableHandle::is_managing() (a JoinableHangle object can safely
 * handle a case where thread.get() does return 0).  This enables a
 * JoinableHandle object to be directly initialized by this
 * constructor from a call to Thread::Thread::start().
 * @sa JoinableHandle::is_managing().
 *
 * Since 1.0.2
 */
  JoinableHandle(std::auto_ptr<Cgu::Thread::Thread> thread, Action act): action(act), detached(false), thread_a(thread) {}
  
/**
 * This constructor initializes a new JoinableHandle object with an
 * existing JoinableHandle object.  This is a move operation which
 * transfers ownership to the new object.
 * @param h The intitializing JoinableHandle object, which will cease
 * to hold a valid Thread::Thread object after the initialization has
 * taken place.
 * @exception Cgu::Thread::MutexError Throws this exception if
 * initialization of the internal mutex fails.  The constructor is
 * strongly exception safe: if Cgu::Thread::MutexError is thrown, the
 * initializing Cgu::Thread::JoinableHandle object will be left
 * unchanged.  (It is often not worth checking for this exception, as
 * it means either memory is exhausted or pthread has run out of other
 * resources to create new mutexes.)
 *
 * Since 1.0.2
 */
  JoinableHandle(JoinableHandle& h): action(h.action), detached(h.detached), thread_a(h.thread_a) {}

/**
 * The destructor will detach a managed thread (if the
 * Thread::JoinableHandle::detach_on_exit flag is set) or join it (if
 * the Thread::JoinableHandle::join_on_exit flag is set), unless it
 * has previously been detached or joined with the detach() or join()
 * methods.  The destructor is thread safe (any thread may destroy the
 * JoinableHandle object).  The destructor will not throw.
 *
 * Since 1.0.2
 */
~JoinableHandle();

/* Only has effect if --with-glib-memory-slices-compat or
 * --with-glib-memory-slices-no-compat option picked */
  CGU_GLIB_MEMORY_SLICES_FUNCS
};

/**
 * @class CancelBlock thread.h c++-gtk-utils/thread.h
 * @brief A class enabling the cancellation state of a thread to be
 * controlled.
 *
 * A class enabling the cancellation state of a thread to be
 * controlled, so as to provide exception safe cancellation state
 * changes.  When the object goes out of scope, cancellation state is
 * returned to the state it was in prior to its construction.
 */

class CancelBlock {
  int starting_state;

  // CancelBlocks cannot be copied
  CancelBlock(const CancelBlock&);
  CancelBlock& operator=(const CancelBlock&);
public:
/**
 * Makes the thread uncancellable, even if the code passes through a
 * cancellation point, while the CancelBlock object exists (when the
 * CancelBlock object ceases to exist, cancellation state is returned
 * to the state prior to it being constructed).  It should only be
 * called by the thread which created the CancelBlock object.  This
 * method will not throw.
 * @param old_state Indicates the cancellation state of the calling
 * thread immediately before this call to block() was made, either
 * PTHREAD_CANCEL_ENABLE (if the thread was previously cancellable) or
 * PTHREAD_CANCEL_DISABLE (if this call did nothing because the thread
 * was already uncancellable).
 * @return 0 if successful, else a value other than 0.
 */
  static int block(int& old_state) {return pthread_setcancelstate(PTHREAD_CANCEL_DISABLE, &old_state);}

/**
 * Makes the thread uncancellable, even if the code passes through a
 * cancellation point, while the CancelBlock object exists (when the
 * CancelBlock object ceases to exist, cancellation state is returned
 * to the state prior to it being constructed).  It should only be
 * called by the thread which created the CancelBlock object.  This
 * method will not throw.
 * @return 0 if successful, else a value other than 0.
 */
  static int block() {int old_state; return block(old_state);}

/**
 * Makes the thread cancellable while the CancelBlock object exists
 * (when the CancelBlock object ceases to exist, cancellation state is
 * returned to the state prior to it being constructed).  It should
 * only be called by the thread which created the CancelBlock object.
 * This method will not throw.
 * @param old_state Indicates the cancellation state of the calling
 * thread immediately before this call to unblock() was made, either
 * PTHREAD_CANCEL_DISABLE (if the thread was previously uncancellable)
 * or PTHREAD_CANCEL_ENABLE (if this call did nothing because the
 * thread was already cancellable).
 * @return 0 if successful, else a value other than 0.
 */
  static int unblock(int& old_state) {return pthread_setcancelstate(PTHREAD_CANCEL_ENABLE, &old_state);}

/**
 * Makes the thread cancellable while the CancelBlock object exists
 * (when the CancelBlock object ceases to exist, cancellation state is
 * returned to the state prior to it being constructed).  It should
 * only be called by the thread which created the CancelBlock object.
 * This method will not throw.
 * @return 0 if successful, else a value other than 0.
 */
  static int unblock() {int old_state; return unblock(old_state);}

/**
 * Restores cancallation state to the state it was in immediately
 * before this CancelBlock object was constructed.  It should only be
 * called by the thread which created the CancelBlock object.  This
 * method will not throw.
 * @param old_state Indicates the cancellation state of the calling
 * thread immediately before this call to restore() was made, either
 * PTHREAD_CANCEL_DISABLE (if the thread was previously uncancellable)
 * or PTHREAD_CANCEL_ENABLE (if this thread was previously
 * cancellable).
 * @return 0 if successful, else a value other than 0.
 * 
 * Since 0.9.1
 */
  int restore(int& old_state) {return pthread_setcancelstate(starting_state, &old_state);}

/**
 * Restores cancallation state to the state it was in immediately
 * before this CancelBlock object was constructed.  It should only be
 * called by the thread which created the CancelBlock object.  This
 * method will not throw.
 * @return 0 if successful, else a value other than 0.
 * 
 * Since 0.9.1
 */
  int restore() {int old_state; return restore(old_state);}

/**
 * The constructor will not throw.
 * @param blocking Whether the CancelBlock object should start
 * in blocking mode.
 */
  CancelBlock(bool blocking = true);

/**
 * The destructor will put the thread in the cancellation state that
 * it was in immediately before the CancelBlock object was constructed
 * (which might be blocking).  It will not throw.
 * 
 * This behaviour is from version 0.9.1.  In version 0.9.0
 * cancellation state was always made non-blocking by the destructor.
 */
  ~CancelBlock() {restore();}

/* Only has effect if --with-glib-memory-slices-compat or
 * --with-glib-memory-slices-no-compat option picked */
  CGU_GLIB_MEMORY_SLICES_FUNCS
};

/**
 * @class Exit thread.h c++-gtk-utils/thread.h
 * @brief A class which can be thrown to terminate the throwing
 * thread.
 *
 * This class can be thrown (instead of calling pthread_exit()) when a
 * thread wishes to terminate itself and also ensure stack unwinding,
 * so that destructors of local objects are called.  It is caught
 * automatically by the implementation of Cgu::Thread::Thread::start()
 * so that it will only terminate the thread throwing it and not the
 * whole process.  See the Cgu::Thread::Thread::cancel() method above,
 * for use when a thread wishes to terminate another one, and the
 * caveats on the use of Cgu::Thread::Thread::cancel().
 */
class Exit {};

} // namespace Thread

} // namespace Cgu

#endif
