/*
   Eiciel - GNOME editor of ACL file permissions.
   Copyright (C) 2004-2005 Roger Ferrer Ibez

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
   */
#include "eiciel_main_window.hpp"

// Constructor
EicielWindow::EicielWindow(EicielMainControler* cont) 
	: Gtk::VBox(),
	ACLusuari(_("User")),
	ACLgrup(_("Group")),
	aclDefault(_("Default")),
	modificarACLDefault(_("Default ACL")),
	mostrarSystem(_("Also show system participants")),
	botonera_mig(Gtk::BUTTONBOX_DEFAULT_STYLE, 2),
	afegirACL(Gtk::Stock::ADD), 
	eliminarACL(Gtk::Stock::REMOVE),
	marc_superior(_("Access Control List")),
	marc_inferior(_("Participants List")),
	modeNomesLectura(false),
	controlador(cont) // Assignem el controlador a la finestra
{
	// Assignem la finestra al controlador
	controlador->finestra = this;

	// Posem la mida de les vores a 4
	set_border_width(4);

	add(zona_principal);
	zona_principal.set_spacing(4);

	zona_principal.add(marc_superior);
	zona_principal.add(marc_inferior);

	marc_superior.add(zona_superior);
	marc_inferior.add(zona_inferior);

	zona_superior.set_border_width(4);
	zona_superior.set_spacing(4);
	zona_inferior.set_border_width(4);
	zona_inferior.set_spacing(4);

	// Obrim les icones
	iconaUsuari = Gdk::Pixbuf::create_from_file(PKGDATADIR "/img/usuari.png");
	iconaUsuariACL = Gdk::Pixbuf::create_from_file(PKGDATADIR "/img/usuari-acl.png");
	iconaGrup = Gdk::Pixbuf::create_from_file(PKGDATADIR "/img/grup.png");
	iconaGrupACL = Gdk::Pixbuf::create_from_file(PKGDATADIR "/img/grup-acl.png");
	iconaAltres = Gdk::Pixbuf::create_from_file(PKGDATADIR "/img/altres.png");
	iconaMascara = Gdk::Pixbuf::create_from_file(PKGDATADIR "/img/mascara.png");

	iconaDefaultUsuari = Gdk::Pixbuf::create_from_file(PKGDATADIR "/img/usuari-default.png");
	iconaDefaultGrup = Gdk::Pixbuf::create_from_file(PKGDATADIR "/img/grup-default.png");
	iconaDefaultAltres = Gdk::Pixbuf::create_from_file(PKGDATADIR "/img/altres-default.png");
	iconaDefaultUsuariACL = Gdk::Pixbuf::create_from_file(PKGDATADIR "/img/usuari-acl-default.png");
	iconaDefaultGrupACL = Gdk::Pixbuf::create_from_file(PKGDATADIR "/img/grup-acl-default.png");
	iconaDefaultMascara = Gdk::Pixbuf::create_from_file(PKGDATADIR "/img/mascara-default.png");

	// Resta de la llista
	refLlistaACL = Gtk::ListStore::create(modelLlistaACL);

	// Posem les columnes de la llista d'ACL's
	vistaLlistaACL.set_reallocate_redraws();
	vistaLlistaACL.set_model(refLlistaACL);
	vistaLlistaACL.append_column("", modelLlistaACL.iconeta);
	vistaLlistaACL.append_column(_("Entry"), modelLlistaACL.nomEntrada);

	CellRendererACL* renderRead = Gtk::manage(new CellRendererACL());
	int numColumnes = vistaLlistaACL.append_column(_("Read"), *renderRead);
	Gtk::TreeViewColumn* columnaLectura = vistaLlistaACL.get_column(numColumnes - 1);

	// Pq hauriem de comprovar aixo ?
	if (columnaLectura != NULL) 
	{
		columnaLectura->add_attribute(renderRead->property_active(), modelLlistaACL.permisLectura);
		columnaLectura->add_attribute(renderRead->property_marcar_fons(), modelLlistaACL.inefectiuLectura);
		renderRead->property_activatable() = true;
	}

	CellRendererACL* renderWrite = Gtk::manage(new CellRendererACL());
	numColumnes = vistaLlistaACL.append_column(_("Write"), *renderWrite);
	Gtk::TreeViewColumn* columnaEscriptura = vistaLlistaACL.get_column(numColumnes - 1);
	if (columnaEscriptura != NULL) 
	{
		columnaEscriptura->add_attribute(renderWrite->property_active(), modelLlistaACL.permisEscriptura);
		columnaEscriptura->add_attribute(renderWrite->property_marcar_fons(), modelLlistaACL.inefectiuEscriptura);
	}

	CellRendererACL* renderExecute = Gtk::manage(new CellRendererACL());
	numColumnes = vistaLlistaACL.append_column(_("Execution"), *renderExecute);
	Gtk::TreeViewColumn* columnaExecucio = vistaLlistaACL.get_column(numColumnes - 1);
	if (columnaExecucio != NULL) 
	{
		columnaExecucio->add_attribute(renderExecute->property_active(), modelLlistaACL.permisExecucio);
		columnaExecucio->add_attribute(renderExecute->property_marcar_fons(), modelLlistaACL.inefectiuExecucio);
	}

	vistaLlistaACL.append_column("", modelLlistaACL.buit);

	contenidorVistaLlistaACL.add(vistaLlistaACL);
	contenidorVistaLlistaACL.set_policy(Gtk::POLICY_NEVER, Gtk::POLICY_AUTOMATIC);

	vistaLlistaACL.set_size_request(-1, 150);

	zona_superior.add(contenidorVistaLlistaACL);

	// Afegim els botons de pujar o baixar
	botonera_mig.pack_start(llegendaInferior, Gtk::PACK_SHRINK, 0);
	botonera_mig.pack_end(eliminarACL, Gtk::PACK_SHRINK, 0);
	botonera_mig.pack_end(modificarACLDefault, Gtk::PACK_SHRINK, 0);


	zona_superior.pack_start(botonera_mig, Gtk::PACK_SHRINK, 0);

	// Agrupem els botons de tipus d'ACL
	Gtk::RadioButton::Group tipusACL = ACLusuari.get_group();
	ACLgrup.set_group(tipusACL);

	selector_participant.pack_start(ACLusuari, Gtk::PACK_SHRINK, 0);
	selector_participant.pack_start(ACLgrup, Gtk::PACK_SHRINK, 0);
	selector_participant.pack_start(aclDefault, Gtk::PACK_SHRINK, 0);
	selector_participant.pack_end(afegirACL, Gtk::PACK_SHRINK, 0);

	zona_inferior.pack_start(selector_participant, Gtk::PACK_SHRINK, 0);

	// Afegim la llista dels participants
	refLlistaParticipants = Gtk::ListStore::create(modelLlistaParticipant);

	vistaLlistaParticipants.set_reallocate_redraws();
	vistaLlistaParticipants.set_model(refLlistaParticipants);
	vistaLlistaParticipants.append_column("", modelLlistaParticipant.iconeta);
	vistaLlistaParticipants.append_column(_("Participant"), modelLlistaParticipant.nomParticipant);

	contenidorVistaLlistaParticipants.add(vistaLlistaParticipants);
	contenidorVistaLlistaParticipants.set_policy(Gtk::POLICY_NEVER, Gtk::POLICY_AUTOMATIC);
	
	zona_inferior.add(contenidorVistaLlistaParticipants);
	zona_inferior.add(mostrarSystem);

	mostrarSystem.signal_toggled().connect(sigc::mem_fun(*this, &EicielWindow::canviMostrarSystem));

	vistaLlistaParticipants.set_size_request(-1, 150);

	// Drag & Drop
	std::list<Gtk::TargetEntry> listTargets;
	listTargets.push_back( Gtk::TargetEntry("participant_acl") );

	// vistaLlistaParticipants es origen del drag
	vistaLlistaParticipants.drag_source_set(listTargets);

	vistaLlistaParticipants.signal_drag_begin().connect( 
			sigc::mem_fun(*this, &EicielWindow::iniciDragAndDrop)
			);

	vistaLlistaParticipants.signal_drag_data_get().connect( 
			sigc::mem_fun(*this, &EicielWindow::establirValorDragAndDrop)
			);

	// vistaLlistaACL s dest del drag
	vistaLlistaACL.drag_dest_set(listTargets);

	vistaLlistaACL.signal_drag_data_received().connect( 
			sigc::mem_fun(*this, &EicielWindow::rebreValorDragAndDrop)
			);


	// Vinculaci dels senyals
	Glib::RefPtr<Gtk::TreeSelection> referenciaSeleccioLlistaACL = vistaLlistaACL.get_selection();

	// Funcio de filtre (aquest cop nomes per al modeNomesLectura)
	referenciaSeleccioLlistaACL->set_select_function( 
			sigc::mem_fun(*this, &EicielWindow::funcioSeleccioACL) 
			);
	
	// Canvi de sellecci ACL
	referenciaSeleccioLlistaACL->signal_changed().connect(
			sigc::mem_fun(*this, &EicielWindow::canviDeSeleccioACL)
			);

	// Canvi de sellecci ACL
	Glib::RefPtr<Gtk::TreeSelection> referenciaSeleccioLlistaParticipant = vistaLlistaParticipants.get_selection();
	referenciaSeleccioLlistaParticipant->signal_changed().connect(
			sigc::mem_fun(*this, &EicielWindow::canviDeSeleccioParticipant)
			);
	
	// Doble clic de la llista d'ACL
	vistaLlistaACL.signal_row_activated().connect(
			sigc::mem_fun(*this, &EicielWindow::dobleClicLlistaACL)
			);

	// Doble clic de la llista de participants
	vistaLlistaParticipants.signal_row_activated().connect(
			sigc::mem_fun(*this, &EicielWindow::dobleClicLlistaParticipants)
			);

	// Boto d'eliminar un ACL
	eliminarACL.signal_clicked().connect(
			sigc::mem_fun(*this, &EicielWindow::eliminarACLSeleccionada)
			);

	// Boto d'afegir un ACL
	afegirACL.signal_clicked().connect(
			sigc::mem_fun(*this, &EicielWindow::afegirParticipantSeleccionat)
			);

	// Detectar un canvi de default ACL
	modificarACLDefault.signal_clicked().connect (
			sigc::mem_fun(*controlador, &EicielMainControler::canviACLDefault)
			);

	// Detectar una edici de la columna 
	Gtk::TreeViewColumn* col = vistaLlistaACL.get_column(2);
	Gtk::CellRenderer* render = col->get_first_cell_renderer(); // Nomes tenim un
	CellRendererACL* toggle_render = dynamic_cast<CellRendererACL*>(render);

	toggle_render->signal_toggled().connect(
			sigc::bind<TipusPermis>( sigc::mem_fun(*this, &EicielWindow::canviPermisos), LECTURA)
			);

	col = vistaLlistaACL.get_column(3);
	render = col->get_first_cell_renderer();
	toggle_render = dynamic_cast<CellRendererACL*>(render);

	toggle_render->signal_toggled().connect(
			sigc::bind<TipusPermis>( sigc::mem_fun(*this, &EicielWindow::canviPermisos), ESCRIPTURA)
			);

	col = vistaLlistaACL.get_column(4);
	render = col->get_first_cell_renderer();
	toggle_render = dynamic_cast<CellRendererACL*>(render);

	toggle_render->signal_toggled().connect(
			sigc::bind<TipusPermis>( sigc::mem_fun(*this, &EicielWindow::canviPermisos), EXECUCIO)
			);

	noHiHaFitxer();
	noHiHaSeleccioACL();
	noHiHaSeleccioParticipant();

	llistaUsuaris = controlador->donaLlistaUsuaris();
	llistaGrups = controlador->donaLlistaGrups();

	ACLusuari.signal_clicked().connect(
			sigc::bind< set<string>*,
			TipusElement, 
			Glib::RefPtr<Gdk::Pixbuf>,
			Glib::RefPtr<Gdk::Pixbuf> > (
				sigc::mem_fun(*this, &EicielWindow::emplenarParticipants),
				&llistaUsuaris,
				ACL_USUARI, 
				iconaUsuariACL, 
				iconaDefaultUsuariACL)
			);

	ACLgrup.signal_clicked().connect(
			sigc::bind< set<string>*,
			TipusElement, 
			Glib::RefPtr<Gdk::Pixbuf>,
			Glib::RefPtr<Gdk::Pixbuf> > (
				sigc::mem_fun(*this, &EicielWindow::emplenarParticipants),
				&llistaGrups,
				ACL_GRUP, 
				iconaGrupACL, 
				iconaDefaultGrupACL)
			);

	aclDefault.signal_clicked().connect(
			sigc::mem_fun(*this, &EicielWindow::canviarTipusParticipant)
			);

	ACLusuari.set_active();
	ACLusuari.clicked();

	mostrarAdmiracio(false);

	show_all_children();
}


void EicielWindow::mostrarAdmiracio(bool b)
{
	if (b)
	{
		llegendaInferior.set_markup(_("<span size=\"smaller\"><span foreground=\"#cc0000\" weight=\"bold\">!</span> means an ineffective permission.</span>"));
		llegendaInferior.set_use_markup(b);
		llegendaInferior.show();
	}
	else
	{
		llegendaInferior.hide();
	}
}

EicielWindow::~EicielWindow()
{
	delete controlador;
}

void EicielWindow::inicialitzar(string s)
{
	controlador->obreFitxer(s);
}

void EicielWindow::activacio(bool b)
{
	zona_principal.set_sensitive(b);
}

void EicielWindow::establirNomesLectura(bool b)
{
	modeNomesLectura = b;
	modificarACLDefault.set_sensitive(!b);
	zona_inferior.set_sensitive(!b);
}

bool EicielWindow::funcioSeleccioACL(const Glib::RefPtr<Gtk::TreeModel>& model,
		const Gtk::TreeModel::Path& path, bool)
{
	return !modeNomesLectura;
}

void EicielWindow::noHiHaFitxer()
{
	zona_principal.set_sensitive(false);
}

void EicielWindow::buidaLlistaACL()
{
	refLlistaACL->clear();
}

void EicielWindow::canviDeSeleccioACL()
{
	Glib::RefPtr<Gtk::TreeSelection> referenciaSeleccioLlista = vistaLlistaACL.get_selection();
	Gtk::TreeModel::iterator iter = referenciaSeleccioLlista->get_selected();
	if (!iter || modeNomesLectura)
	{
		noHiHaSeleccioACL();
	}
	else
	{
		if ( !(*iter)[modelLlistaACL.esborrable] )
			noHiHaSeleccioACL();
		else
			hiHaSeleccioACL();
	}
}

void EicielWindow::canviDeSeleccioParticipant()
{
	Glib::RefPtr<Gtk::TreeSelection> referenciaSeleccioLlista = vistaLlistaParticipants.get_selection();
	Gtk::TreeModel::iterator iter = referenciaSeleccioLlista->get_selected();
	if (!iter)
	{
		noHiHaSeleccioParticipant();
	}
	else
	{
		hiHaSeleccioParticipant();
	}
}

void EicielWindow::eliminarACLSeleccionada()
{
	Glib::RefPtr<Gtk::TreeSelection> referenciaSeleccioLlista = vistaLlistaACL.get_selection();
	Gtk::TreeModel::iterator iter = referenciaSeleccioLlista->get_selected();
	if (iter)
	{
		Gtk::TreeModel::Row row(*iter);
		if (row[modelLlistaACL.esborrable])
		{
			Gtk::TreeModel::Row row(*iter);
			controlador->eliminarACL(Glib::ustring(row[modelLlistaACL.nomEntrada]), 
					TipusElement(row[modelLlistaACL.tipusEntrada]));
		}
	}
}

void EicielWindow::afegirParticipantSeleccionat()
{
	Glib::RefPtr<Gtk::TreeSelection> referenciaSeleccioLlista = vistaLlistaParticipants.get_selection();
	Gtk::TreeModel::iterator iter = referenciaSeleccioLlista->get_selected();
	if (iter)
	{
		Gtk::TreeModel::Row row(*iter);
		controlador->afegirEntradaACL(Glib::ustring(row[modelLlistaParticipant.nomParticipant]), 
				TipusElement(row[modelLlistaParticipant.tipusEntrada]), aclDefault.get_active());
	}
}

void EicielWindow::noHiHaSeleccioACL()
{
	eliminarACL.set_sensitive(false);
}

void EicielWindow::noHiHaSeleccioParticipant()
{
	afegirACL.set_sensitive(false);
}

void EicielWindow::hiHaSeleccioACL()
{
	eliminarACL.set_sensitive(true);
}

void EicielWindow::hiHaSeleccioParticipant()
{
	afegirACL.set_sensitive(true);
}

void EicielWindow::afegeixNoSeleccionable(Glib::ustring titol, 
		bool lectura, bool escriptura, bool execucio, TipusElement e,
		bool efectiuLectura, bool efectiuEscriptura, bool efectiuExecucio)
{
	Gtk::TreeModel::iterator iter = refLlistaACL->append();
	Gtk::TreeModel::Row row(*iter);
	row = *iter;
	afegeixElement(titol, lectura, escriptura, execucio, e, row, 
			efectiuLectura, efectiuEscriptura, efectiuExecucio);
	row[modelLlistaACL.esborrable] = false;
}

void EicielWindow::afegeixSeleccionable(Glib::ustring titol, 
		bool lectura, bool escriptura, bool execucio, TipusElement e,
		bool efectiuLectura, bool efectiuEscriptura, bool efectiuExecucio)
{
	Gtk::TreeModel::iterator iter = refLlistaACL->append();
	Gtk::TreeModel::Row row(*iter);

	afegeixElement(titol, lectura, escriptura, execucio, e, row,
			efectiuLectura, efectiuEscriptura, efectiuExecucio);
	row[modelLlistaACL.esborrable] = true;
}

void EicielWindow::afegeixElement(Glib::ustring titol, bool lectura, bool escriptura, bool execucio,
		TipusElement e, Gtk::TreeModel::Row& row,
		bool efectiuLectura, bool efectiuEscriptura, bool efectiuExecucio)
{
	row[modelLlistaACL.tipusEntrada] = e;
	row[modelLlistaACL.iconeta] = retornaIconaApropiada(e);
	row[modelLlistaACL.nomEntrada] = titol;
	row[modelLlistaACL.permisLectura] = lectura;
	row[modelLlistaACL.permisEscriptura] = escriptura;
	row[modelLlistaACL.permisExecucio] = execucio;

	row[modelLlistaACL.inefectiuLectura] = !efectiuLectura;
	row[modelLlistaACL.inefectiuEscriptura] = !efectiuEscriptura;
	row[modelLlistaACL.inefectiuExecucio] = !efectiuExecucio;
}

Glib::RefPtr<Gdk::Pixbuf> EicielWindow::retornaIconaApropiada(TipusElement e)
{
	switch(e)
	{
		case USUARI : 
			return iconaUsuari;
		case GRUP :
			return iconaGrup;
		case ALTRES :
			return iconaAltres;
		case MASCARA :
			return iconaMascara;
		case ACL_USUARI : 
			return iconaUsuariACL;
		case ACL_GRUP :
			return iconaGrupACL;
		case DEFAULT_USUARI :
			return iconaDefaultUsuari;
		case DEFAULT_GRUP :
			return iconaDefaultGrup;
		case DEFAULT_ALTRES :
			return iconaDefaultAltres;
		case DEFAULT_ACL_USUARI :
			return iconaDefaultUsuariACL;
		case DEFAULT_ACL_GRUP :
			return iconaDefaultGrupACL;
		case DEFAULT_MASCARA :
			return iconaDefaultMascara;
		default :
			return iconaAltres;
	}
}

void EicielWindow::establirNomFitxer(string nomF)
{
	zona_principal.set_sensitive(true);
}

void EicielWindow::canviPermisos(const Glib::ustring& cadena, TipusPermis p)
{
	Gtk::TreeModel::iterator i = refLlistaACL->get_iter(cadena);
	Gtk::TreeModel::Row row(*i);

	if (modeNomesLectura) return;

	switch (p)
	{
		case LECTURA:
			row[modelLlistaACL.permisLectura] = !row[modelLlistaACL.permisLectura];
			break;
		case ESCRIPTURA:
			row[modelLlistaACL.permisEscriptura] = !row[modelLlistaACL.permisEscriptura];
			break;
		case EXECUCIO:
			row[modelLlistaACL.permisExecucio] = !row[modelLlistaACL.permisExecucio];
			break;
	}

	controlador->actualitzaEntradaACL(row[modelLlistaACL.tipusEntrada], 
			Glib::ustring(row[modelLlistaACL.nomEntrada]), row[modelLlistaACL.permisLectura],
			row[modelLlistaACL.permisEscriptura], row[modelLlistaACL.permisExecucio]);
}

void EicielWindow::activarBotoDefaultACL(bool b)
{
	modificarACLDefault.set_sensitive(b);
	aclDefault.set_sensitive(b);
	if (!b)
	{
		aclDefault.set_active(false);
	}
}

void EicielWindow::hiHaDefaultACL(bool b)
{
	modificarACLDefault.set_active(b);
}

bool EicielWindow::donaACLDefault()
{
	return modificarACLDefault.get_active();
}

void EicielWindow::emplenarParticipants(set<string>* participants,
		TipusElement tipus, 
		Glib::RefPtr<Gdk::Pixbuf> iconaNormal,
		Glib::RefPtr<Gdk::Pixbuf> iconaDefecte)
{
	refLlistaParticipants->clear();
	Gtk::TreeModel::iterator iter;

	bool activar_per_defecte = aclDefault.get_active();

	for (set<string>::iterator i = participants->begin(); i != participants->end(); i++)
	{
		iter = refLlistaParticipants->append();
		Gtk::TreeModel::Row row(*iter);
		row[modelLlistaParticipant.iconeta] = 
			activar_per_defecte ? iconaDefecte : iconaNormal;
		row[modelLlistaParticipant.nomParticipant] = *i;
		row[modelLlistaParticipant.tipusEntrada] = tipus;
	}
}

bool EicielWindow::fitxerObert()
{
	return controlador->fitxerObert();
}

Glib::ustring EicielWindow::ultimError()
{
	return controlador->ultimError();
}

void EicielWindow::dobleClicLlistaParticipants(const Gtk::TreeModel::Path& p,
		Gtk::TreeViewColumn* c)
{
	Glib::RefPtr<Gtk::TreeModel> modelLlista = vistaLlistaParticipants.get_model();
	Gtk::TreeModel::iterator iter = modelLlista->get_iter(p);

	if (iter)
	{
		Gtk::TreeModel::Row row(*iter);
		controlador->afegirEntradaACL(Glib::ustring(row[modelLlistaParticipant.nomParticipant]), 
				TipusElement(row[modelLlistaParticipant.tipusEntrada]), aclDefault.get_active());
	}
}

void EicielWindow::dobleClicLlistaACL(const Gtk::TreeModel::Path& p,
		Gtk::TreeViewColumn* c)
{
	Glib::RefPtr<Gtk::TreeModel> modelLlista = vistaLlistaACL.get_model();
	Gtk::TreeModel::iterator iter = modelLlista->get_iter(p);

	if (modeNomesLectura) return;

	if (iter)
	{
		Gtk::TreeModel::Row row(*iter);
		if (row[modelLlistaACL.esborrable])
		{
			controlador->eliminarACL(Glib::ustring(row[modelLlistaACL.nomEntrada]), 
					TipusElement(row[modelLlistaACL.tipusEntrada]));
		}
	}
}

void EicielWindow::seleccionarACL(string s, TipusElement e)
{
	Glib::RefPtr<Gtk::TreeModel> modelLlista = vistaLlistaACL.get_model();
	Gtk::TreeModel::Children children = modelLlista->children();
	bool trobat = false;
	for(Gtk::TreeModel::Children::iterator iter = children.begin(); 
			(iter != children.end()) && !trobat; ++iter)
	{
		Gtk::TreeModel::Row row(*iter);
		if ((row[modelLlistaACL.tipusEntrada] == e) &&
				(row[modelLlistaACL.nomEntrada] == s))
		{
			trobat = true;
			Gtk::TreePath p = modelLlista->get_path(iter);
			vistaLlistaACL.set_cursor(p);
			vistaLlistaACL.scroll_to_row(p, 0.5);
			vistaLlistaACL.grab_focus();
		}
	}
}


void EicielWindow::establirValorDragAndDrop(const Glib::RefPtr<Gdk::DragContext>&, 
		Gtk::SelectionData& selection_data, guint, guint)
{
	selection_data.set("participant_acl", "");
}

void EicielWindow::rebreValorDragAndDrop(const Glib::RefPtr<Gdk::DragContext>& context, 
		int, int, const Gtk::SelectionData& selection_data, guint, guint time)
{
	afegirParticipantSeleccionat();
	context->drag_finish(true, false, time);
}

void EicielWindow::iniciDragAndDrop(const Glib::RefPtr<Gdk::DragContext>& context)
{
	Glib::RefPtr<Gtk::TreeSelection> referenciaSeleccioLlista = vistaLlistaParticipants.get_selection();
	Gtk::TreeModel::iterator iter = referenciaSeleccioLlista->get_selected();
	if (iter)
	{
		Gtk::TreeModel::Row row(*iter);
		Glib::RefPtr<Gdk::Pixbuf> imatge = row[modelLlistaParticipant.iconeta];
		context->set_icon(imatge, -4, -4);
	}
}

void EicielWindow::canviarTipusParticipant()
{
	Glib::RefPtr<Gtk::TreeModel> modelLlista = vistaLlistaParticipants.get_model();

	Gtk::TreeModel::Children children = modelLlista->children();

	bool activarDefault = aclDefault.get_active();

	for(Gtk::TreeModel::Children::iterator iter = children.begin(); 
			iter != children.end(); ++iter)
	{
		Gtk::TreeModel::Row row(*iter);
		if (ACLusuari.get_active())
		{
			row[modelLlistaParticipant.iconeta] = 
				activarDefault ? iconaDefaultUsuariACL : iconaUsuariACL;
		}
		else
		{
			row[modelLlistaParticipant.iconeta] = 
				activarDefault ? iconaDefaultGrupACL : iconaGrupACL;
		}
	}
}

void EicielWindow::canviMostrarSystem()
{
	controlador->mostrarParticipantsSistema(mostrarSystem.get_active());
	llistaUsuaris = controlador->donaLlistaUsuaris();
	llistaGrups = controlador->donaLlistaGrups();

	// Volem forar l'event sense voler memoritzar en quin tipus de llista estem
	if (ACLusuari.get_active()) 
		ACLusuari.clicked();

	if (ACLgrup.get_active()) 
		ACLgrup.clicked();
}
