%%%----------------------------------------------------------------------
%%% File    : mod_shared_roster_ldap_helpers.erl
%%% Author  : Marcin Owsiany <marcin@owsiany.pl>
%%% Purpose : Helper functions used by mod_shared_roster_ldap. The only reason
%%%           they are in a separate file is to make unit testing easier - it
%%%           is not otherwise possible to mock them out.
%%% Created : 7 Feb 2010 by Marcin Owsiany <marcin@owsiany.pl>
%%% Version : 0.5.3
%%%----------------------------------------------------------------------

%%%----------------------------------------------------------------------
%%% Contribution page: http://www.ejabberd.im/mod_shared_roster_ldap
%%% Project page:  https://alioth.debian.org/projects/ejabberd-msrl/
%%% Documentation: https://alioth.debian.org/docman/?group_id=100433
%%%----------------------------------------------------------------------
-module(mod_shared_roster_ldap_helpers).
-author('marcin@owsiany.pl').

-include("eldap/eldap.hrl").
-include("mod_shared_roster_ldap.hrl").

-export([eldap_search/3,
         get_user_to_groups_map/2,
         users_cache_fresh/2,
         groups_cache_fresh/2,
         now/0,
         user_entries_to_dict/3,
         group_entries_to_dict/7]).

%% For a given user, map all his shared roster contacts to groups they are
%% members of. Skip the user himself iff SkipUS is true.
get_user_to_groups_map(US, SkipUS) ->
    {_, Server} = US,
    DisplayedGroups = mod_shared_roster_ldap:get_user_displayed_groups(US),
    lists:foldl(
	fun(Group, Dict1) ->
	    GroupName = mod_shared_roster_ldap:get_group_name(Server, Group),
	    lists:foldl(
		fun(Contact, Dict) ->
		    if
                      SkipUS, Contact == US ->
                         Dict;
		      true ->
                         dict:append(Contact, GroupName, Dict)
		    end
		end, Dict1, mod_shared_roster_ldap:get_group_users(Server, Group))
	end, dict:new(), DisplayedGroups).

%% Pass given FilterParseArgs to eldap_filter:parse, and if successful, run and
%% return the resulting filter, retrieving given AttributesList. Return the
%% result entries. On any error silently return an empty list of results.
%%
%% Eldap server ID and base DN for the query are both retrieved from the State
%% record.
eldap_search(State, FilterParseArgs, AttributesList) ->
    case apply(eldap_filter, parse, FilterParseArgs) of
        {ok, EldapFilter} ->
            % Filter parsing succeeded
            case eldap:search(State#state.eldap_id, [
                     {base, State#state.base},
                     {filter, EldapFilter},
                     {attributes, AttributesList}]) of
                % A result with entries. Return their list.
                #eldap_search_result{entries = Es} -> Es;
                % Something else. Pretend we got no results.
                _ -> []
            end;
        _ ->
            % Filter parsing failed. Pretend we got no results.
            []
    end.

-define(cache_fresh(CacheField, StampField, ValidityField, State, Now),
    Cache = State#state.CacheField,
    Then = State#state.StampField,
    Validity = State#state.ValidityField,
    if
        Cache == undefined -> stale;
        Then + Validity >= Now -> fresh;
        true -> stale
    end).

% Return `fresh` if the cached users are not too stale, or `stale` otherwise.
% Also return `stale` if there is no cache at all.
users_cache_fresh(State, Now) ->
    ?cache_fresh(cached_users, cached_users_timestamp, user_cache_validity, State, Now).
groups_cache_fresh(State, Now) ->
    ?cache_fresh(cached_groups, cached_groups_timestamp, group_cache_validity, State, Now).

% Calls erlang:now(). Placed in this module for mockability.
now() -> erlang:now().

user_entries_to_dict(UserUIDAttr, UserDescAttr, Entries) ->
    lists:foldl(
        fun(#eldap_entry{attributes=Attrs}, Dict) ->
            case {eldap_utils:get_ldap_attr(UserUIDAttr,Attrs),
                  eldap_utils:get_ldap_attr(UserDescAttr,Attrs)}
            of
                % By returning "" get_ldap_attr means "not found"
                {UID, Desc} when UID /= "" ->
                    dict:append(jlib:nodeprep(UID), Desc, Dict);
                _ ->
                    Dict
            end
        end,
        dict:new(),
        Entries
    ).

group_entries_to_dict(GroupIDAttr, GroupDescAttr, GroupMemberAttr, Host, ExtractProcessor, AuthProcessor, Entries) ->
    lists:foldl(
        fun(#eldap_entry{attributes=Attrs}, Dict) ->
            case {eldap_utils:get_ldap_attr(GroupIDAttr, Attrs),
                  eldap_utils:get_ldap_attr(GroupDescAttr, Attrs),
                  lists:keysearch(GroupMemberAttr, 1, Attrs)}
            of
                {ID, Desc, {value, {GroupMemberAttr, Members}}} when ID =/= "" ->
                    JIDs = lists:foldl(
                        fun({ok, UID}, L) ->
                            PUID = jlib:nodeprep(UID),
                            case PUID of
                                error -> L;
                                _ ->
                                    case AuthProcessor(PUID, Host) of
                                        true -> [{PUID, Host} | L];
                                        _ -> L
                                    end
                            end;
                           (_, L) ->
                               L
                        end,
                        [],
                        lists:map(ExtractProcessor, Members)
                    ),
                    NewInfo = #group_info{desc = Desc, members = lists:usort(JIDs)},
                    dict:update(ID,
                        fun(OldInfo) ->
                            OldMembers = OldInfo#group_info.members,
                            NewMembers = NewInfo#group_info.members,
                            OldInfo#group_info{members = lists:umerge(OldMembers, NewMembers)}
                        end,
                        NewInfo, Dict);
                _ ->
                    Dict
            end
        end,
        dict:new(),
        Entries
    ).

