# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.
#
# Authors: Guido Amoruso <guidonte@fluendo.com>


from elisa.core import common

from elisa.core.media_uri import MediaUri
from elisa.core.utils import defer
from elisa.core.utils.cancellable_defer import cancellable_deferred_iterator, \
        CancellableDeferred
from elisa.plugins.base.models.image import ImageModel
from elisa.plugins.base.models.media import PlaylistModel

from twisted.internet import task

from random import shuffle
import datetime
import os.path


class Action(object):
    """Contextul action for a Controller object.

    The intended use is for controllers to subclass this and override the run()
    method, that can do everything you want, assuming to deal with a specific
    controller class.

    @ivar controller: the controller on which to run the action
    @type controller: L{elisa.plugins.pigment.pigment_controller.PigmentController}
    """

    icon = 'elisa.plugins.poblesec.file'
    title = ''
    subtitle = ''

    def __init__(self, controller):
        self.controller = controller

    def run(self):
        raise NotImplementedError()


class PlayAllAction(Action):

    icon = 'elisa.plugins.poblesec.play_all'

    @cancellable_deferred_iterator
    def _build_playlist(self, tracks):
        for item in self.controller.model:
            if isinstance(item, Action):
                continue

            if hasattr(item, 'get_playable_model'):
                dfr = defer.succeed([item])
            else:
                dfr = item.get_tracks()

            dfr.addCallback(lambda result: tracks.extend(result))
            yield dfr

    def run(self):
        tracks = []
        cancellable_iterator = self._build_playlist(tracks)
        def canceller(dfr):
            cancellable_iterator.cancel()
        cancellable = CancellableDeferred(canceller)

        dfr = task.coiterate(iter(cancellable_iterator))
        dfr.addCallback(lambda result: self.order_tracks(tracks))
        dfr.addCallback(self.play)

        dfr.chainDeferred(cancellable)
        return cancellable

    def order_tracks(self, tracks):
        return tracks

    def play(self, items):
        # FIXME: this is not dynamic enough, one needs to first build the list
        # of PlayableModel's before actually starting to play the first item.
        # The playback of the first item should start first and then the list
        # of following items to enqueue in the playlist be built.
        if not items:
            return items

        poblesec = self.controller.frontend.retrieve_controllers('/poblesec')[0]
        player = poblesec.music_player.player
        poblesec.show_music_player()
        self.controller.stop_loading_animation()

        playlist = PlaylistModel()
        playlist.extend(items)
        player.set_playlist(playlist)
        player.play()


class ShuffleAction(PlayAllAction):

    icon = 'elisa.plugins.poblesec.shuffle'

    def order_tracks(self, tracks):
        """Actually shuffle tracks."""
        shuffle(tracks)
        return tracks


class FavoriteNotFoundError(Exception):
    pass


class AddToFavoritesAction(Action):

    toggled = False

    toggled_icon = 'elisa.plugins.poblesec.remove_from_favorites'
    untoggled_icon = 'elisa.plugins.poblesec.add_to_favorites'
    toggled_title = ''
    untoggled_title = ''
    toggled_subtitle = ''
    untoggled_subtitle = ''

    def setup(self, favorites):
        if not favorites:
            self.toggle(False)
        else:
            self.toggle(True)

        return favorites

    def store(self, favorite):
        if favorite is None:
            self.controller.warning("Not storing None favorite")
            return

        self.controller.debug("Storing %s" % favorite)

        common.application.store.add(favorite)
        common.application.store.commit()

        self.controller.stop_loading_animation()
        self.toggle(not self.toggled)

        return favorite

    def remove(self, favorite):
        if favorite is None:
            self.controller.warning("Favorite not found: not removing")
            return

        self.controller.debug("Removing %s" % favorite)

        common.application.store.remove(favorite)
        common.application.store.commit()

        self.controller.stop_loading_animation()
        self.toggle(not self.toggled)

        return favorite

    def toggle(self, status):
        self.controller.debug("Toggling %s" % status)

        self.toggled = status
        self.refresh()

    def refresh(self):
        if self.toggled:
            self.icon = self.toggled_icon
            self.title = self.toggled_title
            self.subtitle = self.toggled_subtitle
        else:
            self.icon = self.untoggled_icon
            self.title = self.untoggled_title
            self.subtitle = self.untoggled_subtitle

        # visually refresh the list item
        # FIXME: uses reassignment to inform the widget that some values have
        # changed
        try:
            index = self.controller.actions.index(self)
            self.controller.actions[index] = self
        except ValueError, exception:
            # the action is not yet in the list of actions
            pass


class AddToPlaylistAction(Action):

    icon = 'elisa.plugins.poblesec.add_to_playlist'
    toggled = False


class ActionPathNotSetError(Exception):
    pass


class LinkAction(Action):
    crumb_title = None
    path = None
    args = {}

    def run(self):
        if self.path is None:
            raise ActionPathNotSetError()

        crumb_title = self.crumb_title
        if crumb_title is None:
            crumb_tile = self.title

        browser = self.controller.frontend.retrieve_controllers('/poblesec/browser')[0]
        dfr = browser.history.append_controller(self.path, crumb_title,
                                                **self.args)

        return dfr


class GoToNode(object):

    def __init__(self):
        self.path = ''
        self.title = ''
        self.args = {}


class ActionNodesNotSetError(Exception):
    pass


class GoToAction(Action):

    def __init__(self, controller):
        super(GoToAction, self).__init__(controller)

        self.label = ''
        self.nodes = []

    def run(self):
        if not self.nodes:
            raise ActionNodesNotSetError()

        browser = self.controller.frontend.retrieve_controllers('/poblesec/browser')[0]
        browser.history.clear()

        def _append_controller(result, *args, **kwargs):
            return browser.history.append_controller(*args, **kwargs)

        dfr = defer.succeed(True)
        for node in self.nodes:
            dfr.addCallback(_append_controller, node.path, node.title, **node.args)

        return dfr


class ViewSlideshowAction(Action):

    icon = 'elisa.plugins.poblesec.play_all'

    def get_items(self, item):
        return defer.succeed([item])

    def run(self):
        photo = getattr(self.controller, 'photo', None)
        photos = getattr(self.controller, 'photos', [])
        if photo and photos:
            self.play(photos, photo)
        else:
            items = [item for item in self.controller.model if not isinstance(item, Action)]
            images = []
            dfrs = []
            for item in items:
                dfr = self.get_items(item)
                dfr.addCallback(lambda the_images: images.extend(the_images))
                dfrs.append(dfr)

            if len(dfrs) == 0:
                self.controller.stop_loading_animation()
                return defer.succeed(None)

            dfr_list = defer.DeferredList(dfrs)
            dfr_list.addCallback(lambda result: self.order_images(images))
            dfr_list.addCallback(self.play, start=photo)

            return dfr_list

    def order_images(self, images):
        return images

    def play(self, images, start=None):
        slideshow_controller = self.controller.frontend.retrieve_controllers('/poblesec/slideshow_player')[0]
        slideshow_controller.player.clear_playlist()

        for image in images:
            img = ImageModel()
            if getattr(image, 'thumbnail', None) is not None:
                thumbnail_uri = MediaUri("file://" + image.thumbnail)
                img.references.append(thumbnail_uri)
            img.references.append(MediaUri("file://" + image.file_path))
            img.title = os.path.basename(image.file_path)
            slideshow_controller.player.enqueue_to_playlist(img)

        if start is not None: 
            # only set the index in case we have a real one
            index = images.index(start)
            slideshow_controller.player.jump_to_index(index)

        main = self.controller.frontend.retrieve_controllers('/poblesec')[0]
        main.show_slideshow_player()
        main.current_player.player.play()
        self.controller.stop_loading_animation()
