# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.
#
# Authors: Guido Amoruso <guidonte@fluendo.com>
#          Benjamin Kampmann <benjamin@fluendo.com>

from elisa.core.media_uri import MediaUri
from elisa.core.log import Loggable
from elisa.core import common
from elisa.core.default_config import PICTURES_CACHE
from elisa.core.utils import defer
from elisa.core.utils.cancellable_defer import CancelledError
from elisa.core.resource_manager import NoMatchingResourceProvider, \
                                        ResourceProviderNotFound

from elisa.plugins.base.models.audio import AlbumModel

from elisa.plugins.poblesec.link import Link
from elisa.plugins.poblesec.base.list import GenericListViewMode

# FIXME: cover retrieval is hardcoded for amazon covers
try:
    from elisa.plugins.amazon.metadata import AmazonCovers
except ImportError:
    AmazonCovers = None

import os

try:
    from hashlib import md5
except ImportError:
    # hashlib is new in Python 2.5
    from md5 import md5


# TODO: factorize this code in the core of Elisa in a caching utilities module
def _thumbnail_file(thumbnail_uri):
    if not os.path.exists(PICTURES_CACHE):
        os.makedirs(PICTURES_CACHE, 0755)
    thumbnail = md5(str(thumbnail_uri)).hexdigest() + '.jpg'
    return os.path.join(PICTURES_CACHE, thumbnail)


# TODO: factorize this code in the core of Elisa in a caching utilities module
def _get_thumbnail(thumbnail_uri):
    thumbnail_file = _thumbnail_file(thumbnail_uri)
    if os.path.exists(thumbnail_file):
        return defer.succeed(thumbnail_file)
    else:
        # Download the thumbnail first
        def got_thumbnail(data_model):
            fd = file(thumbnail_file, 'wb')
            fd.write(data_model.data)
            fd.close()
            return thumbnail_file

        uri = MediaUri(thumbnail_uri)
        model, deferred = common.application.resource_manager.get(uri)
        deferred.addCallback(got_thumbnail)
        return deferred


class AlreadyInQueueError(Exception):
    pass


class ArtistPictureRetriever(Loggable):
    def __init__(self):
        super(ArtistPictureRetriever, self).__init__()
        self.enabled = True
        self.cache = {}

    def _check_for_image_on_model(self, model, model_id):
        if hasattr(model, 'images') and model.images is not None and \
                len(model.images) > 0:
            image_uri = None
            for image in model.images:
                if len(image.references) > 0:
                    image_uri = image.references[1]
                    break

            if image_uri is None:
                return False

            self.cache[model_id] = image_uri
            model.image_uri = image_uri
            return True
        return False

    def get_image_uri(self, model):
        model_id = model.name.replace(" ", "+")
        uri = MediaUri("http://www.discogs.com/artist/%s" % model_id)

        if model_id in self.cache:
            if self.cache[model_id] is not None:
                # We did a request and we still have the result in the cache
                model.image_uri = self.cache[model_id]
                return defer.succeed(model)
            # The request is already in the queue
            return defer.fail(AlreadyInQueueError(uri))

        if not self.enabled:
            # we are disabled
            self._check_for_image_on_model(model, model_id)
            return defer.succeed(model)

        self.cache[model_id] = None

        def got_image_uri(result, model, model_id):
            if result.images:
                image_uri = unicode(result.images[0].references[0])
                model.image_uri = image_uri
                self.cache[model_id] = image_uri
            else:
                self._check_for_image_on_model(model, model_id)
            return model

        def failed(failure, model_id):
            # if it was cancelled, we want to remove the model_id to try again
            # later
            if failure.type is CancelledError:
                self.cache.pop(model_id)
            elif self._check_for_image_on_model(model, model_id):
                return model

            self.debug('Failed to retrieve artist picture for %s: %s' % \
                       (model.name, failure))

        try:
            data_model, dfr = common.application.resource_manager.get(uri)
        except NoMatchingResourceProvider, e:
            self.enabled = False
            return defer.fail(e)

        dfr.addCallback(got_image_uri, model, model_id)
        dfr.addErrback(failed, model_id)
        return dfr

# FIXME: a global instance at this level does not prevent from having multiple
# instances, this should be made a singleton or an attribute of the
# application, or something...
artist_picture_retriever = ArtistPictureRetriever()


class AlbumCoverRetrieval(Loggable):
    def __init__(self):
        super(AlbumCoverRetrieval, self).__init__()
        self.cache = {}
        self.provider = None

        if AmazonCovers is None:
            # plugin missing
            return

        def set_provider(provider):
            self.provider = provider

        try:
            dfr = AmazonCovers.create({})
        except:
            # for some reason the creation failed. As this is called on import
            # time we don't want to stop processing
            dfr = defer.fail()

        dfr.addCallback(set_provider)

    def _check_for_cover_on_model(self, model, model_id):
        # check for the generic cover and set the best reference as cover_uri if
        # available
        if hasattr(model, 'cover') and model.cover is not None and \
                len(model.cover.references) > 0:
            cover_uri = model.cover.references[-1]
            self.cache[model_id] = cover_uri
            model.cover_uri = cover_uri
            return True
        return False

    def get_cover_uri(self, model, artist=None):
        model_id = unicode(model.name)

        if model_id in self.cache:
            if self.cache[model_id] is not None:
                # We did a request and we still have the result in the cache
                model.cover_uri = self.cache[model_id]
                return defer.succeed(model)
            # The request is already in the queue
            return defer.fail(AlreadyInQueueError(model.name))

        if not self.provider:
            # we are disabled
            self._check_for_cover_on_model(model, model_id)
            return defer.succeed(model)

        self.cache[model_id] = None

        album_model = AlbumModel()
        album_model.artist = artist
        album_model.album = model_id.encode('UTF-8')

        def got_cover_uri(result, model, model_id):
            if result.cover:
                cover_uri = unicode(result.cover.references[-1])
                self.cache[model_id] = cover_uri
                model.cover_uri = cover_uri
            else:
                # fallback check on the model
                self._check_for_cover_on_model(model, model_id)

            return model

        def failed(failure, model_id):
            # if it was cancelled, we want to remove the model_id to try again
            # later
            if failure.type is CancelledError:
                self.cache.pop(model_id)
            elif self._check_for_cover_on_model(model, model_id):
                return model

            self.debug('Failed to retrieve album cover for %s: %s' % \
                       (model.name, failure))

        deferred = self.provider.get_metadata(album_model)
        deferred.addCallback(got_cover_uri, model, model_id)
        deferred.addErrback(failed, model_id)
        return deferred

# FIXME: a global instance at this level does not prevent from having multiple
# instances, this should be made a singleton or an attribute of the
# application, or something...
album_cover_retriever = AlbumCoverRetrieval()


class ArtistsViewMode(GenericListViewMode):

    """
    Implementation of the common view modes API.
    """

    def get_label(self, item):
        return defer.succeed(item.name)

    def get_default_image(self, item):
        resource = 'elisa.plugins.poblesec.artist'
        return resource

    def get_image(self, item, theme):
        def got_image_uri(result_model):
            try:
                if result_model.image_uri is None:
                    return None
                return _get_thumbnail(result_model.image_uri)                
            except AttributeError:
                return None

        def _failure(failure):
            failure.trap(AlreadyInQueueError)

        if hasattr(item, 'image_uri') and item.image_uri is not None:
            image_deferred = defer.succeed(item)
        else:
            image_deferred = artist_picture_retriever.get_image_uri(item)
        image_deferred.addCallbacks(got_image_uri, _failure)
        return image_deferred

    def get_preview_image(self, item, theme):
        try:
            thumbnail_file = _thumbnail_file(item.image_uri)
            if not os.path.exists(thumbnail_file):
                return None
            return thumbnail_file
        except AttributeError:
            return None


class AlbumsViewMode(GenericListViewMode):

    """
    Implementation of the common view modes API.
    """

    def get_label(self, item):
        return defer.succeed(item.name)

    def get_default_image(self, item):
        resource = 'elisa.plugins.poblesec.cover'
        return resource

    def get_sublabel(self, item):
        return item.get_artist_name()

    def get_image(self, item, theme):
        def got_artist(artist):
            return album_cover_retriever.get_cover_uri(item, artist)

        def got_image_uri(result_model):
            try:
                if result_model.cover_uri is None:
                    return None
                return _get_thumbnail(result_model.cover_uri)
            except AttributeError:
                return None

        def _failure(failure):
            failure.trap(AlreadyInQueueError)

        if hasattr(item, 'cover_uri') and item.cover_uri is not None:
            image_deferred = defer.succeed(item)
        else:
            image_deferred = item.get_artist_name()
            image_deferred.addCallback(got_artist)
        image_deferred.addCallbacks(got_image_uri, _failure)
        return image_deferred

    def get_preview_image(self, item, theme):
        try:
            thumbnail_file = _thumbnail_file(item.cover_uri)
            if not os.path.exists(thumbnail_file):
                return None
            return thumbnail_file
        except AttributeError:
            return None


class TracksViewMode(GenericListViewMode):

    """
    Implementation of the common view modes API.
    """

    trash_letters = " .,-_"

    def get_label(self, item):
        title = item.title.strip(self.trash_letters)
        return defer.succeed(title)

    def get_sublabel(self, item):
        raise NotImplementedError()

    def get_default_image(self, item):
        resource = 'elisa.plugins.poblesec.cover'
        return resource

    def get_image(self, item, theme):

        def got_image_uri(result_model):
            try:
                if result_model.cover_uri is None:
                    return None
                item.cover_uri = result_model.cover_uri
                return _get_thumbnail(result_model.cover_uri)
            except AttributeError:
                return None

        def _failure(failure):
            failure.trap(AlreadyInQueueError)

        if hasattr(item, 'cover_uri') and item.cover_uri is not None:
            image_deferred = defer.succeed(item)
            image_deferred.addCallbacks(got_image_uri, _failure)
            return image_deferred
        else:
            def got_artist(artist, album):
                return album_cover_retriever.get_cover_uri(album, artist)

            def got_album(album):
                if hasattr(album, 'cover_uri') and album.cover_uri is not None:
                    return defer.succeed(album)
                elif album:
                    dfr = album.get_artist_name()
                    dfr.addCallback(got_artist, album)
                    return dfr

            album_deferred = item.get_album()
            album_deferred.addCallback(got_album)
            album_deferred.addCallbacks(got_image_uri, _failure)
            return album_deferred

    def get_preview_image(self, item, theme):
        try:
            thumbnail_file = _thumbnail_file(item.cover_uri)
            if not os.path.exists(thumbnail_file):
                return None
            return thumbnail_file
        except AttributeError:
            return None
