# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.
#
# Author: Guido Amoruso <guidonte@fluendo.com>

from twisted.web.client import getPage

from elisa.core.media_uri import MediaUri
from elisa.core.default_config import PICTURES_CACHE

from elisa.plugins.poblesec.base.hierarchy import HierarchyController
from elisa.plugins.poblesec.base.list import GenericListViewMode
from elisa.plugins.poblesec.base.preview_list import MenuItemPreviewListController
from elisa.plugins.poblesec.base.coverflow import \
    ImageWithReflectionCoverflowController
from elisa.plugins.poblesec.base.grid import GridItemGridController
from elisa.plugins.poblesec.base.list_switcher import ListSwitcherController
from elisa.plugins.poblesec.actions import Action

from elisa.plugins.rss.actions import SearchAction
from elisa.plugins.rss.models import RssItemModel, RssFeedModel

from elisa.core.utils import defer

import os

try:
    from hashlib import md5
except ImportError:
    # hashlib is new in Python 2.5
    from md5 import md5


# TODO: factorize this code in the core of Elisa in a caching utilities module
def _thumbnail_file(thumbnail_uri):
    if not os.path.exists(PICTURES_CACHE):
        os.makedirs(PICTURES_CACHE, 0755)
    thumbnail = md5(str(thumbnail_uri)).hexdigest() + '.jpg'
    return os.path.join(PICTURES_CACHE, thumbnail)


class RssReaderController(HierarchyController):

    def initialize(self, feed=None, feed_list=[], rss_feed_class=RssFeedModel,
                   media_filter=None, search_enabled=False):
        """
        @param feed: the feed to display
        @type feed: L{elisa.plugins.rss.models.RssFeedModel}
        @param feed_list: a list of feed information in tuple of
                         (name, uri, icon, [etc. whatever needed])
        @type feed_list: tuple of C{str}
        @param rss_feed_class: the class of the RSS feed
        @type rss_feed_class: sublcass of L{elisa.plugins.rss.models.RssFeedModel}
        """
        dfr = super(RssReaderController, self).initialize()

        self.feed = feed
        self.feed_list = feed_list
        self.rss_feed_class = rss_feed_class
        self.media_filter = media_filter

        def add_feeds(self):
            
            if search_enabled:
                self.actions.append(SearchAction(self))
            
            feed_models = []
            for feed in self.feed_list:
                # The actual RSS feed class can implement specialized parsing
                # of the feed
                feed_model = self.rss_feed_class()

                feed_model.name = feed[0]
                feed_model.uri = MediaUri(feed[1])
                feed_models.append(feed_model)
                
            self.model.extend(feed_models)
            return self

        def load_items(self, feed):
            return feed.load()

        def extend_model(result, feed):
            items = feed.items
            if self.media_filter:
                items = [item for item in items \
                         if item.type.startswith(self.media_filter)]
            self.model.extend(items)
            return self

        if feed:
            dfr.addCallback(load_items, feed)
            dfr.addCallback(extend_model, feed)
        else:
            dfr.addCallback(add_feeds)

        return dfr

    def node_clicked(self, widget, item):
        if isinstance(item, Action):
            item.run()

        elif isinstance(item, RssFeedModel):
            browser = self.frontend.retrieve_controllers('/poblesec/browser')[0]
            dfr = browser.history.append_controller(self.path, item.name,
                                                    feed=item,
                                                    rss_feed_class=self.rss_feed_class,
                                                    media_filter=self.media_filter,)

        elif isinstance(item, RssItemModel):
            main = self.frontend.retrieve_controllers('/poblesec')[0]

            if item.playable_model:
                player = self.frontend.retrieve_controllers('/poblesec/video_player')[0]
                player.player.play_model(item.playable_model)
                main.show_video_player()

            elif item.image_model:
                slideshow = self.frontend.retrieve_controllers('/poblesec/slideshow_player')[0]
                slideshow.player.clear_playlist()

                for bit in self.model:
                    if bit.image_model:
                        if bit == item:
                            index = len(slideshow.player.playlist)
                        slideshow.player.enqueue_to_playlist(bit.image_model)

                slideshow.player.jump_to_index(index)
                main.show_slideshow_player()

            self.stop_loading_animation()


class RssReaderViewMode(GenericListViewMode):

    def get_label(self, item):
        if isinstance(item, RssFeedModel):
            return defer.succeed(item.name)
        elif isinstance(item, RssItemModel):
            return defer.succeed(item.title)
        elif isinstance(item, Action):
            return defer.succeed(item.title)

    def get_default_image(self, item):
        if isinstance(item, RssFeedModel):
            resource = 'elisa.plugins.poblesec.folder'
        elif isinstance(item, RssItemModel):
            if item.type.startswith('image'):
                resource = 'elisa.plugins.poblesec.file_picture'
            elif item.type.startswith('audio'):
                resource = 'elisa.plugins.poblesec.file_music'
            elif item.type.startswith('video'):
                resource = 'elisa.plugins.poblesec.file_video'
            else:
                resource = 'elisa.plugins.poblesec.file'
        elif isinstance(item, Action):
            resource = item.icon
        return resource

    # TODO: at one point, we should stop to ship this copied&pasted
    # implementation...
    def get_image(self, item, theme):
        if isinstance(item, RssItemModel):
            if not(item.thumbnails):
                return None

            thumbnail_uri = str(item.thumbnails[0].references[0])
            if not getattr(item, 'thumbail_file', None):
                item.thumbnail_file = _thumbnail_file(thumbnail_uri)

            if os.path.exists(item.thumbnail_file):
                return defer.succeed(item.thumbnail_file)
            else:
                def got_thumbnail(response):
                    fd = file(item.thumbnail_file, 'wb')
                    fd.write(response)
                    fd.close()
                    return item.thumbnail_file

                dfr = getPage(thumbnail_uri)
                dfr.addCallback(got_thumbnail)

                return dfr
        else:
            return None

    def get_preview_image(self, item, theme):
        if isinstance(item, RssItemModel):
            try:
                return item.thumbnail_file
            except AttributeError:
                return None
        else:
            return None


class RssReaderPreviewListController(RssReaderController, MenuItemPreviewListController):
    view_mode = RssReaderViewMode


class RssReaderCoverflowController(RssReaderController, ImageWithReflectionCoverflowController):
    view_mode = RssReaderViewMode


class RssReaderGridController(RssReaderController, GridItemGridController):
    view_mode = RssReaderViewMode


class RssReaderListSwitcherController(ListSwitcherController):
    modes = [RssReaderPreviewListController,
             RssReaderCoverflowController,
             RssReaderGridController]
    default_config = {'view_mode': 'preview_list'}

