# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.
#
# Authors: Olivier Tilloy <olivier@fluendo.com>
#          Philippe Normand <philippe@fluendo.com>

from elisa.core.common import application
from elisa.core.media_uri import MediaUri
from elisa.core.input_event import EventValue, key_values, num_values
from elisa.core.components.model import Model
from elisa.core.default_config import CONFIG_DIR, PICTURES_CACHE
from elisa.core.utils.cancellable_defer import CancellableDeferred, \
                                               CancelledError
from elisa.core.utils.i18n import install_translation

from elisa.plugins.poblesec.link import Link
from elisa.plugins.poblesec.base.hierarchy import HierarchyController
from elisa.plugins.poblesec.base.list import GenericListViewMode
from elisa.plugins.poblesec.base.preview_list import MenuItemPreviewListController
from elisa.plugins.poblesec.base.coverflow import ImageWithReflectionCoverflowController
from elisa.plugins.poblesec.base.grid import GridItemGridController
from elisa.plugins.poblesec.base.list_switcher import ListSwitcherController
from elisa.plugins.poblesec.actions import Action

from elisa.plugins.base.models.image import ImageModel

from elisa.plugins.flickr.models import FlickrResponseModel, \
                                        FlickrPhotoModel, FlickrTagModel, \
                                        FlickrPhotoSetModel, FlickrContactModel
from elisa.plugins.flickr.resource_provider import API_SERVER
from elisa.plugins.flickr.actions import LoginAction, LogoutAction, MyAccountAction
from elisa.plugins.flickr import flickr_api as flickr
from elisa.plugins.flickr.utils import get_flickr_provider

from twisted.internet import defer

import os

try:
    from hashlib import md5
except ImportError:
    # hashlib is new in Python 2.5
    from md5 import md5

import pickle


_ = install_translation('flickr')
_p = install_translation('poblesec')


class FlickrEntryPointModel(Model):
    """ Model representing one of the main Flickr menus.

    @ivar title:     menu title
    @type title:     unicode
    @ivar uri:       entry point URI
    @type uri:       L{elisa.core.media_uri.MediaUri}
    @ivar need_grid: should the entry point be browsed using the grid or not
    @type need_grid: bool
    @ivar icon_resource: resource path of the icon to be displayed for the entry point
    @type icon_resource: unicode
    """

    def __init__(self):
        super(Model, self).__init__()
        self.title = None
        self.uri = None
        self.need_grid = False
        self.icon_resource = None

def flickr_decorator(controller):
    link = Link()
    link.controller_path = '/poblesec/flickr/home'
    link.label = _('Flickr')
    link.icon = 'elisa.plugins.flickr.logo'
    controller.model.append(link)
    return defer.succeed(None)


# TODO: factorize this code in the core of Elisa in a caching utilities module
def _thumbnail_file(thumbnail_uri):
    if not os.path.exists(PICTURES_CACHE):
        os.makedirs(PICTURES_CACHE, 0755)
    thumbnail = md5(str(thumbnail_uri)).hexdigest() + '.jpg'
    return os.path.join(PICTURES_CACHE, thumbnail)

class ControllerMixin(object):
    search_uri = MediaUri('flickr://search')
    account_uri = MediaUri('flickr://account')

    def node_clicked(self, widget, item):
        if isinstance(item, Action):
            item.run()
        elif isinstance(item, FlickrEntryPointModel):
            controllers = self.frontend.retrieve_controllers('/poblesec/browser')
            browser = controllers[0]
            if item.uri == self.search_uri:
                path = '/poblesec/flickr/search'
            elif item.uri == self.account_uri:
                path = '/poblesec/flickr/account'
            elif item.need_grid:
                path = '/poblesec/flickr'
            else:
                path = '/poblesec/flickr/folder'
            dfr = browser.history.append_controller(path,
                                                    item.title, uri=item.uri)

        elif isinstance(item, FlickrTagModel):
            args = {'tags': item.label}
            method = 'flickr.photos.search'
            authenticated = False
            if item.private:
                # only my photos
                args['user_id'] = 'me'
                authenticated = True
            item.uri = MediaUri(flickr.generate_call_uri(method=method,
                                                         arguments=args,
                                                         authenticated=authenticated))
            controllers = self.frontend.retrieve_controllers('/poblesec/browser')
            browser = controllers[0]
            dfr = browser.history.append_controller('/poblesec/flickr', item.label,
                                                    uri=item.uri)

        elif isinstance(item, FlickrPhotoSetModel):
            provider = get_flickr_provider()
            authenticated = provider.auth_token is not None
            item.uri = MediaUri(flickr.generate_call_uri(method='flickr.photosets.getPhotos',
                                                         arguments={'photoset_id': item.flickr_id},
                                                         authenticated=authenticated))
            controllers = self.frontend.retrieve_controllers('/poblesec/browser')
            browser = controllers[0]
            dfr = browser.history.append_controller('/poblesec/flickr', item.title, uri=item.uri)

        elif isinstance(item, FlickrContactModel):
            item.uri = MediaUri(flickr.generate_call_uri(method='flickr.photosets.getList',
                                                         arguments={'user_id': item.nsid},
                                                         authenticated=True))
            controllers = self.frontend.retrieve_controllers('/poblesec/browser')
            browser = controllers[0]
            name = item.realname or item.username
            dfr = browser.history.append_controller('/poblesec/flickr/folder', name,
                                                    uri=item.uri)

        elif isinstance(item, FlickrPhotoModel):
            # We want a Deferred here, too
            self.play_image(item)
            dfr = CancellableDeferred()
            dfr.callback(True)

    def play_image(self, item):
        controllers = self.frontend.retrieve_controllers('/poblesec/slideshow_player')
        slideshow_controller = controllers[0]
        slideshow_controller.player.clear_playlist()

        # Cancel all currently queued requests
        for widget in self.nodes._widgets:
            self.cancel_deferreds(widget)

        def _failure(failure):
            # Swallow errbacks only when the deferred has been cancelled
            failure.trap(CancelledError)

        for flickr_photo in self.model:
            if isinstance(flickr_photo, FlickrPhotoModel):
                if flickr_photo == item:
                    # save the index of the image to be displayed first
                    index = len(slideshow_controller.player.playlist)
                if not hasattr(flickr_photo, 'images'):
                    dfr, photo = _get_image_available_sizes(flickr_photo)
                    self.register_deferred(photo, dfr)
                    dfr.addErrback(_failure)
                else:
                    photo = flickr_photo.images
                    if not photo.references:
                        dfr, photo = _get_image_available_sizes(flickr_photo)
                        self.register_deferred(photo, dfr)
                        dfr.addErrback(_failure)
                photo.title = flickr_photo.title or '<%s>' % flickr_photo.flickr_id
                # we want Flickr's photos to be branded as such in the slideshow
                photo.logo = "elisa.plugins.flickr.logo"
                slideshow_controller.player.enqueue_to_playlist(photo)

        slideshow_controller.player.jump_to_index(index)

        controllers = self.frontend.retrieve_controllers('/poblesec')
        main = controllers[0]
        main.show_slideshow_player()
        self.stop_loading_animation()


    def _resource_loaded(self, resource):
        if isinstance(resource, FlickrResponseModel):
            if hasattr(resource, 'photos'):
                self.model.extend(resource.photos)
            elif hasattr(resource, 'tags'):
                self.model.extend(resource.tags)
            elif hasattr(resource, 'photosets'):
                self.model.extend(resource.photosets)
            elif hasattr(resource, 'contacts'):
                self.model.extend(resource.contacts)
        return self

    def _load_resource(self, result):
        resource, get_deferred = application.resource_manager.get(self.uri)
        return get_deferred

class FlickrController(ControllerMixin, HierarchyController):

    start_uri = MediaUri('http://%s/services/rest/' % API_SERVER)

    def initialize(self, uri=start_uri):
        deferred = super(FlickrController, self).initialize()
        self.uri = uri

        def add_entries(result):
            entries = [(_('Last 7 days interesting'), True,
                        'elisa.plugins.poblesec.recently_viewed_photos',
                        flickr.generate_call_uri(method='flickr.interestingness.getList')),
                       (_('Popular tags'), False,
                        'elisa.plugins.poblesec.tags',
                        flickr.generate_call_uri(method='flickr.tags.getHotList')),
                       (_('Most recent uploads'), True,
                        'elisa.plugins.poblesec.photo',
                        flickr.generate_call_uri(method='flickr.photos.getRecent')),
                       ]

            entry_models = []
            for title, need_grid, icon_resource, url in entries:
                entry_model = FlickrEntryPointModel()
                entry_model.title = title
                entry_model.uri = MediaUri(url)
                entry_model.need_grid = need_grid
                entry_model.icon_resource = icon_resource
                entry_models.append(entry_model)

            self.model.extend(entry_models)
            self.model.append(MyAccountAction(self))
            return self

        def check_for_login_dfr(result):
            provider = get_flickr_provider()
            if provider.login_dfr is not None:
                return provider.login_dfr

        deferred.addCallback(check_for_login_dfr)

        if uri == self.start_uri:
            # List of the entry points
            deferred.addCallback(add_entries)
        else:
            # One particular entry point
            deferred.addCallback(self._load_resource)
            deferred.addCallback(self._resource_loaded)

        return deferred

class FlickrAccountController(ControllerMixin, HierarchyController):


    def initialize(self, uri=ControllerMixin.account_uri):
        deferred = super(FlickrAccountController, self).initialize()
        self.uri = uri

        provider = get_flickr_provider()

        def build_account_menu(self):
            entries = []
            if provider.auth_token:
                entries = [(_('Sets'), False, 'elisa.plugins.poblesec.photo_album',
                            flickr.generate_call_uri(method='flickr.photosets.getList',
                                                     authenticated=True)),
                           (_('Tags'), False, 'elisa.plugins.poblesec.tags',
                            flickr.generate_call_uri(method='flickr.tags.getListUser',
                                                     authenticated=True)),
                           (_('Unorganized photos'), True,
                            'elisa.plugins.poblesec.unorganized_photos',
                            flickr.generate_call_uri(method='flickr.photos.getNotInSet',
                                                     authenticated=True)),
                           (_('Contacts'), False, 'elisa.plugins.poblesec.user_contacts',
                            flickr.generate_call_uri(method='flickr.contacts.getList',
                                                     authenticated=True)),
                           ]

            entry_models = []
            for title, need_grid, icon_resource, url in entries:
                entry_model = FlickrEntryPointModel()
                entry_model.title = title
                entry_model.uri = MediaUri(url)
                entry_model.need_grid = need_grid
                entry_model.icon_resource = icon_resource
                entry_models.append(entry_model)

            # login/logout actions
            if not provider.auth_token:
                entry_models.append(LoginAction(self))
            else:
                entry_models.append(LogoutAction(self))

            self.model.extend(entry_models)
            return self

        if uri == self.account_uri:
            # List of the entry points
            deferred.addCallback(build_account_menu)
        else:
            # One particular entry point
            deferred.addCallback(self._load_resource)
            deferred.addCallback(self._resource_loaded)

        return deferred

def _get_photo_thumbnail(item):
    # Load the photo thumbnail
    thumbnail_uri = item.images.references[0]
    try:
        thumbnail_file = item.thumbnail_file
    except AttributeError:
        item.thumbnail_file = _thumbnail_file(thumbnail_uri)
        thumbnail_file = item.thumbnail_file
    if os.path.exists(thumbnail_file):
        return defer.succeed(thumbnail_file)
    else:
        if thumbnail_uri.scheme == 'file':
            # Image is local (default picture)
            thumbnail_file = thumbnail_uri._path
            return defer.succeed(thumbnail_file)

        # Download the thumbnail first
        def got_thumbnail(data_model):
            f = file(thumbnail_file, 'wb')
            f.write(data_model.data)
            f.close()
            return thumbnail_file

        data_model, dfr = application.resource_manager.get(thumbnail_uri)
        dfr.addCallback(got_thumbnail)
        return dfr


def _cached_sizes_file(item):
    cache = os.path.join(CONFIG_DIR, 'flickr_cache')
    if not os.path.exists(cache):
        os.makedirs(cache, 0755)
    return os.path.join(cache, item.flickr_id)


def _get_image_available_sizes(item):
    # Retrieve the list of available sizes (URLs) for an image
    item.images = ImageModel()
    # TODO: implement image rotation
    item.images.can_rotate = False

    # Check if we have this list in the cache first, will avoid a costly call
    # to the Flickr API
    cache_file = _cached_sizes_file(item)
    if os.path.exists(cache_file):
        fd = open(cache_file, 'rb')
        item.images.references = pickle.load(fd)
        fd.close()
        return (defer.succeed(item.images), item.images)

    provider = get_flickr_provider()
    authenticated = provider.auth_token is not None
    request = flickr.generate_call_uri(method='flickr.photos.getSizes',
                                       arguments={'photo_id': item.flickr_id},
                                       authenticated=authenticated)

    def got_available_sizes(result_model):
        if not item.images.references:
            item.images.references.extend(result_model.sizes.references)
            if not os.path.exists(cache_file):
                fd = open(cache_file, 'wb')
                pickle.dump(item.images.references, fd)
                fd.close()
        return item.images

    result_model, dfr = application.resource_manager.get(request)
    dfr.addCallback(got_available_sizes)
    return (dfr, item.images)


class FlickrViewMode(GenericListViewMode):

    """
    Implementation of the common view modes API.
    """

    def get_label(self, item):
        if isinstance(item, FlickrEntryPointModel):
            title = item.title
        elif isinstance(item, FlickrTagModel):
            title = item.label
        elif isinstance(item, FlickrPhotoModel):
            title = item.title or '<%s>' % item.flickr_id
        elif isinstance(item, FlickrPhotoSetModel):
            title = item.title
        elif isinstance(item, FlickrContactModel):
            title = item.realname or item.username
        elif isinstance(item, Action):
            title = item.title
        return defer.succeed(title)

    def get_default_image(self, item):
        if isinstance(item, FlickrEntryPointModel):
            resource = item.icon_resource
        elif isinstance(item, FlickrTagModel):
            resource = 'elisa.plugins.poblesec.tags'
        elif isinstance(item, FlickrPhotoSetModel):
            resource = 'elisa.plugins.poblesec.photo_album'
        elif isinstance(item, FlickrPhotoModel):
            resource = 'elisa.plugins.poblesec.file_picture'
        elif isinstance(item, FlickrContactModel):
            resource = 'elisa.plugins.poblesec.user'
        elif isinstance(item, Action):
            resource = item.icon
        return resource

    def get_image(self, item, theme):
        if isinstance(item, FlickrPhotoModel):
            # Load the photo thumbnail
            def get_thumbnail(images):
                return _get_photo_thumbnail(item)

            if not hasattr(item, 'images') or not item.images.references:
                images_deferred, images = _get_image_available_sizes(item)
            else:
                images_deferred = defer.succeed(item)
            images_deferred.addCallback(get_thumbnail)
            return images_deferred
        else:
            return None

    def get_preview_image(self, item, theme):
        if isinstance(item, FlickrPhotoModel):
            try:
                return item.thumbnail_file
            except AttributeError:
                return None
        else:
            return None


class FlickrPreviewListController(FlickrController, MenuItemPreviewListController):
    view_mode = FlickrViewMode


class FlickrCoverflowController(FlickrController, ImageWithReflectionCoverflowController):
    view_mode = FlickrViewMode


class FlickrGridController(FlickrController, GridItemGridController):
    view_mode = FlickrViewMode


class FlickrListSwitcherController(ListSwitcherController):
    modes = [FlickrPreviewListController,
             FlickrCoverflowController,
             FlickrGridController]
    crumbs_logo = "elisa.plugins.flickr.crumbs_logo"
    default_mode = FlickrGridController

class FlickrListSwitcherPreviewController(FlickrListSwitcherController):
    default_mode = FlickrPreviewListController


# Account

class FlickrAccountPreviewListController(FlickrAccountController,
                                         MenuItemPreviewListController):
    view_mode = FlickrViewMode


class FlickrAccountCoverflowController(FlickrAccountController,
                                       ImageWithReflectionCoverflowController):
    view_mode = FlickrViewMode


class FlickrAccountGridController(FlickrAccountController, GridItemGridController):
    view_mode = FlickrViewMode


class FlickrAccountListSwitcherController(ListSwitcherController):
    modes = [FlickrAccountPreviewListController,
             FlickrAccountCoverflowController,
             FlickrAccountGridController]
    crumbs_logo = "elisa.plugins.flickr.crumbs_logo"
    default_mode = FlickrAccountPreviewListController


def use_me_hook(frontend):
    """
    'Use me' hook that takes the user to the Pictures/Internet/Flickr section.

    @param frontend: the current frontend (poblesec)
    @type frontend:  L{elisa.plugins.pigment.pigment_frontend.PigmentFrontend}

    @return:         a deferred fired when the action is complete
    @rtype:          L{twisted.internet.defer.Deferred}
    """
    browser = frontend.retrieve_controllers('/poblesec/browser')[0]
    browser.history.clear()

    paths = [('/poblesec/pictures_menu', _p('Pictures')),
             ('/poblesec/pictures/internet', _p('Internet')),
             ('/poblesec/flickr/home', _('Flickr'))]

    # Select the pictures section in Elisa's main menu
    for index, link in enumerate(browser.history.current.model):
        if link.controller_path == paths[0][0]:
            browser.history.current.menu.selected_item_index = index
            break

    def append_controller_cb(result, path, label, **kwargs):
        return browser.history.append_controller(path, label, **kwargs)

    dfr = defer.succeed(None)
    for path, label in paths:
        dfr.addCallback(append_controller_cb, path, label)
    return dfr
