# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006,2007 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 2.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.


__maintainer__ = 'Lionel Martin <lionel@fluendo.com>'
__maintainer2__ = 'Florian Boucault <florian@fluendo.com>'


from elisa.core.component import Component, ComponentError
from elisa.base_components.model import Model
from elisa.core.observers.observer import Observer
from elisa.core.observers.observable import Observable
from elisa.core import common
from elisa.core.input_event import *


class ModelNotSupported(Exception):
    pass


class Controller(Component, Observer, Observable):
    """
    Connects to a compatible L{elisa.base_components.model.Model} and
    holds extra information useful for a L{elisa.base_components.view.View} to
    render the model.
    It notifies the views that are connected to it about all the changes
    occuring in the model and in the extra rendering information so
    that the views are always rendering the up-to-date content.

    DOCME: focus; branch of focused controllers, etc.

    @cvar supported_models:     list of models that are compatible with the
                                controller identified by their path: 'plugin:model'
    @type supported_models:     tuple of strings
    @cvar default_associations: DOCME
    @type default_associations: tuple of 2-tuple of strings
    @ivar model:                model to which the controller is connected;
                                setting it to an incompatible model will fail
    @type model:                L{elisa.base_components.model.Model}
    @ivar parent:               parent controller; None if the controller is a root
    @type parent:               L{elisa.base_components.controller.Controller}
    @ivar focused:              True if the controller has the focus, False
                                otherwise 
    @type focused:              boolean
    @ivar backend:              backend to which the controller belongs
    @type backend:              L{elisa.core.backend.Backend}

    """

    supported_models = ('base:model')

    default_associations = (
    ('base:model','base:controller'),
    )

    def __init__(self):
        Component.__init__(self)
        Observer.__init__(self)
        Observable.__init__(self)

        self._model = None
        self._backend = None
        self._focused = False
        self.parent = None
        self.loading = False


    def focus(self):
        """
        Grab the focus. The controller previously owning the focus loses it.
        """
        self.backend.focused_controller = self

    def focused__get(self):
        return self._focused

    def focused__set(self, focused):
        # FIXME: how/why is it different from focus? This is confusing.
        # Apparently, it is wrong to do controller.focused = True/False since
        # it does not have the same effect as doing controller.focus

        old_focused = self._focused
        self._focused = focused
        self.focused_changed(old_focused, focused)

    def focused_changed(self, old_focused, new_focused):
        """
        Called when L{focused} is set to a new value.

        Override if you wish to react to that change. Do not forget to call
        the parent class method.

        @param old_focused: value of focused before
        @type old_focused:  bool
        @param new_focused: value of focused now
        @type new_focused:  bool
        """
        pass


    def backend__get(self):
        return self._backend

    def backend__set(self, new_backend):
        old_backend, self._backend = self._backend, new_backend
        self.backend_changed(old_backend, new_backend)

    def backend_changed(self, old_backend, new_backend):
        """
        Called when L{backend} is set to a new value.

        Override if you wish to react to that change. Do not forget to call
        the parent class method.

        @param old_backend: value of backend before
        @type old_backend:  L{elisa.core.backend.Backend}
        @param new_backend: value of backend now
        @type new_backend:  L{elisa.core.backend.Backend}
        """
        pass


    def model__get(self):
        return self._model

    def model__set(self, model):
        if self._model == model:
            return

        if model == None:
            self.stop_observing()
        else:
            if not model.path in self.supported_models:
                self.warning("Controller %r does not support model %r" % (self.path, model.path))
                raise ModelNotSupported()
            self.observe(model)

        old_model = self._model
        self._model = model
        self.model_changed(old_model, model)

    def model_changed(self, old_model, new_model):
        """
        Called when L{model} is set to a new value.

        Override if you wish to react to that change. Do not forget to call
        the parent class method.

        @param old_model: value of model before
        @type old_model:  L{elisa.base_components.model.Model}
        @param new_model: value of model now
        @type new_model:  L{elisa.base_components.model.Model}
        """
        pass


    def handle_input(self, input_event):
        """
        Process an input event.
        It can decide that no further processing should be done by returning
        True or let its parent controller process it by returning False.

        @param input_event:  the input event to translate
        @type input_event:   L{elisa.core.input_event.InputEvent}
        @rtype: bool
        """
        if input_event.action == EventAction.EXIT:
            common.application.stop()
            return True

        return False


    def attribute_set(self, key, old_value, new_value):
        """
        Called when an attribute of the model to which it is connected changes.

        @param key:       attribute changed
        @type key:        string
        @param old_value: value of the attribute before being set; None if
                          attribute was not existing
        @type:            any
        @param new_value: value of the attribute after being set
        @type:            any
        """
        # FIXME: needs to be automatic and generic
        if key == 'loading':
            self.loading = new_value

 
    def create_child_controller(self, model):
        """
        Create a controller given a model and set it up as a child of self.
        The newly created controller is then connected to the model.

        @param model: model for which to create a corresponding controller.
        @type model:  L{elisa.base_components.model.Model}

        @raise TypeError: if controller is not a Controller object
        """
        if isinstance(model, Model):
            plugin_registry = common.application.plugin_registry
            controller_path_list = self.get_controller_paths(model.path)
            if len(controller_path_list) == 0:
                self.warning("Cannot create controller for model %r (with " \
                             "controller %r)" % (model.path, self.path))
                raise ComponentError(model.path)
            else:
                controller = plugin_registry.create_component(controller_path_list[0])
                controller.parent = self
                controller.backend = self.backend
                controller.model = model
                return controller
        else:
            self.warning("Cannot create controller for model %r (with " \
                         "controller %r)" % (model.path, self.path))
            raise TypeError

    def get_controller_paths(self, model_path):
        """
        DOCME
        """
        # FIXME: this is going to be changed to externalise the associations
        # model-controller
        controller_paths = []
        for association in self.default_associations:
            if association[0] == model_path:
                controller_paths.append(association[1])
        return controller_paths
