# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006,2007 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 2.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.

"""
Module responsible for starting the Application
"""

__maintainer__ = 'Philippe Normand <philippe@fluendo.com>'

import pkg_resources
import sys, time
import os
import datetime
import locale

from elisa.core import common
common.boot()

from elisa.core import __version__, version_info, config, log
from elisa.core.utils import exception_hook
from elisa.core.bus import bus, bus_message
from elisa.core.utils import i18n
from elisa.core.utils import classinit
from elisa.core import thumbnailer
from elisa.core import media_uri
from elisa.core.utils.mime_getter import MimeGetter

from elisa.core import plugin_registry, config_upgrader
from elisa.core import input_manager, media_manager, service_manager
from elisa.core import metadata_manager
from elisa.core import interface_controller, player_registry

from twisted.internet import reactor, defer, threads
from twisted.python import usage
from twisted.web import client

from elisa.extern.translation import Translator

UPDATES_URL = "http://elisa.fluendo.com/updates/update.php?install_date=%(date)s&version=%(version)s"
CONFIG_DIR = os.path.join(os.path.expanduser('~'), ".elisa")
CONFIG_FILE = "elisa.conf"
TRANS_FILE = "data/translations.lst"

DEFAULT_CONFIG = """\
[general]
version = '%(version)s'
install_date = '%(install_date)s'
media_providers = ['media_ugly:shoutcast_media', 'fspot:fspot_media', 'coherence_plugin:upnp_media', 'media_bad:ipod_media', 'base:local_media', 'media_good:elisa_media', 'media_good:gnomevfs_media', 'audiocd:audiocd_media', 'flickr:flickr_media', 'stage6:stage_media']
metadata_providers = ['media_good:gst_metadata', 'media_good:taglib_metadata', 'media_good:cover_in_dir', 'media_good:cover_cache', 'media_good:amazon_covers']
service_providers = ['hal:hal_service', 'coherence_plugin:coherence_service']
player_engines = ['base:playbin_engine', 'audiocd:cdda_engine']
backends = ['backend1']
frontends = ['frontend1']

[media_scanner]
enabled = '1'
db_backend = 'sqlite'
database = 'elisa.db'
fivemin_location_updates = []
hourly_location_updates = []
daily_location_updates = []
weekly_location_updates = []
unmonitored_locations = []

[backend1]
activity = 'base:elisa_activity'
controller = 'poblenou:elisa_controller'
input_providers = ['input_good:lirc_input']

[frontend1]
backend = 'backend1'
view = 'poblenou:elisa_view'
theme = 'poblenou:chris_theme'
input_providers = ['pigment:pigment_input']

[base:main_menu_activity]
menu_activities = ['base:audio_activity', 'base:video_activity', 'base:image_activity', 'base:config_activity', 'base:service_activity']

[base:audio_activity]
# the audio media locations
locations = []
cover_files = ['front.jpg', 'cover.jpg']

[base:image_activity]
# the picture media locations
locations = []

[base:video_activity]
# the video media locations
locations = []

[input_good:lirc_input]
# filename of the LIRC config map to use
lirc_rc = 'streamzap.lirc'
delay = '4'
repeat = '1'

[coherence_plugin:coherence_service]
logmode = 'none'
controlpoint = 'yes'

[[plugins]]

[base:service_activity]
# a list of activites, which should beappear in the service_menu
service_activities = ['service:about_activity']

[dvd:dvd_activity]
# uri of the dvd. must be file:///* or dvd://
dvd_uri = 'dvd://'

[theme_switcher:theme_switcher_activity]
# a list of themes 'plugin:component' like 'classic:theme'
themes = ['poblenou:tango_theme', 'poblenou:poblenou_theme', 'poblenou:chris_theme']
[elisa_view]
theme = 'poblenou:chris_theme'"""

                    
class Application(log.Loggable):
    """ Application is the entry point of Elisa. It groups all the necessary
    elements needed for Elisa to run. It is in charge of instantiating a
    Config and a PluginRegistry. Application also provides access to
    input events and data, and holds the user interfaces. It creates
    various managers (InputManager, MediaManager...),
    an InterfaceController and a DBBackend.

    @ivar plugin_registry: the PluginRegistry which loads and manages the plugins
    @type plugin_registry: L{elisa.core.plugin_registry.PluginRegistry}
    @ivar config:         Application's config file, storing options
    @type config:         L{elisa.core.config.Config}
    """

    # Allows property fget/fset/fdel/doc overriding
    __metaclass__ = classinit.ClassInitMeta
    __classinit__ = classinit.build_properties

    log_category = "application"

    def __init__(self, config_filename=None, show_tracebacks=False):
        """
        Application constructor. It:

         - loads global localization
         - loads the config file
         - loads the profiler

        @param config_filename: the config filename to use. Can be absolute
                                or relative path
        @type config_filename:  string or None to use default config file
        """
        log.Loggable.__init__(self)
        self.debug("Creating")

        self.show_tracebacks = show_tracebacks
        self.running = False
        self._plugin_registry = None

        self._config = None

        self._load_config(config_filename)
        self._load_exception_hook()
        self._load_profiler()
        self._compile_po_files()
        
        common.set_application(self)

        self._load_translator()
        
        self._plugin_registry = plugin_registry.PluginRegistry(self.config)
        self._plugin_registry.load_plugins()

        self._bus = bus.Bus()

        self._service_manager = service_manager.ServiceManager()
        self._metadata_manager = metadata_manager.MetadataManager()
        self._media_manager = media_manager.MediaManager(self.metadata_manager)
        self._input_manager = input_manager.InputManager()
        self._player_registry = player_registry.PlayerRegistry()
        self._interface_controller = interface_controller.InterfaceController()
        self._thumbnailer = thumbnailer.Thumbnailer()
        self._mime_getter = MimeGetter()

    def _compile_po_files(self):
        if os.path.exists(TRANS_FILE):
            i18n.compile_translations_from_file(TRANS_FILE)

    def _load_translator(self):
        
        # check the locale is supported        
        try:
            locale.getpreferredencoding()
        except locale.Error, error:
            self.warning(error)
            self.warning("Falling back to system locale")
            #locale.setlocale(locale.LC_ALL, '')
            os.environ['LANG'] = 'C'
            
        self._translator = Translator()

    def _load_config(self, config_filename):
        if not config_filename:
            if not os.path.exists(CONFIG_DIR):
                try:
                    os.makedirs(CONFIG_DIR)
                except OSError, e:
                    self.warning("Could not create '%s': %s" % (CONFIG_DIR, e))
                    raise
                
            config_filename = os.path.join(CONFIG_DIR, CONFIG_FILE)

        self.info("Using config file: %r", config_filename)
        self._config_filename = config_filename
        today = datetime.date.today().isoformat()
        default_config = DEFAULT_CONFIG % {'version': __version__,
                                           'install_date': today}

        try:
            cfg = config.Config(config_filename, default_config=default_config)
        except config.ConfigError, error:
            self.warning(error)
            raise

        if not cfg.first_load:
            # ok we might have an old config format here
            upgrader = config_upgrader.ConfigUpgrader(cfg, default_config)
            cfg = upgrader.update_for(version_info)
            
        self._install_date = cfg.get_option('install_date', section='general',
                                            default=today)
        self._config = cfg

    def _load_profiler(self):
        """
        This imports profiling modules (TODO)
        """
        enable = int(self.config.get_option('enable_profiling', default='0'))
        if enable:
            # TODO: import profiling modules and set things ready
            pass

    def _check_updates(self):
        url = UPDATES_URL % {'date': self._install_date, 'version': __version__}
        dfr = client.getPage(url)

        def got_result(result):
            # TODO: notify user about new version
            pass
        
        dfr.addCallback(got_result)
        
    def _load_exception_hook(self):
        """ Override the default system exception hook with our own
        """
        # FIXME: should this be in config?
        logdir = '/tmp'

        params = {'format': 'text', 'logdir': logdir,
                  'file': sys.stderr,
                  'display': self.show_tracebacks}
        self._except_hook = exception_hook.ExceptionHook(**params)
        sys.excepthook = self._except_hook


    # FIXME: given the current use of this method, it should probably be called
    # something like log error to file
    def handle_traceback(self):
        """ Call this to force the exception hook to handle the exception on top
        of the stack. This can be handy to use from and except: block.
        """
        self._except_hook.handle()

    def plugin_registry__get(self):
        """
        PluginRegistry instance accessor.

        @returns: L{elisa.core.plugin_registry.PluginRegistry}
        """
        return self._plugin_registry

    def bus__get(self):
        """
        L{elisa.core.bus.bus.Bus}
        """
        return self._bus

    def translator__get(self):
        """the translator instance accessor
        rtype: L{elisa.extern.translator.Translator}
        """
        return self._translator

    def metadata_manager__get(self):
        """ MetadataManager instance accessor

        @rtype: L{elisa.core.metadata_manager.MetadataManager}
        """
        return self._metadata_manager

    def mime_getter__get(self):
        """ MimeGetter instance accessor

        @rtype: l{elisa.core.utils.mime_getter.MimeGetter}
        """
        return self._mime_getter

    def service_manager__get(self):
        """ ServiceManager instance accessor

        @rtype: L{elisa.core.service_manager.ServiceManager}
        """
        return self._service_manager


    def player_registry__get(self):
        """ PlayerRegistry instance accessor

        @rtype: L{PlayerRegistry}
        """
        return self._player_registry

    def interface_controller__get(self):
        """ InterfaceController instance accessor

        @rtype: L{elisa.core.interface_controller.InterfaceController}
        """
        return self._interface_controller

    def input_manager__get(self):
        """ InputManager instance accessor

        @rtype: L{elisa.core.input_manager.InputManager}
        """
        return self._input_manager

    def media_manager__get(self):
        """ MediaManager instance accessor

        @rtype: L{elisa.core.media_manager.MediaManager}
        """
        return self._media_manager

    def thumbnailer__get(self):
        """ Thumbnailer instance accessor

        @rtype: L{elisa.core.thumbnailer.Thumbnailer}
        """
        return self._thumbnailer

    def config__get(self):
        """ Config instance accessor

        @returns: L{elisa.core.config.Config}
        """
        return self._config

    def initialize(self):
        self._service_manager.initialize()
        self._metadata_manager.initialize()
        self._media_manager.initialize()
        self._input_manager.initialize()
        self._player_registry.initialize()
        self._interface_controller.initialize()

        self.bus.send_message(bus_message.ComponentsLoaded())

    def start(self):
        """ Execute the application. Start the Managers and the
        InterfaceController.
        """
        threads.deferToThread(self._check_updates)
        
        self.running = True
        self.info("Starting")
        self.bus.start()
        self.input_manager.start()
        self.media_manager.start(seconds=10,resume_scan=True)
        self.service_manager.start()
        self.interface_controller.start()


    def restart(self):
        return self.stop(restart=True)

    def stop(self, stop_reactor=True, restart=False):
        """ Close the application. Good idea to save the configuration
        here, since it's probable it has been updated.

        @param restart: tell the application to restart or not
        @type restart:  bool
        @rtype:         L{twisted.internet.defer.Deferred}
        """

        def stopped(r):
            self.player_registry.deinitialize()
            self.thumbnailer.stop()
            self.service_manager.stop()
            self.media_manager.stop()
            self.input_manager.stop()
            self.bus.stop()

            self.running = False

            if self.config:
                self.config.write()

            if restart:
                self.info("Restarting")
                self.running = True
                args = sys.argv
                args.insert(0, sys.executable)
                if reactor.running:
                    reactor.disconnectAll()
                    reactor.stop()
                os.spawnvpe(os.P_WAIT, sys.executable, args, os.environ)

            if stop_reactor and reactor.running:
                reactor.stop()

        if self.running:
            self.info("Stopping")
            dfr = self.interface_controller.stop()
            dfr.addCallback(stopped)
        else:
            dfr = defer.succeed(None)
            
        return dfr

class Options(usage.Options):
    """
    Application's command-line options definitions
    """

    optFlags = [['version', '', 'show elisa version'],
                ['twisted-version', '', 'show twisted version'],
                ['tracebacks', 't', 'display tracebacks'],
                ]

    tracebacks = False

    def parseArgs(self, config_file=None):
        self['config_file'] = config_file

    def opt_twisted_version(self):
        return usage.Options.opt_version(self)

    def opt_version(self):
        print 'Elisa version %s' % __version__
        sys.exit(0)


def main(args=None):
    """ Parse command-line options and start a new Application in the
    Twisted's reactor.
    """
    if not args:
        args = sys.argv

    options = Options()
    try:
        options.parseOptions(args[1:])
    except usage.UsageError, errortext:
        print '%s: %s' % (args[0], errortext)
        print '%s: Try --help for usage details.' % (args[0])
        sys.exit(1)

    try:
        app = Application(options['config_file'],
                          show_tracebacks=options['tracebacks'])
    except Exception, exc:
        print 'Error', exc
        if options['tracebacks']:
            raise
    else:
        app.initialize()
        app.start()
        reactor.run()
        app.stop()
        reactor.stop()
