# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006,2007 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 2.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.


__maintainer__ = 'Florian Boucault <florian@fluendo.com>'


from elisa.extern.log import log

class Observable(log.Loggable, object):
    """Observable objects can be observed by multiple
    L{elisa.core.observers.observer.Observer} instances, each of them
    getting notified of any changes occuring to the former.
    """

    _observers = []

    def __init__(self):
        self._observers = []

    def add_observer(self, observer):
        """Attach an observer which will then be notified of all the changes
        applied to the observable.

        @param observer: observer to attach
        @type observer:  L{elisa.core.observers.observer.Observer}
        """
        self._observers.append(observer)

    def remove_observer(self, observer):
        """Detach an observer which will not be notified anymore of changes
        applied to the observable.

        @param observer: observer to detach
        @type observer:  L{elisa.core.observers.observer.Observer}
        """
        self._observers.remove(observer)

    def __setattr__(self, key, new_value):
        try:
            old_value = object.__getattribute__(self, key)
        except:
            old_value = None
        object.__setattr__(self, key, new_value)

        # ignore private attributes (those starting with '_')
        if key[0] != '_':
            # notifies all the attached observers
            for observer in self._observers:
                observer.attribute_set(key, old_value, new_value)
