# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006,2007 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 2.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.

"""
Plugin base class, the mother of all Elisa plugins
"""

__maintainer__ = 'Philippe Normand <philippe@fluendo.com>'

from elisa.core.utils import classinit, misc
from elisa.core import log
from elisa.core.config import Config
from elisa.core import component

import os
import inspect

class Plugin(log.Loggable):
    """
    Plugin Base Class

    This class acts as a L{elisa.core.component.Component} factory. All the Plugin's class
    variables can be stored in a plugin.conf file which has the
    following syntax::

        [general]
        name="base"
        version="0.1"
        plugin_dependencies=["foo"]
        external_dependencies=["pgm"]
        description="base elisa components"

        # that's a component!
        [media_providers.local_media:LocalMedia]
        description="To access file:// media"
        platforms=["linux", "posix"]
        external_dependencies=["gnomevfs",]
        component_dependencies = ["base:coherence_service"]

    Each option of the "general" section will be mapped to it's corresponding
    class variable in the Plugin.

    @cvar config_file:           Plugin's directory relative path to plugin.conf
    @type config_file:           string
    @cvar config:                Config generated by reading L{config_file}
    @type config:                L{elisa.core.config.Config}
    @cvar components:            Component implementations provided by the
                                 Plugin
    @type components:            dict mapping L{elisa.core.component.Component}
                                 names to L{elisa.core.component.Component}
                                 instances
    @cvar name:                  plugin's name. Should be unique
    @type name:                  string
    @cvar external_dependencies: external Python dependencies specified using
                                 dot-syntax
    @type external_dependencies: list
    @cvar plugin_dependencies:   Elisa plugins dependencies specified using
                                 colon-syntax
    @type plugin_dependencies:   list
    @cvar version:               Plugin's version as a dot-syntax string
    @type version:               string
    @cvar description:           one line Plugin's description
    @type description:           string
    @cvar directory:             absolute path to the directory storing the
                                 Plugin's source code file (or Egg)
    @type directory:             string
    """

    __metaclass__ = classinit.ClassInitMeta
    __classinit__ = classinit.build_properties

    config = None
    config_file = None
    
    directory = ""
    components = {}
    name = ""
    plugin_dependencies = []
    external_dependencies = []
    version = ""
    description = ""

    def __init__(self):
        # configure the log category based on my name
        self.log_category = self.name

        log.Loggable.__init__(self)
        self.debug("Creating")

    @classmethod
    def initialize(cls):
        """
        Check Plugin's python dependencies and check each Component of
        the Plugin. Invalid components are removed from the Plugin.

        @raise InitializeFailure: if the Plugin is not well structured
        @raise UnMetDependency:   if the plugin has un-met dependencies
        @rtype:                   list of L{ComponentError}
        """
        log.debug(cls.name, "Initializing")
        errors = []
        invalid_components = []

        # check plugin's python dependencies
        component.check_python_dependencies(cls.name,
                                            cls.external_dependencies)

        if not isinstance(cls.components, dict):
            reason = "%s.components must be a dictionnary" % cls.__name__
            raise component.InitializeFailure(cls.name, reason)

        # check platform and python dependencies for each component
        for component_name, informations in cls.components.iteritems():
            if component_name == cls.name:
                reason = "Component %s has the same name as its plugin. "\
                         "Please rename it." % component_name
                raise component.InitializeFailure(cls.name, reason)
            
            platforms = informations.get('platforms',[])
            deps = informations.get('external_dependencies',[])

            try:
                component.check_platforms(component_name, platforms)
                component.check_python_dependencies(component_name, deps)
            except component.ComponentError, error:
                errors.append(error)
                invalid_components.append(component_name)

        # remove invalid components
        for component_name in invalid_components:
            del cls.components[component_name]

        return errors

    @classmethod
    def load_config(cls):
        """ Load the L{config_file} if defined

        Fill L{config} class variable if L{config_file} points to a
        valid config filename. Also fill L{name}, L{version},
        L{description}, L{plugin_dependencies},
        L{external_dependencies} and L{components}.
        """
        components = {}
                
        if cls.config:
            log.debug(cls.name, "Config already loaded")
        elif cls.config_file:
            log.debug(cls.name, "Loading config from file %r", cls.config_file)
            plugin_dir = os.path.dirname(inspect.getsourcefile(cls))
            config_file = os.path.join(plugin_dir, cls.config_file)
            cls.config = Config(config_file)
            general_section = cls.config.get_section('general',{})

            name = general_section.get('name','')
            if name:
                cls.name = name

            version = general_section.get('version','')
            if version:
                cls.version = version

            description = general_section.get('description','')
            if description:
                cls.description = description

            deps = general_section.get('plugin_dependencies',[])
            if deps:
                cls.plugin_dependencies = deps

            ext_deps = general_section.get('external_dependencies',[])
            cls.external_dependencies = ext_deps

            sections = cls.config.as_dict()
            del sections['general']

            # scan components
            for component_path, section in sections.iteritems():
                if 'name' not in section:
                    path = component_path.split(':')
                    if len(path) < 2:
                        continue
                    path = path[1]
                    un_camel = misc.un_camelify(path)
                    section['name'] = un_camel
                component_name = section['name']
                section['path'] = component_path

                components.update({component_name:section})

            if components:
                cls.components = components
        else:
            cls.config = Config()

    @classmethod
    def load_translations(cls, translator):
        """ Load the translation files supported by the Plugin into a
        translator, usually the Application's translator. This method
        uses the config class attribute, so be sure to have called
        L{load_config} before calling this method. Translation files
        are loaded from the i18n plugin.conf section (inside general
        section).

        @param translator: The translator to load i18n files into
        @type translator:  L{elisa.extern.translation.Translator}
        """
        i18n = cls.config.get_option('i18n', section='general', default={})

        for key, value in i18n.iteritems():
            trans_path = os.path.join(cls.directory, value)
            translator.addLocaleDir(key, trans_path)
            log.debug(cls.name, "Adding %s to domain %s", key, trans_path)
