# Elisa - Home multimedia server
# Copyright (C) 2006,2007 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 2.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.

from twisted.trial import unittest
from elisa.core.tests.elisa_test_case import ElisaTestCase
from elisa.core import plugin_registry
from elisa.core import config
from elisa.core import plugin
from elisa.base_components import activity
import inspect, os
import tempfile, shutil

class SomeActivity(activity.Activity):
    name = 'some_activity'

class SomePlugin(plugin.Plugin):
    name = 'test'
    components = {'some_activity':{'path':SomeActivity}}

class FooPlugin(plugin.Plugin):
    name = 'foo'

SOME_PLUGIN="""\
from elisa.core import plugin
from elisa.base_components import activity

class SomeActivity(activity.Activity):
    name = 'some_activity'

class SomePlugin(plugin.Plugin):
    name = 'test'
    components = {'some_activity':{'path':SomeActivity}}
"""

FOO_PLUGIN="""\
from elisa.core import plugin

class FooPlugin(plugin.Plugin):
    name = 'foo'
"""

class TestPluginRegistry(ElisaTestCase):


    def setUp(self):
        ElisaTestCase.setUp(self)
        self._plugins_dir = tempfile.mkdtemp()

    def tearDown(self):
        ElisaTestCase.tearDown(self)
        shutil.rmtree(self._plugins_dir)


    def test_plugin_registration(self):
        """ Check plugin classes registering/unregistering works
        """
        manager = plugin_registry.PluginRegistry(config.Config('empty.conf'))

        self.assertEquals(manager.plugins, {})
        self.assertEquals(manager.plugin_classes, {})

        manager.register_plugin(SomePlugin)
        self.assertEquals(manager.plugin_classes, {'test': SomePlugin})

        manager.unregister_plugin(SomePlugin)
        self.assertEquals(manager.plugin_classes, {})

    def test_plugin_loading(self):
        """ Check the plugins declared in the config are loaded
        """
        conf = config.Config('empty.conf')
        os.environ['ELISA_PLUGIN_PATH'] = self._plugins_dir
        section = {'plugins':['test', 'foo']}
        conf.set_section('plugin_registry', section)

        init = os.path.join(self._plugins_dir, '__init__.py')
        open(init,'w').close()


        # write plugins code to files
        f = open(os.path.join(self._plugins_dir, 'foo.py'),'w')
        f.write(FOO_PLUGIN)
        f.close()

        f = open(os.path.join(self._plugins_dir, 'some.py'),'w')
        f.write(SOME_PLUGIN)
        f.close()

        manager = plugin_registry.PluginRegistry(conf)

        # load plugins from files
        manager.load_plugins()

        self.assertEquals(manager.plugins, {})
        plugins = manager.plugin_classes.keys()
        self.failUnless('test' in plugins)
        self.failUnless('foo' in plugins)

        test_plugin = manager.plugin_classes['test']
        foo_plugin = manager.plugin_classes['foo']
        self.assertEquals(test_plugin.__name__, 'SomePlugin')
        self.assertEquals(foo_plugin.__name__, 'FooPlugin')

    def test_plugin_instantiation(self):
        """ Check Plugin instances are lazily referenced by the manager
        """
        conf = config.Config('empty.conf')
        manager = plugin_registry.PluginRegistry(conf)
        manager.register_plugin(SomePlugin)

        plugin1 = manager.get_plugin_with_name('test')
        plugin2 = manager.get_plugin_with_name('test')

        self.failUnless(isinstance(plugin1, SomePlugin))
        self.failUnless(plugin1 == plugin2)

    def test_component_creation(self):
        """ Check Components are correctly instantiated by the manager
        """
        conf = config.Config('empty.conf')
        conf.set_option('activities', ['test:some_activity',],
                        'plugin_registry')
        manager = plugin_registry.PluginRegistry(conf)
        manager.register_plugin(SomePlugin)

        activities = manager.create_components(['test:some_activity',])
        self.assertEqual(len(activities), 1)
        self.failUnless(isinstance(activities[0], activity.Activity))
        self.failUnless(isinstance(activities[0], SomeActivity))

    def test_multiple_component_creation(self):
        """ Check the manager can create multiple instance of the same component
        """
        conf_activities = ['test:some_activity:1',
                           'test:some_activity:2']
        conf = config.Config('empty.conf')
        conf.set_option('activities', conf_activities,
                        'plugin_registry')
        manager = plugin_registry.PluginRegistry(conf)
        manager.register_plugin(SomePlugin)

        activities = manager.create_components(conf_activities)
        self.assertEqual(len(activities), 2)
        index = 1
        for component in activities:
            self.assertEqual(component.id, index)
            self.failUnless(isinstance(component, activity.Activity))
            self.failUnless(isinstance(component, SomeActivity))
            index += 1

    def test_create_component(self):
        """
        """
        conf = config.Config('empty.conf')
        conf.set_option('activities', ['test:some_activity:1',
                                       'test:some_activity:2',
                                       ],
                        'plugin_registry')
        manager = plugin_registry.PluginRegistry(conf)
        manager.register_plugin(SomePlugin)

        act = manager.create_component('test:some_activity')
        self.failUnless(act)
        self.failUnless(isinstance(act, activity.Activity))
        self.failUnless(isinstance(act, SomeActivity))

    def test_get_component_class(self):
        """
        """
        conf = config.Config('empty.conf')
        conf.set_option('activities', ['test:some_activity:1',
                                       'test:some_activity:2',
                                       ],
                        'plugin_registry')
        manager = plugin_registry.PluginRegistry(conf)
        manager.register_plugin(SomePlugin)

        class App:
            pass

        app = App()
        app.plugin_registry = manager

        from elisa.core import common
        common.set_application(app)

        c = plugin_registry.get_component_class('test:some_activity')
        self.failUnless(c)
        self.failUnless(issubclass(c, activity.Activity))
        self.assertEqual(c, SomeActivity)


    def test_split_component_path(self):
        """
        """
        conf = config.Config('empty.conf')
        conf.set_option('activities', ['test:some_activity:1',
                                       'test:some_activity:2',
                                       ],
                        'plugin_registry')
        manager = plugin_registry.PluginRegistry(conf)

        self.assertEqual((), manager._split_component_path('Invalid'))
        self.assertEqual(('A', 'B', 0, 'A:B'), manager._split_component_path('A:B'))
        self.assertEqual(('A', 'B', 1, 'A:B:1'), manager._split_component_path('A:B:1'))
        self.assertEqual((), manager._split_component_path('A:B:c'))

    def test_get_plugin_path(self):
        """
        """
        conf = config.Config('empty.conf')
        manager = plugin_registry.PluginRegistry(conf)
        manager.register_plugin(SomePlugin)

        path = manager.get_plugin_path('test')
        abspath = os.path.dirname(inspect.getsourcefile(self.__class__))
        self.assertEqual(abspath, path)
