# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006,2007 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 2.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.


__maintainer__ = 'Lionel Martin <lionel@fluendo.com>'


from elisa.base_components.view import View
from elisa.core import common
from elisa.core import player
from twisted.internet import reactor
from elisa.core.input_event import *

try:
    from dvd_player import *
    FLUENDO_DVD_INSTALLED = True
except:
    FLUENDO_DVD_INSTALLED = False

class DvdPlayerView(View):
    """
    This class implements a view which can create a
    Player instance and manage it.
    This class can be inherited to provide the
    toolkit dependent display, as well as the
    GStreamer video and audio sinks
    """

    default_config={'gpl': 0}
    config_doc = {'gpl': 'wether or not to use GPL GStreamer elements'}
                    

    def initialize(self):
        View.initialize(self)
        if FLUENDO_DVD_INSTALLED:
            gpl = int(self.config.get('gpl', '0'))
            self.player = ElisaDVDPlayer(gpl)
            self.player.in_menu.connect(self._in_menu)
        else:
            self.player = common.application.player_registry.create_player()
            
    def _in_menu(self, value):
        self.controller.model.in_menu = value
        
    def attribute_set(self, key, old_value, new_value):
        if key == 'state':
            if new_value == player.STATES.PLAYING:
                self.state_changed(new_value)
                self.player.play()
                if self.player.is_menu and FLUENDO_DVD_INSTALLED:
                    self.player.fire_action(DVD_ACTION_MENU_ROOT_OR_RESUME)
            elif new_value == player.STATES.PAUSED:
                self.state_changed(new_value)
                self.player.pause()
            elif new_value == player.STATES.STOPPED:
                self.state_changed(new_value)
                self.player.stop()
        elif key == 'uri':
            self.player.load_uri(new_value)
            self.uri_changed(new_value)
        elif key == 'seek_to':
            self.seek_to_changed(new_value)
        elif key == 'volume':
            self.player.volume = new_value
            self.volume_changed(new_value)
        elif key == 'muted':
            self.player.muted = new_value
        elif key == 'last_navigation_action' and FLUENDO_DVD_INSTALLED:
            if new_value == EventAction.GO_UP:
                self.player.fire_action(DVD_ACTION_BUTTON_UP)
            
            elif new_value == EventAction.GO_DOWN:
                self.player.fire_action(DVD_ACTION_BUTTON_DOWN)
            
            elif new_value == EventAction.GO_LEFT:
                if self.player.is_menu:
                    self.player.fire_action(DVD_ACTION_BUTTON_LEFT)
                else:
                    self.player.fire_action(DVD_ACTION_SEARCH_PREV_PG)
            
            elif new_value == EventAction.GO_RIGHT:
                if self.player.is_menu:
                    self.player.fire_action(DVD_ACTION_BUTTON_RIGHT)
                else:
                    self.player.fire_action(DVD_ACTION_SEARCH_NEXT_PG)
            
            elif new_value == EventAction.DVD_MENU:
                if not self.player.is_menu:
                    self.player.fire_action(DVD_ACTION_MENU_ROOT_OR_RESUME)
            
            elif new_value == EventAction.OK:
                self.player.fire_action(DVD_ACTION_BUTTON_ACTIVATE)

    def dvd_menu_visible_changed(self, is_visible):
        self.player.fire_action(DVD_ACTION_MENU_ROOT_OR_RESUME)


    def state_changed(self, state):
        self.player.volume = self.controller.model.volume
        self.player.muted = self.controller.model.muted
        self.controller.model.duration = self.player.duration
        self._update_model_position()

    def _update_model_position(self):
        if self.controller.model.duration != self.player.duration:
            self.controller.model.duration = self.player.duration
        self.controller.model.position = self.player.position
        if self.controller.model.state == player.STATES.PLAYING:
            reactor.callLater(0.5, self._update_model_position )

    def uri_changed(self, uri):
        pass

    def seek_to_changed(self, position):
        pass

    def volume_changed(self, volume):
        pass

    def controller_changed(self):
        if self.controller.uri != None:
            self.player.load_uri(self.controller.uri)

            state = self.controller.state
            if state == player.STATES.PLAYING:
                self.player.play()
            elif state == player.STATES.PAUSED:
                self.player.pause()
            elif state == player.STATES.STOPPED:
                self.player.stop()


    def _end_of_stream(self, uri):
        self.controller.model.state = player.STATES.STOPPED

