# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006,2007 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 2.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.


__maintainer__ = 'Philippe Normand <philippe@fluendo.com>'
__maintainer2__ = 'Florian Boucault <florian@fluendo.com>'


from menu_activity import MenuActivity
from elisa.core.observers.list import ListObserver, ListObservable
from elisa.core.observers.dict import DictObserver, DictObservable
from elisa.core.media_manager import MediaProviderNotFound
from elisa.core import common, log
from elisa.core.media_uri import MediaUri
from elisa.core.utils import misc
from elisa.core.bus import bus_message
from elisa.base_components.media_provider import NotifyEvent
from twisted.internet import defer

import os, re, time

from elisa.extern.translation import gettexter, N_

T_ = gettexter('elisa-base')


class MetadataObserver(DictObserver):

    def __init__(self, model):
        DictObserver.__init__(self)
        self._model = model

    def modified(self, key, value):
        self._model.activity.metadata_changed(self._model, key, value)

class FilteringUriObserver(ListObserver):

    def __init__(self, model):
        ListObserver.__init__(self)
        self._model = model

    def inserted(self, elements, position):
        def insert_model(model):
            if model != None:
                if not self._model.has_children:
                    self._model.has_children = True
                    self._model.loading = False
                    self._model.activity.set_directory_actions(self._model,
                                                               self._model.uri)
                index = self._model.activity.files_index_start+position
                self._model.insert(index, model)
            else:
                if self._model.loading:
                    self._model.loading = False

        for elt in elements:
            time.sleep(0.01)
            dfr = self._model.activity.create_menu_for_uri(elt,
                                                           parent=self._model)
            dfr.addCallback(insert_model)

    def removed(self, elements, position):
        index = self._model.activity.files_index_start+position
        self._model[index:index+len(elements)] = []

        if len(self._model) == self._model.activity.files_index_start:
            self._model[:] = []
            self._model.has_children = False


class MediaMenuActivity(MenuActivity):
    """

    Notes:

     - MediaMenuActivities should declare in their default config a
       "locations" (type: list) option
     - All menu models created by this activity:

       * have their icon name prefixed with the media_type value and a dash

    Example: the "Folders" menu of the AudioActivity would have the
    "audio-by-folder" icon

    @cvar menu_name:      Media menu name, should be human readable
    @type menu_name:      string
    @cvar menu_icon_name: Menu icon name
    @type menu_icon_name: string
    @cvar media_types:    Media types handled by the activity
    @type media_type:     list
    @ivar root_menu:      MenuModel representing the menu tree of the activity
    @type root_menu:      L{elisa.plugins.base.models.menu_model.MenuModel}
    @ivar loadmore_callbacks: the loadmore callbacks for the menu_models handled by the activity
    @type loadmore_callbacks: dict mapping menu_models to callables
    """

    slideshow_model = None
    player_model = None

    menu_name = ""
    menu_icon_name = ""
    media_types = []
    xdg_var_name = ''

    def __init__(self):
        MenuActivity.__init__(self)
        self.files_index_start = 0
        self.root_menu = None
        self.loadmore_callbacks = {}
        self._media_locations = []
        self._dynamic_sources = []
        self.model_cache = {}
        self.meta_observers = {}
        self.byfolders_cache = []

    def initialize(self):
        MenuActivity.initialize(self)
        bus = common.application.bus
        bus.register(self._got_bus_message, bus_message.MediaLocation)

        self.root_menu = MenuActivity._create_menu_model(self,
                                                         self.menu_name,
                                                         self.menu_icon_name)

        self._by_folder = self._create_menu_model(T_(N_("Folders")),
                                                  "by_folder")
        self._by_location = self._create_menu_model(T_(N_("Home network")),
                                                    "browse_by_network")
        self._by_internet = self._create_menu_model(T_(N_("Internet")),
                                                    "browse_by_internet")

        self.loadmore_callbacks = {self._by_folder: self._loadmore_by_folder}

        self._load_media_locations()

    # Activity implementation

    def get_model(self):
        self.root_menu.activity = self
        return self.root_menu


    def set_directory_actions(self, menu_model, uri):
        """
        Add some menu_items at the beginning of the directory
        folder. Each of these menu_items can have customized activate
        actions.

        This method needs to update L{file_index_start} instance
        variable so that the activity knows when action items ends in
        the children list.

        @param menu_model: the directory model to add actions in
        @type menu_model: L{elisa.plugins.base.models.menu_node_model.MenuNodeModel}
        @param uri: directory URI
        @type uri: L{elisa.core.media_uri.MediaUri}
        """
        menu_model.uri = uri
 
        registry = common.application.plugin_registry
        action = registry.create_component('base:enqueue_action')
        action.uri = uri
        action.player_model = self.player_model
        action.media_types = self.media_types
        action_model = self._create_menu_model(T_(N_('Play all')), 'all')
        action_model.uri = uri
        action_model.activate_action = action
        menu_model.insert(0, action_model)

        self.files_index_start = 1
 
    def set_activate_action(self, menu_model, uri, file_type):
        """
        Add an activate action to menu items that are not directories.

        @param menu_model: the model to add an action in
        @type menu_model: L{elisa.plugins.base.models.menu_node_model.MenuNodeModel}
        @param uri: file URI
        @type uri: L{elisa.core.media_uri.MediaUri}
        @param file_type: the type of the file
        @type file_type: string
        """
        registry = common.application.plugin_registry
        action = registry.create_component('base:play_action')
        action.uri = uri
        action.media_type = file_type
        action.player_model = self.player_model
        menu_model.activate_action = action

    # MenuActivity implementation

    def loadmore(self, model):
        dfr = None
        callback = self.loadmore_callbacks.get(model)
        if callback == None:
            callback = self._loadmore_for_model
        if callback != None:
            self.debug("Calling %r", callback)
            dfr = callback(model)

        return dfr

    def set_hover_action(self, menu_model, uri, file_type='directory'):
        if uri:
            registry = common.application.plugin_registry

            if file_type != 'directory':
                action_type = 'base:preview_play_action'

                action = registry.create_component(action_type)
                action.uri = uri
                action.player_model = self.player_model
                menu_model.hover_action = action

    def _create_menu_model(self, text, icon_name, thumbnail_source=None,
                           parent=None):
        if self.menu_icon_name:
            icon_name = "%s_%s" % (self.menu_icon_name, icon_name)
        return MenuActivity._create_menu_model(self, text, icon_name,
                                               thumbnail_source, parent)

    def create_menu_for_uri(self, uri_with_info, parent=None, icon=None):
        """
        @param uri_with_info:   the URI and its optional metadata
        @type uri_with_info:    tuple (MediaUri, dict)
        """
        media_manager = common.application.media_manager
        uri = uri_with_info[0]

        def create(media_type, uri_with_info, icon):
            metadata = uri_with_info[1]
            supported_types = ['directory',] + self.media_types

            file_type = media_type['file_type']

            if file_type != None and file_type not in supported_types:
                self.debug("%s of type %r is an invalid media for this \
                            activity" % (uri, file_type))
                return None

            if not icon:
                if file_type == "directory":
                    icon = "directory"
                else:
                    icon = "file"

            if not metadata.has_key('default_image'):
                if file_type == "directory" or file_type == "audio":
                    thumbnail_source = None
                else:
                    thumbnail_source = uri
                metadata['default_image'] = None
            else:
                thumbnail_source = metadata['default_image']

            menu_model = self._create_menu_model(uri.label, icon,
                                                 thumbnail_source,
                                                 parent=parent)
            menu_model.uri = uri

            if file_type != 'directory':
                self.set_activate_action(menu_model, uri, file_type)

            self._observe_metadata_dict(menu_model, metadata)

            metadata_manager = common.application.metadata_manager
            metadata_manager.get_metadata(metadata)

            def update_model(has_children):
                menu_model.has_children = has_children
                return menu_model

            self.set_hover_action(menu_model, uri, file_type)

            dfr = media_manager.has_children_with_types(uri,
                                                        supported_types)
            dfr.addCallback(update_model)
            return dfr

        dfr = media_manager.get_media_type(uri)
        dfr.addCallback(create, uri_with_info, icon)
        return dfr

    # Private methods

    def _got_bus_message(self, message, sender):
        root_location_messages = (bus_message.DeviceAction,
                                  bus_message.ForeignApplication)
        
        def got_model(menu_model):
            if menu_model == None:
                return

            menu_model.has_children = True

            self._dynamic_sources.append(menu_model)
            if isinstance(message, bus_message.DeviceAction):
                self.root_menu.insert(0, menu_model)
                self.root_menu.has_children = True

            elif isinstance(message, bus_message.ForeignApplication):
                # insert right after "Folders" if existing
                if self._by_folder in self.root_menu:
                    index = self.root_menu.index(self._by_folder)+1
                else:
                    index = 0
                self.root_menu.insert(index, menu_model)
                self.root_menu.has_children = True

            elif isinstance(message, bus_message.InternetLocation):
                if self._by_internet not in self.root_menu:
                    self.root_menu.append(self._by_internet)
                    self.root_menu.has_children = True

                self._by_internet.append(menu_model)
                self._by_internet.has_children = True

            elif isinstance(message, bus_message.LocalNetworkLocation):
                if self._by_location not in self.root_menu:
                    self.root_menu.append(self._by_location)
                    self.root_menu.has_children = True

                self._by_location.append(menu_model)
                self._by_location.has_children = True

            else:
                self.warning("Don't know where to store model: %r", menu_model)
                
        if set(self.media_types).intersection(set(message.media_types)):
            uri = MediaUri(message.mount_point)
            uri.label = message.name
            ActionType = bus_message.MediaLocation.ActionType
            if message.action == ActionType.LOCATION_ADDED:

                if message.fstype in ('daap', 'upnp'):
                    icon = "computer"
                elif message.fstype == 'file':
                    # FIXME: how can fstype be file? and why should it trigger
                    # the usb icon?
                    icon = "usb"
                else:
                    icon = message.fstype

                uri_with_info = (uri, {})
                try:
                    dfr = self.create_menu_for_uri(uri_with_info, icon=icon)
                except MediaProviderNotFound, error:
                    self.warning(error)
                else:
                    dfr.addCallback(got_model)
                    media_manager = common.application.media_manager
                    media_manager.add_source(uri, self.media_types)

            elif message.action == ActionType.LOCATION_REMOVED:
                if str(uri) in self.model_cache.keys():
                    children, filter_uri, dfr = self.model_cache[str(uri)]
                    del self.model_cache[str(uri)]
                    dfr.pause()
                    filter_uri.stop_observing()
                menu_model = None
                for model in self._dynamic_sources:
                    if model.uri == uri:
                        menu_model = model
                        self._dynamic_sources.remove(menu_model)
                        for child in menu_model[:]:
                            menu_model.remove(child)
                        if isinstance(message, root_location_messages):
                            self.root_menu.remove(menu_model)
                        else:
                            self._by_location.remove(menu_model)
                            if len(self._by_location) == 0:
                                self._by_location.has_children = False
                                self.root_menu.remove(self._by_location)
                        break

                children_count = len(self.root_menu)
                self.root_menu.has_children = children_count > 0

    def _load_xdg_env(self):
        config_home = os.path.expanduser(os.getenv('XDG_CONFIG_HOME',
                                                   '~/.config'))
        user_dirs_file = os.path.join(config_home, 'user-dirs.dirs')
        if os.path.exists(user_dirs_file):
            variables = filter(lambda i: i.startswith('XDG'),
                               os.environ.keys())
            if self.xdg_var_name not in variables:
                self.debug('XDG variables not found in os.environ, loading %r',
                           user_dirs_file)
                key_val_re = re.compile('(.*)="(.*)"')
                for line in open(user_dirs_file).readlines():
                    match = key_val_re.search(line)
                    if match and not match.groups()[0].startswith('#'):
                        var, value = match.groups()
                        self.debug('Found XDG variable %r, injecting in os.environ',
                                   var)
                        value = misc.env_var_expand(value)
                        os.environ[var] = value
            else:
                self.debug("XDG variables %r already in os.environ",
                           variables)
                
    def _load_media_locations(self):
        self._load_xdg_env()
        
        # check XDG variables
        if self.xdg_var_name and self.xdg_var_name in os.environ:
            local_directory = os.environ[self.xdg_var_name]
            location_uri = MediaUri('file://%s' % local_directory)
            self.add_location(location_uri)
        else:
            self.warning("XDG user-dir disabled")
            
        for location in self.config.get('locations', []):
            # TODO: handle /*

            location_options = self.config.get(location, {})

            if location.endswith('/'):
                location = location[:-1]
                
            location_uri = MediaUri(location.decode('utf-8'))
            if 'label' in location_options:
                location_uri.label = location_options['label']
                
            self.add_location(location_uri)

    def add_location(self, location_uri):
        media_manager = common.application.media_manager

        def got_result(is_directory):
            ok = True
            if not is_directory:
                handle = media_manager.blocking_open(location_uri)
                if handle is None:
                    ok = False
                else:
                    handle.blocking_close()
            if ok:
                media_manager.add_source(location_uri, self.media_types)
                self.info("Adding location: %r", location_uri)
                self._media_locations.append(location_uri)
                self._by_folder.has_children = len(self._media_locations) > 0
                if self._by_folder.has_children and \
                   self._by_folder not in self.root_menu:
                    self.root_menu.insert(0, self._by_folder)
                    self.root_menu.has_children = True

        try:
            dfr = media_manager.is_directory(location_uri)
            dfr.addCallback(got_result)
        except MediaProviderNotFound, error:
            self.warning(error)
            
    def _start_monitoring(self, monitored_uri, children_with_info, model):
        media_manager = common.application.media_manager
        if media_manager.uri_is_monitorable(monitored_uri):
            media_manager.monitor_uri(monitored_uri, self._handle_media_event,
                                      children_with_info)
        return children_with_info

    def _handle_media_event(self, uri, metadata, event, monitored_uri,
                            children):
        self.debug('Monitoring event received in %s : %s %s' % (monitored_uri,
                                                                uri, event))
        media_manager = common.application.media_manager

        # FIXME: hack 'cause "file:///uri/" != "file:///uri"
        parent_uri = MediaUri(uri.parent[:-1])

        if parent_uri == monitored_uri:
            if event == NotifyEvent.ADDED:
                children.insert(0, (uri, metadata))
            elif event == NotifyEvent.REMOVED:
                found = None
                for infos in children:
                    if infos[0] == uri:
                        found = infos
                        break
                if found != None:
                    children.remove(found)
            if media_manager.enabled:
                media_manager.handle_notify_event(uri, event,
                                                  self.media_types)

    # loadmore specialized callbacks

    def _loadmore_search_by_folder(self, model):
        # TODO: implement
        pass

    def _loadmore_for_model(self, model):
        dfr = None
        uri = model.uri

        def got_children(children, uri, model, filter_uri):
            if len(children) == 0:
                model.has_children = False
                model.loading = False

            self._start_monitoring(uri, children, model)
            
        if uri:
            if str(uri) not in self.model_cache.keys():
                self.debug('Not in cache: %r', model)
                children = ListObservable()
                filter_uri = FilteringUriObserver(model)
                filter_uri.observe(children)

                model.has_children = False
                model.loading = True
                media_manager = common.application.media_manager
                dfr = media_manager.get_direct_children(uri, children)
                if dfr != None:
                    self.model_cache[str(uri)] = (children, filter_uri, dfr)
                    dfr.addCallback(got_children, uri, model, filter_uri)
                else:
                    model.has_children = False
                    model.loading = False
            else:
                self.debug('In cache: %r' % model)
                dfr = defer.Deferred()
                children, filter_uri, dfr = self.model_cache[str(uri)]
                dfr.callback(children)
        else:
            self.debug("No URI for %r", model)
        return dfr

    def unload(self, model):
        uri = model.uri
        if model.loading:
            model.has_children = True
        model.loading = False
        if uri:
            if str(uri) in self.model_cache.keys():
                self.debug('Emptying: %r', model)
                children, filter_uri, dfr = self.model_cache[str(uri)]
                del self.model_cache[str(uri)]
                dfr.pause()
                filter_uri.stop_observing()
                for i in xrange(len(model)):
                    m = model.pop()

    def _loadmore_by_folder(self, model):
        children = ListObservable()
        filter_uri = FilteringUriObserver(model)
        filter_uri.observe(children)

        dfr = defer.Deferred()
        for location in self._media_locations:
            if location not in self.byfolders_cache:
                children.append((location, {}))
                self.byfolders_cache.append(location)

        dfr.callback(children)
        return dfr

    def metadata_changed(self, model, key, value):
        if key == 'default_image' and value is not None:
            model.thumbnail_source = value

    def _observe_metadata_dict(self, model, metadata):
        if isinstance(metadata, DictObservable):
            meta_observer = MetadataObserver(model)
            meta_observer.observe(metadata)
            self.meta_observers[model] = meta_observer
