# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006,2007 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 2.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.


__maintainer__ = 'Florian Boucault <florian@fluendo.com>'


from elisa.base_components.controller import Controller

from list_controller import ListController
from elisa.core.input_event import *

class InvalidPath(Exception):
    pass

class NodeController(ListController):

    supported_models = ('base:menu_node_model')

    default_associations = (
    ('base:menu_node_model','base:node_controller'),
    )

    def __init__(self):
        ListController.__init__(self)

        # if False, loadmore are done when right arrow is pressed
        # if True, on selection (down or up arrow)
        self.loadmore_on_selection = False

        # FIXME: understand what self.expanded is
        self.expanded = False
        self.thumbnail_source = None
        self.has_children = None
        self._selected = False
        self.selected_controller = self
        self._root = self

    def __repr__(self):
        if self.model == None:
            return "<%s: current_index=%r %s items>" % (self.__class__.__name__,
                                                       self._current_index,
                                                       len(self))
        else:
            return "<%s: %s current_index=%r %s items>" % (self.__class__.__name__,
                                                       self.model.text,
                                                       self._current_index,
                                                       len(self))

    def attribute_set(self, key, old_value, new_value):
        ListController.attribute_set(self, key, old_value, new_value)
        if key == 'thumbnail_source':
            self.thumbnail_source = new_value
        elif key == 'has_children':
            self.has_children = new_value

    def model_changed(self, old_model, new_model):
        if isinstance(self.parent, NodeController) and self.parent.root != None:
            self.root = self.parent.root
            self.thumbnail_source = self.model.thumbnail_source
            self.has_children = self.model.has_children
        ListController.model_changed(self, old_model, new_model)


    def root__set(self, root):
        self._root = root
        for child_controller in self:
            child_controller.root = self.root

    def root__get(self):
        return self._root


    def removed(self, elements, position):
        """
        DOCME

        called when a child_model is removed in controller model
        """
        self.debug("%s got a REMOVED at %s for %s elements" % (self, position,
                                                                len(elements)))
        end = position+len(elements)
        # current index needs updating
        if self.current_index >= position:
            # current index is set to the end of the remaining children
            if end >= len(self):
                self.current_index = position - 1
            else:
                # current index still points to the same item
                if self.current_index >= end:
                    self.current_index -= len(elements)
                else:
                # current index is set to the child after the removed ones
                    self.current_index = position

        for elem in elements:
            if position >= len(self):
                break
            to_del = self[position]
            self.debug("%s performs a remove of %s" % (self, to_del))
            self.remove(to_del)
            if isinstance(to_del, NodeController):
                if to_del.selected:
                    self.selected = True
                    self.focus()
 

        if len(self) == 0 and self.selected:
            if self.parent != None:
                self.parent.selected = True
                self.parent.focus()
            else:
                self.warning("%s is selected but has no parents to give the"
                               "selection to. This will break the Tree!" %
                               self)


    def child_controller_creating(self, controller, model, position):
        if self.root != None:
            controller.root = self.root
        ListController.child_controller_creating(self, controller, model, position)


    def selected__get(self):
        return self._selected

    def current_index__set(self, index):
        ListController.current_index__set(self, index)
        if self.loadmore_on_selection == True and len(self) > 0:
            child = self[index]
            if child.model.has_children:
                child.loadmore()

    def selected__set(self, value):
        """
        set controller selected
        only one controller can be selected at same time
        the previous controller is unselected automatically
        """
        previous_selection = self.root.selected_controller
        if value == True:
            if isinstance(previous_selection, NodeController):
                previous_selection.selected = False
            if isinstance(self.parent, NodeController):
                self.parent.current_index = self.parent.index(self)
            self.root.selected_controller = self
            self._selected = True

        elif previous_selection == self:
            self._selected = False
            self.root.selected_controller = None


    def handle_input(self, input_event):
        if input_event.action == EventAction.GO_LEFT:
            if isinstance(self.parent, NodeController):
                self.parent.focus()
                self.parent.selected = True
            return True

        elif input_event.action == EventAction.GO_RIGHT:
            if len(self) == 0:
                return True

            child_controller = self[self.current_index]
            if child_controller.model.has_children:
                child_controller.loadmore()
                child_controller.focus()
                child_controller.selected = True

                if self.loadmore_on_selection == True and len(child_controller) > 0:
                    grand_child_controller = child_controller[child_controller.current_index]
                    if grand_child_controller.model.has_children:
                        grand_child_controller.loadmore()

            return True

        elif input_event.action == EventAction.GO_UP:
            new_selection = self.current_index - 1
            if new_selection >= 0 and new_selection < len(self):
                self.current_index = new_selection
            return True

        elif input_event.action == EventAction.GO_DOWN:
            new_selection = self.current_index + 1
            if new_selection >= 0 and new_selection < len(self):
                self.current_index = new_selection
            return True

        elif input_event.action == EventAction.OK:
            child_controller = self[self.current_index]
            child_controller.model.activate_action(self, input_event.origin)
            return True

        return False

    def remove(self, element):
        if isinstance(element, Controller):
            if isinstance(element, ListController):
                self.debug("%s saying %s to remove the children" % (self,
                                                            element))
                self._remove_children(element)
                if isinstance(element, NodeController):
                    if element.selected:
                        self.selected = True
                        self.focus()

            element.parent = None
            element.backend = None

        ListController.remove(self, element)

    def _remove_children(self, parent):
        # Does anyone know why 'for child in parent' does not work correctly?
        # Probably because you were trying to remove element from parent
        # within the loop.
        while len(parent):
            child = parent[0]
            self.debug("%s:%s should remove %s" %  (self, parent, child))
            parent.remove(child)
            if isinstance(child, NodeController):
                if child.selected:
                    parent.selected = True
                    parent.focus()
