# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006,2007 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 2.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.


__maintainer__ = 'Lionel Martin <lionel@fluendo.com>'
__maintainer2__ = 'Florian Boucault <florian@fluendo.com>'


from elisa.base_components.controller import Controller
from twisted.internet import reactor
from elisa.core.input_event import *


class SlideshowController(Controller):

    supported_models = ('base:slideshow_model')

    current_index = None
    playing = False
    preview_mode = True
    duration = 0.0

    def __init__(self):
        Controller.__init__(self)
        self._call_later = None

    def model_changed(self, old_model, new_model):
        self.current_index = self.model.current_index
        self.playing = self.model.playing
        self.preview_mode = self.model.preview_mode

    def handle_input(self, input_event):
        if input_event.action == EventAction.GO_LEFT:
            if len(self.model.playlist) > 0:
                self.previous_image()
            return True

        elif input_event.action == EventAction.GO_RIGHT:
            if len(self.model.playlist) > 0:
                self.next_image()
            return True

        elif input_event.action == EventAction.OK:
            self.model.playing = not self.model.playing
            return True

        elif input_event.action == EventAction.PLAY:
            self.model.playing = True
            return True
        
        elif input_event.action == EventAction.PAUSE:
            self.model.playing = False
            return True
        
        elif input_event.action == EventAction.STOP:
            self.model.playing = False
            self.model.current_index = 0
            return True

        return False
    
    def attribute_set(self, key, old_value, new_value):
        Controller.attribute_set(self, key, old_value, new_value)

        if key == 'current_index':
            self.current_index = new_value

        elif key == 'playing':
            self.playing = new_value
            if self.model.playing:
                self._call_later = reactor.callLater(self.model.duration,
                                                     self.next_image)
            else:
                self._cancel_last_call_later()

        elif key == 'preview_mode':
            self.preview_mode = new_value
            if not new_value:
                self.focus()

        elif key == 'duration':
            self.duration = new_value


    def _cancel_last_call_later(self):
        if self._call_later != None and self._call_later.active():
            self._call_later.cancel()

    def focused_changed(self, old_focused, new_focused):
        if new_focused == False and self.model.playing == True:
            self._cancel_last_call_later()
            self.model.playing = False


    def next_image(self):
        self._cancel_last_call_later()

        index = self.model.current_index + 1
        if index < len(self.model.playlist):
            self.info("Loading image at index %r", index)
            self.model.current_index = index
            if self.model.playing == True:
                self._call_later = reactor.callLater(self.model.duration,
                                                     self.next_image)
            return True
        else:
            self.info("Reached end of slideshow")
            return False


    def previous_image(self):
        self._cancel_last_call_later()

        index = self.model.current_index - 1
        if index < len(self.model.playlist) and index >= 0:
            self.info("Loading image at index %r", index)
            self.model.current_index = index
            if self.model.playing == True:
                self._call_later = reactor.callLater(self.model.duration,
                                                     self.previous_image)
            return True
        else:
            self.info("Reached beginning of slideshow")
            return False
