# Elisa - Home multimedia server
# Copyright (C) 2006,2007 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 2.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.

from elisa.core.tests import component_test_case
from elisa.core import media_uri, db_backend
import os
from twisted.internet import defer
import tempfile

# TODO: refactor next_location() tests with deferred generators.

class FspotTest(component_test_case.ComponentTestCase):
    component_class = 'fspot_media:FspotMedia'
    component_path = '..'
    dump_path = None

    def __init__(self, methodName='runTests'):
        component_test_case.ComponentTestCase.__init__(self, methodName)
        if self.dump_path is None:
            self.skip = "no db dump to load"
    
    def setUp(self):
    
        # load the SQL dump
        dump_path = os.path.join(self.directory, self.dump_path)
        fd, self._db_path = tempfile.mkstemp()

        db = db_backend.DBBackend(db_backend='sqlite', database=self._db_path)
        for statement in open(dump_path).read().split(';'):
            db.sql_execute(statement)
        db.save_changes()
        db.disconnect()
        
        self.component_class.default_config = {'db_path': self._db_path}
        component_test_case.ComponentTestCase.setUp(self)

    def tearDown(self):
        component_test_case.ComponentTestCase.tearDown(self)
        os.unlink(self._db_path)
    
class TestKaleoFspot(FspotTest):

    dump_path = 'data/kaleo.sql'


    def test_next_location_from_root(self):
    
        start = media_uri.MediaUri("fspot:///")

        def got_next1(next):
            self.assertEquals(next, media_uri.MediaUri(u'fspot:///Acontecimientos?id=5&tag_id=-1'))

            dfr2 = self.component.next_location(next, root=start)
            dfr2.addCallback(got_next2)
            return dfr2
        
        def got_next2(next):
            self.assertEquals(next, media_uri.MediaUri(u'fspot:///Favoritos?id=1&tag_id=-1'))

            dfr = self.component.next_location(next, root=start)
            dfr.addCallback(got_next3)
            return dfr

        def got_next3(next):
            self.assertEquals(next, media_uri.MediaUri(u'fspot:///home/kaleo/Desktop/media/pictures/NATURE-ChristmasOnMaui_1280x1024.jpg?tag_id=1#6'))

            dfr = self.component.next_location(next, root=start)
            dfr.addCallback(got_next4)
            return dfr

        def got_next4(next):
            self.assertEquals(next, media_uri.MediaUri(u'fspot:///home/kaleo/Desktop/media/pictures/Julia.png?tag_id=3#5'))
            dfr = self.component.next_location(next, root=start)
            dfr.addCallback(got_next5)
            return dfr

        def got_next5(next):
            self.assertEquals(next, media_uri.MediaUri(u'fspot:///home/kaleo/Desktop/media/pictures/369113572_24fc34005a_o_d.jpg?tag_id=3#2'))
            dfr = self.component.next_location(next, root=start)
            dfr.addCallback(got_next6)
            return dfr

        def got_next6(next):
            self.assertEquals(next, media_uri.MediaUri(u'fspot:///home/kaleo/Desktop/media/pictures/308567765_b904571e9f_o.jpg?tag_id=4#1'))
            dfr = self.component.next_location(next, root=start)
            dfr.addCallback(got_next7)
            return dfr

        def got_next7(next):
            self.assertEquals(next, media_uri.MediaUri(u'fspot:///home/kaleo/Desktop/media/pictures/NATURE-KosametThailand_1280x1024.jpg?tag_id=4#7'))
            dfr = self.component.next_location(next, root=start)
            dfr.addCallback(got_next8)
            return dfr

        def got_next8(next):
            # TODO: finish me (or remove photos from kaleo's db)
            self.failIf(next is None)

        dfr = self.component.next_location(start, root=start)
        dfr.addCallback(got_next1)
        return dfr


class TestFspotMedia0_3(FspotTest):

    dump_path = 'data/fspot_0_3.sql'


    def _from_tmp_tag(self):
        start = media_uri.MediaUri("fspot:///subtmp?id=7&tag_id=6")

        def got_next1(next):
            self.assertEquals(next, media_uri.MediaUri(u'fspot:///home/phil/Photos/2001/2/17/duck.jpg?tag_id=7#19'))

            dfr2 = self.component.next_location(next, root=start)
            dfr2.addCallback(got_next2)
            return dfr2

        def got_next2(next):
            self.assertEquals(next, media_uri.MediaUri(u'fspot:///home/phil/Photos/2001/9/26/dogs.jpg?tag_id=7#18'))

            dfr = self.component.next_location(next, root=start)
            dfr.addCallback(got_next3)
            return dfr

        def got_next3(next):
            self.assertEquals(next, media_uri.MediaUri(u'fspot:///home/phil/Photos/2002/4/1/lambs.jpg?tag_id=7#22'))
            got_next3.called = True
            return next
        
        got_next3.called = False

        def check_called(uri):
            self.check_called(got_next3)
            return uri

        dfr = self.component.next_location(start, root=start)
        dfr.addCallback(got_next1)
        dfr.addCallback(check_called)
        return dfr

    def test_simple_next_location(self):
        dfr = self._from_tmp_tag()
        return dfr

    def test_next_location_from_root(self):

        start = media_uri.MediaUri("fspot:///")

        def got_next1(next):
            self.assertEquals(next, media_uri.MediaUri(u'fspot:///Events?id=5&tag_id=-1'))

            dfr2 = self.component.next_location(next, root=start)
            dfr2.addCallback(got_next2)
            return dfr2

        def got_next2(next):
            self.assertEquals(next, media_uri.MediaUri(u'fspot:///Favorites?id=1&tag_id=-1'))

            dfr = self.component.next_location(next, root=start)
            dfr.addCallback(got_next3)
            return dfr


        def got_next3(next):
            self.assertEquals(next, media_uri.MediaUri(u'fspot:///Hidden?id=2&tag_id=-1'))

            dfr = self.component.next_location(next, root=start)
            dfr.addCallback(got_next4)
            return dfr

        def got_next4(next):
            self.assertEquals(next, media_uri.MediaUri(u'fspot:///People?id=3&tag_id=-1'))

            dfr = self.component.next_location(next, root=start)
            dfr.addCallback(got_next5)
            return dfr
        
        def got_next5(next):
            self.assertEquals(next, media_uri.MediaUri(u'fspot:///tmp?id=6&tag_id=3'))

            dfr = self._from_tmp_tag()
            dfr.addCallback(lambda uri: self.component.next_location(uri,
                                                                     root=start))
            dfr.addCallback(got_next6)
            return dfr
        
        def got_next6(next):
            self.assertEquals(next, media_uri.MediaUri(u'fspot:///home/phil/Photos/2004/7/30/rabbit.jpg?tag_id=6#25'))

            dfr = self.component.next_location(next, root=start)
            dfr.addCallback(got_next7)
         
            return dfr
        
        def got_next7(next):
            self.assertEquals(next, media_uri.MediaUri(u'fspot:///home/phil/Photos/2004/7/30/rabbit.jpg?tag_id=6#10'))

            dfr = self.component.next_location(next, root=start)
            dfr.addCallback(got_next8)
            return dfr
        
        def got_next8(next):
            self.assertEquals(next, media_uri.MediaUri(u'fspot:///home/phil/Photos/2006/8/22/leopard.jpg?tag_id=6#23'))

            dfr = self.component.next_location(next, root=start)
            dfr.addCallback(got_next9)
            return dfr

        def got_next9(next):
            self.assertEquals(next, media_uri.MediaUri(u'fspot:///home/phil/Photos/2003/1/29/hippo.jpg?tag_id=3#20'))
            got_next9.called = True

        got_next9.called = False

        dfr = self.component.next_location(start, root=start)
        dfr.addCallback(got_next1)
        dfr.addCallback(lambda r: self.check_called(got_next9))
        return dfr

    def test_get_media_type(self):

        def check_file(result):
            self.assertEquals(result['file_type'], 'image')
            self.assertEquals(result['mime_type'], 'image/jpeg')

        u1 = media_uri.MediaUri(u'fspot:///home/phil/Photos/2003/1/29/hippo.jpg?tag_id=3#20')
        d1 = self.component.get_media_type(u1)
        d1.addCallback(check_file)

        def check_dir(result):
            self.assertEquals(result['file_type'], 'directory')
            self.assertEquals(result['mime_type'], '')
            
        u2 = media_uri.MediaUri(u'fspot:///tmp?id=6&tag_id=3')
        d2 = self.component.get_media_type(u2)
        d2.addCallback(check_dir)

        d = defer.DeferredList([d1,d2])
        return d
        
    def test_has_children(self):

        def check_file(result):
            self.assertEquals(result, False)

        u1 = media_uri.MediaUri(u'fspot:///home/phil/Photos/2003/1/29/hippo.jpg?tag_id=3#20')
        d1 = self.component.has_children_with_types(u1,['image',])
        d1.addCallback(check_file)

        def check_dir(result):
            self.assertEquals(result, True)
      
        u2 = media_uri.MediaUri(u'fspot:///tmp?id=6&tag_id=3')
        d2 = self.component.has_children_with_types(u2, ['image',])
        d2.addCallback(check_dir)

        def check_dir2(result):
            self.assertEquals(result, True)
      
        u3 = media_uri.MediaUri(u'fspot:///')
        d3 = self.component.has_children_with_types(u3, ['directory',])
        d3.addCallback(check_dir2)

        u4 = media_uri.MediaUri("fspot:///subtmp?id=7&tag_id=6")
        d4 = self.component.has_children_with_types(u4, ['directory', 'image'])
        d4.addCallback(check_dir2)

        d = defer.DeferredList([d1,d2,d3,d4])
        return d

    def test_get_direct_children(self):

        def got_children(children):
            expected = ['fspot:///Events?id=5&tag_id=-1',
                        'fspot:///Favorites?id=1&tag_id=-1',
                        'fspot:///Hidden?id=2&tag_id=-1',
                        'fspot:///People?id=3&tag_id=-1',
                        'fspot:///Places?id=4&tag_id=-1']
            children = [ str(uri) for uri, metadata in children ]
            self.assertEquals(expected, children)

        d1 = self.component.get_direct_children(media_uri.MediaUri('fspot:///'),
                                                [])
        d1.addCallback(got_children)
        return d1
    
    def test_get_direct_children_tmp(self):

        def got_children(children):
            expected = ['fspot:///home/phil/Photos/2001/2/17/duck.jpg?tag_id=7#19',
                        'fspot:///home/phil/Photos/2001/9/26/dogs.jpg?tag_id=7#18',
                        'fspot:///home/phil/Photos/2002/4/1/lambs.jpg?tag_id=7#22']
            children = [ str(uri) for uri, metadata in children ]
            self.assertEquals(expected, children)

        d1 = self.component.get_direct_children(media_uri.MediaUri("fspot:///subtmp?id=7&tag_id=6"),
                                                [])
        d1.addCallback(got_children)
        return d1

class TestFspotMedia0_4(TestFspotMedia0_3):

    dump_path = 'data/fspot_0_4.sql'
