# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006,2007 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 2.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.


__maintainer__ = 'Lionel Martin <lionel@fluendo.com>'
__maintainer2__ = 'Florian Boucault <florian@fluendo.com>'


from elisa.base_components.context import Context
from elisa.core.utils import signal

import pgm,os

from twisted.internet import reactor

class PigmentContext(Context):

    default_config = {'screen_ratio' : 'auto',
                      'window_width': '0',
                      'touchscreen': '0'}
    config_doc = {'screen_ratio' : 'The ratio of the screen. This is a string'
                                   ' containing a relation value, separated'
                                   ' by a colon (:). Common values are 16:9,'
                                   ' 4:3, 16:10, but it could be any other,'
                                   ' too. You can have special values : mm,'
                                   ' based on screen size in milimeters,'
                                   ' sr, based on window manager resolution,'
                                   ' or auto. auto will use mm special value.',
                  'window_width' : 'Here you can set the width in pixels the'
                                   ' window should have at startup. The height'
                                   ' is computed using screen_ratio.'
                                   ' If this value is 0, we decide'
                                   ' automatically.',
                  'touchscreen' : 'If set to 1, the mouse behaviour will be'
                                  ' adapted for touchscreen equipped hardware.'
    }


    def initialize(self):
        Context.initialize(self)
        config_screen_ratio = self.config.get('screen_ratio', 'auto')

        self.touchscreen =  self.config.get('touchscreen', '1') == '1'
        try:    
            # FIXME: that is only working on Linux with ps installed
            self._compiz_check()
        except:
            pass

        # OpenGL viewport creation
        factory = pgm.ViewportFactory('opengl')
        viewport = factory.create()
        self.viewport_handle = viewport
        viewport.title = 'Elisa Media Center'
        self._window_managet_screen_size_mm = viewport.screen_size_mm
 
        # Canvas creation
        self.canvas = pgm.Canvas()
        
        try:
            width = int(self.config.get('window_width', '0'))
            if width != 0:
               viewport.width = width
        except Exception, e:
            self.warning("The configuration value of window_width seems to be"
                         " no number value in pixels: %s" % e)

        # delay the resize of the canvas to avoid crippling the CPU
        self._resize_delay = 0.200
        self._resize_delayed = None
        self.canvas_resized = signal.Signal('canvas_resized', tuple)
        self.aspect_ratio_changed(None, config_screen_ratio)

        # Bind the canvas to the OpenGL viewport
        viewport.set_canvas(self.canvas)
        
        # show the window
        viewport.show()

        viewport.connect('configure-event', self._configure_callback)

        # hide the cursor
        self.viewport_handle.cursor = pgm.VIEWPORT_NONE

        if not self.touchscreen:
            # delay the hiding of the mouse cursor
            self._cursor_delay = 1.000
            self._cursor_delayed = None
            # signal triggered when the canvas gets resized
            viewport.connect('motion-notify-event',
                             self._motion_notify_callback)
            

    def aspect_ratio_changed(self, old_aspect_ratio, new_aspect_ratio):
        if old_aspect_ratio is not None:
            self.debug("aspect ratio changed from %r to %r", old_aspect_ratio,
                       new_aspect_ratio)
        else:
            self.info("will set the aspect ratio with parameter %r", new_aspect_ratio)
            
        screen_ratio = self._get_screen_size_from_ratio(new_aspect_ratio)
        self.debug("screen ratio is %r", screen_ratio)
        
        pixel_aspect_ratio = self._get_pixel_aspect_ratio(screen_ratio)
        self.info("used pixel aspect ratio is %r", pixel_aspect_ratio)

        canvas_height = self.canvas.width / screen_ratio
        self.debug("new canvas height: %r", canvas_height)
        self.canvas.height = canvas_height
        
        # set the window aspect ratio to the screen aspect ratio
        viewport_height = self.viewport_handle.width / (screen_ratio * pixel_aspect_ratio)
        self.debug("new viewport height: %r", viewport_height)
        self.viewport_handle.height = viewport_height

        #w2,l2 = self.viewport_handle.size
        #self.viewport_handle.screen_size_mm = (int(w), int(h))
        self.viewport_size = self.viewport_handle.size
        self.debug("viewport size: %r", self.viewport_size)

        config_screen_ratio = self.config.get('screen_ratio', 'auto')
        if config_screen_ratio != new_aspect_ratio:
            self.config['screen_ratio'] = new_aspect_ratio
            
        self.canvas_resized.emit(self.canvas.size)
            
    def update(self):
        self.viewport_handle.update()

    def clean(self):
        # FIXME: should unref self.viewport_handle
        pass
#        self.viewport_handle.setcanvas(None)
#        print len(gc.get_referrers(self.viewport_handle))
#        del self.viewport_handle
#        gc.collect()


    def context_handle__set(self, context_handle):
        self._context_handle = context_handle
        if isinstance(context_handle, pgm.Drawable):
            self.canvas.add(pgm.DRAWABLE_MIDDLE, context_handle)

    def context_handle__get(self):
        return self._context_handle

    def _compiz_check(self):
        # FIXME: this is a work-a-round against the flickering in compiz
        ps_aux = os.popen('ps aux')

        compiz = False

        for line in ps_aux:
            if 'compiz' in line:
                compiz = True

        if compiz:
            self.warning("Compiz detected, trying to hide all other windows.")
            try:
                import wnck
            except ImportError:
                self.warning("Could not import WNCK. Windows hiding disabled.")
            else:
                s = wnck.screen_get_default()
                s.force_update()
                lst = s.get_windows()
                cur_workspace = s.get_active_workspace()
                for window in lst:
                    if window.is_in_viewport(cur_workspace):
                        window.minimize()

        else:
            self.info('Compiz not found')

    def _get_screen_size_from_ratio(self, config_screen_ratio):
        # retrieve the screen aspect ratio
        if config_screen_ratio == 'sr':
            self.info("using window manager screen resolution as screen aspect ratio: %r",
                      self.viewport_handle.screen_resolution)
            w, h = self.viewport_handle.screen_resolution
            self.viewport_handle.screen_size_mm = (int(w), int(h) )
        elif config_screen_ratio.find(':') != -1:
            self.info("Using configured aspect screen ratio: %r", config_screen_ratio)
            w, h = config_screen_ratio.split(':')
            self.viewport_handle.screen_size_mm = (int(w), int(h))
        else:
            self.info("Using the milimeter based screen aspect ratio : %r",
                      self.viewport_handle.screen_size_mm)
            w, h = self._window_managet_screen_size_mm
            self.viewport_handle.screen_size_mm =  (int(w), int(h))
        width = float(w)
        height = float(h)
        self.debug("screen size: (%r,%r)", width, height)
        
        return width / float(height)
    
    def _get_pixel_aspect_ratio(self, screen_aspect_ratio):
        w, h = self.viewport_handle.screen_resolution
        return ( w / float(h) ) / float(screen_aspect_ratio)
                
    def _configure_callback(self, viewport, event):
        # (re)schedule the resize of the canvas
        if self._resize_delayed != None and self._resize_delayed.active():
            self._resize_delayed.reset(self._resize_delay)
        else:
            self._resize_delayed = reactor.callLater(self._resize_delay,
                                                     self._resize_canvas)

    def _resize_canvas(self):
        # only reacts if the size of the viewport changed
        if self.viewport_size != self.viewport_handle.size:
            self.debug("regeneration of the drawables")
            self.viewport_size = self.viewport_handle.size
            self.canvas_resized.emit(self.canvas.size)

        """
        # resize the canvas to follow the aspect ratio of the viewport
        viewport_ratio = float(self.viewport_handle.width)/float(self.viewport_handle.height)
        canvas_ratio = float(self.canvas.width)/float(self.canvas.height)

        # only resize the canvas if the aspect ratio of the viewport has changed
        if viewport_ratio != canvas_ratio:
            # FIXME: only one of width/height needs to change (multiplied by
            # viewport_ratio)
            self.canvas.size = (self.viewport_handle.width/100.0,
                                self.viewport_handle.height/100.0)
            self.debug("The canvas is getting resized. New size = (%s, %s)" % \
                        (self.canvas.size[0], self.canvas.size[1]))
        """

    def _motion_notify_callback(self, viewport, event):
        # show the cursor
        viewport.cursor = pgm.VIEWPORT_INHERIT

        # (re)schedule the hiding of the cursor
        if self._cursor_delayed != None and self._cursor_delayed.active():
            self._cursor_delayed.reset(self._cursor_delay)
        else:
            self._cursor_delayed = reactor.callLater(self._cursor_delay, self._hide_cursor)

    def _hide_cursor(self):
        self.viewport_handle.cursor = pgm.VIEWPORT_NONE
