/* 
   elmo - ELectronic Mail Operator

   Copyright (C) 2003 rzyjontko

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; version 2.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  

   ----------------------------------------------------------------------

   
*/
/****************************************************************************
 *    IMPLEMENTATION HEADERS
 ****************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "address.h"
#include "rarray.h"
#include "raddress.h"
#include "rstring.h"
#include "xmalloc.h"
#include "str.h"
#include "memchunk.h"

/****************************************************************************
 *    IMPLEMENTATION PRIVATE DEFINITIONS / ENUMERATIONS / SIMPLE TYPEDEFS
 ****************************************************************************/

#define LINE_MAX_LEN 70

/****************************************************************************
 *    IMPLEMENTATION PRIVATE CLASS PROTOTYPES / EXTERNAL CLASS REFERENCES
 ****************************************************************************/
/****************************************************************************
 *    IMPLEMENTATION PRIVATE STRUCTURES / UTILITY CLASSES
 ****************************************************************************/
/****************************************************************************
 *    IMPLEMENTATION REQUIRED EXTERNAL REFERENCES (AVOID)
 ****************************************************************************/
/****************************************************************************
 *    IMPLEMENTATION PRIVATE DATA
 ****************************************************************************/
/****************************************************************************
 *    INTERFACE DATA
 ****************************************************************************/
/****************************************************************************
 *    IMPLEMENTATION PRIVATE FUNCTION PROTOTYPES
 ****************************************************************************/
/****************************************************************************
 *    IMPLEMENTATION PRIVATE FUNCTIONS
 ****************************************************************************/

static char *
find_comma (char *str)
{
        int in_quotes = 0;

        while (*str){
                if (*str == '"')
                        in_quotes = ! in_quotes;
                if (*str == ',' && ! in_quotes)
                        return str;
                str++;
        }
        return NULL;
}


static int
is_not_abook (void *ptr)
{
        address_t *addr = (address_t *) ptr;

        return ! addr->flags.bits.abook;
}

/****************************************************************************
 *    INTERFACE FUNCTIONS
 ****************************************************************************/

raddress_t *
raddress_create_size (int size)
{
        return (raddress_t *) rarray_create_size (size);
}




raddress_t *
raddress_create (void)
{
        return (raddress_t *) rarray_create_size (2);
}




void
raddress_destroy (raddress_t *ptr)
{
        rarray_destroy ((rarray_t *) ptr);
}



void
raddress_add (raddress_t *ptr, address_t *x)
{
        rarray_add ((rarray_t *) ptr, x);
}



void
raddress_remove (raddress_t *ptr, unsigned index)
{
        rarray_remove ((rarray_t *) ptr, index);
}



void
raddress_shrink (raddress_t *ptr)
{
        rarray_shrink ((rarray_t *) ptr);
}



raddress_t *
raddress_join (raddress_t *x, raddress_t *y)
{
        return (raddress_t *) rarray_join ((rarray_t *) x, (rarray_t *) y);
}



void
raddress_add_array (raddress_t *x, raddress_t *y)
{
        rarray_add_array ((rarray_t *) x, (rarray_t *) y, is_not_abook);
}


raddress_t *
raddress_get_from_header (char *header)
{
        raddress_t *result;
        address_t     *addr;
        char       *seek  = header;
        char       *comma = NULL;

        result = raddress_create ();

        while (seek && *seek){
                comma = find_comma (seek);
                if (comma)
                        *comma = '\0';
  
                addr = address_from_string (seek);
                raddress_add (result, addr);

                if (comma){
                        *comma = ',';
                        seek   = comma + 1;
                }
                else
                        break;
        }

        return result;
}



rstring_t *
raddress_to_strings (raddress_t *ptr)
{
        rstring_t  *result = rstring_create_size (ptr->count + 1);
        address_t    **addr;

        for (addr = ptr->array; *addr; addr++){
                rstring_add (result, (*addr)->full);
        }

        return result;
}



char *
raddress_list (raddress_t *ptr, const char *header, int line_break,
               const char *line_indent)
{
        int     i;
        int     len;
        int     line_len = 0;
        int     first    = 1;
        str_t  *str;
        address_t *addr;

        if (ptr == NULL)
                return NULL;

        str = str_create ();
  
        if (header){
                line_len = str_sprintf (str, "%s: ", header);
        }

        for (i = 0; i < ptr->count; i++){
                addr = ptr->array[i];

                if (addr->full == NULL)
                        continue;

                if (first){
                        line_len += str_sprintf (str, "%s", addr->full);
                        first     = 0;
                }
                else {
                        len = strlen (addr->full);
                        if (line_break && (line_len + len + 3 > LINE_MAX_LEN)){
                                str_put_char (str, ',');
                                str_put_char (str, '\n');
                                line_len = 0;
                        }
                        else {
                                str_put_char (str, ',');
                                line_len++;
                        }
                        if (line_len == 0)
                                line_len += str_sprintf (str, "%s%s",
                                                         line_indent,
                                                         addr->full);
                        else
                                line_len += str_sprintf (str, " %s",
                                                         addr->full);
                }
        }
  
        return str_finished (str);
}



raddress_t *
raddress_read (memchunk_t *chunk)
{
        int         i;
        int         count;
        address_t  *addr;
        raddress_t *ptr;

        count = memchunk_intget (chunk);
        if (count == 0)
                return NULL;

        ptr = raddress_create_size (count + 1);
        for (i = 0; i < count; i++){
                addr = address_read (chunk);
                raddress_add (ptr, addr);
        }
        return ptr;
}



void
raddress_dump (raddress_t *ptr, memchunk_t *chunk)
{
        int i;
        
        if (ptr == NULL){
                memchunk_intdump (chunk, 0);
                return;
        }

        memchunk_intdump (chunk, ptr->count);
        for (i = 0; i < ptr->count; i++){
                address_dump (ptr->array[i], chunk);
        }
}


/****************************************************************************
 *    INTERFACE CLASS BODIES
 ****************************************************************************/
/****************************************************************************
 *
 *    END MODULE raddress.c
 *
 ****************************************************************************/
