;;; code-pages.el --- coding systems for assorted codepages  -*-coding: utf-8;-*-

;; Copyright (C) 2001, 2002, 2003, 2004, 2005  Free Software Foundation, Inc.
;; Copyright (C) 2004, 2005
;;   National Institute of Advanced Industrial Science and Technology (AIST)
;;   Registration Number H14PRO021

;; Author: Dave Love <fx@gnu.org>
;; Keywords: i18n

;; This file is part of GNU Emacs.

;; GNU Emacs is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; GNU Emacs is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with GNU Emacs; see the file COPYING.  If not, write to the
;; Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
;; Boston, MA 02110-1301, USA.

;;; Commentary:

;; Definitions of miscellaneous 8-bit coding systems based on ASCII
;; (we can't cope properly with EBCDIC, for instance), mainly for PC
;; `code pages'.  They are decoded into Latin-1 and mule-unicode
;; charsets rather than (lossily) into single iso8859 charsets à la
;; codepage.el.  The utility `cp-make-coding-system' derives them from
;; simple tables.

;; Those covered are: cp437, cp737, cp720, cp775, cp850, cp851, cp852,
;; cp855, cp857, cp860, cp861, cp862, cp863, cp864, cp865, cp866,
;; cp869, cp874, cp1125, windows-1250, windows-1253, windows-1254,
;; windows-1255, windows-1256, windows-1257, windows-1258, next,
;; iso-8859-6, iso-8859-10, iso-8859-11, iso-8859-16, koi8-t,
;; georgian-ps.  This is meant to include all the single-byte ones
;; relevant to GNU (used in glibc-defined locales); we don't yet get
;; all the multibyte ones in base Emacs.

;; Note that various of these can clash with definitions in
;; codepage.el; we try to avoid damage from that.  A few CPs from
;; codepage.el (770, 773, 774) aren't covered (in the absence of
;; translation tables to Unicode).

;; Compile this, to avoid loading `ccl' at runtime.

;; Although the tables used here aren't very big, it might be worth
;; splitting the file and autoloading the coding systems if/when my
;; (or similar) autoloading code is installed.

;;; Code:

;; The defsubsts here are just so that language files can use
;; `cp-make-coding-system' and not require functions from this file
;; at runtime.

(defsubst cp-make-translation-table (v)
  "Return a translation table made from 128-long vector V.
V comprises characters encodable by mule-utf-8."
  (let ((encoding-vector (make-vector 256 0)))
    (dotimes (i 128)
      (aset encoding-vector i i))
    (dotimes (i 128)
      (aset encoding-vector (+ i 128) (aref v i)))
    ;; Add equivalent characters to the encoder so that we can unify
    ;; on encoding.
    (let* ((tab (make-translation-table-from-vector encoding-vector))
	   ;; Translation table used for encoding:
	   (encode-table (char-table-extra-slot tab 0)))
      (map-char-table (lambda (c v)
			(if v
			    (let ((c1 (aref encode-table v)))
			      (if c1	; we encode that unicode
				  (aset encode-table c c1)))))
		      ucs-mule-to-mule-unicode)
      tab)))

(defsubst cp-valid-codes (v)
  "Derive a valid-codes list for translation vector V.
See `make-coding-system'."
  (let (pairs
	(i 128)				; index into v
	(start 0)			; start of a valid range
	(end 127))			; end of a valid range
    (while (< i 256)
      (if (aref v (- i 128))		; start or extend range
	  (progn
	    (setq end i)
	    (unless start (setq start i)))
	(if start
	    (push (cons start end) pairs))
	(setq start nil))
      (setq i (1+ i)))
    (if start (push (cons start end) pairs))
    (nreverse pairs)))

;; Fix things that have been, or might be, done by codepage.el.
(eval-after-load "codepage"
  '(progn

;; Semi-dummy version for the stuff in codepage.el which we don't
;; define here.  (Used by mule-diag.)
(defun cp-supported-codepages ()
  "Return an alist of supported codepages.

Each association in the alist has the form (NNN . CHARSET), where NNN is the
codepage number, and CHARSET is the MULE charset which is the closest match
for the character set supported by that codepage.

A codepage NNN is supported if a variable called `cpNNN-decode-table' exists,
is a vector, and has a charset property."
  '(("774" . latin-iso8859-4) ("770" . latin-iso8859-4)
    ("773" . latin-iso8859-4)))

;; A version which doesn't override the coding systems set up by this
;; file.  It could still be used for the few missing ones from
;; codepage.el.
(defun codepage-setup (codepage)
  "Create a coding system cpCODEPAGE to support the IBM codepage CODEPAGE.

These coding systems are meant for encoding and decoding 8-bit non-ASCII
characters used by the IBM codepages, typically in conjunction with files
read/written by MS-DOS software, or for display on the MS-DOS terminal."
  (interactive
   (let ((completion-ignore-case t)
	 (candidates (cp-supported-codepages)))
     (list (completing-read "Setup DOS Codepage (default 437): " candidates
			    nil t nil nil "437"))))
  (let ((cp (format "cp%s" codepage)))
    (unless (coding-system-p (intern cp))
      (cp-make-coding-systems-for-codepage
       cp (cp-charset-for-codepage cp) (cp-offset-for-codepage cp))))))
)					; eval-after-load

;; Macro to allow ccl compilation at byte-compile time, avoiding
;; loading ccl.
;;;###autoload
(defmacro cp-make-coding-system (name v &optional doc-string mnemonic)
  "Make coding system NAME for and 8-bit, extended-ASCII character set.
V is a 128-long vector of characters to translate the upper half of
the character set.  DOC-STRING and MNEMONIC are used as the
corresponding args of `make-coding-system'.  If MNEMONIC isn't given,
?* is used.
Return an updated `non-iso-charset-alist'."
  (let* ((encoder (intern (format "encode-%s" name)))
	 (decoder (intern (format "decode-%s" name)))
	 (ccl-decoder
	  (ccl-compile
	   `(4
	     ((loop
	       (read r1)
	       (if (r1 < 128) ;; ASCII
		   (r0 = ,(charset-id 'ascii))
		 (if (r1 < 160)
		     (r0 = ,(charset-id 'eight-bit-control))
		   (r0 = ,(charset-id 'eight-bit-graphic))))
	       (translate-character ,decoder r0 r1)
	       ;; Allow fragmentation on decoding -- relevant for
	       ;; Cyrillic, Greek and, possibly Arabic and Hebrew.
	       (translate-character utf-translation-table-for-decode r0 r1)
	       (write-multibyte-character r0 r1)
	       (repeat))))))
	 (ccl-encoder
	  (ccl-compile
	   `(1
	     ((loop
	       (read-multibyte-character r0 r1)
	       (translate-character ,encoder r0 r1)
	       (if (r0 != ,(charset-id 'ascii))
		   (if (r0 != ,(charset-id 'eight-bit-graphic))
		       (if (r0 != ,(charset-id 'eight-bit-control))
			   (r1 = ??))))
	       (write-repeat r1)))))))
    `(let ((translation-table (cp-make-translation-table ,v))
	   (codes (cp-valid-codes ,v)))
       (define-translation-table ',decoder translation-table)
       (define-translation-table ',encoder
	 (char-table-extra-slot translation-table 0))
       (make-coding-system
	',name 4 ,(or mnemonic ?*)
	(or ,doc-string (format "%s encoding" ',name))
	(cons ,ccl-decoder ,ccl-encoder)
	(list (cons 'safe-chars (get ',encoder 'translation-table))
	      (cons 'valid-codes codes)
	      (cons 'mime-charset ',name)
	      ;; For Quail translation.  Fixme: this should really be
	      ;; a separate table that only translates the coding
	      ;; system's safe-chars.
	      (cons 'translation-table-for-input 'ucs-mule-to-mule-unicode)))
       (let ((slot (assq ',name non-iso-charset-alist))
	     (elt (list nil			; charset list
			',decoder
			(let (l)		; code range
			  (dolist (elt (reverse codes))
			    (push (cdr elt) l)
			    (push (car elt) l))
			  (list l)))))
	 (if (not slot)
	     (push (cons ',name elt) non-iso-charset-alist)
	   (setcdr slot elt)
	   non-iso-charset-alist)))))

(eval-when-compile (defvar non-iso-charset-alist))

;; These tables were mostly derived by running somthing like
;; `recode -f cpxxx/..utf-8' on a binary file filled by
;; `(dotimes (i 128) (insert ?? ?\\ (+ 128 i) ?\n))' and then
;; exchanging the ?\� entries for nil.  iconv was used instead in some
;; cases.

;; Fixme: Do better for mode-line mnemonics?

;;;###autoload(autoload-coding-system 'cp437 '(require 'code-pages))
(cp-make-coding-system
 cp437
 [?\Ç
  ?\ü
  ?\é
  ?\â
  ?\ä
  ?\à
  ?\å
  ?\ç
  ?\ê
  ?\ë
  ?\è
  ?\ï
  ?\î
  ?\ì
  ?\Ä
  ?\Å
  ?\É
  ?\æ
  ?\Æ
  ?\ô
  ?\ö
  ?\ò
  ?\û
  ?\ù
  ?\ÿ
  ?\Ö
  ?\Ü
  ?\¢
  ?\£
  ?\¥
  ?\₧
  ?\ƒ
  ?\á
  ?\í
  ?\ó
  ?\ú
  ?\ñ
  ?\Ñ
  ?\ª
  ?\º
  ?\¿
  ?\⌐
  ?\¬
  ?\½
  ?\¼
  ?\¡
  ?\«
  ?\»
  ?\░
  ?\▒
  ?\▓
  ?\│
  ?\┤
  ?\╡
  ?\╢
  ?\╖
  ?\╕
  ?\╣
  ?\║
  ?\╗
  ?\╝
  ?\╜
  ?\╛
  ?\┐
  ?\└
  ?\┴
  ?\┬
  ?\├
  ?\─
  ?\┼
  ?\╞
  ?\╟
  ?\╚
  ?\╔
  ?\╩
  ?\╦
  ?\╠
  ?\═
  ?\╬
  ?\╧
  ?\╨
  ?\╤
  ?\╥
  ?\╙
  ?\╘
  ?\╒
  ?\╓
  ?\╫
  ?\╪
  ?\┘
  ?\┌
  ?\█
  ?\▄
  ?\▌
  ?\▐
  ?\▀
  ?\α
  ?\ß
  ?\Γ
  ?\π
  ?\Σ
  ?\σ
  ?\µ
  ?\τ
  ?\Φ
  ?\Θ
  ?\Ω
  ?\δ
  ?\∞
  ?\φ
  ?\ε
  ?\∩
  ?\≡
  ?\±
  ?\≥
  ?\≤
  ?\⌠
  ?\⌡
  ?\÷
  ?\≈
  ?\°
  ?\·
  ?\•
  ?\√
  ?\ⁿ
  ?\²
  ?\■
  ?\ ])

;;;###autoload(autoload-coding-system 'cp737 '(require 'code-pages))
(cp-make-coding-system
 cp737
 [?\Α
  ?\Β
  ?\Γ
  ?\Δ
  ?\Ε
  ?\Ζ
  ?\Η
  ?\Θ
  ?\Ι
  ?\Κ
  ?\Λ
  ?\Μ
  ?\Ν
  ?\Ξ
  ?\Ο
  ?\Π
  ?\Ρ
  ?\Σ
  ?\Τ
  ?\Υ
  ?\Φ
  ?\Χ
  ?\Ψ
  ?\Ω
  ?\α
  ?\β
  ?\γ
  ?\δ
  ?\ε
  ?\ζ
  ?\η
  ?\θ
  ?\ι
  ?\κ
  ?\λ
  ?\μ
  ?\ν
  ?\ξ
  ?\ο
  ?\π
  ?\ρ
  ?\σ
  ?\ς
  ?\τ
  ?\υ
  ?\φ
  ?\χ
  ?\ψ
  ?\░
  ?\▒
  ?\▓
  ?\│
  ?\┤
  ?\╡
  ?\╢
  ?\╖
  ?\╕
  ?\╣
  ?\║
  ?\╗
  ?\╝
  ?\╜
  ?\╛
  ?\┐
  ?\└
  ?\┴
  ?\┬
  ?\├
  ?\─
  ?\┼
  ?\╞
  ?\╟
  ?\╚
  ?\╔
  ?\╩
  ?\╦
  ?\╠
  ?\═
  ?\╬
  ?\╧
  ?\╨
  ?\╤
  ?\╥
  ?\╙
  ?\╘
  ?\╒
  ?\╓
  ?\╫
  ?\╪
  ?\┘
  ?\┌
  ?\█
  ?\▄
  ?\▌
  ?\▐
  ?\▀
  ?\ω
  ?\ά
  ?\έ
  ?\ή
  ?\ϊ
  ?\ί
  ?\ό
  ?\ύ
  ?\ϋ
  ?\ώ
  ?\Ά
  ?\Έ
  ?\Ή
  ?\Ί
  ?\Ό
  ?\Ύ
  ?\Ώ
  ?\±
  ?\≥
  ?\≤
  ?\Ϊ
  ?\Ϋ
  ?\÷
  ?\≈
  ?\°
  ?\∙
  ?\·
  ?\√
  ?\ⁿ
  ?\²
  ?\■
  ?\ ])
(coding-system-put 'cp737 'mime-charset nil) ; not in IANA list

;;;###autoload(autoload-coding-system 'cp775 '(require 'code-pages))
(cp-make-coding-system
 cp775
 [?\Ć
  ?\ü
  ?\é
  ?\ā
  ?\ä
  ?\ģ
  ?\å
  ?\ć
  ?\ł
  ?\ē
  ?\Ŗ
  ?\ŗ
  ?\ī
  ?\Ź
  ?\Ä
  ?\Å
  ?\É
  ?\æ
  ?\Æ
  ?\ō
  ?\ö
  ?\Ģ
  ?\¢
  ?\Ś
  ?\ś
  ?\Ö
  ?\Ü
  ?\ø
  ?\£
  ?\Ø
  ?\×
  ?\¤
  ?\Ā
  ?\Ī
  ?\ó
  ?\Ż
  ?\ż
  ?\ź
  ?\”
  ?\¦
  ?\©
  ?\®
  ?\¬
  ?\½
  ?\¼
  ?\Ł
  ?\«
  ?\»
  ?\░
  ?\▒
  ?\▓
  ?\│
  ?\┤
  ?\Ą
  ?\Č
  ?\Ę
  ?\Ė
  ?\╣
  ?\║
  ?\╗
  ?\╝
  ?\Į
  ?\Š
  ?\┐
  ?\└
  ?\┴
  ?\┬
  ?\├
  ?\─
  ?\┼
  ?\Ų
  ?\Ū
  ?\╚
  ?\╔
  ?\╩
  ?\╦
  ?\╠
  ?\═
  ?\╬
  ?\Ž
  ?\ą
  ?\č
  ?\ę
  ?\ė
  ?\į
  ?\š
  ?\ų
  ?\ū
  ?\ž
  ?\┘
  ?\┌
  ?\█
  ?\▄
  ?\▌
  ?\▐
  ?\▀
  ?\Ó
  ?\ß
  ?\Ō
  ?\Ń
  ?\õ
  ?\Õ
  ?\µ
  ?\ń
  ?\Ķ
  ?\ķ
  ?\Ļ
  ?\ļ
  ?\ņ
  ?\Ē
  ?\Ņ
  ?\’
  ?\­
  ?\±
  ?\“
  ?\¾
  ?\¶
  ?\§
  ?\÷
  ?\„
  ?\°
  ?\∙
  ?\·
  ?\¹
  ?\³
  ?\²
  ?\■
  ?\ ])

;;;###autoload(autoload-coding-system 'cp850 '(require 'code-pages))
(cp-make-coding-system
 cp850
 [?\Ç
  ?\ü
  ?\é
  ?\â
  ?\ä
  ?\à
  ?\å
  ?\ç
  ?\ê
  ?\ë
  ?\è
  ?\ï
  ?\î
  ?\ì
  ?\Ä
  ?\Å
  ?\É
  ?\æ
  ?\Æ
  ?\ô
  ?\ö
  ?\ò
  ?\û
  ?\ù
  ?\ÿ
  ?\Ö
  ?\Ü
  ?\ø
  ?\£
  ?\Ø
  ?\×
  ?\ƒ
  ?\á
  ?\í
  ?\ó
  ?\ú
  ?\ñ
  ?\Ñ
  ?\ª
  ?\º
  ?\¿
  ?\®
  ?\¬
  ?\½
  ?\¼
  ?\¡
  ?\«
  ?\»
  ?\░
  ?\▒
  ?\▓
  ?\│
  ?\┤
  ?\Á
  ?\Â
  ?\À
  ?\©
  ?\╣
  ?\║
  ?\╗
  ?\╝
  ?\¢
  ?\¥
  ?\┐
  ?\└
  ?\┴
  ?\┬
  ?\├
  ?\─
  ?\┼
  ?\ã
  ?\Ã
  ?\╚
  ?\╔
  ?\╩
  ?\╦
  ?\╠
  ?\═
  ?\╬
  ?\¤
  ?\ð
  ?\Ð
  ?\Ê
  ?\Ë
  ?\È
  ?\ı
  ?\Í
  ?\Î
  ?\Ï
  ?\┘
  ?\┌
  ?\█
  ?\▄
  ?\¦
  ?\Ì
  ?\▀
  ?\Ó
  ?\ß
  ?\Ô
  ?\Ò
  ?\õ
  ?\Õ
  ?\µ
  ?\þ
  ?\Þ
  ?\Ú
  ?\Û
  ?\Ù
  ?\ý
  ?\Ý
  ?\¯
  ?\´
  ?\­
  ?\±
  ?\‗
  ?\¾
  ?\¶
  ?\§
  ?\÷
  ?\¸
  ?\°
  ?\¨
  ?\·
  ?\¹
  ?\³
  ?\²
  ?\■
  ?\ ])

;;;###autoload(autoload-coding-system 'cp851 '(require 'code-pages))
(cp-make-coding-system
 cp851
 [?\Ç
  ?\ü
  ?\é
  ?\â
  ?\ä
  ?\à
  ?\Ά
  ?\ç
  ?\ê
  ?\ë
  ?\è
  ?\ï
  ?\î
  ?\Έ
  ?\Ä
  ?\Ή
  ?\Ί
  nil
  ?\Ό
  ?\ô
  ?\ö
  ?\Ύ
  ?\û
  ?\ù
  ?\Ώ
  ?\Ö
  ?\Ü
  ?\ά
  ?\£
  ?\έ
  ?\ή
  ?\ί
  ?\ϊ
  ?\ΐ
  ?\ό
  ?\ύ
  ?\Α
  ?\Β
  ?\Γ
  ?\Δ
  ?\Ε
  ?\Ζ
  ?\Η
  ?\½
  ?\Θ
  ?\Ι
  ?\«
  ?\»
  ?\░
  ?\▒
  ?\▓
  ?\│
  ?\┤
  ?\Κ
  ?\Λ
  ?\Ν
  ?\Μ
  ?\╣
  ?\║
  ?\╗
  ?\╝
  ?\Ξ
  ?\Ο
  ?\┐
  ?\└
  ?\┴
  ?\┬
  ?\├
  ?\─
  ?\┼
  ?\Π
  ?\Ρ
  ?\╚
  ?\╔
  ?\╩
  ?\╦
  ?\╠
  ?\═
  ?\╬
  ?\Σ
  ?\Τ
  ?\Υ
  ?\Φ
  ?\Χ
  ?\Ψ
  ?\Ω
  ?\α
  ?\β
  ?\γ
  ?\┘
  ?\┌
  ?\█
  ?\▄
  ?\δ
  ?\ε
  ?\▀
  ?\ζ
  ?\η
  ?\θ
  ?\ι
  ?\κ
  ?\λ
  ?\μ
  ?\ν
  ?\ξ
  ?\ο
  ?\π
  ?\ρ
  ?\σ
  ?\ς
  ?\τ
  ?\´
  ?\­
  ?\±
  ?\υ
  ?\φ
  ?\χ
  ?\§
  ?\ψ
  ?\˛
  ?\°
  ?\¨
  ?\ω
  ?\ϋ
  ?\ΰ
  ?\ώ
  ?\■
  ?\ ])

;;;###autoload(autoload-coding-system 'cp852 '(require 'code-pages))
(cp-make-coding-system
 cp852
 [?\Ç
  ?\ü
  ?\é
  ?\â
  ?\ä
  ?\ů
  ?\ć
  ?\ç
  ?\ł
  ?\ë
  ?\Ő
  ?\ő
  ?\î
  ?\Ź
  ?\Ä
  ?\Ć
  ?\É
  ?\Ĺ
  ?\ĺ
  ?\ô
  ?\ö
  ?\Ľ
  ?\ľ
  ?\Ś
  ?\ś
  ?\Ö
  ?\Ü
  ?\Ť
  ?\ť
  ?\Ł
  ?\×
  ?\č
  ?\á
  ?\í
  ?\ó
  ?\ú
  ?\Ą
  ?\ą
  ?\Ž
  ?\ž
  ?\Ę
  ?\ę
  ?\¬
  ?\ź
  ?\Č
  ?\ş
  ?\«
  ?\»
  ?\░
  ?\▒
  ?\▓
  ?\│
  ?\┤
  ?\Á
  ?\Â
  ?\Ě
  ?\Ş
  ?\╣
  ?\║
  ?\╗
  ?\╝
  ?\Ż
  ?\ż
  ?\┐
  ?\└
  ?\┴
  ?\┬
  ?\├
  ?\─
  ?\┼
  ?\Ă
  ?\ă
  ?\╚
  ?\╔
  ?\╩
  ?\╦
  ?\╠
  ?\═
  ?\╬
  ?\¤
  ?\đ
  ?\Đ
  ?\Ď
  ?\Ë
  ?\ď
  ?\Ň
  ?\Í
  ?\Î
  ?\ě
  ?\┘
  ?\┌
  ?\█
  ?\▄
  ?\Ţ
  ?\Ů
  ?\▀
  ?\Ó
  ?\ß
  ?\Ô
  ?\Ń
  ?\ń
  ?\ň
  ?\Š
  ?\š
  ?\Ŕ
  ?\Ú
  ?\ŕ
  ?\Ű
  ?\ý
  ?\Ý
  ?\ţ
  ?\´
  ?\­
  ?\˝
  ?\˛
  ?\ˇ
  ?\˘
  ?\§
  ?\÷
  ?\¸
  ?\°
  ?\¨
  ?\˙
  ?\ű
  ?\Ř
  ?\ř
  ?\■
  ?\ ])

;;;###autoload(autoload-coding-system 'cp855 '(require 'code-pages))
(cp-make-coding-system
 cp855
 [?\ђ
  ?\Ђ
  ?\ѓ
  ?\Ѓ
  ?\ё
  ?\Ё
  ?\є
  ?\Є
  ?\ѕ
  ?\Ѕ
  ?\і
  ?\І
  ?\ї
  ?\Ї
  ?\ј
  ?\Ј
  ?\љ
  ?\Љ
  ?\њ
  ?\Њ
  ?\ћ
  ?\Ћ
  ?\ќ
  ?\Ќ
  ?\ў
  ?\Ў
  ?\џ
  ?\Џ
  ?\ю
  ?\Ю
  ?\ъ
  ?\Ъ
  ?\а
  ?\А
  ?\б
  ?\Б
  ?\ц
  ?\Ц
  ?\д
  ?\Д
  ?\е
  ?\Е
  ?\ф
  ?\Ф
  ?\г
  ?\Г
  ?\«
  ?\»
  ?\░
  ?\▒
  ?\▓
  ?\│
  ?\┤
  ?\х
  ?\Х
  ?\и
  ?\И
  ?\╣
  ?\║
  ?\╗
  ?\╝
  ?\й
  ?\Й
  ?\┐
  ?\└
  ?\┴
  ?\┬
  ?\├
  ?\─
  ?\┼
  ?\к
  ?\К
  ?\╚
  ?\╔
  ?\╩
  ?\╦
  ?\╠
  ?\═
  ?\╬
  ?\¤
  ?\л
  ?\Л
  ?\м
  ?\М
  ?\н
  ?\Н
  ?\о
  ?\О
  ?\п
  ?\┘
  ?\┌
  ?\█
  ?\▄
  ?\П
  ?\я
  ?\▀
  ?\Я
  ?\р
  ?\Р
  ?\с
  ?\С
  ?\т
  ?\Т
  ?\у
  ?\У
  ?\ж
  ?\Ж
  ?\в
  ?\В
  ?\ь
  ?\Ь
  ?\´
  ?\­
  ?\ы
  ?\Ы
  ?\з
  ?\З
  ?\ш
  ?\Ш
  ?\э
  ?\Э
  ?\щ
  ?\Щ
  ?\ч
  ?\Ч
  nil
  ?\■
  ?\ ])

;;;###autoload(autoload-coding-system 'cp857 '(require 'code-pages))
(cp-make-coding-system
 cp857
 [?\Ç
  ?\ü
  ?\é
  ?\â
  ?\ä
  ?\à
  ?\å
  ?\ç
  ?\ê
  ?\ë
  ?\è
  ?\ï
  ?\î
  ?\ı
  ?\Ä
  ?\Å
  ?\É
  ?\æ
  ?\Æ
  ?\ô
  ?\ö
  ?\ò
  ?\û
  ?\ù
  ?\İ
  ?\Ö
  ?\Ü
  ?\ø
  ?\£
  ?\Ø
  ?\Ş
  ?\ş
  ?\á
  ?\í
  ?\ó
  ?\ú
  ?\ñ
  ?\Ñ
  ?\Ğ
  ?\ğ
  ?\¿
  ?\®
  ?\¬
  ?\½
  ?\¼
  ?\¡
  ?\«
  ?\»
  ?\░
  ?\▒
  ?\▓
  ?\│
  ?\┤
  ?\Á
  ?\Â
  ?\À
  ?\©
  ?\╣
  ?\║
  ?\╗
  ?\╝
  ?\¢
  ?\¥
  ?\┐
  ?\└
  ?\┴
  ?\┬
  ?\├
  ?\─
  ?\┼
  ?\ã
  ?\Ã
  ?\╚
  ?\╔
  ?\╩
  ?\╦
  ?\╠
  ?\═
  ?\╬
  ?\¤
  ?\º
  ?\ª
  ?\Ê
  ?\Ë
  ?\È
  nil
  ?\Í
  ?\Î
  ?\Ï
  ?\┘
  ?\┌
  ?\█
  ?\▄
  ?\¦
  ?\Ì
  ?\▀
  ?\Ó
  ?\ß
  ?\Ô
  ?\Ò
  ?\õ
  ?\Õ
  ?\µ
  nil
  ?\×
  ?\Ú
  ?\Û
  ?\Ù
  ?\ì
  ?\ÿ
  ?\—
  ?\´
  ?\­
  ?\±
  nil
  ?\¾
  ?\¶
  ?\§
  ?\÷
  ?\˛
  ?\°
  ?\¨
  ?\˙
  ?\¹
  ?\³
  ?\²
  ?\■
  ?\ ])

;;;###autoload(autoload-coding-system 'cp860 '(require 'code-pages))
(cp-make-coding-system
 cp860
 [?\Ç
  ?\ü
  ?\é
  ?\â
  ?\ã
  ?\à
  ?\Á
  ?\ç
  ?\ê
  ?\Ê
  ?\è
  ?\Î
  ?\Ô
  ?\ì
  ?\Ã
  ?\Â
  ?\É
  ?\À
  ?\È
  ?\ô
  ?\õ
  ?\ò
  ?\Ú
  ?\ù
  ?\Ì
  ?\Õ
  ?\Ü
  ?\¢
  ?\£
  ?\Ù
  ?\₧
  ?\Ò
  ?\á
  ?\í
  ?\ó
  ?\ú
  ?\ñ
  ?\Ñ
  ?\ª
  ?\º
  ?\¿
  ?\Ó
  ?\¬
  ?\½
  ?\¼
  ?\¡
  ?\«
  ?\»
  ?\░
  ?\▒
  ?\▓
  ?\│
  ?\┤
  ?\╡
  ?\╢
  ?\╖
  ?\╕
  ?\╣
  ?\║
  ?\╗
  ?\╝
  ?\╜
  ?\╛
  ?\┐
  ?\└
  ?\┴
  ?\┬
  ?\├
  ?\─
  ?\┼
  ?\╞
  ?\╟
  ?\╚
  ?\╔
  ?\╩
  ?\╦
  ?\╠
  ?\═
  ?\╬
  ?\╧
  ?\╨
  ?\╤
  ?\╥
  ?\╙
  ?\╘
  ?\╒
  ?\╓
  ?\╫
  ?\╪
  ?\┘
  ?\┌
  ?\█
  ?\▄
  ?\▌
  ?\▐
  ?\▀
  ?\α
  ?\ß
  ?\Γ
  ?\π
  ?\Σ
  ?\σ
  ?\µ
  ?\τ
  ?\Φ
  ?\Θ
  ?\Ω
  ?\δ
  ?\∞
  ?\φ
  ?\ε
  ?\∩
  ?\≡
  ?\±
  ?\≥
  ?\≤
  ?\⌠
  ?\⌡
  ?\÷
  ?\≈
  ?\°
  ?\·
  ?\•
  ?\√
  ?\ⁿ
  ?\²
  ?\■
  ?\ ])

;;;###autoload(autoload-coding-system 'cp861 '(require 'code-pages))
(cp-make-coding-system
 cp861
 [?\Ç
  ?\ü
  ?\é
  ?\â
  ?\ä
  ?\à
  ?\å
  ?\ç
  ?\ê
  ?\ë
  ?\è
  ?\Ð
  ?\ð
  ?\Þ
  ?\Ä
  ?\Å
  ?\É
  ?\æ
  ?\Æ
  ?\ô
  ?\ö
  ?\þ
  ?\û
  ?\Ý
  ?\ý
  ?\Ö
  ?\Ü
  ?\ø
  ?\£
  ?\Ø
  ?\₧
  ?\Ò
  ?\á
  ?\í
  ?\ó
  ?\ú
  ?\Á
  ?\Í
  ?\Ó
  ?\Ú
  ?\¿
  nil
  ?\¬
  ?\½
  ?\¼
  ?\¡
  ?\«
  ?\»
  ?\░
  ?\▒
  ?\▓
  ?\│
  ?\┤
  ?\╡
  ?\╢
  ?\╖
  ?\╕
  ?\╣
  ?\║
  ?\╗
  ?\╝
  ?\╜
  ?\╛
  ?\┐
  ?\└
  ?\┴
  ?\┬
  ?\├
  ?\─
  ?\┼
  ?\╞
  ?\╟
  ?\╚
  ?\╔
  ?\╩
  ?\╦
  ?\╠
  ?\═
  ?\╬
  ?\╧
  ?\╨
  ?\╤
  ?\╥
  ?\╙
  ?\╘
  ?\╒
  ?\╓
  ?\╫
  ?\╪
  ?\┘
  ?\┌
  ?\█
  ?\▄
  ?\▌
  ?\▐
  ?\▀
  ?\α
  ?\ß
  ?\Γ
  ?\π
  ?\Σ
  ?\σ
  ?\µ
  ?\τ
  ?\Φ
  ?\Θ
  ?\Ω
  ?\δ
  ?\∞
  ?\φ
  ?\ε
  ?\∩
  ?\≡
  ?\±
  ?\≥
  ?\≤
  ?\⌠
  ?\⌡
  ?\÷
  ?\≈
  ?\°
  ?\·
  ?\•
  ?\√
  ?\ⁿ
  ?\²
  ?\■
  ?\ ])

;;;###autoload(autoload-coding-system 'cp862 '(require 'code-pages))
(cp-make-coding-system
 cp862
 [?\א
  ?\ב
  ?\ג
  ?\ד
  ?\ה
  ?\ו
  ?\ז
  ?\ח
  ?\ט
  ?\י
  ?\ך
  ?\כ
  ?\ל
  ?\ם
  ?\מ
  ?\ן
  ?\נ
  ?\ס
  ?\ע
  ?\ף
  ?\פ
  ?\ץ
  ?\צ
  ?\ק
  ?\ר
  ?\ש
  ?\ת
  ?\¢
  ?\£
  ?\Ù
  ?\₧
  ?\Ò
  ?\á
  ?\í
  ?\ó
  ?\ú
  ?\ñ
  ?\Ñ
  ?\ª
  ?\º
  ?\¿
  nil
  ?\¬
  ?\½
  ?\¼
  ?\¡
  ?\«
  ?\»
  ?\░
  ?\▒
  ?\▓
  ?\│
  ?\┤
  ?\╡
  ?\╢
  ?\╖
  ?\╕
  ?\╣
  ?\║
  ?\╗
  ?\╝
  ?\╜
  ?\╛
  ?\┐
  ?\└
  ?\┴
  ?\┬
  ?\├
  ?\─
  ?\┼
  ?\╞
  ?\╟
  ?\╚
  ?\╔
  ?\╩
  ?\╦
  ?\╠
  ?\═
  ?\╬
  ?\╧
  ?\╨
  ?\╤
  ?\╥
  ?\╙
  ?\╘
  ?\╒
  ?\╓
  ?\╫
  ?\╪
  ?\┘
  ?\┌
  ?\█
  ?\▄
  ?\▌
  ?\▐
  ?\▀
  ?\α
  ?\ß
  ?\Γ
  ?\π
  ?\Σ
  ?\σ
  ?\µ
  ?\τ
  ?\Φ
  ?\Θ
  ?\Ω
  ?\δ
  ?\∞
  ?\φ
  ?\ε
  ?\∩
  ?\≡
  ?\±
  ?\≥
  ?\≤
  ?\⌠
  ?\⌡
  ?\÷
  ?\≈
  ?\°
  ?\·
  ?\•
  ?\√
  ?\ⁿ
  ?\²
  ?\■
  ?\ ])

;;;###autoload(autoload-coding-system 'cp863 '(require 'code-pages))
(cp-make-coding-system
 cp863
 [?\Ç
  ?\ü
  ?\é
  ?\â
  ?\Â
  ?\à
  ?\¶
  ?\ç
  ?\ê
  ?\ë
  ?\è
  ?\ï
  ?\î
  ?\ì
  ?\À
  ?\§
  ?\É
  ?\È
  ?\Ê
  ?\ô
  ?\Ë
  ?\Ï
  ?\û
  ?\ù
  ?\¤
  ?\Ô
  ?\Ü
  ?\¢
  ?\£
  ?\Ù
  ?\Û
  ?\ƒ
  ?\¦
  ?\´
  ?\ó
  ?\ú
  ?\¨
  ?\¸
  ?\³
  ?\¯
  ?\Î
  ?\⌐
  ?\¬
  ?\½
  ?\¼
  ?\¾
  ?\«
  ?\»
  ?\░
  ?\▒
  ?\▓
  ?\│
  ?\┤
  ?\╡
  ?\╢
  ?\╖
  ?\╕
  ?\╣
  ?\║
  ?\╗
  ?\╝
  ?\╜
  ?\╛
  ?\┐
  ?\└
  ?\┴
  ?\┬
  ?\├
  ?\─
  ?\┼
  ?\╞
  ?\╟
  ?\╚
  ?\╔
  ?\╩
  ?\╦
  ?\╠
  ?\═
  ?\╬
  ?\╧
  ?\╨
  ?\╤
  ?\╥
  ?\╙
  ?\╘
  ?\╒
  ?\╓
  ?\╫
  ?\╪
  ?\┘
  ?\┌
  ?\█
  ?\▄
  ?\▌
  ?\▐
  ?\▀
  ?\α
  ?\ß
  ?\Γ
  ?\π
  ?\Σ
  ?\σ
  ?\µ
  ?\τ
  ?\Φ
  ?\Θ
  ?\Ω
  ?\δ
  ?\∞
  ?\∅
  ?\ε
  ?\∩
  ?\≡
  ?\±
  ?\≥
  ?\≤
  ?\⌠
  ?\⌡
  ?\÷
  ?\≈
  ?\∘
  ?\·
  ?\•
  ?\√
  ?\ⁿ
  ?\²
  ?\■
  ?\ ])

;;;###autoload(autoload-coding-system 'cp864 '(require 'code-pages))
(cp-make-coding-system
 cp864
 [?\°
  ?\·
  ?\∘
  ?\√
  ?\▒
  ?\─
  ?\│
  ?\┼
  ?\┤
  ?\┬
  ?\├
  ?\┴
  ?\┐
  ?\┌
  ?\└
  ?\┘
  ?\ß
  ?\∞
  ?\ø
  ?\±
  ?\½
  ?\¼
  ?\≈
  ?\«
  ?\»
  ?\ﻷ
  ?\ﻸ
  nil
  nil
  ?\ﻻ
  ?\ﻼ
  ?\
  nil
  ?\­
  ?\ﺂ
  ?\£
  ?\¤
  ?\ﺄ
  nil
  nil
  ?\ﺎ
  ?\ب
  ?\ت
  ?\ث
  ?\،
  ?\ج
  ?\ح
  ?\خ
  ?\٠
  ?\١
  ?\٢
  ?\٣
  ?\٤
  ?\٥
  ?\٦
  ?\٧
  ?\٨
  ?\٩
  ?\ڤ
  ?\؛
  ?\س
  ?\ش
  ?\ص
  ?\؟
  ?\¢
  ?\ء
  ?\آ
  ?\أ
  ?\ؤ
  ?\ﻊ
  ?\ئ
  ?\ا
  ?\ﺑ
  ?\ة
  ?\ﺗ
  ?\ﺛ
  ?\ﺟ
  ?\ﺣ
  ?\ﺧ
  ?\د
  ?\ذ
  ?\ر
  ?\ز
  ?\ﺳ
  ?\ﺷ
  ?\ﺻ
  ?\ﺿ
  ?\ط
  ?\ظ
  ?\ﻋ
  ?\ﻏ
  ?\¦
  ?\¬
  ?\÷
  ?\×
  ?\ع
  ?\ـ
  ?\ﻒ
  ?\ﻖ
  ?\ﻛ
  ?\ﻞ
  ?\ﻣ
  ?\ﻦ
  ?\ﻫ
  ?\و
  ?\ى
  ?\ﻳ
  ?\ض
  ?\ﻢ
  ?\ﻎ
  ?\غ
  ?\م
  ?\ﹽ
  ?\ّ
  ?\ن
  ?\ه
  ?\ﻬ
  ?\ﻰ
  ?\ﻲ
  ?\ف
  ?\ق
  ?\ﻵ
  ?\ﻶ
  ?\ل
  ?\ك
  ?\ي
  ?\■
  ?\ ])

;;;###autoload(autoload-coding-system 'cp865 '(require 'code-pages))
(cp-make-coding-system
 cp865
 [?\Ç
  ?\ü
  ?\é
  ?\â
  ?\ä
  ?\à
  ?\å
  ?\ç
  ?\ê
  ?\ë
  ?\è
  ?\ï
  ?\î
  ?\ì
  ?\Ä
  ?\Å
  ?\É
  ?\æ
  ?\Æ
  ?\ô
  ?\ö
  ?\ò
  ?\û
  ?\ù
  ?\ÿ
  ?\Ö
  ?\Ü
  ?\ø
  ?\£
  ?\Ø
  ?\₧
  ?\ƒ
  ?\á
  ?\í
  ?\ó
  ?\ú
  ?\ñ
  ?\Ñ
  ?\ª
  ?\º
  ?\¿
  ?\⌐
  ?\¬
  ?\½
  ?\¼
  ?\¡
  ?\«
  ?\»
  ?\░
  ?\▒
  ?\▓
  ?\│
  ?\┤
  ?\╡
  ?\╢
  ?\╖
  ?\╕
  ?\╣
  ?\║
  ?\╗
  ?\╝
  ?\╜
  ?\╛
  ?\┐
  ?\└
  ?\┴
  ?\┬
  ?\├
  ?\─
  ?\┼
  ?\╞
  ?\╟
  ?\╚
  ?\╔
  ?\╩
  ?\╦
  ?\╠
  ?\═
  ?\╬
  ?\╧
  ?\╨
  ?\╤
  ?\╥
  ?\╙
  ?\╘
  ?\╒
  ?\╓
  ?\╫
  ?\╪
  ?\┘
  ?\┌
  ?\█
  ?\▄
  ?\▌
  ?\▐
  ?\▀
  ?\α
  ?\ß
  ?\Γ
  ?\π
  ?\Σ
  ?\σ
  ?\µ
  ?\τ
  ?\Φ
  ?\Θ
  ?\Ω
  ?\δ
  ?\∞
  ?\∅
  ?\ε
  ?\∩
  ?\≡
  ?\±
  ?\≥
  ?\≤
  ?\⌠
  ?\⌡
  ?\÷
  ?\≈
  ?\∘
  ?\·
  ?\•
  ?\√
  ?\ⁿ
  ?\²
  ?\■
  ?\ ])

;;;###autoload(autoload-coding-system 'cp866 '(require 'code-pages))
(cp-make-coding-system
 cp866
 [?\А
  ?\Б
  ?\В
  ?\Г
  ?\Д
  ?\Е
  ?\Ж
  ?\З
  ?\И
  ?\Й
  ?\К
  ?\Л
  ?\М
  ?\Н
  ?\О
  ?\П
  ?\Р
  ?\С
  ?\Т
  ?\У
  ?\Ф
  ?\Х
  ?\Ц
  ?\Ч
  ?\Ш
  ?\Щ
  ?\Ъ
  ?\Ы
  ?\Ь
  ?\Э
  ?\Ю
  ?\Я
  ?\а
  ?\б
  ?\в
  ?\г
  ?\д
  ?\е
  ?\ж
  ?\з
  ?\и
  ?\й
  ?\к
  ?\л
  ?\м
  ?\н
  ?\о
  ?\п
  ?\░
  ?\▒
  ?\▓
  ?\│
  ?\┤
  ?\╡
  ?\╢
  ?\╖
  ?\╕
  ?\╣
  ?\║
  ?\╗
  ?\╝
  ?\╜
  ?\╛
  ?\┐
  ?\└
  ?\┴
  ?\┬
  ?\├
  ?\─
  ?\┼
  ?\╞
  ?\╟
  ?\╚
  ?\╔
  ?\╩
  ?\╦
  ?\╠
  ?\═
  ?\╬
  ?\╧
  ?\╨
  ?\╤
  ?\╥
  ?\╙
  ?\╘
  ?\╒
  ?\╓
  ?\╫
  ?\╪
  ?\┘
  ?\┌
  ?\█
  ?\▄
  ?\▌
  ?\▐
  ?\▀
  ?\р
  ?\с
  ?\т
  ?\у
  ?\ф
  ?\х
  ?\ц
  ?\ч
  ?\ш
  ?\щ
  ?\ъ
  ?\ы
  ?\ь
  ?\э
  ?\ю
  ?\я
  ?\Ё
  ?\ё
  ?\Є
  ?\є
  ?\Ї
  ?\ї
  ?\Ў
  ?\ў
  ?\°
  ?\∙
  ?\·
  ?\√
  ?\№
  ?\¤
  ?\■
  ?\ ]
 "CP866 (Cyrillic)."
 ?A)

;;;###autoload(autoload-coding-system 'cp869 '(require 'code-pages))
(cp-make-coding-system
 cp869
 [nil
  nil
  nil
  nil
  nil
  nil
  ?\Ά
  nil
  ?\·
  ?\¬
  ?\¦
  ?\‛
  ?\’
  ?\Έ
  ?\—
  ?\Ή
  ?\Ί
  ?\Ϊ
  ?\Ό
  nil
  nil
  ?\Ύ
  ?\Ϋ
  ?\©
  ?\Ώ
  ?\²
  ?\³
  ?\ά
  ?\£
  ?\έ
  ?\ή
  ?\ί
  ?\ϊ
  ?\ΐ
  ?\ό
  ?\ύ
  ?\Α
  ?\Β
  ?\Γ
  ?\Δ
  ?\Ε
  ?\Ζ
  ?\Η
  ?\½
  ?\Θ
  ?\Ι
  ?\«
  ?\»
  ?\░
  ?\▒
  ?\▓
  ?\│
  ?\┤
  ?\Κ
  ?\Λ
  ?\Μ
  ?\Ν
  ?\╣
  ?\║
  ?\╗
  ?\╝
  ?\Ξ
  ?\Ο
  ?\┐
  ?\└
  ?\┴
  ?\┬
  ?\├
  ?\─
  ?\┼
  ?\Π
  ?\Ρ
  ?\╚
  ?\╔
  ?\╩
  ?\╦
  ?\╠
  ?\═
  ?\╬
  ?\Σ
  ?\Τ
  ?\Υ
  ?\Φ
  ?\Χ
  ?\Ψ
  ?\Ω
  ?\α
  ?\β
  ?\γ
  ?\┘
  ?\┌
  ?\█
  ?\▄
  ?\δ
  ?\ε
  ?\▀
  ?\ζ
  ?\η
  ?\θ
  ?\ι
  ?\κ
  ?\λ
  ?\μ
  ?\ν
  ?\ξ
  ?\ο
  ?\π
  ?\ρ
  ?\σ
  ?\ς
  ?\τ
  ?\´
  ?\­
  ?\±
  ?\υ
  ?\φ
  ?\χ
  ?\§
  ?\ψ
  ?\΅
  ?\°
  ?\¨
  ?\ω
  ?\ϋ
  ?\ΰ
  ?\ώ
  ?\■
  ?\ ])

;;;###autoload(autoload-coding-system 'cp874 '(require 'code-pages))
(cp-make-coding-system
 cp874
 [?\€
  nil
  nil
  nil
  nil
  ?\…
  nil
  nil
  nil
  nil
  nil
  nil
  nil
  nil
  nil
  nil
  nil
  ?\‘
  ?\’
  ?\“
  ?\”
  ?\•
  ?\–
  ?\—
  nil
  nil
  nil
  nil
  nil
  nil
  nil
  nil
  ?\ 
  ?\ก
  ?\ข
  ?\ฃ
  ?\ค
  ?\ฅ
  ?\ฆ
  ?\ง
  ?\จ
  ?\ฉ
  ?\ช
  ?\ซ
  ?\ฌ
  ?\ญ
  ?\ฎ
  ?\ฏ
  ?\ฐ
  ?\ฑ
  ?\ฒ
  ?\ณ
  ?\ด
  ?\ต
  ?\ถ
  ?\ท
  ?\ธ
  ?\น
  ?\บ
  ?\ป
  ?\ผ
  ?\ฝ
  ?\พ
  ?\ฟ
  ?\ภ
  ?\ม
  ?\ย
  ?\ร
  ?\ฤ
  ?\ล
  ?\ฦ
  ?\ว
  ?\ศ
  ?\ษ
  ?\ส
  ?\ห
  ?\ฬ
  ?\อ
  ?\ฮ
  ?\ฯ
  ?\ะ
  ?\ั
  ?\า
  ?\ำ
  ?\ิ
  ?\ี
  ?\ึ
  ?\ื
  ?\ุ
  ?\ู
  ?\ฺ
  nil
  nil
  nil
  nil
  ?\฿
  ?\เ
  ?\แ
  ?\โ
  ?\ใ
  ?\ไ
  ?\ๅ
  ?\ๆ
  ?\็
  ?\่
  ?\้
  ?\๊
  ?\๋
  ?\์
  ?\ํ
  ?\๎
  ?\๏
  ?\๐
  ?\๑
  ?\๒
  ?\๓
  ?\๔
  ?\๕
  ?\๖
  ?\๗
  ?\๘
  ?\๙
  ?\๚
  ?\๛
  nil
  nil
  nil
  nil])

;;;###autoload(autoload-coding-system 'windows-1250 '(require 'code-pages))
;;;###autoload(autoload-coding-system 'cp1250 '(require 'code-pages))
(cp-make-coding-system
 windows-1250
 [?\€
  nil
  ?\‚
  nil
  ?\„
  ?\…
  ?\†
  ?\‡
  nil
  ?\‰
  ?\Š
  ?\‹
  ?\Ś
  ?\Ť
  ?\Ž
  ?\Ź
  nil
  ?\‘
  ?\’
  ?\“
  ?\”
  ?\•
  ?\–
  ?\—
  nil
  ?\™
  ?\š
  ?\›
  ?\ś
  ?\ť
  ?\ž
  ?\ź
  ?\ 
  ?\ˇ
  ?\˘
  ?\Ł
  ?\¤
  ?\Ą
  ?\¦
  ?\§
  ?\¨
  ?\©
  ?\Ş
  ?\«
  ?\¬
  ?\­
  ?\®
  ?\Ż
  ?\°
  ?\±
  ?\˛
  ?\ł
  ?\´
  ?\µ
  ?\¶
  ?\·
  ?\¸
  ?\ą
  ?\ş
  ?\»
  ?\Ľ
  ?\˝
  ?\ľ
  ?\ż
  ?\Ŕ
  ?\Á
  ?\Â
  ?\Ă
  ?\Ä
  ?\Ĺ
  ?\Ć
  ?\Ç
  ?\Č
  ?\É
  ?\Ę
  ?\Ë
  ?\Ě
  ?\Í
  ?\Î
  ?\Ď
  ?\Đ
  ?\Ń
  ?\Ň
  ?\Ó
  ?\Ô
  ?\Ő
  ?\Ö
  ?\×
  ?\Ř
  ?\Ů
  ?\Ú
  ?\Ű
  ?\Ü
  ?\Ý
  ?\Ţ
  ?\ß
  ?\ŕ
  ?\á
  ?\â
  ?\ă
  ?\ä
  ?\ĺ
  ?\ć
  ?\ç
  ?\č
  ?\é
  ?\ę
  ?\ë
  ?\ě
  ?\í
  ?\î
  ?\ď
  ?\đ
  ?\ń
  ?\ň
  ?\ó
  ?\ô
  ?\ő
  ?\ö
  ?\÷
  ?\ř
  ?\ů
  ?\ú
  ?\ű
  ?\ü
  ?\ý
  ?\ţ
  ?\˙])

;;;###autoload(autoload-coding-system 'windows-1253 '(require 'code-pages))
;;;###autoload(autoload-coding-system 'cp1253 '(require 'code-pages))
(cp-make-coding-system
 windows-1253
 [?\€
  nil
  ?\‚
  ?\ƒ
  ?\„
  ?\…
  ?\†
  ?\‡
  nil
  ?\‰
  nil
  ?\‹
  nil
  nil
  nil
  nil
  nil
  ?\‘
  ?\’
  ?\“
  ?\”
  ?\•
  ?\–
  ?\—
  nil
  ?\™
  nil
  ?\›
  nil
  nil
  nil
  nil
  ?\ 
  ?\΅
  ?\Ά
  ?\£
  ?\¤
  ?\¥
  ?\¦
  ?\§
  ?\¨
  ?\©
  nil
  ?\«
  ?\¬
  ?\­
  ?\®
  ?\―
  ?\°
  ?\±
  ?\²
  ?\³
  ?\΄
  ?\µ
  ?\¶
  ?\·
  ?\Έ
  ?\Ή
  ?\Ί
  ?\»
  ?\Ό
  ?\½
  ?\Ύ
  ?\Ώ
  ?\ΐ
  ?\Α
  ?\Β
  ?\Γ
  ?\Δ
  ?\Ε
  ?\Ζ
  ?\Η
  ?\Θ
  ?\Ι
  ?\Κ
  ?\Λ
  ?\Μ
  ?\Ν
  ?\Ξ
  ?\Ο
  ?\Π
  ?\Ρ
  nil
  ?\Σ
  ?\Τ
  ?\Υ
  ?\Φ
  ?\Χ
  ?\Ψ
  ?\Ω
  ?\Ϊ
  ?\Ϋ
  ?\ά
  ?\έ
  ?\ή
  ?\ί
  ?\ΰ
  ?\α
  ?\β
  ?\γ
  ?\δ
  ?\ε
  ?\ζ
  ?\η
  ?\θ
  ?\ι
  ?\κ
  ?\λ
  ?\μ
  ?\ν
  ?\ξ
  ?\ο
  ?\π
  ?\ρ
  ?\ς
  ?\σ
  ?\τ
  ?\υ
  ?\φ
  ?\χ
  ?\ψ
  ?\ω
  ?\ϊ
  ?\ϋ
  ?\ό
  ?\ύ
  ?\ώ
  nil]
 nil ?g) ;; Greek

;;;###autoload(autoload-coding-system 'windows-1254 '(require 'code-pages))
;;;###autoload(autoload-coding-system 'cp1254 '(require 'code-pages))
(cp-make-coding-system
 windows-1254
 [?\€
  nil
  ?\‚
  ?\ƒ
  ?\„
  ?\…
  ?\†
  ?\‡
  ?\ˆ
  ?\‰
  ?\Š
  ?\‹
  ?\Œ
  nil
  nil
  nil
  nil
  ?\‘
  ?\’
  ?\“
  ?\”
  ?\•
  ?\–
  ?\—
  ?\˜
  ?\™
  ?\š
  ?\›
  ?\œ
  nil
  nil
  ?\Ÿ
  ?\ 
  ?\¡
  ?\¢
  ?\£
  ?\¤
  ?\¥
  ?\¦
  ?\§
  ?\¨
  ?\©
  ?\ª
  ?\«
  ?\¬
  ?\­
  ?\®
  ?\¯
  ?\°
  ?\±
  ?\²
  ?\³
  ?\´
  ?\µ
  ?\¶
  ?\·
  ?\¸
  ?\¹
  ?\º
  ?\»
  ?\¼
  ?\½
  ?\¾
  ?\¿
  ?\À
  ?\Á
  ?\Â
  ?\Ã
  ?\Ä
  ?\Å
  ?\Æ
  ?\Ç
  ?\È
  ?\É
  ?\Ê
  ?\Ë
  ?\Ì
  ?\Í
  ?\Î
  ?\Ï
  ?\Ğ
  ?\Ñ
  ?\Ò
  ?\Ó
  ?\Ô
  ?\Õ
  ?\Ö
  ?\×
  ?\Ø
  ?\Ù
  ?\Ú
  ?\Û
  ?\Ü
  ?\İ
  ?\Ş
  ?\ß
  ?\à
  ?\á
  ?\â
  ?\ã
  ?\ä
  ?\å
  ?\æ
  ?\ç
  ?\è
  ?\é
  ?\ę
  ?\ë
  ?\ė
  ?\í
  ?\î
  ?\ī
  ?\ğ
  ?\ñ
  ?\ò
  ?\ó
  ?\ô
  ?\õ
  ?\ö
  ?\÷
  ?\ø
  ?\ù
  ?\ú
  ?\û
  ?\ü
  ?\ı
  ?\ş
  ?\ÿ])

;; yi_US
;;;###autoload(autoload-coding-system 'windows-1255 '(require 'code-pages))
;;;###autoload(autoload-coding-system 'cp1255 '(require 'code-pages))
(cp-make-coding-system
 windows-1255
 [?\€
  nil
  ?\‚
  ?\ƒ
  ?\„
  ?\…
  ?\†
  ?\‡
  ?\ˆ
  ?\‰
  nil
  ?\‹
  nil
  nil
  nil
  nil
  nil
  ?\‘
  ?\’
  ?\“
  ?\”
  ?\•
  ?\–
  ?\—
  ?\˜
  ?\™
  nil
  ?\›
  nil
  nil
  nil
  nil
  ?\ 
  ?\¡
  ?\¢
  ?\£
  ?\₪
  ?\¥
  ?\¦
  ?\§
  ?\¨
  ?\©
  ?\×
  ?\«
  ?\¬
  ?\­
  ?\®
  ?\¯
  ?\°
  ?\±
  ?\²
  ?\³
  ?\´
  ?\µ
  ?\¶
  ?\·
  ?\¸
  ?\¹
  ?\÷
  ?\»
  ?\¼
  ?\½
  ?\¾
  ?\¿
  ?\ְ
  ?\ֱ
  ?\ֲ
  ?\ֳ
  ?\ִ
  ?\ֵ
  ?\ֶ
  ?\ַ
  ?\ָ
  ?\ֹ
  nil
  ?\ֻ
  ?\ּ
  ?\ֽ
  ?\־
  ?\ֿ
  ?\׀
  ?\ׁ
  ?\ׂ
  ?\׃
  ?\װ
  ?\ױ
  ?\ײ
  ?\׳
  ?\״
  nil
  nil
  nil
  nil
  nil
  nil
  nil
  ?\א
  ?\ב
  ?\ג
  ?\ד
  ?\ה
  ?\ו
  ?\ז
  ?\ח
  ?\ט
  ?\י
  ?\ך
  ?\כ
  ?\ל
  ?\ם
  ?\מ
  ?\ן
  ?\נ
  ?\ס
  ?\ע
  ?\ף
  ?\פ
  ?\ץ
  ?\צ
  ?\ק
  ?\ר
  ?\ש
  ?\ת
  nil
  nil
  ?\‎
  ?\‏
  nil]
 nil ?h) ;; Hebrew

;;;###autoload(autoload-coding-system 'windows-1256 '(require 'code-pages))
;;;###autoload(autoload-coding-system 'cp1256 '(require 'code-pages))
(cp-make-coding-system
 windows-1256
 [?\€
  ?\پ
  ?\‚
  ?\ƒ
  ?\„
  ?\…
  ?\†
  ?\‡
  ?\ˆ
  ?\‰
  ?\ٹ
  ?\‹
  ?\Œ
  ?\چ
  ?\ژ
  ?\ڈ
  ?\گ
  ?\‘
  ?\’
  ?\“
  ?\”
  ?\•
  ?\–
  ?\—
  ?\ک
  ?\™
  ?\ڑ
  ?\›
  ?\œ
  ?\‌
  ?\‍
  ?\ں
  ?\ 
  ?\،
  ?\¢
  ?\£
  ?\¤
  ?\¥
  ?\¦
  ?\§
  ?\¨
  ?\©
  ?\ھ
  ?\«
  ?\¬
  ?\­
  ?\®
  ?\¯
  ?\°
  ?\±
  ?\²
  ?\³
  ?\´
  ?\µ
  ?\¶
  ?\·
  ?\¸
  ?\¹
  ?\؛
  ?\»
  ?\¼
  ?\½
  ?\¾
  ?\؟
  ?\ہ
  ?\ء
  ?\آ
  ?\أ
  ?\ؤ
  ?\إ
  ?\ئ
  ?\ا
  ?\ب
  ?\ة
  ?\ت
  ?\ث
  ?\ج
  ?\ح
  ?\خ
  ?\د
  ?\ذ
  ?\ر
  ?\ز
  ?\س
  ?\ش
  ?\ص
  ?\ض
  ?\×
  ?\ط
  ?\ظ
  ?\ع
  ?\غ
  ?\ـ
  ?\ف
  ?\ق
  ?\ك
  ?\à
  ?\ل
  ?\â
  ?\م
  ?\ن
  ?\ه
  ?\و
  ?\ç
  ?\è
  ?\é
  ?\ê
  ?\ë
  ?\ى
  ?\ي
  ?\î
  ?\ï
  ?\ً
  ?\ٌ
  ?\ٍ
  ?\َ
  ?\ô
  ?\ُ
  ?\ِ
  ?\÷
  ?\ّ
  ?\ù
  ?\ْ
  ?\û
  ?\ü
  ?\‎
  ?\‏
  ?\ے]
 nil ?a) ;; Arabic

;;;###autoload(autoload-coding-system 'windows-1257 '(require 'code-pages))
;;;###autoload(autoload-coding-system 'cp1257 '(require 'code-pages))
(cp-make-coding-system
 windows-1257
 [?\€
  nil
  ?\‚
  nil
  ?\„
  ?\…
  ?\†
  ?\‡
  nil
  ?\‰
  nil
  ?\‹
  nil
  nil
  nil
  nil
  nil
  ?\‘
  ?\’
  ?\“
  ?\”
  ?\•
  ?\–
  ?\—
  nil
  ?\™
  nil
  ?\›
  nil
  nil
  nil
  nil
  ?\ 
  nil
  ?\¢
  ?\£
  ?\¤
  nil
  ?\¦
  ?\§
  ?\Ø
  ?\©
  ?\Ŗ
  ?\«
  ?\¬
  ?\­
  ?\®
  ?\Æ
  ?\°
  ?\±
  ?\²
  ?\³
  nil
  ?\µ
  ?\¶
  ?\·
  ?\ø
  ?\¹
  ?\ŗ
  ?\»
  ?\¼
  ?\½
  ?\¾
  ?\æ
  ?\Ą
  ?\Į
  ?\Ā
  ?\Ć
  ?\Ä
  ?\Å
  ?\Ę
  ?\Ē
  ?\Č
  ?\É
  ?\Ź
  ?\Ė
  ?\Ģ
  ?\Ķ
  ?\Ī
  ?\Ļ
  ?\Š
  ?\Ń
  ?\Ņ
  ?\Ó
  ?\Ō
  ?\Õ
  ?\Ö
  ?\×
  ?\Ų
  ?\Ł
  ?\Ś
  ?\Ū
  ?\Ü
  ?\Ż
  ?\Ž
  ?\ß
  ?\ą
  ?\į
  ?\ā
  ?\ć
  ?\ä
  ?\å
  ?\ę
  ?\ē
  ?\č
  ?\é
  ?\ź
  ?\ė
  ?\ģ
  ?\ķ
  ?\ī
  ?\ļ
  ?\š
  ?\ń
  ?\ņ
  ?\ó
  ?\ō
  ?\õ
  ?\ö
  ?\÷
  ?\ų
  ?\ł
  ?\ś
  ?\ū
  ?\ü
  ?\ż
  ?\ž
  nil])

;;;###autoload(autoload-coding-system 'windows-1258 '(require 'code-pages))
;;;###autoload(autoload-coding-system 'cp1258 '(require 'code-pages))
(cp-make-coding-system
 windows-1258
 [?\€
  nil
  ?\‚
  ?\ƒ
  ?\„
  ?\…
  ?\†
  ?\‡
  ?\ˆ
  ?\‰
  nil
  ?\‹
  ?\Œ
  nil
  nil
  nil
  nil
  ?\‘
  ?\’
  ?\“
  ?\”
  ?\•
  ?\–
  ?\—
  ?\˜
  ?\™
  nil
  ?\›
  ?\œ
  nil
  nil
  ?\Ÿ
  ?\ 
  ?\¡
  ?\¢
  ?\£
  ?\¤
  ?\¥
  ?\¦
  ?\§
  ?\¨
  ?\©
  ?\ª
  ?\«
  ?\¬
  ?\­
  ?\®
  ?\¯
  ?\°
  ?\±
  ?\²
  ?\³
  ?\´
  ?\µ
  ?\¶
  ?\·
  ?\¸
  ?\¹
  ?\º
  ?\»
  ?\¼
  ?\½
  ?\¾
  ?\¿
  ?\À
  ?\Á
  ?\Â
  ?\Ă
  ?\Ä
  ?\Å
  ?\Æ
  ?\Ç
  ?\È
  ?\É
  ?\Ê
  ?\Ë
  ?\̀
  ?\Í
  ?\Î
  ?\Ï
  ?\Đ
  ?\Ñ
  ?\̉
  ?\Ó
  ?\Ô
  ?\Ơ
  ?\Ö
  ?\×
  ?\Ø
  ?\Ù
  ?\Ú
  ?\Û
  ?\Ü
  ?\Ư
  ?\̃
  ?\ß
  ?\à
  ?\á
  ?\â
  ?\ă
  ?\ä
  ?\å
  ?\æ
  ?\ç
  ?\è
  ?\é
  ?\ê
  ?\ë
  ?\́
  ?\í
  ?\î
  ?\ï
  ?\đ
  ?\ñ
  ?\̣
  ?\ó
  ?\ô
  ?\ơ
  ?\ö
  ?\÷
  ?\ø
  ?\ù
  ?\ú
  ?\û
  ?\ü
  ?\ư
  ?\₫
  ?\ÿ])

;;;###autoload(autoload-coding-system 'next '(require 'code-pages))
(cp-make-coding-system
 next
 [?\ 
  ?\À
  ?\Á
  ?\Â
  ?\Ã
  ?\Ä
  ?\Å
  ?\Ç
  ?\È
  ?\É
  ?\Ê
  ?\Ë
  ?\Ì
  ?\Í
  ?\Î
  ?\Ï
  ?\Ð
  ?\Ñ
  ?\Ò
  ?\Ó
  ?\Ô
  ?\Õ
  ?\Ö
  ?\Ù
  ?\Ú
  ?\Û
  ?\Ü
  ?\Ý
  ?\Þ
  ?\µ
  ?\×
  ?\÷
  ?\©
  ?\¡
  ?\¢
  ?\£
  ?\⁄
  ?\¥
  ?\ƒ
  ?\§
  ?\¤
  nil
  ?\“
  ?\«
  nil
  nil
  ?\ﬁ
  ?\ﬂ
  ?\®
  ?\–
  ?\†
  ?\‡
  ?\·
  ?\¦
  ?\¶
  ?\•
  nil
  nil
  ?\”
  ?\»
  ?\…
  ?\‰
  ?\¬
  ?\¿
  ?\¹
  ?\ˋ
  ?\´
  ?\ˆ
  ?\˜
  ?\¯
  ?\˘
  ?\˙
  ?\¨
  ?\²
  ?\˚
  ?\¸
  ?\³
  ?\˝
  ?\˛
  ?\ˇ
  ?\—
  ?\±
  ?\¼
  ?\½
  ?\¾
  ?\à
  ?\á
  ?\â
  ?\ã
  ?\ä
  ?\å
  ?\ç
  ?\è
  ?\é
  ?\ê
  ?\ë
  ?\ì
  ?\Æ
  ?\í
  ?\ª
  ?\î
  ?\ï
  ?\ð
  ?\ñ
  ?\Ł
  ?\Ø
  ?\Œ
  ?\º
  ?\ò
  ?\ó
  ?\ô
  ?\õ
  ?\ö
  ?\æ
  ?\ù
  ?\ú
  ?\û
  ?\ı
  ?\ü
  ?\ý
  ?\ł
  ?\ø
  ?\œ
  ?\ß
  ?\þ
  ?\ÿ
  nil
  nil]
 "NeXTstep encoding." ?N)

;;;###autoload(autoload-coding-system 'koi8-t '(require 'code-pages))
(cp-make-coding-system
 koi8-t					; used by glibc for tg_TJ
 [?\қ
  ?\ғ
  ?\‚
  ?\Ғ
  ?\„
  ?\…
  ?\†
  ?\‡
  nil
  ?\‰
  ?\ҳ
  ?\‹
  ?\Ҳ
  ?\ҷ
  ?\Ҷ
  nil
  ?\Қ
  ?\‘
  ?\’
  ?\“
  ?\”
  ?\•
  ?\–
  ?\—
  nil
  ?\™
  nil
  ?\›
  nil
  nil
  nil
  nil
  nil
  ?\ӯ
  ?\Ӯ
  ?\ё
  ?\¤
  ?\ӣ
  ?\¦
  ?\§
  nil
  nil
  nil
  ?\«
  ?\¬
  ?\­
  ?\®
  nil
  ?\°
  ?\±
  ?\²
  ?\Ё
  nil
  ?\Ӣ
  ?\¶
  ?\·
  nil
  ?\№
  nil
  ?\»
  nil
  nil
  nil
  ?\©
  ?\ю
  ?\а
  ?\б
  ?\ц
  ?\д
  ?\е
  ?\ф
  ?\г
  ?\х
  ?\и
  ?\й
  ?\к
  ?\л
  ?\м
  ?\н
  ?\о
  ?\п
  ?\я
  ?\р
  ?\с
  ?\т
  ?\у
  ?\ж
  ?\в
  ?\ь
  ?\ы
  ?\з
  ?\ш
  ?\э
  ?\щ
  ?\ч
  ?\ъ
  ?\Ю
  ?\А
  ?\Б
  ?\Ц
  ?\Д
  ?\Е
  ?\Ф
  ?\Г
  ?\Х
  ?\И
  ?\Й
  ?\К
  ?\Л
  ?\М
  ?\Н
  ?\О
  ?\П
  ?\Я
  ?\Р
  ?\С
  ?\Т
  ?\У
  ?\Ж
  ?\В
  ?\Ь
  ?\Ы
  ?\З
  ?\Ш
  ?\Э
  ?\Щ
  ?\Ч
  ?\Ъ]
 "Unicode-based KOI8-T encoding for Cyrillic")
(coding-system-put 'koi8-t 'mime-charset nil) ; not in the IANA list
(define-coding-system-alias 'cyrillic-koi8-t 'koi8-t)

;;   Online final ISO draft:

;;     http://www.evertype.com/standards/iso8859/fdis8859-16-en.pdf

;;   Equivalent National Standard:
;;     Romanian Standard SR 14111:1998, Romanian Standards Institution
;;     (ASRO).
 
;; Intended usage:

;;  "This set of coded graphic characters is intended for use in data and
;;   text processing applications and also for information interchange. The
;;   set contains graphic characters used for general purpose applications in
;;   typical office environments in at least the following languages:
;;   Albanian, Croatian, English, Finnish, French, German, Hungarian, Irish
;;   Gaelic (new orthography), Italian, Latin, Polish, Romanian, and
;;   Slovenian. This set of coded graphic characters may be regarded as a
;;   version of an 8-bit code according to ISO/IEC 2022 or ISO/IEC 4873 at
;;   level 1." [ISO 8859-16:2001(E), p. 1]
  
;;   This charset is suitable for use in MIME text body parts.
  
;;   ISO 8859-16 was primarily designed for single-byte encoding the Romanian
;;   language. The UTF-8 charset is the preferred and in today's MIME software
;;   more widely implemented encoding suitable for Romanian.
;;;###autoload(autoload-coding-system 'iso-8859-16 '(require 'code-pages))
(cp-make-coding-system
 iso-latin-10				; consistent with, e.g. Latin-1
 [nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil
  nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil
  ?\ 
  ?\Ą
  ?\ą
  ?\Ł
  ?\€
  ?\„
  ?\Š
  ?\§
  ?\š
  ?\©
  ?\Ș
  ?\«
  ?\Ź
  ?\­
  ?\ź
  ?\Ż
  ?\°
  ?\±
  ?\Č
  ?\ł
  ?\Ž
  ?\”
  ?\¶
  ?\·
  ?\ž
  ?\č
  ?\ș
  ?\»
  ?\Œ
  ?\œ
  ?\Ÿ
  ?\ż
  ?\À
  ?\Á
  ?\Â
  ?\Ă
  ?\Ä
  ?\Ć
  ?\Æ
  ?\Ç
  ?\È
  ?\É
  ?\Ê
  ?\Ë
  ?\Ì
  ?\Í
  ?\Î
  ?\Ï
  ?\Đ
  ?\Ń
  ?\Ò
  ?\Ó
  ?\Ô
  ?\Ő
  ?\Ö
  ?\Ś
  ?\Ű
  ?\Ù
  ?\Ú
  ?\Û
  ?\Ü
  ?\Ę
  ?\Ț
  ?\ß
  ?\à
  ?\á
  ?\â
  ?\ă
  ?\ä
  ?\ć
  ?\æ
  ?\ç
  ?\è
  ?\é
  ?\ê
  ?\ë
  ?\ì
  ?\í
  ?\î
  ?\ï
  ?\đ
  ?\ń
  ?\ò
  ?\ó
  ?\ô
  ?\ő
  ?\ö
  ?\ś
  ?\ű
  ?\ù
  ?\ú
  ?\û
  ?\ü
  ?\ę
  ?\ț
  ?\ÿ]
 "Unicode-based encoding for Latin-10 (MIME: ISO-8859-16)"
 ?r) ;; Romanian
(coding-system-put 'iso-latin-10 'mime-charset 'iso-8859-16)
(define-coding-system-alias 'iso-8859-16 'iso-latin-10)
(define-coding-system-alias 'latin-10 'iso-latin-10)

;; Unicode-based alternative which has the possible advantage of
;; having its relative sparseness specified.
;;;###autoload(autoload-coding-system 'iso-8859-6 '(require 'code-pages))
(cp-make-coding-system
 ;; The base system uses arabic-iso-8bit, but that's not a MIME charset.
 iso-8859-6
 [nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil
  nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil
  ?\ 
  ?\¤
  ?\،
  ?\­
  ?\؛
  ?\؟
  ?\ء
  ?\آ
  ?\أ
  ?\ؤ
  ?\إ
  ?\ئ
  ?\ا
  ?\ب
  ?\ة
  ?\ت
  ?\ث
  ?\ج
  ?\ح
  ?\خ
  ?\د
  ?\ذ
  ?\ر
  ?\ز
  ?\س
  ?\ش
  ?\ص
  ?\ض
  ?\ط
  ?\ظ
  ?\ع
  ?\غ
  ?\ـ
  ?\ف
  ?\ق
  ?\ك
  ?\ل
  ?\م
  ?\ن
  ?\ه
  ?\و
  ?\ى
  ?\ي
  ?\ً
  ?\ٌ
  ?\ٍ
  ?\َ
  ?\ُ
  ?\ِ
  ?\ّ
  ?\ْ
  nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil
  nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil
  nil nil nil nil nil nil nil nil nil nil nil]
 "Unicode-based Arabic ISO/IEC 8859-6 (MIME: ISO-8859-6)"
 ?6)
(define-coding-system-alias 'arabic-iso-8bit 'iso-8859-6)

;;;###autoload(autoload-coding-system 'iso-8859-10 '(require 'code-pages))
(cp-make-coding-system
 iso-latin-6
 [nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil
  nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil
  ?\ 
  ?\Ą
  ?\Ē
  ?\Ģ
  ?\Ī
  ?\Ĩ
  ?\Ķ
  ?\§
  ?\Ļ
  ?\Đ
  ?\Š
  ?\Ŧ
  ?\Ž
  ?\­
  ?\Ū
  ?\Ŋ
  ?\°
  ?\ą
  ?\ē
  ?\ģ
  ?\ī
  ?\ĩ
  ?\ķ
  ?\·
  ?\ļ
  ?\đ
  ?\š
  ?\ŧ
  ?\ž
  ?\―
  ?\ū
  ?\ŋ
  ?\Ā
  ?\Á
  ?\Â
  ?\Ã
  ?\Ä
  ?\Å
  ?\Æ
  ?\Į
  ?\Č
  ?\É
  ?\Ę
  ?\Ë
  ?\Ė
  ?\Í
  ?\Î
  ?\Ï
  ?\Ð
  ?\Ņ
  ?\Ō
  ?\Ó
  ?\Ô
  ?\Õ
  ?\Ö
  ?\Ũ
  ?\Ø
  ?\Ų
  ?\Ú
  ?\Û
  ?\Ü
  ?\Ý
  ?\Þ
  ?\ß
  ?\ā
  ?\á
  ?\â
  ?\ã
  ?\ä
  ?\å
  ?\æ
  ?\į
  ?\č
  ?\é
  ?\ę
  ?\ë
  ?\ė
  ?\í
  ?\î
  ?\ï
  ?\ð
  ?\ņ
  ?\ō
  ?\ó
  ?\ô
  ?\õ
  ?\ö
  ?\ũ
  ?\ø
  ?\ų
  ?\ú
  ?\û
  ?\ü
  ?\ý
  ?\þ
  ?\ĸ]
 "Unicode-based encoding for Latin-6 (MIME: ISO-8859-10)")
(coding-system-put 'iso-latin-6 'mime-charset 'iso-8859-10)
(define-coding-system-alias 'iso-8859-10 'iso-latin-6)
(define-coding-system-alias 'latin-6 'iso-latin-6)

;; used by lt_LT, lv_LV, mi_NZ
;;;###autoload(autoload-coding-system 'iso-8859-13 '(require 'code-pages))
(cp-make-coding-system
 iso-latin-7
 [nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil
  nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil
  ?\ 
  ?\”
  ?\¢
  ?\£
  ?\¤
  ?\„
  ?\¦
  ?\§
  ?\Ø
  ?\©
  ?\Ŗ
  ?\«
  ?\¬
  ?\­
  ?\®
  ?\Æ
  ?\°
  ?\±
  ?\²
  ?\³
  ?\“
  ?\µ
  ?\¶
  ?\·
  ?\ø
  ?\¹
  ?\ŗ
  ?\»
  ?\¼
  ?\½
  ?\¾
  ?\æ
  ?\Ą
  ?\Į
  ?\Ā
  ?\Ć
  ?\Ä
  ?\Å
  ?\Ę
  ?\Ē
  ?\Č
  ?\É
  ?\Ź
  ?\Ė
  ?\Ģ
  ?\Ķ
  ?\Ī
  ?\Ļ
  ?\Š
  ?\Ń
  ?\Ņ
  ?\Ó
  ?\Ō
  ?\Õ
  ?\Ö
  ?\×
  ?\Ų
  ?\Ł
  ?\Ś
  ?\Ū
  ?\Ü
  ?\Ż
  ?\Ž
  ?\ß
  ?\ą
  ?\į
  ?\ā
  ?\ć
  ?\ä
  ?\å
  ?\ę
  ?\ē
  ?\č
  ?\é
  ?\ź
  ?\ė
  ?\ģ
  ?\ķ
  ?\ī
  ?\ļ
  ?\š
  ?\ń
  ?\ņ
  ?\ó
  ?\ō
  ?\õ
  ?\ö
  ?\÷
  ?\ų
  ?\ł
  ?\ś
  ?\ū
  ?\ü
  ?\ż
  ?\ž
  ?\’
  ]
 "Unicode-based encoding for Latin-7 (MIME: ISO-8859-13)"
 ?l) ;; Lithuanian/Latvian
(coding-system-put 'iso-latin-7 'mime-charset 'iso-8859-13)
(define-coding-system-alias 'iso-8859-13 'iso-latin-7)
(define-coding-system-alias 'latin-7 'iso-latin-7)

;; Fixme: check on the C1 characters which libiconv includes.  They
;; are reproduced below, but are probably wrong.  I can't find an
;; official definition of georgian-ps.
;;;###autoload(autoload-coding-system 'georgian-ps '(require 'code-pages))
(cp-make-coding-system
 georgian-ps				; used by glibc for ka_GE
 [?\
  ?\
  ?\‚
  ?\ƒ
  ?\„
  ?\…
  ?\†
  ?\‡
  ?\ˆ
  ?\‰
  ?\Š
  ?\‹
  ?\Œ
  ?\
  ?\
  ?\
  ?\
  ?\‘
  ?\’
  ?\“
  ?\”
  ?\•
  ?\–
  ?\—
  ?\˜
  ?\™
  ?\š
  ?\›
  ?\œ
  ?\
  ?\
  ?\Ÿ
  ?\ 
  ?\¡
  ?\¢
  ?\£
  ?\¤
  ?\¥
  ?\¦
  ?\§
  ?\¨
  ?\©
  ?\ª
  ?\«
  ?\¬
  ?\­
  ?\®
  ?\¯
  ?\°
  ?\±
  ?\²
  ?\³
  ?\´
  ?\µ
  ?\¶
  ?\·
  ?\¸
  ?\¹
  ?\º
  ?\»
  ?\¼
  ?\½
  ?\¾
  ?\¿
  ?\ა
  ?\ბ
  ?\გ
  ?\დ
  ?\ე
  ?\ვ
  ?\ზ
  ?\ჱ
  ?\თ
  ?\ი
  ?\კ
  ?\ლ
  ?\მ
  ?\ნ
  ?\ჲ
  ?\ო
  ?\პ
  ?\ჟ
  ?\რ
  ?\ს
  ?\ტ
  ?\ჳ
  ?\უ
  ?\ფ
  ?\ქ
  ?\ღ
  ?\ყ
  ?\შ
  ?\ჩ
  ?\ც
  ?\ძ
  ?\წ
  ?\ჭ
  ?\ხ
  ?\ჴ
  ?\ჯ
  ?\ჰ
  ?\ჵ
  ?\æ
  ?\ç
  ?\è
  ?\é
  ?\ê
  ?\ë
  ?\ì
  ?\í
  ?\î
  ?\ï
  ?\ð
  ?\ñ
  ?\ò
  ?\ó
  ?\ô
  ?\õ
  ?\ö
  ?\÷
  ?\ø
  ?\ù
  ?\ú
  ?\û
  ?\ü
  ?\ý
  ?\þ
  ?\ÿ]
 nil ?G)
(coding-system-put 'georgian-ps 'mime-charset nil) ; not in IANA list

;;;###autoload(autoload-coding-system 'cp720 '(require 'code-pages))
;; From http://www.microsoft.com/globaldev/reference/oem/720.htm
(cp-make-coding-system
 cp720
 [nil
  nil
  ?\é
  ?\â
  nil
  ?\à
  nil
  ?\ç
  ?\ê
  ?\ë
  ?\è
  ?\ï
  ?\î
  nil
  nil
  nil
  nil
  ?\ّ
  ?\ْ
  ?\ô
  ?\¤
  ?\ـ
  ?\û
  ?\ù
  ?\ء
  ?\آ
  ?\أ
  ?\ؤ
  ?\£
  ?\إ
  ?\ئ
  ?\ا
  ?\ب
  ?\ة
  ?\ت
  ?\ث
  ?\ج
  ?\ح
  ?\خ
  ?\د
  ?\ذ
  ?\ر
  ?\ز
  ?\س
  ?\ش
  ?\ص
  ?\«
  ?\»
  ?\░
  ?\▒
  ?\▓
  ?\│
  ?\┤
  ?\╡
  ?\╢
  ?\╖
  ?\╕
  ?\╣
  ?\║
  ?\╗
  ?\╝
  ?\╜
  ?\╛
  ?\┐
  ?\└
  ?\┴
  ?\┬
  ?\├
  ?\─
  ?\┼
  ?\╞
  ?\╟
  ?\╚
  ?\╔
  ?\╩
  ?\╦
  ?\╠
  ?\═
  ?\╬
  ?\╧
  ?\╨
  ?\╤
  ?\╥
  ?\╙
  ?\╘
  ?\╒
  ?\╓
  ?\╫
  ?\╪
  ?\┘
  ?\┌
  ?\█
  ?\▄
  ?\▌
  ?\▐
  ?\▀
  ?\ض
  ?\ط
  ?\ظ
  ?\ع
  ?\غ
  ?\ف
  ?\µ
  ?\ق
  ?\ك
  ?\ل
  ?\م
  ?\ن
  ?\ه
  ?\و
  ?\ى
  ?\ي
  ?\≡
  ?\ً
  ?\ٌ
  ?\ٍ
  ?\َ
  ?\ُ
  ?\ِ
  ?\≈
  ?\°
  ?\∙
  ?\·
  ?\√
  ?\ⁿ
  ?\²
  ?\■
  ?\ ])
(coding-system-put 'cp720 'mime-charset nil) ; not in IANA list

;;;###autoload(autoload-coding-system 'cp1125 '(require 'code-pages))
;; http://oss.software.ibm.com/cvs/icu/charset/data/ucm/ibm-1125_P100-2000.ucm
(cp-make-coding-system
 cp1125
 [?\А
  ?\Б
  ?\В
  ?\Г
  ?\Д
  ?\Е
  ?\Ж
  ?\З
  ?\И
  ?\Й
  ?\К
  ?\Л
  ?\М
  ?\Н
  ?\О
  ?\П
  ?\Р
  ?\С
  ?\Т
  ?\У
  ?\Ф
  ?\Х
  ?\Ц
  ?\Ч
  ?\Ш
  ?\Щ
  ?\Ъ
  ?\Ы
  ?\Ь
  ?\Э
  ?\Ю
  ?\Я
  ?\а
  ?\б
  ?\в
  ?\г
  ?\д
  ?\е
  ?\ж
  ?\з
  ?\и
  ?\й
  ?\к
  ?\л
  ?\м
  ?\н
  ?\о
  ?\п
  ?\░
  ?\▒
  ?\▓
  ?\│
  ?\┤
  ?\╡
  ?\╢
  ?\╖
  ?\╕
  ?\╣
  ?\║
  ?\╗
  ?\╝
  ?\╜
  ?\╛
  ?\┐
  ?\└
  ?\┴
  ?\┬
  ?\├
  ?\─
  ?\┼
  ?\╞
  ?\╟
  ?\╚
  ?\╔
  ?\╩
  ?\╦
  ?\╠
  ?\═
  ?\╬
  ?\╧
  ?\╨
  ?\╤
  ?\╥
  ?\╙
  ?\╘
  ?\╒
  ?\╓
  ?\╫
  ?\╪
  ?\┘
  ?\┌
  ?\█
  ?\▄
  ?\▌
  ?\▐
  ?\▀
  ?\р
  ?\с
  ?\т
  ?\у
  ?\ф
  ?\х
  ?\ц
  ?\ч
  ?\ш
  ?\щ
  ?\ъ
  ?\ы
  ?\ь
  ?\э
  ?\ю
  ?\я
  ?\Ё
  ?\ё
  ?\Ґ
  ?\ґ
  ?\Є
  ?\є
  ?\І
  ?\і
  ?\Ї
  ?\ї
  ?\·
  ?\√
  ?\№
  ?\¤
  ?\■
  ?\ ])
(define-coding-system-alias 'ruscii 'cp1125)
;; Original name for cp1125, says Serhii Hlodin <hlodin@lutsk.bank.gov.ua>
(define-coding-system-alias 'cp866u 'cp1125)
(coding-system-put 'cp1125 'mime-charset nil)

;; Suggested by Anton Zinoviev <anton@lml.bas.bg>: Bulgarian DOS
;; codepage.  Table at
;; <http://en.wikipedia.org/wiki/MIK_Code_page>.
;;;###autoload(autoload-coding-system 'mik '(require 'code-pages))
(cp-make-coding-system
 mik
 [?А ?Б ?В ?Г ?Д ?Е ?Ж ?З ?И ?Й ?К ?Л ?М ?Н ?О ?П ?Р ?С ?Т ?У ?Ф ?Х ?Ц
  ?Ч ?Ш ?Щ ?Ъ ?Ы ?Ь ?Э ?Ю ?Я ?а ?б ?в ?г ?д ?е ?ж ?з ?и ?й ?к ?л ?м ?н
  ?о ?п ?р ?с ?т ?у ?ф ?х ?ц ?ч ?ш ?щ ?ъ ?ы ?ь ?э ?ю ?я ?└ ?┴ ?┬ ?├ ?─
  ?┼ ?╣ ?║ ?╚ ?╔ ?╩ ?╦ ?╠ ?═ ?╬ ?┐ ?░ ?▒ ?▓ ?│ ?┤ ?№ ?§ ?╗ ?╝ ?┘ ?┌ ?█
  ?▄ ?▌ ?▐ ?▀ ?α ?ß ?Γ ?π ?Σ ?σ ?µ ?τ ?Φ ?Θ ?Ω ?δ ?∞ ?φ ?ε ?∩ ?≡ ?± ?≥
  ?≤ ?⌠ ?⌡ ?÷ ?≈ ?° ?∙ ?· ?√ ?ⁿ ?² ?■ ? ])
(coding-system-put 'mik 'mime-charset nil)

;; Suggested by Anton Zinoviev <anton@lml.bas.bg>: similar to CP1251
;; and used for some non-Slavic Cyrillic languages.  Table found at
;; <URL:ftp://ftp.logic.ru/pub/logic/linux/cyr-asian/PT154>.  See also
;; <URL:http://lists.w3.org/Archives/Public/ietf-charsets/2002AprJun/0092.html,
;; which suggests it's used in an Asian Cyrillic context.
;;;###autoload(autoload-coding-system 'pt154 '(require 'code-pages))
(cp-make-coding-system
 pt154
 [?Җ ?Ғ ?Ӯ ?ғ ?„ ?… ?Ҷ ?Ү ?Ҳ ?ү ?Ҡ ?Ӣ ?Ң ?Қ ?Һ ?Ҹ ?җ ?‘ ?’ ?“ ?” ?• ?–
  ?— ?ҳ ?ҷ ?ҡ ?ӣ ?ң ?қ ?һ ?ҹ ?  ?Ў ?ў ?Ј ?Ө ?Ҙ ?Ұ ?§ ?Ё ?© ?Ә ?\« ?¬ ?ӯ
  ?® ?Ҝ ?° ?ұ ?І ?і ?ҙ ?ө ?¶ ?· ?ё ?№ ?ә ?\» ?ј ?Ҫ ?ҫ ?ҝ ?А ?Б ?В ?Г ?Д
  ?Е ?Ж ?З ?И ?Й ?К ?Л ?М ?Н ?О ?П ?Р ?С ?Т ?У ?Ф ?Х ?Ц ?Ч ?Ш ?Щ ?Ъ ?Ы
  ?Ь ?Э ?Ю ?Я ?а ?б ?в ?г ?д ?е ?ж ?з ?и ?й ?к ?л ?м ?н ?о ?п ?р ?с ?т
  ?у ?ф ?х ?ц ?ч ?ш ?щ ?ъ ?ы ?ь ?э ?ю ?я])
  
;;;###autoload(autoload-coding-system 'iso-8859-11 '(require 'code-pages))
(cp-make-coding-system
 iso-8859-11
 [nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil
  nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil
  ?   ?ก  ?ข  ?ฃ  ?ค  ?ฅ  ?ฆ  ?ง  ?จ  ?ฉ  ?ช  ?ซ  ?ฌ  ?ญ  ?ฎ  ?ฏ
  ?ฐ  ?ฑ  ?ฒ  ?ณ  ?ด  ?ต  ?ถ  ?ท  ?ธ  ?น  ?บ  ?ป  ?ผ  ?ฝ  ?พ  ?ฟ
  ?ภ  ?ม  ?ย  ?ร  ?ฤ  ?ล  ?ฦ  ?ว  ?ศ  ?ษ  ?ส  ?ห  ?ฬ  ?อ  ?ฮ  ?ฯ
  ?ะ  ?ั  ?า  ?ำ  ?ิ  ?ี  ?ึ  ?ื  ?ุ  ?ู  ?ฺ  nil nil nil nil ?฿
  ?เ  ?แ  ?โ  ?ใ  ?ไ  ?ๅ  ?ๆ  ?็  ?่  ?้  ?๊  ?๋  ?์  ?ํ  ?๎  ?๏
  ?๐  ?๑  ?๒  ?๓  ?๔  ?๕  ?๖  ?๗  ?๘  ?๙  ?๚  ?๛   nil nil nil nil]
 "ISO-8859-11.  This is `thai-tis620' with the addition of no-break-space.")

(dotimes (i 9)
  (let ((w (intern (format "windows-125%d" i)))
	(c (intern (format "cp125%d" i))))
    ;; Define cp125* as aliases for all windows-125*, so on Windows
    ;; we can just concat "cp" to the ANSI codepage we get from the system
    ;; and not have to worry about whether it should be "cp" or "windows-".
    (when (coding-system-p w)
      (define-coding-system-alias c w)
      ;; Compatibility with codepage.el, though cp... are not the
      ;; canonical names.
      (if (not (assq c non-iso-charset-alist))
	  (let ((slot (assq w non-iso-charset-alist)))
	    (if slot
		(push (cons c (cdr slot)) non-iso-charset-alist)))))))

(provide 'code-pages)

;;; arch-tag: 8b6e3c73-b271-4198-866d-ea6d0ceff1b2
;;; code-pages.el ends here
