;;; emacs-wiki-journal.el --- Maintain a weblog style journal with emacs-wiki

;; Copyright (C) 2003, 2004 Gary V. Vaughan (gary AT gnu DOT org)
;; Copyright (C) 2004 Ole Arndt (ole AT sugarshark DOT com)
;; Copyright (C) 2004 Hoan Ton-That (hoan AT ton-that DOT org)
;; Copyright (C) 2004 Jose A. Ortega Ruiz (jao AT gnu DOT org)
;; Copyright (C) 2004 Michael Olson (mwolson AT gnu DOT org)

;; Emacs Lisp Archive Entry
;; Filename: emacs-wiki-journal.el
;; Version: 2.66
;; Keywords: hypermedia
;; Author: Gary V. Vaughan (gary AT gnu DOT org)
;; Maintainer: Michael Olson (mwolson AT gnu DOT org)
;; Description: Maintain weblog style journal in a local Emacs Wiki
;; URL: http://www.mwolson.org/projects/EmacsWiki.html
;; Compatibility: XEmacs21

;; This file is not part of GNU Emacs.
;;
;; This is free software; you can redistribute it and/or modify it
;; under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.
;;
;; This is distributed in the hope that it will be useful, but WITHOUT
;; ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
;; FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
;; for more details.
;;
;; You should have received a copy of the GNU General Public License
;; along with GNU Emacs; see the file COPYING.  If not, write to the
;; Free Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,
;; MA 02111-1307, USA.

;;;_* Commentary

;; This file is the part of the Emacs Wiki project that allows you to
;; maintain a journal.

;;;_ + Startup

;; To begin using emacs-wiki-journal, put this in your .emacs file:
;;
;;   (require 'emacs-wiki-journal)
;;
;; Now you can add an entry to your journal with M-x
;; emacs-wiki-journal-add-entry, give it a WikiName for a category to
;; index the entry under (usually beginning "Category") and a header
;; line for the new entry.  A suitable header for today's entry will
;; added to emacs-wiki-journal-wiki where you add the body for the
;; entry and a cross reference will be added to the category index.
;; If you have a PNG icon in the emacs-wiki-category-icons-url then
;; the icon named after the CategoryWiki will be inserted in the
;; journal header.
;;
;; You may need to set up a stylesheet to layout the journal page as
;; you want it.
;;
;; You should also type M-x customize-group, and give the name
;; "emacs-wiki-journal".  Change it to suit your preferences.  Each of
;; the options has its own documentation.

;;;_ + Key Bindings

;; You might want to bind emacs-wiki-journal-add-entry to a key in your
;; global keymap:
;;
;;   (define-key ctl-x-4-map
;;               "j" 'emacs-wiki-journal-add-entry-other-window)
;;
;; And also in emacs-wiki mode:
;;
;;   (add-hook 'emacs-wiki-mode-hook
;;             (lambda ()
;;               (local-set-key "C-cj" 'emacs-wiki-journal-add-entry)))

;;;_ + Code:

;; The parts of this code:
;;
;; * Customization group setup
;; * Category Index maintenance
;; * JournalWiki maintenance

;;;_ + Contributors

;; Gary Vaughan (gary AT gnu DOT org) started the Journal in 2003.
;; The original date for his 0.0.3 version is Fri, 2 May 2003
;;
;; Ole Arndt (ole AT sugarshark DOT com) was the maintainer from April
;; 2004 to September 2004.  He added Category support.
;;
;; Hoan Ton-That (hoan AT ton-that DOT org) contributed multiple
;; journal support.
;;
;; Jose A. Ortega Ruiz (jao AT gnu DOT org) contributed
;; journal-splitting support, which allows the number of entries to
;; keep on a page to be specified.  Old entries are moved to another
;; page.
;;
;; Yamagata Yoriyuki (yoriyuki AT mbg DOT ocn DOT ne DOT jp)
;; contributed a patch that fixed some breakage in
;; `emacs-wiki-journal-category-alist'.

;;;_ + Todo

;; Only add category icons if the icon file exists
;; Search for category icon using emacs-wiki-image-regexp
;; Read in the CategoryWiki using emacs-wiki-read-name
;; Split the JournalWiki up by year and generate annual chronological indices

;;;_* Prerequisites
(require 'cl)
(require 'emacs-wiki)
(require 'parse-time)
(defvar emacs-wiki-journal-loaded nil)

;;;_* Options

(defun emacs-wiki-journal-option-customized (sym val)
  (set sym val)
  (when emacs-wiki-journal-loaded
    (emacs-wiki-journal-update-wiki-project)))

(defgroup emacs-wiki-journal nil
  "Options controlling the behaviour of emacs-wiki journaling.
See `emacs-wiki-journal-add-entry' for more information."
  :group 'emacs-wiki)

(defvar emacs-wiki-journal-project-default-name "JournalWiki"
  "Default name of Journal project.
This is used by `emacs-wiki-journal-update-wiki-project' to make
sure that any old entries are removed correctly.")

(defcustom emacs-wiki-journal-project
  emacs-wiki-journal-project-default-name
  "The name of this project, used when referencing it from other
emacs-wiki projects."
  :type 'string
  :group 'emacs-wiki-journal)

(defcustom emacs-wiki-journal-directory "~/Blog"
  "The directory that contains your journal entries."
  :type 'directory
  :set 'emacs-wiki-journal-option-customized
  :group 'emacs-wiki-journal)

(defcustom emacs-wiki-journal-publishing-directory
  "~/website/blog"
  "The directory where all journal entries are published to."
  :type 'directory
  :set 'emacs-wiki-journal-option-customized
  :group 'emacs-wiki-journal)

(defcustom emacs-wiki-journal-server-prefix
  "../blog/"
  "The location of the publishing directory with respect to the
locations of the publishing directories of other emacs-wiki
projects."
  :type 'directory
  :set 'emacs-wiki-journal-option-customized
  :group 'emacs-wiki-journal)

(defcustom emacs-wiki-journal-wiki "MyJournal"
  "Default prefix of name of the file to which journal entries
are added."
  :type 'string
  :set 'emacs-wiki-journal-option-customized
  :group 'emacs-wiki-journal)

(defcustom emacs-wiki-journal-default-category "CategoryMisc"
  "The default category used when adding entries."
  :type 'string
  :group 'emacs-wiki-journal)

(defcustom emacs-wiki-journal-use-other-window nil
  "If non-nil, emacs-wiki-journal will open in another window."
  :type 'boolean
  :group 'emacs-wiki-journal)

(defcustom emacs-wiki-journal-icons-subdirectory "images"
  "Default base url to a directory containing category icons."
  :type 'string
  :group 'emacs-wiki-journal)

(defcustom emacs-wiki-journal-time-format "%a, %e %b. %2y"
  "Format for the date string of journal entries.
See `format-time-string' for more information."
  :type 'string
  :group 'emacs-wiki-journal)

(defcustom emacs-wiki-journal-category-regexp
  "^Category\\([A-Z][a-z]+\\)+$"
  "Each of the category index Wiki files start with this prefix."
  :type 'string
  :group 'emacs-wiki-journal)

(defcustom emacs-wiki-journal-category-time-format nil
  "Format for the date string of category page entries.
If `nil', `emacs-wiki-journal-time-format' is used.

See `format-time-string' for more information."
  :type '(choice (string :tag "Time string format: ")
                 (const :tag "nil" nil))
  :group 'emacs-wiki-journal)

(defcustom emacs-wiki-journal-title nil
  "Title of the main journal page. If `nil' no title is inserted."
  :type '(choice (string :tag "Title string: ")
                 (const :tag "Not title" nil))
  :group 'emacs-wiki-journal)

(defcustom emacs-wiki-journal-old-title-prefix "Entries "
  "Title prefix of the old journal pages. If `nil' no title is
inserted."
  :type 'string
  :group 'emacs-wiki-journal)

(defcustom emacs-wiki-journal-entries-per-page 4
  "Maximum number of journal entries per page."
  :type 'integer
  :group 'emacs-wiki-journal)

(defcustom emacs-wiki-journal-more-entries-link "(more entries...)"
  "Text used for the more entries link at the end of a journal page."
  :type 'string
  :group 'emacs-wiki-journal)

;;;_* Internal Functions

(defvar emacs-wiki-journal-index-title-threshold t
  "*If nil, filenames are always used in the index.
This is faster, but less informative. If a positive integer,
only that many bytes will be scanned for a #title directive.
Else, the entire wiki file is scanned for a #title.")


;; From planner.el
(defun emacs-wiki-journal-find-file (page &optional command)
  "Open the emacs-wiki PAGE by name.
If COMMAND is non-nil, it is the function used to visit the file."
  (make-directory emacs-wiki-journal-directory t)
  (funcall (or command 'find-file)
           (expand-file-name page emacs-wiki-journal-directory)))

(defun emacs-wiki-journal-category-alist (&optional no-check-p)
  "Return possible category index Wikis in `emacs-wiki-journal-directory'.
If NO-CHECK-P is non-nil, then don't check for changes in the
directories to decide whether to re-read the cached alist, just
re-read the disk."
  (let ((file-alist
         (cadr (assoc emacs-wiki-journal-project
                      emacs-wiki-file-alist)))
        (category-alist nil))
    (save-match-data
      (mapc
       (lambda (file)
         (if (string-match emacs-wiki-journal-category-regexp
                           (car file))
             (setq category-alist (cons file category-alist))))
       file-alist))
    category-alist))

(defun emacs-wiki-journal-prompt-for-category-wiki ()
  "Prompt for a category index file."
  (let ((file-alist (emacs-wiki-journal-category-alist))
        (emacs-wiki-default-page emacs-wiki-journal-default-category))
    (emacs-wiki-read-name file-alist "Category: ")))

(defalias 'emacs-wiki-journal-make-link 'emacs-wiki-make-link)

(defun emacs-wiki-journal-add-category-entry
  (wiki target-url &optional link-description)
  "Find category index file and add an entry for today."
  (emacs-wiki-journal-find-file wiki)
  (undo-boundary)
  (goto-char (point-min))

  ;; skip to first heading
  (let ((p (point)))
    (if (search-forward-regexp "^\\* " (point-max) t)
        (progn  (beginning-of-line)
                (forward-line 1))
      (goto-char p)))

  (goto-char
   (or (search-forward-regexp "^ ?- " (point-max) t) (point)))
  (beginning-of-line)
  (insert (concat " - "
                  (format-time-string
                   (or emacs-wiki-journal-category-time-format
                       emacs-wiki-journal-time-format))
                  " "
                  (emacs-wiki-journal-make-link
                   target-url link-description)
                  "\n")))

(defun emacs-wiki-journal-1+-string (value)
  "Increment an ascii encoded number."
  (int-to-string (1+ (string-to-int value))))

(defun emacs-wiki-journal-update-wiki-project ()
  "Update the \"emacs-wiki-journal\" project in `emacs-wiki-projects'."
  ;; Remove the entry associated with Journal
  (setq emacs-wiki-projects
        (delq (assoc emacs-wiki-journal-project emacs-wiki-projects)
              emacs-wiki-projects))
  ;; Remove an entry that uses the default Journal project name
  (setq emacs-wiki-projects
        (delq (assoc emacs-wiki-journal-project-default-name
                     emacs-wiki-projects)
              emacs-wiki-projects))
  ;; Assign new contents to Journal entry
  (add-to-list 'emacs-wiki-projects
               `(,emacs-wiki-journal-project
                 . ((emacs-wiki-directories
                     . (,emacs-wiki-journal-directory))
                    (emacs-wiki-home-page
                     . ,emacs-wiki-journal-wiki)
                    (emacs-wiki-publishing-directory
                     . ,emacs-wiki-journal-publishing-directory)
                    (emacs-wiki-project-server-prefix
                     . ,emacs-wiki-journal-server-prefix)
                    (emacs-wiki-index-title-threshold
                     . ,emacs-wiki-journal-index-title-threshold))))
  (emacs-wiki-update-project-interwikis))

(defun emacs-wiki-journal-prepare-journal ()
  "Check if maximum number of journal entries reached
and react accordingly."
  (when (and (> emacs-wiki-journal-entries-per-page 0)
             (>= (emacs-wiki-journal-entries)
                emacs-wiki-journal-entries-per-page))
    (emacs-wiki-journal-store-journal-file)
    (emacs-wiki-journal-update-anchors)))

(defun emacs-wiki-journal-entries ()
  "Return the number of journal entries in the main journal file."
  (let ((jf (expand-file-name emacs-wiki-journal-wiki
                              emacs-wiki-journal-directory)))
    (if (file-exists-p jf)
      (with-current-buffer (find-file-noselect jf)
        (save-excursion
          (labels
              ((count-ent (n)
                          (if (search-forward-regexp
                               "^\\*\\* " (point-max) t)
                              (count-ent (1+ n))
                            n)))
            (goto-char (point-min))
            (count-ent 0))))
      0)))

(defun emacs-wiki-journal-store-journal-file ()
  "Move current journal file to a numbered one."
  (let* ((current (expand-file-name
                   emacs-wiki-journal-wiki
                   emacs-wiki-journal-directory))
         (no (emacs-wiki-journal-pages))
         (new (format "%s%d" current (1+ no))))
    (when (file-exists-p current)
      (copy-file current new)
      (emacs-wiki-journal-update-old-journal-title new)
      (with-current-buffer (find-file-noselect current)
        (delete-region (point-min) (point-max))))))

(defun emacs-wiki-journal-update-old-journal-title (file)
  "Update the title of an old journal page."
  (labels
      ((nextday ()
                (and (search-forward-regexp
                      "^\\* \\(.+\\)" (point-max) t)
                     (match-string 1)))
       (lastday (previous)
                (let ((next (nextday)))
                  (if next (lastday next) previous)))
       (mktime (s)
               (let ((tim (parse-time-string s)))
                 (format "%d-%02d-%02d"
                         (nth 5 tim) (nth 4 tim) (nth 3 tim)))))
    (when emacs-wiki-journal-old-title-prefix
      (with-current-buffer (find-file-noselect file)
        (goto-char (point-min))
        (when (search-forward-regexp "^#title " (point-max) t)
          (beginning-of-line)
          (let ((kill-whole-line t)) (kill-line)))
        (let* ((to (nextday))
               (from (and to (lastday to))))
          (goto-char (point-min))
          (insert "#title " emacs-wiki-journal-old-title-prefix
                  (if from (concat (if to "from " " ")
                                   (mktime from) " ") "")
                  (if to (concat (if from "to " " ")
                                 (mktime to)) "")
                  "\n\n")))
      (save-buffer (current-buffer))
      (kill-buffer (current-buffer)))))

(defun emacs-wiki-journal-pages ()
  "Return the current number of journal wiki pages."
  (cdr (emacs-wiki-journal-last-page)))

(defun emacs-wiki-journal-last-page ()
  "Return a pair with the last journal page name and number."
  (let ((final (car-safe
                (last (directory-files emacs-wiki-journal-directory
                                       nil
                                       (concat emacs-wiki-journal-wiki
                                               "[0-9]+\\'"))))))
    (if (or (not final) (not (string-match "[0-9]+\\'" final)))
        (cons "" 0)
      (cons (file-name-nondirectory final)
            (string-to-int (match-string 0 final))))))

(defun emacs-wiki-journal-update-anchors ()
  "Update anchors in category pages to journal wiki file to the
last one stored."
  (mapc
   (lambda (f)
     (let ((buffer (find-file-noselect f))
           (from-string (concat emacs-wiki-journal-wiki "#"))
           (replacement (format "%s%d#"
                                emacs-wiki-journal-wiki
                                (emacs-wiki-journal-pages))))
       (when buffer
         (with-current-buffer buffer
           (goto-char (point-min))
           (while (re-search-forward from-string nil t)
             (replace-match replacement nil nil))
           (save-buffer buffer)
           (kill-buffer nil)))))
   (mapcar 'cdr (emacs-wiki-journal-category-alist))))

(defun emacs-wiki-journal-more-entries-anchor ()
  "Return a wiki anchor to last stored journal page."
  (emacs-wiki-journal-make-link (car (emacs-wiki-journal-last-page))
                                emacs-wiki-journal-more-entries-link))

(defun emacs-wiki-journal-publish-index ()
  "Replacement for `emacs-wiki-publish-index'."
  (interactive)
  (if (not (string-equal emacs-wiki-project
                         emacs-wiki-journal-project))
      (emacs-wiki-publish-index)
    (emacs-wiki-generate-index)))

;;;_* User Functions

;;;###autoload
(defun emacs-wiki-journal-add (category-wiki journal-entry-heading)
  "Add a journal entry under category with a heading"
  (interactive)
  (let* ((category-anchor-base (downcase category-wiki))
         (anchor-regexp (concat "^#"
                                (regexp-quote category-anchor-base)
                                "\\([0-9][0-9]*\\)"))
         (anchor-ord "0"))

    (emacs-wiki-journal-find-file
     emacs-wiki-journal-wiki
     (when emacs-wiki-journal-use-other-window
       'find-file-other-window))

    (goto-char (point-min))

    (when emacs-wiki-journal-title
      (when (not (search-forward-regexp "^#title " (point-max) t))
        (insert "#title " emacs-wiki-journal-title "\n")
        (forward-line)))

    (save-excursion
      (if (search-forward-regexp anchor-regexp (point-max) t)
          (setq anchor-ord
                (emacs-wiki-journal-1+-string
                 (buffer-substring (match-beginning 1)
                                   (match-end 1))))
        (if (and (> emacs-wiki-journal-entries-per-page 0)
                 (> (emacs-wiki-journal-pages) 0)
                 (zerop (emacs-wiki-journal-entries)))
            (insert "\n\n" (emacs-wiki-journal-more-entries-anchor)
                    "\n"))))

    (save-excursion
      (emacs-wiki-journal-add-category-entry
       category-wiki
       (concat emacs-wiki-journal-wiki "#" category-anchor-base anchor-ord)
       journal-entry-heading))

    ;; skip # lines and blanks at the start of the buffer
    (while (and (looking-at "^\\(#\\|\n\\)")
                (equal 0 (forward-line 1))))

    ;; skip to first heading
    (goto-char
     (or (search-forward-regexp "^\\* " (point-max) t) (point)))
    (beginning-of-line)

    (let* ((time-string (format-time-string
                         emacs-wiki-journal-time-format))
           (icon-file-name
            (concat emacs-wiki-journal-icons-subdirectory "/"
                    category-wiki ".png"))
           (icon-link (if (file-exists-p icon-file-name)
                          (concat (emacs-wiki-journal-make-link
                            icon-file-name) " ")
                        nil)))
      ;; add the anchor first so that user is taken above entry
      (insert (concat "#" category-anchor-base anchor-ord "\n"))
      ;; only add a new date if different from the top entry
      (if (not (looking-at (regexp-quote (concat "* " time-string))))
          (insert (concat "* " time-string "\n"))
        (forward-line 1))

      (open-line 1)
      (insert
       (concat
        "** " journal-entry-heading "\n"
        (if icon-link
            (concat icon-link "\n"))
        "*** " category-wiki "\n\n\n")))

    (emacs-wiki-journal-find-file emacs-wiki-journal-wiki)
    ;; move to insertion point
    (forward-line -1)))

;;;###autoload
(defun emacs-wiki-journal-add-entry ()
  "Find journal file and add an entry and category index for today."
  (interactive)
  (let ((category (emacs-wiki-journal-prompt-for-category-wiki))
        (heading (read-from-minibuffer "Journal Heading: ")))
    (emacs-wiki-journal-prepare-journal)
    (emacs-wiki-journal-add category heading)))

;;;###autoload
(defun emacs-wiki-journal-add-entry-other-window ()
  "Find category index file in another window and add an entry
for today."
  (interactive)
  (let ((emacs-wiki-journal-use-other-window t))
    (emacs-wiki-journal-add-entry)))

;;;_* Initialization

(setq emacs-wiki-journal-loaded t)
(emacs-wiki-journal-update-wiki-project)
(provide 'emacs-wiki-journal)

;;;_* Local emacs vars.

;; Local variables:
;; allout-layout: (* 0 : )
;; End:

;;; emacs-wiki-journal.el ends here
