# This file is part of Email-Reminder.
#
# Email-Reminder is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License as
# published by the Free Software Foundation; either version 2 of the
# License, or (at your option) any later version.
#
# Email-Reminder is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with Email-Reminder; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
# 02110-1301, USA.

package EmailReminder::AnniversaryEvent;

use strict;
use warnings;

use EmailReminder::Utils;
use EmailReminder::BirthdayEvent;

our @ISA = ("EmailReminder::BirthdayEvent");

# XML tags, attributes and values
my $PARTNER_NAME_TAG  = 'partner_name';
my $PARTNER_EMAIL_TAG = 'partner_email';

# Hard-coded value for this event's type (class method)
sub get_type
{
    return 'anniversary';
}

# Number of fields this event adds to its parent (class method)
sub get_nb_fields
{
    return EmailReminder::BirthdayEvent->get_nb_fields() + 2;
}

sub get_partner_name
{
    my ($this) = @_;
    return EmailReminder::Utils::get_node_value($this->{XML_NODE}, $PARTNER_NAME_TAG);
}

sub set_partner_name
{
    my ($this, $new_value) = @_;
    return EmailReminder::Utils::set_node_value($this->{XML_NODE}, $PARTNER_NAME_TAG, $new_value);
}

sub get_partner_email
{
    my ($this) = @_;
    return EmailReminder::Utils::get_node_value($this->{XML_NODE}, $PARTNER_EMAIL_TAG);
}

sub set_partner_email
{
    my ($this, $new_value) = @_;
    return EmailReminder::Utils::set_node_value($this->{XML_NODE}, $PARTNER_EMAIL_TAG, $new_value);
}

sub get_message_body
{
    my $this = shift;

    # destination user
    my $first_name = shift;

    # people involved
    my $name          = $this->get_name();
    my $email         = $this->get_email();
    my $partner_name  = $this->get_partner_name();
    my $partner_email = $this->get_partner_email();
    my $occurence     = $this->get_occurence();
    my $th            = get_th($occurence);
    my $special_name  = get_special_name($occurence) || '';
    my $when          = $this->{"WHEN"};

    my $subject = $occurence ? "${occurence}$th anniversary of $name and $partner_name" : "Anniversary of $name and $partner_name";
    my $occurence_string = $occurence ? "${occurence}$th " : "";

    my $email_message = "";
    if ($email && $partner_email) {
        $email_message = "\n\nYou can reach them at $email and $partner_email respectively.";
    } elsif ($email) {
        $email_message = "\n\nYou can reach $name at $email.";
    } elsif ($partner_email) {
        $email_message = "\n\nYou can reach $partner_name at $partner_email.";
    }

    my $message = <<MESSAGEEND;
Subject: $subject

Hi $first_name,

I just want to remind you that the ${occurence_string}anniversary ${special_name}of $name and $partner_name is $when.$email_message
MESSAGEEND

    return $message;
}

# Returns the traditional name for this occurence of the anniversary 
# if applicable (e.g. Silver, Golden, Diamond)
# (see http://www.the-inspirations-store.com/acatalog/anniversary.html)
sub get_special_name
{
    my $occurence = shift;
    return undef unless defined($occurence);

    my @names = ("Paper",   # 1st 
		 "Cotton",  # 2nd 
		 "Leather", # 3rd
		 "Linen",   # 4th
		 "Wood",    # 5th
		 "Iron",    # 6th
		 "Copper",  # 7th
		 "Bronze",  # 8th
		 "Pottery", # 9th
		 "Tin",     # 10th
		 "Steel",   # 11th
		 "Silk",    # 12th
		 "Lace",    # 13th
		 "Ivory",   # 14th
		 "Crystal", # 15th
		 undef, undef, undef, undef, # 16th - 19th
		 "China",   # 20th
		 undef, undef, undef, undef, # 21th - 24th
		 "Silver",  # 25th
		 undef, undef, undef, undef, # 26th - 29th
		 "Pearl",   # 30th
		 undef, undef, undef, undef, # 31th - 34th
		 "Jade",    # 35th
		 undef, undef, undef, undef, # 36th - 39th
		 "Ruby",    # 40th
		 undef, undef, undef, undef, # 41th - 44th
		 "Sapphire",# 45th
		 undef, undef, undef, undef, # 46th - 49th
		 "Golden",  # 50th
		 undef, undef, undef, undef, # 51th - 54th
		 "Emerald", # 55th
		 undef, undef, undef, undef, # 56th - 59th
		 "Diamond", # 60th
		 );

    my $name = $names[$occurence - 1];

    return defined($name) ? "($name) " : undef;
}

1;
