;;; emms-source-file.el --- EMMS sources from the filesystem.

;; Copyright (C) 2003  Jorgen Schfer

;; Author: Jorgen Schfer <forcer@forcix.cx>
;; Keywords: emms, mp3, mpeg, multimedia

;; This file is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.
;;
;; GNU Emacs is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.
;;
;; You should have received a copy of the GNU General Public License
;; along with GNU Emacs; see the file COPYING.  If not, write to the
;; Free Software Foundation, Inc., 59 Temple Place - Suite 330,
;; Boston, MA 02111-1307, USA.

;;; Commentary:

;; This file contains a track source for EMMS that is based on the
;; file system. You can retrieve single files or whole directories.
;; Also, this file offers the commands to play from these sources.

;; TODO:

;;; Code:

;; Version control
(defvar emms-source-file-version "0.2 $Revision: 1.16 $"
  "emms-source-file.el version string")
;; $Id: emms-source-file.el,v 1.16 2004/04/03 11:57:30 forcer Exp $

;;; User Customization

(require 'emms)
(eval-when-compile 
  (condition-case nil
      (require 'locate)
    (error nil)))

(defgroup emms-source-file nil
  "*Sources for EMMS that use the file system."
  :prefix "emms-source-file-"
  :group 'emms-source)

(defcustom emms-source-file-default-directory nil
  "*The default directory to look for media files."
  :type 'string
  :group 'emms-source-file)

(defcustom emms-source-file-directory-tree-function 'emms-source-file-directory-tree-internal
  "*A function to call that searches in a given directory all files
that match a given regex. DIR and REGEX are the only arguments passed
to this function.
You have two build-in options:
`emms-source-file-directory-tree-internal' will work always, but might
be slow.
`emms-source-file-directory-tree-find' will work only if you have GNU
find, but it's faster."
  :type 'function
  :options '(emms-source-file-directory-tree-internal
             emms-source-file-directory-tree-find)
  :group 'emms-source-file)

(defcustom emms-source-file-gnu-find "find"
  "*The program name for GNU find."
  :type 'string
  :group 'emms-source-file)

;;; User Interface

;;;###autoload
(defun emms-play-file ()
  "*Play a file using EMMS."
  (interactive)
  (emms-source-play 'emms-source-file))

;;;###autoload
(defun emms-play-directory ()
  "*Play a single directory using EMMS."
  (interactive)
  (emms-source-play 'emms-source-directory))

;;;###autoload
(defun emms-play-directory-tree ()
  "*Play all files in a directory tree using EMMS."
  (interactive)
  (emms-source-play 'emms-source-directory-tree))

;;;###autoload
(defun emms-play-find ()
  "*Play all files in `emms-source-file-default-directory' that match
a specific regular expression."
  (interactive)
  (emms-source-play 'emms-source-find))

;;;###autoload
(defun emms-play-dired ()
  "Play marked files from the current dired buffer"
  (interactive)
  (emms-source-play 'emms-source-dired))

;;;###autoload
(defun emms-play-playlist-file ()
  "Play all files from a playlist-file."
  (interactive)
  (emms-source-play 'emms-source-playlist-file))

;;; Sources

;;;###autoload
(defun emms-source-file (&optional file)
  "An EMMS source for a single file - either FILE, or queried from the
user."
  (let ((file (or file
                  (read-file-name "Play file: "
                                  emms-source-file-default-directory
                                  emms-source-file-default-directory
                                  t))))
    (if (file-directory-p file)
        (emms-source-directory file)
      (list (emms-track 'file (expand-file-name file))))))

;;;###autoload
(defun emms-source-files (files)
  "An EMMS source for a list of FILES."
  (mapcar (lambda (elt) (car (emms-source-file elt))) files))

;;;###autoload
(defun emms-source-directory (&optional dir)
  "An EMMS source for a whole directory tree - either DIR, or queried
from the user"
  (let ((dir (expand-file-name
              (or dir
                  (read-file-name "Play directory: "
                                  emms-source-file-default-directory
                                  emms-source-file-default-directory
                                  t)))))
    (mapcar (lambda (file)
              (emms-track 'file file))
            (directory-files dir t (emms-source-file-regex)))))

;;;###autoload
(defun emms-source-directory-tree (&optional dir)
  "An EMMS source for multiple directory trees - either DIR, or the
value of `emms-source-file-default-directory'."
  (let ((dir (expand-file-name
              (or dir
                  (read-file-name "Play directory: "
                                  emms-source-file-default-directory
                                  emms-source-file-default-directory
                                  t)))))
    (mapcar (lambda (file)
              (emms-track 'file file))
            (emms-source-file-directory-tree dir (emms-source-file-regex)))))

;;;###autoload
(defun emms-source-find (&optional dir regex)
  "An EMMS source that will find files in DIR or
`emms-source-file-default-directory' that match REGEXP."
  (let ((dir (expand-file-name
              (or dir
                  (read-file-name "Find in directory: "
                                  emms-source-file-default-directory
                                  emms-source-file-default-directory
                                  t))))
        (regex (or regex
                   (read-from-minibuffer "Find files matching: "))))
    (mapcar (lambda (file)
              (emms-track 'file file))
            (emms-source-file-directory-tree dir regex))))

;;; Helper functions

;;;###autoload
(defun emms-source-file-directory-tree (dir regex)
  "Return a list of all files under DIR that match REGEX.
This function uses `emms-source-file-directory-tree-function'."
  (message "Building playlist...")
  (let ((pl (funcall emms-source-file-directory-tree-function
                     dir
                     regex)))
    (message "Building playlist...done")
    pl))

(defun emms-source-file-directory-tree-internal (dir regex)
  "Return a list of all files under DIR that match REGEX.
This function uses only emacs functions, so it might be a bit slow."
  (let ((files '())
        (dirs (list dir)))
    (while dirs
      (cond
       ((file-directory-p (car dirs))
        (if (string-match "/\\.\\.?$" (car dirs))
            (setq dirs (cdr dirs))
          (setq dirs
                (condition-case nil
                    (append (cdr dirs)
                            (directory-files (car dirs)
                                             t nil t))
                  (error
                   (cdr dirs))))))
       ((string-match regex (car dirs))
        (setq files (cons (car dirs) files)
              dirs (cdr dirs)))
       (t
        (setq dirs (cdr dirs)))))
    files))

(defun emms-source-file-directory-tree-find (dir regex)
  "Return a list of all files under DIR that match REGEX.
This function uses the external find utility. The name for GNU find
may be supplied using `emms-source-file-gnu-find'."
  (with-temp-buffer
    (call-process emms-source-file-gnu-find
                  nil t nil
                  (expand-file-name dir)
                  "-type" "f"
                  "-iregex" (concat ".*\\(" regex "\\).*"))
    (delete ""
            (split-string (buffer-substring (point-min)
                                            (point-max))
                          "\n"))))

;;;###autoload
(defun emms-source-playlist-file (&optional file)
  "Return all files from a playlist-file."
  (let ((file (expand-file-name
              (or file
                  (read-file-name "Play playlist-file: "
                                  emms-source-file-default-directory
				  nil
                                  t)))))
    (emms-source-files 
     (with-temp-buffer
       (insert-file-contents-literally file)
       (mapcar
        (function (lambda (line)
                    ;; check if it's relative or absolute
                    (if (or (string-match "^/" line) (string= "" line))
                        ;; absolute, don't touch
                        line
                      (concat (file-name-directory file) line)))) ;; relative, add base path
        (remove "" (split-string (buffer-substring (point-min)
                                        (point-max)) "\n")))))))


;;;###autoload
(defun emms-source-dired ()
  "Return all marked files of a dired buffer"
  (emms-source-files (dired-get-marked-files)))

;;;###autoload
(defun emms-source-file-regex ()
  "Return a regexp that matches everything any player (that supports
files) can play."
  (mapconcat (lambda (player)
               (if (emms-player-do player 'has 'regex)
                   (emms-player-do player 'regex)
                 ""))
             emms-player-list
             "\\|"))

;; Really don't know where to put this, but as the functions for
;; important and playing a playlist are in ths file i suppose it a
;; good place for it.

(defun emms-save-playlist (filename)
  "Export the current playlist as to FILENAME. See also:
`emms-pbi-import-playlist'."
  (interactive "FFile to save playlist as: ")
  (with-temp-file filename
    (mapc (lambda (elt) (insert elt "\n")) emms-playlist)))

;; emms-locate should be part of a once to be emms-dired, with maybe
;; file rename after tag functions and so on, but till then i park it
;; here... :)

;;;###autoload
(defun emms-locate (regexp)
  "Search for REGEXP and display the results in a locate buffer"
  (interactive "sRegexp to search for: ")
  (require 'locate)
  (save-window-excursion 
    (set-buffer (get-buffer-create "*EMMS Find*")) 
    (locate-mode) 
    (erase-buffer)
    (mapc (lambda (elt) (insert (cdr (assoc 'name elt)) "\n")) 
	  (emms-source-find emms-source-file-default-directory regexp)) 
    (locate-do-setup regexp))
  (and (not (string-equal (buffer-name) "*EMMS Find*")) 
       (switch-to-buffer-other-window "*EMMS Find*")) 
  (run-hooks 'dired-mode-hook) 
  (dired-next-line 2))

(provide 'emms-source-file)
;;; emms-source-file.el ends here
