/* *****************************************************************************
 *******************************************************************************
 *
 * Copyright (c) 2007 eMusic.com Inc.
 *
 *   This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your
 * option) any later version.  The GNU Lesser General Public License can be
 * viewed by clicking on the following link:
 *
 *   http://www.gnu.org/licenses/lgpl.html#SEC4 . 
 *
 *   This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more
 * details.
 *
 *   You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the
 *
 *   Free Software Foundation, Inc.,
 *   51 Franklin Street, Fifth Floor,
 *   Boston, MA  02110-1301  USA
 *
 *******************************************************************************
 *
 * Javascript source for the DLM sync services.
 *
 *******************************************************************************
 ******************************************************************************/

/*******************************************************************************
 *******************************************************************************
 *
 * DLM sync configuration.
 *
 *******************************************************************************
 ******************************************************************************/

/*
 * syncCompName                 DLM sync component name.
 * syncAppName                  DLM sync application name.
 * testSyncDir                  Directory to use for sync testing.
 */

const addMediaThreadIfc = new Components.Constructor(ThreadContractID, "nsIThread", "init");

var DLMSyncConfig = {}

DLMSyncConfig.syncCompName = "";
DLMSyncConfig.syncAppName = ""

//DLMSyncConfig.syncCompName = "@emusic.com/DLM/iTunes;1";
//DLMSyncConfig.syncAppName = "iTunes"

//DLMSyncConfig.syncCompName = "@emusic.com/DLM/WMP;1";
//DLMSyncConfig.syncAppName = "WMP"

//DLMSyncConfig.syncCompName = "@emusic.com/DLM/Winamp;1";
//DLMSyncConfig.syncAppName = "Winamp"

/*******************************************************************************
 *******************************************************************************
 *
 * DLM sync services.
 *
 *******************************************************************************
 ******************************************************************************/

var DLMSync =
{
    /***************************************************************************
     *
     * DLM sync services fields.
     *
     **************************************************************************/

    /*
     * mDLMSync                 DLM sync component object.
     * mFileProtocolHandler     nsIFileProtocolHandler object.
     */

    mDLMSync: null,
    mFileProtocolHandler: null,


    /***************************************************************************
     *
     * DLM sync services functions.
     *
     **************************************************************************/

    /*
     * initialize
     *
     *   This function initializes the DLM sync services.
     */

    initialize: function() {
        try { this.initialize1(); }
        catch (err) { debug ("Exception: " + err + "\n"); }
    },

    initialize1: function() {
        /* Get an instance of the DLM sync component and initialize it. */
        this.mDLMSync = Components.classes[DLMSyncConfig.syncCompName].
                            createInstance(Components.interfaces.emIDLMSync);
        this.mDLMSync.initialize();

        /* Do nothing if sync target application is not present. */
        if (!this.mDLMSync.isAppPresent())
        {
            debug(DLMSyncConfig.syncAppName + " not present.\n");
            return;
        }

        /* Get an nsIFileProtocolHandler. */
        this.mFileProtocolHandler =
            Components.classes["@mozilla.org/network/protocol;1?name=file"].
                createInstance(Components.interfaces.nsIFileProtocolHandler);

    },


    /*
     * addMedia
     *
     *   --> mediaList              List of media file paths to add.
     *
     *   This function adds the media items whose file paths are contained in
     * the list specified by mediaPathList.
     *   The paths may either be individual file paths or whole directory paths,
     * in which case all of the media files within the directory will be added.
     */

    addMedia: function(mediaList, trackGenre, albumArt) {
        var                         exMediaList = [];
        var                         i;
        var                         result;

        /* Expand the media list. */
        for (i = 0; i < mediaList.length; i++) {
            //debug(mediaList[i])
            result = this.expandMediaList(exMediaList, mediaList[i]);
        }

        /* Add the media. */
        debug(exMediaList[0])
        //this.mDLMSync.addMedia(exMediaList.length, exMediaList, this);

        if (eMusicPlatform.ident == "WINNT") {
          var th = new addMediaThread({run: function() { DLMSync.mDLMSync.addMedia(exMediaList.length, exMediaList, null, trackGenre, albumArt); }});
        } else {
          this.mDLMSync.addMedia(exMediaList.length, exMediaList, null, trackGenre, albumArt);
        }
    },

    /***************************************************************************
     *
     * DLM sync services emIDLMSyncListener functions.
     *
     **************************************************************************/

    /*
     * addMediaCompleted
     *
     *   --> mediaPath              File path of added media.
     *   --> status                 Status of added media.
     *
     *   This function is called when addition of the media with the file path
     * specified by mediaPath completes, successfully or not.  The final status
     * is specified by status and is set to one of the values in the add media
     * completion status definition set.
     */

    addMediaCompleted: function(mediaPath, status) {
        if (status == Components.interfaces.emIDLMSyncListener.STATUS_SUCCEEDED) {
            dump(  "Added \"" + mediaPath + "\" to "
            + DLMSyncConfig.syncAppName + "\n");
        } else {
            debug(  "Failed to add \"" + mediaPath + "\" to "
                 + DLMSyncConfig.syncAppName + "\n");
        }
    },


    /***************************************************************************
     *
     * Private DLM sync services functions.
     *
     **************************************************************************/

    /*
     * expandMediaList
     *
     *   --> mediaList              Media list into which to expand.
     *   --> mediaPath              Media file path.
     *
     *   This function expands the media list specified by mediaList with the
     * media whose file path is specified by mediaPath.  If the media file path
     * is for a file, it is simply added to the media list.  If the media file
     * path is a directory, its media content is added to the media list.
     */

    expandMediaList: function(mediaList, mediaPath) {
        var                         mediaFile;
        var                         mediaSubFile;
        var                         fileEnum;

        /* Get the media file object. */
        mediaFile = Components.classes["@mozilla.org/file/local;1"].
                            createInstance(Components.interfaces.nsILocalFile);
        mediaFile.initWithPath(mediaPath);

        /* If the media file path is a directory, expand its contents. */
        /* Otherwise, add it to the list if it's not already present.  */
        if (mediaFile.isDirectory()) {
            fileEnum = mediaFile.directoryEntries;
            while (fileEnum.hasMoreElements()) {
                mediaSubFile = fileEnum.getNext().
                                QueryInterface(Components.interfaces.nsIFile);
                this.expandMediaList(mediaList, mediaSubFile.path);
            }
        } else {
            if (   this.isMediaFile(mediaFile)
                && !this.mDLMSync.isMediaPresent(mediaPath)) {
                mediaList.push(mediaPath);
            }
        }
    },


    /*
     * isMediaFile
     *
     *   --> mediaFile              Media nsIFile object.
     *
     *   <--                        True if the file is a media file.
     *
     *   This function checks if the file specified by mediaFile is a media
     * file.  If it is, this function returns true; otherwise, it returns false.
     */

    isMediaFile: function(mediaFile) {
        var                         mediaURI;
        var                         mediaURL;
        var                         fileExt;
        var                         isMediaFile = false;

        /* Get a media nsIURL object. */
        mediaURI = this.mFileProtocolHandler.newFileURI(mediaFile);
        mediaURL = Components.classes["@mozilla.org/network/standard-url;1"].
                        createInstance(Components.interfaces.nsIStandardURL);
        mediaURL.init(Components.interfaces.nsIStandardURL.URLTYPE_STANDARD,
                      0,
                      null,
                      null,
                      mediaURI);
        mediaURL = mediaURL.QueryInterface(Components.interfaces.nsIURL);

        /* Check the file extension for a media type. */
        fileExt = mediaURL.fileExtension.toLowerCase();

        if (fileExt == "mp3")
            isMediaFile = true;

        return (isMediaFile);
    }
};

function addMediaThread(runable) {
  var th = new addMediaThreadIfc(runable, 0, 
                         nsIThread.PRIORITY_HIGH,
                         nsIThread.SCOPE_LOCAL,
                         nsIThread.STATE_JOINABLE);
  return th;
}
