#include <linux/major.h>
#ifndef UNIX98_PTY_MAJOR_COUNT
  #define UNIX98_PTY_MAJOR_COUNT 8
  #ifndef UNIX98_NR_MAJORS
    #define UNIX98_NR_MAJORS=UNIX98_PTY_MAJOR_COUNT
  #endif
#endif

#include <linux/module.h>

#if defined(__GNUC__) && __GNUC__ >= 2
#define _LOOSE_KERNEL_NAMES
#endif

#include <linux/version.h>
#ifndef KERNEL_VERSION
#define KERNEL_VERSION(a,b,c)  (((a) << 16) + ((b) << 8) + (c))
#endif

#include <linux/fs.h>
#include <linux/stat.h>
#include <linux/errno.h>

#include <asm/segment.h>
#if LINUX_VERSION_CODE > KERNEL_VERSION(2,1,0)
#include <asm/uaccess.h>          /* PTB - when did this arrive in kernel? */
#endif
#include <asm/byteorder.h>
#if LINUX_VERSION_CODE > KERNEL_VERSION(2,1,0)
#include <linux/wrapper.h>
#endif

#define MAJOR_NR NBD_MAJOR
static int major = MAJOR_NR;

#include <linux/proc_fs.h>
#include <linux/genhd.h>
#include <linux/hdreg.h>
#if LINUX_VERSION_CODE > KERNEL_VERSION(2,1,0)
#include <linux/file.h>           /* PTB - when did this arrive in kernel? */
#endif
#include <linux/smp_lock.h>
#if LINUX_VERSION_CODE > KERNEL_VERSION(2,2,15)
#include <linux/devfs_fs_kernel.h>
#endif
#include <linux/sysctl.h>
#include <linux/init.h>
#include <linux/blkdev.h>
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,0)
#include <linux/iobuf.h>
#endif
#include <linux/delay.h>
#include <linux/timer.h>

    /* PTB we are included from the kernel nbd.c so put kernel stuff here */

#include <linux/locks.h>
#include <linux/config.h>

#define __NBD_FILE "enbd_raw.c"
#ifndef DEBUG
#define ENBD_COMPILED_DEBUG_LEVEL 0
#else
#define ENBD_COMPILED_DEBUG_LEVEL DEBUG
#endif
#include <linux/enbd.h>

static int
wait_on_condition_timeout(wait_queue_head_t *wq, int (*f)(void *), 
        void *data, unsigned long timeout) {
    unsigned long start_time = jiffies;

    while (! f (data)) {

	if (jiffies >= start_time + timeout) {
            return -ETIME;
        }
        interruptible_sleep_on_timeout (wq, start_time + timeout - jiffies);
    }
    return 0;
}

/*
 * Transfer an iovec
 */
static int
enbd_rw_iovec (struct enbd_device *lo, struct kiobuf *iobuf, int rw,
	      int sector, int nsectors)
{
	struct request *req;
	struct page *page;
	int offset = iobuf->offset, ndone = 0, pageno, result;
        const unsigned long timeout = lo->req_timeo * HZ;

        int local_condition(struct enbd_device *lo) {
                if (down_trylock (&lo->req_sem) == 0)
                        return 1;
                return 0;
        }

        // PTB we're waiting to be able to write the req safely
        if (wait_on_condition_timeout(&lo->req_wq,
                (int(*)(void*))local_condition, lo, timeout) < 0) {
                // PTB it takes too long to get the ioctl lock
                ENBD_ALERT("took too long to get a spare req: TIMEOUT\n");
                return -ETIME;
        }

        // PTB semaphore is now held, prepare the fake request for our queue

        req = &lo->req;
        memset(req, 0, sizeof(*req));

	/* Perform I/O on each sector */
	req->sector = sector;
	req->current_nr_sectors = 1;
	req->cmd = rw;

	for (pageno = 0; pageno < iobuf->nr_pages; pageno++) {
		page = iobuf->maplist[pageno];
		while (ndone < nsectors) {
			/* Fake up a request structure for the operation */
			req->buffer = (void *) (kmap (page) + offset);
                       // PTB queue request for treatment and wait till treated
#ifdef DECLARE_COMPLETION
                        init_completion(&lo->req_x);
                        req->waiting = &lo->req_x;
#endif
                        req->rq_status = RQ_ACTIVE;

                        enbd_enqueue (lo, req);
                        result = enbd_local_wait_on_request_timeout(req, timeout);

			kunmap (page);
			if (result < 0) {
                                up(&lo->req_sem);
				return ndone;
                        }
			/* Move on to the next sector */
			ndone++;
			req->sector++;
			offset += 512;
			if (offset >= PAGE_SIZE) {
				offset = 0;
				break;
			}
		}
	}
        up(&lo->req_sem);
	return ndone;

}

static int
enbd_raw_transfer (struct enbd_device *lo, char *buf, size_t count,
		 loff_t * offset, int rw)
{
	struct kiobuf *iobuf;
	int result;

	/* Only block alignment and size allowed */
	if ((*offset & (512 - 1)) || (count & (512 - 1)))
		return -EINVAL;
	if ((unsigned long) buf & (512 - 1))
		return -EINVAL;

	/* Allocate an I/O vector */
	result = alloc_kiovec (1, &iobuf);
	if (result)
		return result;

	/* Map the user I/O buffer and do the I/O. */
	result = map_user_kiobuf (rw, iobuf, (unsigned long) buf, count);
	if (result) {
		free_kiovec (1, &iobuf);
		return result;
	}
	result =
	 enbd_rw_iovec (lo, iobuf, rw, *offset >> 9, count >> 9);

	/* Clean up and return. */
	unmap_kiobuf (iobuf);
	free_kiovec (1, &iobuf);
	if (result > 0)
		*offset += result << 9;
	return result << 9;
}

/*
 *  Raw read and write syscalls.
 */
ssize_t enbd_raw_read(struct file *filp, char *buf, size_t size, loff_t *off)
{
        int minor = MINOR(filp->f_dentry->d_inode->i_rdev);
        int nbd = minor >> ENBD_SHIFT;
        struct enbd_device *lo = enbd_get_device(nbd);
        return enbd_raw_transfer(lo, buf, size, off, READ);
}

ssize_t enbd_raw_write(struct file *filp, const char *buf, size_t size,
                        loff_t *off)
{
        int minor = MINOR(filp->f_dentry->d_inode->i_rdev);
        int nbd = minor >> ENBD_SHIFT;
        struct enbd_device *lo = enbd_get_device(nbd);
        return enbd_raw_transfer(lo, (char *)buf, size, off, WRITE);
}

static struct file_operations enbd_raw_fops = {
	read:           enbd_raw_read,
	write:          enbd_raw_write,
	open:           enbd_open,
	release:        enbd_release,
	ioctl:          enbd_ioctl,
};
static int enbd_raw_registered = 0;

static int __init
enbd_raw_init (void)
{
	int result;
	SET_MODULE_OWNER (&enbd_raw_fops);
	result = register_chrdev (major, "enbd", &enbd_raw_fops);
	if (result < 0) {
                ENBD_ERROR("could not register char device on major %d\n", MAJOR_NR);
                return -EIO;
        }
        enbd_raw_registered = 1;
        return 0;
}

static void __exit
enbd_raw_cleanup(void)
{
        if (enbd_raw_registered)
                unregister_chrdev(major, "enbd");
}

#ifdef MODULE
  #if LINUX_VERSION_CODE > KERNEL_VERSION(2,1,0)
    MODULE_AUTHOR ("Peter T. Breuer, Andres Marin");
    MODULE_DESCRIPTION ("Enhanced Network Block Device raw character interface"
            ENBD_VERSION);
    #ifdef MODULE_LICENSE
      MODULE_LICENSE("GPL");
    #endif
  #endif
#endif		/* MODULE */



module_init (enbd_raw_init);
module_exit (enbd_raw_cleanup);

