
#include <linux/enbd.h>
#include <linux/proc_fs.h>

extern struct enbd_device * enbd_get(int i);

static void
do_reset (int reset, int i) {

	void do_r (void) {
		struct enbd_device *lo = enbd_get(i);
		if (reset != 0) {
			lo->reset (lo, i);
			return;
		};
	};
	if (i >= 0 && i < MAX_NBD) {
		do_r ();
		return;
	}
}

/*
 * PTB This is just to get a nice limited width integer printout in proc!
 * use endpos (<= 8) spaces at most. We serve from a static buffer size 16.
 */
char *
display (unsigned n, int endpos)
{
	// PTB  use endpos (<= 8) spaces at most
	static char buf[16];
	int units = 0;
	int decimals = 0;
	int decpos = endpos;
	int wholepart = n, fractionpart = 0;
	buf[endpos--] = 0;
	// PTB  find the right units to display. U or K or M or G.
	while (n >= 1 << 10) {
		decimals = n & ((1 << 10) - 1);
		n >>= 10;
		units++;
	}
	switch (units) {
	  case 0:
		break;
	  case 1:
		buf[endpos--] = 'K';
		break;
	  case 2:
		buf[endpos--] = 'M';
		break;
	  case 3:
		buf[endpos--] = 'G';
		break;
	  case 4:
		buf[endpos--] = 'T';
		break;
	}
	// after this wholepart = n && fractionpart = decimals
	fractionpart = wholepart & ((1 << (units * 10)) - 1);
	wholepart >>= units * 10;
	// PTB  write the whole digits (something between 0 and 1023 inclusive)
	if (n == 0) {
		buf[endpos--] = '0';
	} else {
		while (endpos >= 0 && n > 0) {
			buf[endpos--] = '0' + n % 10;
			n /= 10;
		}
	}
	// PTB if there is space and cause, add decimal digits
	if (endpos >= 1 && units > 0) {
		int k = 0;
		char unitchar = buf[--decpos];
		buf[decpos + k++] = '.';
		while (endpos >= k) {
			int digit = (decimals * 10) >> 10;
			buf[decpos + k++] = '0' + digit;
			decimals -= (digit << 10) / 10;
			decimals *= 10;
		}
		buf[decpos + k++] = unitchar;
		buf[decpos + k] = 0;
	}
	// PTB report the start position
	return buf + endpos + 1;
}


static void
set_generic (int x, int i, int X)
{
	void set_x (void) {
		struct enbd_device *lo = enbd_get(i);
                if (lo->magic != ENBD_DEV_MAGIC)
                        return;
		if (x != 0) {
			atomic_set_mask (X, &lo->flags);
			return;
		};
                atomic_clear_mask (X, &lo->flags);
	};

	if (i >= 0 && i < MAX_NBD) {
		set_x ();
		return;
	}
	for (i = 0; i < MAX_NBD; i++) {
		set_x ();
	}
}

static void
set_sync_intvl (int sync_intvl, int i)
{
    set_generic(sync_intvl, i, ENBD_SYNC);
}


static void
set_show_errs (int show_errs, int i)
{
    set_generic(show_errs, i, ENBD_SHOW_ERRS);
}

static void
set_md5sum (int md5sum, int i)
{
    set_generic(md5sum, i, ENBD_MD5SUM);
}


static void
set_enable (int enable, int i)
{
	void set_e (void) {
		struct enbd_device *lo = enbd_get(i);
                if (!lo || lo->magic != ENBD_DEV_MAGIC)
                        return;
		if (enable != 0) {
                        if (!(atomic_read (&lo->flags) & ENBD_ENABLED)) {
			        lo->enable (lo);
			        return;
                        }
		};
                lo->disable (lo);
	};

	if (i >= 0 && i < MAX_NBD) {
		set_e ();
		return;
	}
	for (i = 0; i < MAX_NBD; i++) {
		set_e ();
	}
}

static void
set_direct (int direct, int i)
{
        set_generic(direct, i, ENBD_DIRECT);
}

#ifndef NO_BUFFERED_WRITES
static void
set_buffer_writes (int buffer_writes, int i)
{
        set_generic(buffer_writes, i, ENBD_BUFFERWR);
}
#endif

static void
set_merge_requests (int mr, int i)
{
	void set_mr (void) {
		struct enbd_device *lo = enbd_get(i);
		if (lo->magic != ENBD_DEV_MAGIC)
			 return;
		 atomic_set (&lo->merge_requests, mr);
	}
        
        if (i >= 0 && i < MAX_NBD) {
		set_mr ();
		return;
	}
	for (i = 0; i < MAX_NBD; i++) {
		set_mr ();
	}
}

int
enbd_read_proc (char *buf, char **start, off_t offset, int len, int *eof,
	       void *data)
{

#ifndef MIN
#define MIN(x,y) ((x)<(y)?(x):(y))
#endif

	const int limit = MIN (PAGE_SIZE, len) - 80;
	static int i;
	struct enbd_device *lo;
	struct enbd_acct *acct;
	static int last;
	static void *next_label;
	static char *next_label_name;
	static int total;
	unsigned long flags;

	if (offset > 0 && !next_label) {
		*eof = 1;
		*start = buf;
		return 0;
	}

	if (offset <= 0) {
		// PTB do static inits first time through
		last = -1;
		i = 0;
		next_label = NULL;
		next_label_name = NULL;
		total = 0;
	}

	// PTB  start this bytecount
	len = 0;

#define ENBD_PROC_LABEL(n) \
        next_label = &&label_##n; \
        next_label_name = "label_" #n; \
        if (len > limit) { \
            *start = (char *) (unsigned long) len; \
            total += len; \
            return len;\
        } \
        label_##n:

	for ( /* static init */ ; i < MAX_NBD; i++) {

		char *devnam;

		lo = enbd_get(i);
		devnam = lo->devnam;
                acct = &lo->acct;
		if (lo->nslot <= 0) {
			next_label = NULL;
			continue;
		}

		// PTB computed goto next not-done
		if (next_label) {
			void *label = next_label;
			next_label = NULL;
			next_label_name = NULL;
			len = 0;
			goto *label;
		}

		ENBD_PROC_LABEL (1);

		if (last == i - 2) {
                        struct enbd_device * lo = enbd_get (i - 1);
			char *prevdevnam = lo->devnam;
			len +=
			 sprintf (buf + len, "Device %s:\tClosed\n",
				  prevdevnam);
		}
		if (last < i - 2) {
                        struct enbd_device * llo = enbd_get (last + 1);
                        struct enbd_device * plo = enbd_get (i - 1);
			char lastdevnam[3];
			char prevdevnam[3];
			strncpy (lastdevnam, llo->devnam, 3);
			strncpy (prevdevnam, plo->devnam, 3);
			len +=
			 sprintf (buf + len, "Device %s-%s:\tClosed\n",
				  lastdevnam, prevdevnam);
		}

		ENBD_PROC_LABEL (2);

		len +=
		 sprintf (buf + len, "Device %s:\tOpen " "\n", devnam);

		ENBD_PROC_LABEL (3);

		len += sprintf (buf + len,
				"[%s] State:\t%s%s%s%s%s%s%s%s%s%s%s%slast error %d, lives %d, bp %d\n",
				devnam,
                                atomic_read (&lo->flags)
				& ENBD_INITIALISED ? "" : "uninitialized, ",
				atomic_read (&lo->flags)
				& ENBD_WRITE_NOCHK ? "noverify, " : "verify, ",
				atomic_read (&lo->flags)
                                & ENBD_READ_ONLY ? "ro, " : "rw, ",
				atomic_read(&lo->merge_requests) ? "merge requests, " : "",
#ifndef NO_BUFFERED_WRITES
				atomic_read (&lo->flags)
				& ENBD_BUFFERWR ? "buffer writes, " : "",
#else
				"",
#endif		/* NO_BUFFERED_WRITES */
				atomic_read (&lo->flags)
				& ENBD_ENABLED ? "enabled, " : "disabled, ",
				atomic_read (&lo->flags)
				& ENBD_VALIDATED ? "validated, " : "",
                                atomic_read (&lo->flags)
				& ENBD_REMOTE_INVALID ? "remote invalid, " : "",
				atomic_read (&lo->flags)
				& ENBD_SHOW_ERRS ? "show_errs, " : "",
			        atomic_read (&lo->flags)
                                & ENBD_DIRECT ? "direct, " : "",
				atomic_read (&lo->flags)
				& ENBD_SYNC ? "sync, " : "",
				atomic_read (&lo->flags)
				& ENBD_MD5SUM ? "md5sum, " : "",
				lo->harderror,
				lo->lives - ((atomic_read (&lo->flags) & ENBD_ENABLED) ?  1 : 0), 0);

		ENBD_PROC_LABEL (4);

		do {		// PTB begin long do once block
			int countq[2] = { 0, 0 };
			int cmd;

			struct list_head *pos;

			read_lock_irqsave (&lo->queue_lock, flags);

			list_for_each (pos, &lo->queue) {
				struct request *req =
				 list_entry (pos, struct request, queuelist);
				if (countq[READ] + countq[WRITE] > 1000)
					break;

				cmd = rq_data_dir (req);
				countq[cmd]++;
			}

			read_unlock_irqrestore (&lo->queue_lock, flags);

			len += sprintf (buf + len,
					"[%s] Queued:\t+%dR/%dW curr (check %dR/%dW) +%dR/%dW max\n",
					devnam,
					atomic_read (&acct->countq[READ]),
					atomic_read (&acct->countq[WRITE]),
					countq[READ], countq[WRITE],
					atomic_read (&acct->maxq[READ]),
					atomic_read (&acct->maxq[WRITE]));
		} while (0);	// PTB end long do once block

		ENBD_PROC_LABEL (5);

		len += sprintf (buf + len,
				"[%s] Buffersize:\t%d\t(sectors=%d, blocks=%d)\n",
				devnam, lo->bufsiz, lo->max_sectors,
				lo->max_sectors / (lo->blksize >> 9));
		len +=
		 sprintf (buf + len, "[%s] Blocksize:\t%d\t(log=%d)\n",
			  devnam, lo->blksize, lo->logblksize);
		len +=
		 sprintf (buf + len, "[%s] Size:\t%luKB\n", devnam,
			  (unsigned long) (lo->bytesize >> 10));
		len +=
		 sprintf (buf + len, "[%s] Blocks:\t%u\n", devnam,
			  lo->size >> (lo->logblksize - 10));

		ENBD_PROC_LABEL (6);

		len +=
		 sprintf (buf + len, "[%s] Sockets:\t%d", devnam, lo->nslot);

		ENBD_PROC_LABEL (7);

		do {		// PTB begin short do once block
			int j;
			for (j = 0; j < lo->nslot; j++) {
				struct enbd_slot *slotj = &lo->slots[j];
				if (j != atomic_read (&lo->islot)) {
					len +=
					 sprintf (buf + len, "\t(%s)",
						  slotj->file ? "+" : "-");
                                } else {
					len +=
					 sprintf (buf + len, "\t(%s)",
						  slotj->file ? "*" : ".");
                                }
			}
		} while (0);	// PTB end short do once block

		len += sprintf (buf + len, "\n");

		ENBD_PROC_LABEL (8);

		len += sprintf (buf + len, "[%s] Requested:\t%s", devnam,
		        display (atomic_read(&acct->requests_in[READ]) +
				 atomic_read(&acct->requests_in[WRITE]), 7));

		ENBD_PROC_LABEL (9);

		do {		// PTB begin short do once block
			int j;
			char buff[2][8];
			for (j = 0; j < lo->nslot; j++) {
				struct enbd_slot *slotj = &lo->slots[j];
				len +=
				 sprintf (buf + len, "\t(%s)",
					  display (slotj->in, 5));
			}
			strncpy (buff[0],
				 display (atomic_read
					  (&acct->requests_in[READ]), 6), 7);
			strncpy (buff[1],
				 display (atomic_read
					  (&acct->requests_in[WRITE]), 6),
				 7);
			len +=
			 sprintf (buf + len, "\t%sR/%sW", buff[0],
				  buff[1]);
			lo->set_speed (lo);
			len += sprintf (buf + len, "\tmax %d",
					atomic_read (&acct->maxreqblks));
		} while (0);	// PTB end short do once block

		len += sprintf (buf + len, "\n");
		len += sprintf (buf + len, "[%s] Despatched:\t%s", devnam,
			display (atomic_read(&acct->requests_out[READ]) +
			         atomic_read(&acct->requests_out[WRITE]), 7));

		ENBD_PROC_LABEL (10);

		do {		// PTB begin short do once block
			int j;
			char buff[2][8];
			for (j = 0; j < lo->nslot; j++) {
				struct enbd_slot *slotj = &lo->slots[j];
				len +=
				 sprintf (buf + len, "\t(%s)",
					  display (slotj->out, 5));
			}
			strncpy (buff[0],
			 display (atomic_read(&acct->requests_out[READ]), 6), 7);
			strncpy (buff[1],
		         display (atomic_read(&acct->requests_out[WRITE]), 6), 7);
			len +=
			 sprintf (buf + len, "\t%sR/%sW", buff[0], buff[1]);
			len +=
			 sprintf (buf + len, "\tmd5 %sW",
				  display (atomic_read(&lo->wrequests_5to), 5));
			len +=
			 sprintf (buf + len, " (%s eq,",
				 display (atomic_read (&lo->wrequests_5so), 5));
			len +=
			 sprintf (buf + len, " %s ne,",
				 display (atomic_read(&lo->wrequests_5wo), 5));
			len +=
			 sprintf (buf + len, " %s dn)",
				 display (atomic_read (&lo->wrequests_5eo), 5));
		} while (0);	// PTB end short do once block

		len += sprintf (buf + len, "\n");
		len += sprintf (buf + len, "[%s] Errored:\t%s", devnam,
				display (atomic_read (&acct->requests_err), 7));

		ENBD_PROC_LABEL (11);

		do {		// PTB begin short do once block
			int j;
			char buff[2][8];
			int toterrs = 0;

			for (j = 0; j < lo->nslot; j++) {
				struct enbd_slot *slotj = &lo->slots[j];
				len +=
				 sprintf (buf + len, "\t(%s)",
					  display (slotj->err, 5));
				toterrs += slotj->err;
			}
			strncpy (buff[0], display (toterrs, 6), 7);
			strncpy (buff[1],
		         display (atomic_read (&acct->requests_err)-toterrs, 6), 7);
			len +=
			 sprintf (buf + len, "\t%s+%s\n", buff[0],
				  buff[1]);
		} while (0);	// PTB end short do once block

		ENBD_PROC_LABEL (12);

		do {		// PTB begin long do once block
			int pending_rblks = 0;	/* PTB  reads not reached the slots yet */
			int pending_wblks = 0;	/* PTB  writes not reached the slots yet */
			int blks = 0;

			read_lock_irqsave (&lo->queue_lock, flags);

			do {	// PTB begin short do once block
				struct list_head *pos;

				int count = 0;
				struct request *req;

				list_for_each (pos, &lo->queue) {
					req =
					 list_entry (pos, struct request,
						     queuelist);
					if (count++ > 1000)
						break;
					blks = req->nr_sectors / lo->blksize;
					if (blks > 0) {
						switch (rq_data_dir (req)) {
						  case READ:
							pending_rblks +=
							 blks;
							break;
						  case WRITE:
							pending_wblks +=
							 blks;
							break;
						}
					}
				}
			} while (0);	// PTB end short do once block

			read_unlock_irqrestore (&lo->queue_lock, flags);
			len +=
			 sprintf (buf + len, "[%s] Pending:\t%d", devnam,
				  atomic_read (&acct->requests_req[READ]) +
				  atomic_read (&acct->requests_req[WRITE]));

			do {	// PTB begin short do once block
				int j;
				for (j = 0; j < lo->nslot; j++) {
					struct enbd_slot *slotj =
					 &lo->slots[j];
					len +=
					 sprintf (buf + len, "\t(%d)",
						  slotj->req);
				}
			} while (0);	// PTB end short do once block

			len += sprintf (buf + len,
					"\t%dR/%dW+%dR/%dW\n",
					atomic_read (&acct->requests_req[READ]),
					atomic_read (&acct->requests_req[WRITE]),
					pending_rblks, pending_wblks);

		} while (0);	// PTB end long do once block

		ENBD_PROC_LABEL (13);

		do {		// PTB begin long do once block
			char buff[10][8];
			int shift = lo->logblksize;

			strncpy (buff[0],
			 display (atomic_read (&lo->wspeed.speed) << shift, 5), 7);
			strncpy (buff[1],
			 display (atomic_read(&lo->wspeed.speedav) << shift, 5), 7);
			strncpy (buff[2],
			 display (atomic_read(&lo->wspeed.speedmax) << shift,
					  5), 7);

			strncpy (buff[3],
			 display (atomic_read (&lo->rspeed.speed) << shift, 5), 7);
			strncpy (buff[4],
			 display (atomic_read (&lo->rspeed.speedav) << shift, 5), 7);
			strncpy (buff[5],
			 display (atomic_read (&lo->rspeed.speedmax) << shift, 5), 7);

			strncpy (buff[6],
			 display (atomic_read (&lo->tspeed.speed) << shift, 5), 7);
			strncpy (buff[7],
			 display (atomic_read (&lo->tspeed.speedav) << shift, 5), 7);
			strncpy (buff[8],
			 display (atomic_read (&lo->tspeed.speedmax) << shift, 5), 7);

			len +=
			 sprintf (buf + len, "[%s] B/s now:", devnam);
			len +=
			 sprintf (buf + len, "\t%s\t(%sR+%sW)\n", buff[6],
				  buff[3], buff[0]);
			len +=
			 sprintf (buf + len, "[%s] B/s ave:", devnam);
			len +=
			 sprintf (buf + len, "\t%s\t(%sR+%sW)\n", buff[7],
				  buff[4], buff[1]);
			len +=
			 sprintf (buf + len, "[%s] B/s max:", devnam);
			len +=
			 sprintf (buf + len, "\t%s\t(%sR+%sW)\n", buff[8],
				  buff[5], buff[2]);
		} while (0);	// PTB end long do once block

		do {		// PTB begin short do once block
			int blks;
			int tot_reqs = 0;

			len +=
			 sprintf (buf + len, "[%s] Spectrum:", devnam);
			for (blks = 0;
			     blks <= atomic_read (&acct->maxreqblks); blks++) {
				tot_reqs +=
				 atomic_read (&acct->req_in[READ][blks]) +
				 atomic_read (&acct->req_in[WRITE][blks]);
			}

			for (blks = 0;
			     blks <= atomic_read (&acct->maxreqblks); blks++) {
				int req_blks =
				 atomic_read (&acct->req_in[READ][blks])
				 + atomic_read (&acct->req_in[WRITE][blks]);
				int percent =
				 tot_reqs >
				 0 ? (100 * req_blks) / tot_reqs : 0;
				if (percent <= 0)
					continue;
				len +=
				 sprintf (buf + len, "\t%u%%%d", percent,
					  blks);
			}
			len += sprintf (buf + len, "\n");
		} while (0);	// PTB end short do once block

		ENBD_PROC_LABEL (14);

		len += sprintf (buf + len, "[%s] Kthreads:\t%d", devnam,
				atomic_read (&acct->kthreads));
		len +=
		 sprintf (buf + len, "\t(%d waiting/%d running/%d max)\n",
			  atomic_read (&acct->kwaiters),
			  atomic_read (&acct->kthreads) -
			  atomic_read (&acct->kwaiters),
			  atomic_read (&acct->kmax));

		ENBD_PROC_LABEL (15);

		len += sprintf (buf + len, "[%s] Cthreads:\t%d", devnam,
				atomic_read (&acct->cthreads));

		ENBD_PROC_LABEL (16);

		do {
			int j;
			for (j = 0; j < lo->nslot; j++) {
				struct enbd_slot *slotj = &lo->slots[j];
				int state =
				 ((slotj->flags & ENBD_SLOT_RUNNING) ? 1 :
				  0) +
				 ((slotj->flags & ENBD_SLOT_WAITING) ? 2 :
				  0);
				char *desc = "?";
				switch (state) {
				  case 0:
					desc = "-";
					break;	/* PTB not in */
				  case 1:
					desc = "*";
					break;	/* PTB in and not waiting */
				  case 2:
					desc = "?";
					break;	/* PTB impossible */
				  case 3:
					desc = "+";
					break;	/* PTB in and waiting */
				}
				len += sprintf (buf + len, "\t(%s)", desc);
			}
		} while (0);

		len += sprintf (buf + len, "\n");

		ENBD_PROC_LABEL (17);

		last = i;
		len += sprintf (buf + len, "[%s] Cpids:\t%d", devnam,
				atomic_read (&acct->cthreads));

		do {
			int j;
			for (j = 0; j < lo->nslot; j++) {
				struct enbd_slot *slotj = &lo->slots[j];
				len +=
				 sprintf (buf + len, "\t(%u)", slotj->pid);
			}
			len += sprintf (buf + len, "\n");
		} while (0);

		do {
			int j, k;
			for (j = 0; j < lo->nslot; j++) {
				struct enbd_slot *slotj = &lo->slots[j];
				if (slotj->spid != 0)
					break;
			}
			if (j < lo->nslot) {
				len +=
				 sprintf (buf + len, "[%s] Kpids:\t%d",
					  devnam,
					  atomic_read (&acct->cthreads));
				for (k = 0; k < lo->nslot; k++) {
					struct enbd_slot *slotk =
					 &lo->slots[k];
					len +=
					 sprintf (buf + len, "\t(%u)",
						  slotk->spid);
				}
				len += sprintf (buf + len, "\n");
			}
		} while (0);

		ENBD_PROC_LABEL (18);

		ENBD_PROC_LABEL (19);

		// PTB have to tell loop head that we are not reentering 
		next_label = NULL;
		next_label_name = NULL;
	}

	ENBD_PROC_LABEL (20);

	if (last == i - 2) {
                struct enbd_device * lo = enbd_get (i - 1);
		char *prevnam = lo->devnam;
		len +=
		 sprintf (buf + len, "Device %s:\tClosed\n", prevnam);
	}

	if (last < i - 2) {
		char lastnam[3];
		char prevnam[3];
                struct enbd_device * llo = enbd_get (last + 1);
                struct enbd_device * plo = enbd_get (i - 1);
		strncpy (lastnam, llo->devnam, 3);
		strncpy (prevnam, plo->devnam, 3);
		len += sprintf (buf + len, "Device %s-%s:\tClosed\n",
				lastnam, prevnam);
	}

	ENBD_PROC_LABEL (21);

	// PTB re-init vital statistics for next time 
	next_label = NULL;
	next_label_name = NULL;

	*eof = 1;
	*start = buf;
	total += len;

	return len;
}

/*
 * PTB read an int from a string. Return number of ints read (0 or 1).
 */
static int
sscani (char *buf, int len, int *n)
{

	int i, a = 0;
	short has_digits = 0;
	short is_signed = 0;

	// PTB look for first significant character
	for (i = 0; i < len; i++) {
		char c = buf[i];
		if (c == ' ' || c == '\t') {
			if (is_signed)
				return 0;
		} else if (c == '-') {
			if (is_signed)
				return 0;
			is_signed = -1;
		} else if (c == '+') {
			if (is_signed)
				return 0;
			is_signed = 1;
		} else if (c >= '0' && c <= '9') {
			is_signed = 1;
			has_digits = 1;
			break;
		} else {
			return 0;
		}
	}
	// PTB i now points at first digit if there is one
	if (!has_digits)
		return 0;
	for (; i < len; i++) {
		char c = buf[i];
		if (c < '0' || c > '9')
			break;
		a *= 10;
		a += c - '0';
	}
	if (is_signed >= 0) {
		*n = a;
        } else {
		*n = -a;
        }
	return 1;
}

/*
 * look for a 1 or 2 letter device code ("a" or "aa") and save the
 * device number to which it refers. Return number of device letter
 * codes found (0 or 1).
 */
static int
sscana (char *buf, int len, int *n)
{

	int i, a = 0;
	short has_letters = 0;

	for (i = 0; i < len; i++) {
		char c = buf[i];
		if (c >= 'a' && c <= 'z') {
			has_letters = 1;
			break;
		} else if (c == ' ') {
			if (has_letters)
				return 0;
		} else {
			return 0;
		}
	}
	if (!has_letters)
		return 0;
	for (; i < len; i++) {
		char c = buf[i];
		if (c < 'a' || c > 'z')
			break;
		a *= 26;
		a += c - 'a';
	}
	*n = a;
	return 1;
}

/*
 * read an integer (or 2-letter ascii) arg into an int. Return numner
 * of integers read (0 or 1) and -1 for no keymatch. The first arg is a
 * preceding key.
 * @i is the integer value that results
 * @j is an index if one one supplied (foo[j] = i ), else -1
 */
static int
getarg (const char *buffer, int buflen, const char *key, int *i, int *j)
{

	int keylen;

	void skip_ws (void) {
		while (buflen > 0) {
			if (*buffer != ' ' && *buffer != '\t')
				break;
			buffer++;
			buflen--;
	        }
        };

        skip_ws ();

	keylen = strlen (key);
	if (strncmp (buffer, key, keylen))
		return -1;

	buffer += keylen;
	buflen -= keylen;

	skip_ws ();

	*j = -1;
	if (*buffer == '[') {
		char *closing;
		int indexlen;

		buffer++;
		buflen--;

		skip_ws ();

		closing = strchr (buffer, ']');
		if (!closing)
			return -1;
		indexlen = closing - buffer;
		*closing = 0;

		if (sscani ((char *) buffer, indexlen, j) < 1)
			return 0;
		if (sscana ((char *) buffer, buflen, j) < 1)
			return 0;

		buffer = closing;
		buflen -= indexlen;

		buffer++;
		buflen--;

		skip_ws ();
	}

	if (*buffer != '=')
		return -1;

	buffer++;
	buflen--;

	skip_ws ();

	if (sscani ((char *) buffer, buflen, i) < 1)
		return 0;
	if (sscana ((char *) buffer, buflen, i) < 1)
		return 0;
	return 1;
}

/*  
 * PTB - write a 0 with echo -n 0 to /proc/nbdinfo to do a hard reset.
 */
static int
enbd_write_proc (struct file *file, const char *buffer, unsigned long count,
		void *data)
{

	switch (count) {

		int i;

	  case 2:
		if (buffer[1] != '\n')
			break;
		/* else fallthru to case 1 */
	  case 1:
		switch (*buffer) {
		  case '1':
			for (i = 0; i < MAX_NBD; i++) {
				struct enbd_device *lo = enbd_get(i);
			        lo->hard_reset (lo);
			}
			break;
		  case '0':
			for (i = 0; i < MAX_NBD; i++) {
				//  PTB this takes the io spinlock and our spinlock.
				struct enbd_device *lo = enbd_get(i);
				lo->soft_reset (lo);
                                lo->reenable_delay(lo, 5);
			}
			break;
		}
		break;
	  default:
		do {
			int index;
                        int merge_requests;
                        int sync_intvl;
                        int show_errs;
                        int md5sum;
#ifndef NO_BUFFERED_WRITES
                        int buffer_writes;
#endif
                        int enable;
                        int direct;
                        int reset;

			if (getarg (buffer, count, "merge_requests",
				    &merge_requests, &index) >= 0) {
				// merge_requests
				set_merge_requests (merge_requests, index);
				break;
			}
			if (getarg (buffer, count, "sync_intvl",
				    &sync_intvl, &index) >= 0
			    || getarg (buffer, count, "sync",
				       &sync_intvl, &index) >= 0) {
				// sync_intvl
				set_sync_intvl (sync_intvl, index);
				break;
			}
			if (getarg (buffer, count, "show_errs",
				    &show_errs, &index) >= 0) {
				// show_errs
				set_show_errs (show_errs, index);
				break;
			}
			if (getarg (buffer, count, "md5sum",
				    &md5sum, &index) >= 0) {
				// md5sum
				set_md5sum (md5sum, index);
				break;
			}
#ifndef NO_BUFFERED_WRITES
			if (getarg (buffer, count, "buffer_writes",
				    &buffer_writes, &index) >= 0) {
				// buffer_writes
				set_buffer_writes (buffer_writes, index);
				break;
			}
#endif		/* NO_BUFFERED_WRITES */
			if (getarg (buffer, count, "enable",
				    &enable, &index) >= 0) {
				// enable
				set_enable (enable, index);
				break;
			}
			if (getarg (buffer, count, "direct",
				    &direct, &index) >= 0) {
				// enable
                                set_direct(direct, index);
				break;
                        }
			if (getarg (buffer, count, "reset",
				    &reset, &index) >= 0) {
				// reset
				do_reset(reset, index);
				break;
			}
			ENBD_ERROR ("illegal %ld character command\n",
				   count);
			return -EINVAL;
		} while (0);
		break;
	}
	return count;
}

void
enbd_init_proc(struct proc_dir_entry *res) {
        res->read_proc = enbd_read_proc;
        res->write_proc = enbd_write_proc;
}

