/*
 * Open connection for network block device
 *
 * Copyright 1997,1998 Pavel Machek, distribute under GPL
 *  <pavel@atrey.karlin.mff.cuni.cz>
 *
 * Version 1.0 - 64bit issues should be fixed, now
 * Version 2.1 - enhancements from Peter Breuer, <ptb@it.uc3m.es>
 * Version 2.2 - SSL additions by Andres Marin, <amarin@it.uc3m.es>
 *             - more improvements from ptb, also negotiate cleanups
 *          and improvements from James MacLean <macleajb@EDnet.NS.CA>
 * Version 2.4 - negotiations moved into master daemons <ptb@it.uc3m.es>
 *             - clientside caching <ptb@it.uc3m.es>
 *             - using a single port with listen, accept <ptb@it.uc3m.es>
 *             - 64 bit fixes <ptb@it.uc3m.es>
 */

#ifndef MY_NAME
#define MY_NAME "enbd-client"
#endif
#define ISCLIENT

#define _USE_BSD
#include <sys/ioctl.h>
#include <sys/socket.h>
#include <sys/types.h>
#include <sys/resource.h>	/* wait4 */
#include <sys/wait.h>
#include <unistd.h>		/* get/setpgid stuff */

extern pid_t getpgid (pid_t pid);

#include <netinet/tcp.h>
#include <netinet/in.h>		/* sockaddr_in, htons, in_addr */
#include <netdb.h>		/* hostent, gethostby*, getservby* */
#include <stdio.h>
#include <string.h>
#include <fcntl.h>
#include <syslog.h>
#include <stdlib.h>
#include <signal.h>
#include <setjmp.h>
#include <sys/time.h>
#include <sys/errno.h>
#include <sys/stat.h>
#include <ctype.h>
#include <sys/mman.h>
#include <dirent.h>              /* readdir, DIR */

//#include <pthread.h>

#ifdef USING_SSL
#include "sslincl.h"
#endif

#ifndef __GNUC__
#error I need GCC to work
#endif

#include <linux/ioctl.h>
#include <linux/enbd.h>
#include "cliserv.h"

/* lifted from fs.h */
#define BLKRRPART  _IO(0x12,95)
#define BLKGETSIZE _IO(0x12,96)
#define BLKFLSBUF  _IO(0x12,97)
#define BLKBSZGET  _IOR(0x12,112,int)
#define BLKBSZSET  _IOW(0x12,113,int)
#define BLKROSET   _IO(0x12,93)	/* set device read-only (0 = read-write) */
#define BLKROGET   _IO(0x12,94)	/* get read-only status (0 = read_write) */

#define HELPME 1

#include "stream.h"
#include "server_stub.h"
#include "pidfile.h"
#include "enbd-client.h"
#include "logging.h"
#include "socket.h"
#include "alarm.h"
#include "select.h"
#include "md5.h"
#include "ioctl.h"
#include "file.h"
#include "time.h"

static int sig[ENBD_SIGLEN / sizeof (int)];
				 /* PTB whole device signature - shared */
int using_ssl = 0;		/* PTB modified later if we can and want */
int debug_level = DEBUG_DFLT;
				 /* PTB for debugging */

// forward decls
static int negotiate (struct nbd_client *self);
extern int errno;

// for desperate measures on restart
static sigjmp_buf jmp_start;

#ifdef USING_SSL		/* AMARIN shared variables for SSL */
extern int set_cert_stuff (SSL_CTX * ctx, char *cert_file, char *key_file);
static int verify_depth = 0;
BIO *bio_err = NULL;
	    /* AMARIN showing SSL errors - shared 
	       BIO stands for blocking I/O */
static char *cert_file = NULL;
static char *key_file = NULL;
static char *CApath = NULL;
static char *CAfile = NULL;
static int verify = SSL_VERIFY_NONE;
static SSL_METHOD *meth = NULL;
static SSL_CTX *ctx = NULL;
static char *cipher = NULL;
#endif


static
struct nbd_client *self; //= & manager.client;

// forward decl
static void sigchldhandler ();
static void setsighandlers ();

static void
setself (struct nbd_client *this)
{
    self = this;
}

static void
unplug (struct nbd_client *self)
{

    // PTB - not from an unsetup child daemon
    if (!(self->flags & ENBD_CLIENT_NEGOTIATED))
	return;

    // PTB not from a session master
    if (!(self->flags & ENBD_CLIENT_SLAVE))
	return;

    // PTB - not if we didn't yet open a nbd device
    if (!(self->flags & ENBD_CLIENT_OPEN)) {
	return;
    }
    if (self->dev < 0)
	return;

    // PTB it's OK from running daemons 
    DEBUG ("unplug daemon (%d) from kernel slot %d and clear its socket\n",
	   self->i, self->slot);
    PERR ("requested unplug (%d) %m on %s:%d"
	           " so clear socket\n", self->i, self->hostname, self->port);
    self->ioctl (self, ENBD_CLEAR_SOCK, (char *)getpid() /*dummy */ );
}

static void
report_last_error ()
{
    MSG ("client (%d) last error %m\n", self->i);
}

static void
unplug_from_kernel ()
    // PTB unplug this client daemon from the kernel slot in which it is sitting
    // Done on exit so we have to figure out who we are
{
    unplug (self);
}

static void
slave_sighandler(struct nbd_client * self, int n) {

        if (n < 0)
            return;

        switch (n) {

          case SIGTTIN:		// PTB ignore
          case SIGTTOU:
          case SIGPIPE:
	    PERR ("client (%d) ignores signal %d \n", self->slot, n);
	    break;
          case SIGHUP:
	    break;
          case SIGUSR1:
	    if (self->dev >= 0) {
	        MSG ("client (%d) delivers soft reset\n", self->slot);
	        self->ioctl (self, ENBD_RESET, NULL /*dummy */ );
	    }
	    break;
          case SIGUSR2:
	    if (self->dev >= 0) {
	        MSG ("client (%d) delivers hard reset\n", self->slot);
	        self->ioctl (self, ENBD_HARD_RESET, NULL /*dummy */ );
	    }
	    break;
          case SIGTERM:
          case SIGINT:
	    MSG ("client (%d) terminates\n", self->slot);
	    exit (0);
	    break;
          case SIGPWR:
	    //MSG ("client (%d) unplugs from kernel\n", self->slot);
	    //unplug (self);
	    if (self->dev >= 0) {
	        MSG ("client (%d) delivers soft reset\n", self->slot);
	        self->ioctl (self, ENBD_RESET, NULL /*dummy */ );
	    }
	    microsleep (5000000);
	    MSG ("client (%d) terminates\n", self->slot);
	    exit (0);
	    break;
          case SIGCHLD:
	    // PTB ordinary exit will cause renegotiate
	    PERR ("client (%d) ignores exit %d in child\n", self->slot, n);
	    break;
          default:
	    MSG ("client (%d) terminates\n", self->slot);
	    exit (0);
	    break;
        }
}

static void
slavesighandler (int n)
{

    if (!self)
        return;

    if (self->sigvector && (1 << n)) {
	return;
    }
    if (self->sighandler) {
        self->sigvector |= 1<<n;
        // PTB handle the signal inline for the mo
        self->sighandler(self, n);
        self->sigvector &= ~(1<<n);
	return;
    }

    DEBUG ("slave sighandler received signal %d\n", n);

}

/*
 * pass a signal to all live slaves in the session
 */
static void
propagate_signal (struct nbd_client_session *self, int n)
{
    int i;
    for (i = 0; i < self->nport; i++) {
	int pid = self->pids[i].pid;
	if (pid <= 0)
	    continue;
	if (kill (pid, 0) < 0) {
	    self->pids[i].pid = -1;
	    continue;
        }
	kill (pid, n);
	if (n == SIGKILL)
	    self->pids[i].pid = -1;
    }
}

static int sigchldhandler_needed;
static int longjmp_needed;

/*
 * This handles events while in the main "coasting" loop
 * */
static void
managersighandler (int n)
{

    static int lock;
    struct nbd_client_session *session;

    //if (self != &manager.client) {
    //	PERR ("slave tried to run managers sighandler on signal %d\n", n);
    //	return;
    //}
    if (!self) {
        PERR ("called with self = NULL!\n");
        return;
    }
    session = self->session; //&manager;

    if (++lock > 1) {
	lock--;
	return;
    }

    PERR ("received signal %d\n", n);

    switch (n) {

      case SIGTTIN:		// PTB ignore
      case SIGTTOU:
      case SIGPIPE:
	PERR ("ignores exit %d in manager\n", n);
	break;
      case SIGHUP:
	PERR ("propogates HUP signal from manager\n");
	propagate_signal (session, n);
	break;
      case SIGCHLD:
	if (longjmp_needed <= 0) {
	    MSG ("sighandler relaunches child from manager \n");
	    sigchldhandler_needed++;
	}
	break;
      case SIGTERM:
      case SIGINT:
	propagate_signal (session, n);
	microsleep (500000);
	propagate_signal (session, SIGKILL);
	MSG ("sighandler terminates manager safely\n");
	exit (0);
	break;
      case SIGPWR:
	propagate_signal (session, n);
	microsleep (500000);
	propagate_signal (session, SIGKILL);
	// PTB That forget about the children, who now should be dead
	longjmp_needed++;
	microsleep (500000);
	MSG ("sighandler asks manager to break off all and restart\n");
	break;
      case SIGUSR1:
	if (self->dev >= 0) {
	    MSG ("sighandler delivers soft reset from manager\n");
	    self->ioctl (self, ENBD_RESET, NULL /*dummy */ );
	}
	propagate_signal (session, n);
	break;
      case SIGUSR2:
	if (self->dev >= 0) {
	    PERR ("sighandler delivers hard reset from manager\n");
	    self->ioctl (self, ENBD_HARD_RESET, NULL /*dummy */ );
	}
	propagate_signal (session, n);
	break;
      default:			// PTB ignore
	PERR ("sighandler ignores exit %d in manager\n", n);
	break;
    }
    lock--;
}

static char usage_str[] =
            "Usage: host[:]port { hostalias  .. | -n nchan } "
	    "[-b blksize] [-t timeout] [-x retries] [-r] "
	    "[-p pulse_intvl ] [-e] "
#ifdef HAVE_MLOCKALL
            "[-s] "
#endif
#ifdef USING_SSL
	    "[-cert certfile] [-key keyfile] "
	    "[-CApath CApath] [-CAfile CAfile] " "[-verify depth] "
#endif
	    "nbd_device [mirror_device ..]\n" "(error %d)\n";

static void
usage (int err)
{
    printf (usage_str);
}

static int
int_arg (int *res, char *a, int dflt)
{

    char *err;
    *res = strtol (a, &err, 10);
    if (!*a || *err) {
	*res = dflt;
	return -2;
    }
    return 0;
}

typedef union {
    int i;
    char *s;
} INT_OR_STR;

static void
set_ro (struct nbd_client_session *self, int dummy)
{
    // PTB the local device is ro for sure if flagged on cmdline
    self->client.flags |= ENBD_CLIENT_READONLY;
}
static void
set_async (struct nbd_client_session *self, int dummy)
{
    self->client.flags |= ENBD_CLIENT_ASYNC;
}
static void
set_md5sum (struct nbd_client_session *self, int dummy)
{
    self->client.flags |= ENBD_CLIENT_MD5SUM;
}
static void
set_neg_timeout (struct nbd_client_session *self, int timeout)
{
    self->client.negotiate_timeout = timeout;
}
static void
set_max_lives (struct nbd_client_session *self, int max_lives)
{
    self->client.max_lives = max_lives;
}
static void
set_pulse_intvl (struct nbd_client_session *self, int pulse_intvl)
{
    self->client.pulse_intvl = pulse_intvl;
}
static void
set_show_errs (struct nbd_client_session *self, int dummy)
{
    self->client.flags |= ENBD_CLIENT_SHOW_ERRS;
}
static void
set_channels (struct nbd_client_session *self, int n)
{

    if (!self->client.hostname) {
	PERR ("client ignores channel multiplier %d while server name"
	      " still undefined\n", n);
	return;
    }

    // PTB the zeroth copy comes from the session, rest from prior example
    while (n-- > 0) {
        struct nbd_client * client = &self->clients[self->nport];
        struct nbd_client * prev_client =
            self->nport > 0 ? &self->clients[self->nport - 1] : &self->client ;
	client->hostname = prev_client->hostname;
	client->ctlp     = 
	client->port     = prev_client->port;
	DEBUG ("client sees target %d is %s:%d\n",
                self->nport, client->hostname, client->port);
	client->i        = self->nport++;
    }
}
static void
set_blksize (struct nbd_client_session *self, int blksize)
{
    self->client.blksize = blksize;
}

static int
set_portnum (struct nbd_client_session *self, int portnum)
{

    if (self->client.port < 0) {
	DEBUG ("client says ctrl port is %d\n", portnum);
	self->client.ctlp = portnum;
	// should phase this out for manager
	self->client.port = portnum;
	return 0;
    }
    // this is illegal. Only one port now allowed
    PERR ("client ignores illegal cmdline arg \"%d\"\n", portnum);
    return -EINVAL;
}

static void
set_nbddevice (struct nbd_client_session *self, char *device)
{
    self->client.name = device;
}

static void
set_servername (struct nbd_client_session *self, char *server)
{
    /* If this is the first -S option, then the requirements
     * are straighforward - simply set the hostname...
     */
    if (!self->client.hostname) {
	self->client.hostname = server;
    } else {
	/* For other connections things are a little more 
	 * tricky...
         */
	self->clients[self->nport].hostname = server;
	DEBUG ("client says host name %d is %s\n", self->nport, server);
	self->clients[self->nport].port = self->client.port;
	self->clients[self->nport].ctlp = self->client.port;
	DEBUG ("client says port %d is %d\n", self->nport, self->port);
	self->clients[self->nport].i = self->nport;
	self->nport++;
    }
}

#ifdef HAVE_MLOCKALL
static void
set_swap (struct nbd_client_session *self, int dummy)
{
    self->client.flags |= ENBD_CLIENT_SWAPPING;
}
#endif
static void
set_buffer_writes (struct nbd_client_session *self, int dummy)
{
    self->client.flags |= ENBD_CLIENT_BUFFERWR;
}

struct optarg {
    char *x;			// name
    char *xx;			// long name
    char *t;			// type
    INT_OR_STR *a;		// target
    INT_OR_STR d;		// default
    char *m;			// message
    void (*f) ();		// action
};

#ifndef HAVE_STRSEP
static char *
strsep(char **stringp, const char *delim) {
    char *s = *stringp, *t;
    t = strpbrk(s, delim);
    if (! t)
        return NULL;
    *t = 0;
    *stringp = t+1;
    return s;
}
#endif
#ifndef HAVE_STRTOK_R
static char *
strtok_r(char *s, const char * delim, char **ptr) {
    if (s) {
      *ptr = s;
      return strsep(ptr, delim);
    }
    return strsep(ptr, delim);
}
#endif

static int
cmdline (struct nbd_client_session *session, int argc, char *argv[])
{
    int i;

    //struct nbd_client * self = &session->client;
    // PTB need some statics just to keep the compiler happy
    static char *clientid = 0;
    static int negotiate_timeout = ENBD_NEGOTIATE_TIMEOUT_DFLT;
    static int ro = ENBD_RO_DFLT;
                                /* PTB is the device forced ro (no) */
    static int async = ENBD_ASYNC_DFLT;
                                /* PTB is the device forced async (no) */
    static int cksum = ENBD_CKSUM_DFLT;
                                /* PTB is the device forced checksumming (no) */
    static int show_errs = ENBD_SHOW_ERRS_DFLT;
                                /* PTB is the device in raid mode? (no) */
    static int pulse_intvl = ENBD_PULSE_INTVL_DFLT;
    static int nchan = -1;	/* PTB in case we want plain channels
				   only to the server addr n times */
    static int blksize = ENBD_BLKSIZE_DFLT;
    static int portnum = -1;           /* SE We must supply a port number */
    static char *servername = NULL;    /* SE -S option value...           */
    static char *nbddevice  = NULL;    /* SE -D option value...           */
    static int buffer_writes = ENBD_BUFFER_WRITES_DFLT;
    /* PTB will device buffer writes? */
#ifdef HAVE_MLOCKALL
    static int swap = ENBD_SWAP_DFLT;
                                /* PTB are we swapping over the device? */
#endif
    static int max_lives = ENBD_MAX_LIVES_DFLT;

    int do_extra_hostname (char *hostname) {

	// control connection already setup, so this counts as
	// a new connection to the hostname

        char *off, *pp, *ptr;
        int port, err;

	off = strchr (hostname, ':');
        if (off) {
            // ctl port redfined
 	    *off = 0;
	    err = int_arg (&port, off + 1, -1);
	    if (err < 0) {
	        PERR ("client ignores illegal port number %s\n", off + 1);
	        return 2;
	    };
            // PTB try passing the new value along
            session->client.port = port;
        } else {
            // copy previous value
	    port = session->client.port;
        }

        for (ptr = strtok_r(hostname, ",", &pp); ptr; ptr = strtok_r(NULL, ",", &pp)) {

            set_servername(session, ptr);

        }
	return 0;
    }

    int do_string_arg (char *x, struct optarg *opt) {
	if (opt->a)
	    opt->a->s = x;
	syslog (LOG_DAEMON | LOG_INFO | LOG_CONS | LOG_PERROR, opt->m, x);
	if (opt->f)
	    opt->f (session, x);
	return 0;
    }

    int do_int_arg (char *x, struct optarg *opt) {
	int y;
	int err = int_arg (&y, x, opt->d.i);
	if (err < 0)
	     return 2;
	if (opt->a)
	     opt->a->i = y;
	 syslog (LOG_DAEMON | LOG_INFO | LOG_CONS | LOG_PERROR, opt->m, y);
	if (opt->f)
	     opt->f (session, y);
	 return 0;
    }

    int do_bool_arg (struct optarg *opt) {
	int y = 1;
	if (opt->a)
	     opt->a->i = y;
	 syslog (LOG_DAEMON | LOG_INFO | LOG_CONS | LOG_PERROR, opt->m, y);
	if (opt->f)
	     opt->f (session, y);
	 return 0;
    }

    int do_first_hostname (char *hostname) {
	char *off, *ptr, *pp;
	int err = 0;
	int port;
        int count = 0;


	/* look for a hostname */
	 off = strchr (hostname, ':');
	if (off == NULL)	/* port will be next arg */
	     return 1;
	*off = 0;
	 err = int_arg (&port, off + 1, -1);
	if (err < 0) {
	    PERR ("client ignores illegal port number %s\n", off + 1);
	    return 2;
	};
	if (session->client.port >= 0) {
	    // this is illegal. Only one port now allowed
	    PERR ("client ignores illegal cmdline arg \"%d\"\n", port);
	    return 3;
	};
	// it's the control  connection
        for (ptr = strtok_r(hostname, ",", &pp); ptr; ptr = strtok_r(NULL, ",", &pp)) {
            if (count++ <= 0) {
	        session->client.hostname = ptr;
	        session->client.ctlp = port;
	        session->client.port = port;
	        MSG ("client says target %d is %s:%d\n", session->nport, ptr, port);
	        //session->client.i = 0;
                // session->nport = 1;

            } else {
                do_extra_hostname(ptr);
            }
        }
	// should phase this out for manager
	return 0;
    }

    int do_port (char *x) {
	int port;
	int err = 0;

	 err = int_arg (&port, x, -1);
	if (err < 0)
	     return 2;
        if (set_portnum(session, port) != 0)
            return 1;
        return 0;
    }

    static struct optarg options[] = {
      {"P", "port", "i", (INT_OR_STR *) & portnum, {i: -1},
        MY_NAME ": client says port is %d\n", (void(*)())set_portnum},
      {"S", "server", "s", (INT_OR_STR *) & servername, {s:NULL},
        MY_NAME ": client says server is %s\n", set_servername},
      {"D", "device", "s", (INT_OR_STR *) & nbddevice, {s:NULL},
        MY_NAME ": client says device is %s\n", set_nbddevice},
      {"b", "blocksize", "i", (INT_OR_STR *) & blksize, {i: ENBD_BLKSIZE_DFLT},
	 MY_NAME ": client says blksize is %d\n", set_blksize},
      {"t", "timeout", "i", (INT_OR_STR *) & negotiate_timeout,
         {i: ENBD_NEGOTIATE_TIMEOUT_DFLT},
	 MY_NAME ": client negotiation timeout is %d\n", set_neg_timeout},
      {"x", "retries", "i", (INT_OR_STR *) & max_lives, {i: ENBD_MAX_LIVES_DFLT},
	 MY_NAME ": client negotiation timeout is %d\n", set_max_lives},
      {"p", "pulse", "i", (INT_OR_STR *) & pulse_intvl,
         {i: ENBD_PULSE_INTVL_DFLT},
	 MY_NAME ": client pulse intvl is %d\n", set_pulse_intvl},
      {"i", "identifier", "s", (INT_OR_STR *) & clientid, {s:NULL},
	 MY_NAME ": client identifier is %s\n", NULL},
      {"r", "readonly", "o", (INT_OR_STR *) & ro, {i: ENBD_RO_DFLT},
	 MY_NAME ": client ro flag is %d\n", set_ro},
      {"a", "async", "o", (INT_OR_STR *) & async, {i: ENBD_ASYNC_DFLT},
	 MY_NAME ": client async flag is %d\n", set_async},
      {"m", "md5sum", "o", (INT_OR_STR *) & cksum, {i: ENBD_CKSUM_DFLT},
	 MY_NAME ": client cksum flag is %d\n", set_md5sum},
      {"e", "errors", "o", (INT_OR_STR *) & show_errs, {i: ENBD_SHOW_ERRS_DFLT},
	 MY_NAME ": client errors flag is %d\n", set_show_errs},
      {"d", "debug", "i", (INT_OR_STR *) & debug_level, {i:0},
	 MY_NAME ": client debug level is %d\n", NULL},
      {"n", "channels", "i", (INT_OR_STR *) & nchan, {i:1},
	 MY_NAME ": client channels is %d\n", set_channels},
#ifdef HAVE_MLOCKALL
      {"s", "swap", "o", (INT_OR_STR *) & swap, {i: ENBD_SWAP_DFLT},
	 MY_NAME ": client swap flag is %d\n", set_swap},
#endif
      {"c", "cache", "o", (INT_OR_STR *) & buffer_writes,
         {i: ENBD_BUFFER_WRITES_DFLT},
	 MY_NAME ": client cache flag is %d\n", set_buffer_writes},
      {"j", "journal", "s", (INT_OR_STR *) & buffer_writes, {s:NULL},
	  // PTB backwards compatibility
	  MY_NAME ": client cache flag is %d\n", set_buffer_writes},
#ifdef USING_SSL
      {"cert", "cert", "s+", (INT_OR_STR *) & cert_file, {s:NULL},
	 MY_NAME ": client cert file is %s\n", NULL},
      {"cipher", "cipher", "s+", (INT_OR_STR *) & cipher, {s:NULL},
	 MY_NAME ": client cipher: %s\n", NULL},
      {"keyfile", "keyfile", "s+", (INT_OR_STR *) & key_file, {s:
	 NULL},
	  MY_NAME ": key cert file is %s\n", NULL},
      {"CApath", "CApath", "s+", (INT_OR_STR *) & CApath, {s:NULL},
	 MY_NAME ": client CApath is %s\n", NULL},
      {"CAfile", "CAfile", "s+", (INT_OR_STR *) & CAfile, {s:NULL},
	 MY_NAME ": client CAfile is %s\n", NULL},
      {"verify", "verify", "i+", (INT_OR_STR *) & verify_depth, {i:0},
	  MY_NAME ": client verify_depth is %d\n", NULL},
#endif
      {NULL, NULL, NULL, NULL, {i:0}, NULL, NULL},
    };

    if (argc < 2) {
	return 10;
    }

    for (i = 1; i < argc; i++) {

	// first nonopt argument
	if (*argv[i] != '-' && !session->client.hostname) {
	    do_first_hostname (argv[i]);
	    continue;
	}

	// second nonopt arg if was not already part of first after colon
	if (*argv[i] != '-' && session->client.port < 0) {
	    /* look for a port */
	    if (do_port (argv[i]) == 0)
		continue;
	}

	// further nonopt args but not the last arg
	// (we backtrack later for the last and make it the device)
	if (*argv[i] != '-') {
	    /* extra hostname for new channel */
	    if (session->client.hostname) {
		do_extra_hostname (argv[i]);
		continue;
	    }
	}

	/* option */
	if (*argv[i] == '-') {

	    int j;
	    int success = 0;
	    int err = 9;

	    for (j = 0; options[j].x; j++) {
		if (i + 1 >= argc && options[j].t[0] != 'o')
		    continue;
		if (i >= argc)
		    continue;
		if (argv[i][1] != '-'
		    && strcmp (argv[i] + 1, options[j].x)) continue;
		if (argv[i][1] == '-'
		    && strcmp (argv[i] + 2, options[j].xx)) continue;
		switch (options[j].t[0]) {
		  case 'o':
		    err = do_bool_arg (&options[j]);
		    break;
		  case 'i':
		    err = do_int_arg (argv[++i], &options[j]);
		    break;
		  case 's':
		    err = do_string_arg (argv[++i], &options[j]);
		    break;
		}		/* end sw */
		if (err == 0)
		    success = 1;
#ifdef USING_SSL
		if (options[j].t[0] && options[j].t[1])	/* == '+' */
		    using_ssl = 1;
#endif
		if (success)
		    break;
	    }			/* end for j */
	    if (!success)
		return 9;
	    /* next option i */
	    continue;
	}			// if *argv[i] == '-'

    }				// for i

    // fixup device name. Was the last nonopt arg
    if (!session->client.name) {
        if (session->nport > 0) {
	    set_nbddevice(session, session->clients[--session->nport].hostname);
	    /* device */
	    DEBUG ("client says device name is %s\n", session->client.name);
        }
        // else condition is warned about below
    }

    if (session->nport <= 0) {
	// still nobody has set the number of channels
	if (nchan >= 0) { // -n has been given
	    set_channels (session, nchan);	// sets self->nport
	} else {
	    PERR ("client specified no route nor number of channels!"
              " Try -n 2. Assuming 2.\n");
	    set_channels (session, 2);	// sets self->nport
	}
    }
    if (!session->client.name) {
	PERR ("client specified no nbd device! Try /dev/nda.\n");
    }
    session->client.cid = clientid;
    return 0;

}

int
getproto (struct nbd_client *self)
{
    int err = 0;
#ifdef MY_NBD_REG_BUF
    int old = 0;
    int dev = self->dev;
    if (err = ioctl (dev, MY_NBD_REG_BUF, 0), err < 0)
	old = 1;
#else
    int old = 1;
#endif
    return old;
}

static int
server_check (struct nbd_client *self)
{

    struct nbd_stub_server *server = &self->server;
    //struct enbd_reply reply;
    int err;
    //reply.error = 0;
    //err = server->check(server, &reply);
    err = server->check (server);
    //if (err < 0)
    return err;
    //if (reply.error != 0)
    //return 1;
    //return 0;
}

static int
server_write (struct nbd_client *self, struct enbd_request *req,
	      struct enbd_reply *reply, char *buf)
{
    int err = 0;
    struct nbd_stub_server *server = &self->server;
    enbd_digest_t digest, reply_digest;
    unsigned long reply_flags = 0;

    static int nreqin, nmd5reqin, nreqout, nreqerr,
     nmd5reqerr, nmd5reqeq, nmd5reqneq;

    // PTB save a couple of parts of the req for the reply
    u32 handle = req->handle;
    // PTB testing
    int len = req->len;
    unsigned long long from = req->from;
    int type = req->type;
    unsigned long seqno = req->seqno;
    unsigned long flags = req->flags;

    DEBUG
     ("server_write entered req type %d for %dB at sector %Ld from buf %#x\n",
      type, len, from >> 9, (unsigned)buf);

    nreqin++;

    // PTB simple write protocol, without md5
    if (/* PTB testing 1 || */ !(flags & ENBD_REQUEST_MD5SUM)) {
	DEBUG ("server_write writes for request %d sector %Ld len %d\n",
	       seqno, from >> 9, len);

	// PTB out of band data first
	server->ioctl (server, MY_NBD_SET_RQ_SEQNO,
			(char *) (unsigned long) seqno);
	server->ioctl (server, MY_NBD_SET_RQ_HANDLE,
			(void *) (unsigned long) handle);

	// PTB now the write!
	err = server->write (server, buf, len, from);
        if (err < len) {
                PERR("server write of %dB at sector %Ld returned %d\n",
                        len, from >> 9, err);
        }

	reply_flags &= ~ENBD_REPLY_MD5SUM;
	reply_flags &= ~ENBD_REPLY_MD5_OK;	// 00

        if (0 && err < len) {
            // PTB testing
            PERR("bounced req type %d locally for sectors %ld-%ld\n",
                type, 
                (long) (from >> 9),
                (long)((from + len - 1) >> 9));
            err = len;

        }

	goto create_reply;
    }

    // PTB otherwise get the checksum first and add it to the request

    nmd5reqin++;
    md5_buffer (buf, len, &digest[0]);
    DEBUG ("using client md5sum %.8x%.8x%.8x%.8x\n",
	   digest[0], digest[1], digest[2], digest[3]);

    // PTB save the checksum into the outgoing request
    // memcpy(&req->data.digest[0], &digest[0], sizeof(digest));

    server->ioctl (server, MY_NBD_SET_RQ_SEQNO,
		    (char *) (unsigned long) seqno);
    server->ioctl (server, MY_NBD_SET_RQ_HANDLE,
		    (void *) (unsigned long) handle);
    // PTB ask the server for a checksum 
    err = server->sum (server, &reply_digest[0], len, from);
    DEBUG ("received server md5sum %.8x%.8x%.8x%.8x\n",
	   reply_digest[0], reply_digest[1], reply_digest[2], reply_digest[3]);

    // PTB the 4 poss result codes here are passed back to the kernel,
    //     in the reply struct, where they are used for accounting

    // PTB if we errored on the checksum, we have to really send data
    if (err < 0) {
	nmd5reqerr++;

	// PTB out of band data first
	server->ioctl (server, MY_NBD_SET_RQ_SEQNO,
			(char *) (unsigned long) req->seqno);
	server->ioctl (server, MY_NBD_SET_RQ_HANDLE,
			(void *) (unsigned long) handle);

	// PTB now the write!
	err = server->write (server, buf, len, from);

	reply_flags |= ENBD_REPLY_MD5_OK;	// 01
	reply_flags &= ~ENBD_REPLY_MD5SUM;
	goto create_reply;
    }

    // PTB if the checksum is wrong, we have to send data
    if (memcmp (&digest[0], &reply_digest[0], ENBD_DIGEST_LENGTH)) {
	DEBUG ("digests differ so writing %dB\n", len);

	// PTB out of band data first
	server->ioctl (server, MY_NBD_SET_RQ_SEQNO,
			(char *) (unsigned long) req->seqno);
	server->ioctl (server, MY_NBD_SET_RQ_HANDLE,
			(void *) (unsigned long) handle);

	// PTB now the write!
	err = server->write (server, buf, len, from);

	nmd5reqneq++;
	reply_flags &= ~ENBD_REPLY_MD5_OK;	// 10
	reply_flags |= ENBD_REPLY_MD5SUM;
	goto create_reply;
    }

    DEBUG ("digests equal so skipping %dB\n", len);
    // PTB we need to pretend we sent it
    nmd5reqeq++;
    reply_flags |= ENBD_REPLY_MD5_OK;	// 11
    reply_flags |= ENBD_REPLY_MD5SUM;
    err = len;
    goto create_reply;

  create_reply:
    // PTB create reply for kernel
    reply->handle = handle;
    reply->flags = reply_flags;
    reply->magic = ENBD_REPLY_MAGIC;
    reply->error = 0;

    // PTB maybe tell the kernel we errored, if we did
    //
    // PTB these are serious errors, no?
    if (err < len) {
	reply->error++;
	reply->flags |= ENBD_REPLY_ERRORED;
	nreqerr++;
        DEBUG ("exit FAIL (%d)\n", err);
	return err;
    }
    // PTB tell the kernel the transaction is good
    nreqout++;
    DEBUG ("exit OK (%d)\n", err);
    return err;			// 0 <= req->len <= err
}

static int
server_read (struct nbd_client *self, struct enbd_request *req,
	     struct enbd_reply *reply, char *buf)
{
    int err = 0;
    struct nbd_stub_server *server = &self->server;
    u32 handle;

    static int nreqin, nreqout, nreqerr;

    // PTB save a couple of parts of the req for the reply
    handle = req->handle;

    DEBUG ("server_read entered for %dB at sector %Ld\n",
	   req->len, req->from >> 9);

    nreqin++;

    DEBUG ("server_read reads for request len %d\n", req->len);
    err = server->read (server, buf, req->len, req->from);

    // PTB create reply for kernel
    reply->handle = handle;
    reply->flags = 0;
    reply->magic = ENBD_REPLY_MAGIC;
    reply->error = 0;

    // PTB maybe tell the kernel we errored, if we did

    if (err < req->len) {
	reply->error++;
	reply->flags |= ENBD_REPLY_ERRORED;
	nreqerr++;
	return err;
    }
    // PTB tell the kernel the transaction is good
    nreqout++;
    return err;			// 0 <= req->len <= err

}

static int
server_ioctl (struct nbd_client *self, struct enbd_request *req,
	      struct enbd_reply *reply, char *buf)
{
    int err = 0;
    struct nbd_stub_server *server = &self->server;
    u32 handle;
    char *arg = (char *) (long) (req->from & 0xffffffffL); // PTB low 32 bits
    unsigned ioctl = (req->from >> 32) & 0xffffffffL;	   // PTB hi  32 bits
    int size = 0;

    // PTB save a couple of parts of the req for the reply
    handle = req->handle;

    DEBUG ("server_ioctl entered for %dB for ioctl %u arg %#x handle %#lx\n",
	   req->len, ioctl, (unsigned)arg, (unsigned long)handle);

    // PTB "ioctl" has already been fixed up by the kernel driver

    if (ioctl == -1) {
	// PTB the ioctl was not in the whitelist and we refuse
	err = -EINVAL;
	goto create_reply;
    }

    if (_IOC_DIR (ioctl) & _IOC_READ) {
	// PTB save the arg contents locally and
	//     rewrite the arg to point at our cache
	if (req->len <= 0) {
	    arg = NULL;
	    DEBUG("server_ioctl indirect 0 bytes so set ioctl buffer NULL\n");
	} else {
	    size = req->len;
	    arg = buf;
	    DEBUG ("server_ioctl indirect size %dB so use buffer %#x (%#x..)\n",
		   size, (unsigned)arg, *(unsigned *)arg);
	    if (_IOC_DIR (ioctl) & _IOC_WRITE) {
                DEBUG("server_ioctl write buffer contains %#x..\n",
                        *(unsigned *)buf);
	    }
	}
    }

    DEBUG ("server_ioctl prepares to send ioctl handle %#lx id %#x arg %#x\n",
      (unsigned long) handle, ioctl, (unsigned) arg);

    // PTB if the arg was indirected, we replaced and rewrote it
    err = server->ioctl (server, ioctl, arg);
    DEBUG ("server_ioctl remote ioctl returned %d\n", err);
    // PTB if the ioctl wrote our data, it wrote into ctldata

  create_reply:
    // PTB create reply for kernel
    reply->handle = handle;
    reply->flags = ENBD_REPLY_IOCTL;
    reply->magic = ENBD_REPLY_MAGIC;
    reply->error = 0;

    // PTB maybe tell the kernel we errored, if we did.
    //
    // PTB but we want to only tell the kernel the ioctl failed,
    //     not that the remote is down, so we return 0
    if (err < 0) {
	DEBUG ("server_ioctl errored out for ioctl handle %#lx\n",
                (unsigned long) handle);
	reply->error++;
	reply->flags |= ENBD_REPLY_ERRORED;
	return 0;
    }

    // PTB success
    if (_IOC_DIR (ioctl) & _IOC_READ) {
	DEBUG ("server_ioctl recovers %d bytes (%d) of "
	       "data (%#x..) OK! in %#x for indirect ioctl\n",
	       size, err, *(unsigned *) arg, (unsigned) arg);
	// PTB save the arg contents locally. Note the offset.
	//     this is in agreement with what the kernel wants
	if (req->len > 0) {
	    // PTB if buffer placement is not the same for
	    //     request and reply, we need to shift the data
	    //     along the buffer. But it should be the same.
	    if (buf != arg) {
		PERR ("recv data buf %p != send buf %p!\n", buf, arg);
		memmove (buf, arg, req->len);
	    }
	} 
    }
    return req->len;

}

static int
server_cksum (struct nbd_client *self, struct enbd_request *req,
	      struct enbd_reply *reply)
{

    // PTB create reply for kernel
    reply->handle = req->handle;
    reply->flags = 0;
    reply->magic = ENBD_REPLY_MAGIC;
    reply->error = 0;

    // PTB tell the kernel the transaction is good
    return 0;
}

static int
server_special (struct nbd_client *self, struct enbd_request *req,
	      struct enbd_reply *reply)
{

    struct nbd_stub_server *server = &self->server;
    int err;
    // PTB create reply for kernel
    reply->handle = req->handle;
    MSG("special %#lx type %d with from=%Ld len=%d goes out\n",
            (unsigned long)req->handle, req->type,
	    (unsigned long long)req->from, req->len);

    err = server->special (server, req->type, req->len, req->from);
    reply->flags = 0;
    reply->magic = ENBD_REPLY_MAGIC;
    reply->error = err;
    MSG("special %#lx type %d with from=%Ld len=%d comes back err=%d\n",
            (unsigned long)req->handle, req->type,
	    (unsigned long long)req->from, req->len, err);

    // PTB tell the kernel the transaction is good
   return req->len;
}


/*
 * Handle the read and write requests got from the kernel. Return a
 * reply struct for use by the kernel. Also ioctl requests.
 *
 * Looks like it returns the number of bytes transfered, not including
 * the request and reply structs themselves. So a "0" probably
 * indicates a remote error, while a -ve result could be a net error.
 */
static int
server_cmd (struct nbd_client *self, struct enbd_request *req,
	    struct enbd_reply *reply, char *buf)
{
    int cmd = req->type;

    if (0 && cmd == WRITE) {
        // PTB testing
        PERR("bounced req type %d for sectors %ld-%ld\n", cmd, 
                (long) (req->from >> 9), (long)((req->from + req->len - 1) >> 9));
        reply->handle = req->handle;
        reply->magic = ENBD_REPLY_MAGIC;
        reply->error = 0;
        reply->flags = 0;
        return req->len;
    }

    switch (cmd) {
      case WRITE:
	return server_write (self, req, reply, buf);
      case READ:
	return server_read (self, req, reply, buf);
      case IOCTL:
	return server_ioctl (self, req, reply, buf);
      case CKSUM:
	return server_cksum (self, req, reply);
      case SPECIAL:
	return server_special (self, req, reply);
      default:
        // PTB error handled below
        break;
    }

    PERR ("impossible request type %d\n", cmd);
    reply->handle = req->handle;
    reply->magic = ENBD_REPLY_MAGIC;
    reply->error++;
    reply->flags = 0;
    reply->flags |= ENBD_REPLY_ERRORED;
    return -EINVAL;
}

/*
 * treat the return from a server check. Get 0 for all OK,
 * -ve for a net err (we send a clera sock to the kernel),
 * +ve for a remote err (we send a flshbuf to the kernel)
 */
static void
handle_server_check_err (struct nbd_client *self, int err)
{

    if (err < 0) {

	// PTB network error, syslog the first three times
	PERR ("Server_check (%d) failed %m on %s:%d"
	      " so clear socket\n", self->i, self->hostname, self->port);
	// PTB rollback kernel requests, unplug from kernel
        unplug(self);
	//return err = -EAGAIN; // PTB renegotiate
	return;

    }
    if (err > 0) {

	// PTB remote error, net is good
	PERR ("(%d) remote error on %s:%d\n", self->i,
                self->hostname, self->port);
	if (!(self->flags & ENBD_CLIENT_INVALID)) {
	    self->flags |= ENBD_CLIENT_INVALID;
	    DEBUG (" (%d) flush buffers\n", self->i);
	    self->ioctl (self, MY_NBD_INVALIDATE, (char *) 1);
	}
	return;

    }

}

static void
handle_server_cmd_err (struct nbd_client *self, struct enbd_request *req,
		       struct enbd_reply *reply, int err)
{

    if (err < 0) {
	// PTB network error

	DEBUG ("Network error. Read_stat (%d) failed %m port %d so clr sock\n",
	      self->i, self->port);

	// PTB rollback kernel request
	self->ioctl (self, MY_NBD_CLR_REQ, NULL /* dummy */ );
	PERR ("Server_cmd (%d) failed %m on %s:%d"
	      " so clear socket\n", self->i, self->hostname, self->port);
        unplug(self);

	// We die.  We do nothing about the request at
	// all and simply wait for the network to come back up.
	return;

    }
    if (err < req->len) {
	// PTB remote error, net is good

	// PTB the reply must have an error code
	if (reply->error == 0) {
	    PERR ("bug! Short reply should carry error, but did not\n");
	    reply->error = 1;
	}
	// PTB we probably want to reemit the request once more in case
	// the error indicates a media change at the other end, and we
	// really don't want to lose the first request on a new device

	// PTB the reply has the error flag set, so error results
	err = self->ioctl (self, MY_NBD_ACK, (char *) reply);
	// PTB kernel can think we're crazy. Harmless
	if (!(self->flags & ENBD_CLIENT_INVALID)) {
	    self->flags |= ENBD_CLIENT_INVALID;
	    DEBUG (" (%d) flush buffers\n", self->i);
	    self->ioctl (self, MY_NBD_INVALIDATE, (char *) 1);
	}
	return;
    }
}

static int
newproto (struct nbd_client *self)
{

    int err = 0;
    int i;
    const int retries = 3;

    struct enbd_request *req = (struct enbd_request *) self->ioctlbuf;
    struct enbd_reply *reply = (struct enbd_reply *) self->ioctlbuf;
    // PTB the offset takes data beyond both request and reply areas
    char *buf = (char *) self->ioctlbuf + ENBD_BUFFER_DATA_OFFSET;

    goto get_req;

  get_req:
    // PTB try three times to get and send a request at 1s intervals

    i = 0;
    while (1) {

        if (i++ >= retries) {
            // PTB fourth consecutive kernel error that's not a timeout
            PERR ("kernel errored %d times when we asked for a new req: %m\n",
                    i);
            // PTB rollback nonexistent kernel request
            self->ioctl (self, MY_NBD_CLR_REQ, NULL /* dummy */ );
            // PTB this will cause a daemon death and rebirth
            return -EINVAL;
        }

	err = self->ioctl (self, MY_NBD_GET_REQ, (char *) req);

	if (err >= 0)
	    break;	// PTB all OK

	// PTB something is wrong or timeout when waiting for work
	if (err < 0) {
            switch (errno) {
                case ETIME:

	        err = server_check (self);
	        // PTB either -ve net err (die) or remote err (flush buffs)

	        if (err != 0) {
		    handle_server_check_err (self, err);
		    return err;
	        }

	        DEBUG ("dummy CHK_REQ (%d) received and discarded OK\n",
		   self->i);
	        self->flags &= ~ENBD_CLIENT_INVALID;
	        DEBUG (" (%d) revalidate device\n", self->i);
	        self->ioctl (self, MY_NBD_INVALIDATE, (char *) 0);
	        return 0;
                break;

                default:
	        // PTB weird  kernel error - try again three times
	        microsleep (1000000);
                continue;
                break;
            }
	}

    } // ewhile
    goto ack_req;

  ack_req:
    // PTB normal continuation
    DEBUG ("got request type %d offset %Ld sectors\n",
	   req->type, req->from >> 9);

    err = server_cmd (self, req, reply, buf);

    DEBUG ("server_cmd returns %d\n", err);

    if (err < 0) {

	handle_server_cmd_err (self, req, reply, err);
	goto finish;		// network err - request still outstanding

    }
    if (err < req->len) {

	handle_server_cmd_err (self, req, reply, err);
	microsleep (1000000);	// PTB sleep, maybe retry, remote error 
	DEBUG ("remote errored on req len %d offset %Ld: %m\n",
	      req->len, req->from);
	goto finish;
	// PTB invalidate buffers is already done in handler

    }
    if (err >= req->len) {

	// PTB all is good
	if (self->flags & ENBD_CLIENT_INVALID) {
	    self->flags &= ~ENBD_CLIENT_INVALID;
	    DEBUG (" (%d) revalidate device\n", self->i);
	    self->ioctl (self, MY_NBD_INVALIDATE, (char *) 0);
	}
	DEBUG ("request handle %#lx received OK\n",
	       (unsigned long) reply->handle);
	err = self->ioctl (self, MY_NBD_ACK, (char *) reply);
	// PTB kernel can think we're crazy. Harmless
	DEBUG ("acked kernel for req handle %#lx with result %d\n",
	       (unsigned long) reply->handle, err);
        // PTB an -EAGAIN here is acceptable, but a 0 is better as
        // -EAGAIN will cause a half second delay
	goto finish;		// PTB hunky-dory

    }

  finish:
    // return and reenter later. We will probably pick up the same
    // request and try and send it out.
    DEBUG ("exited %d on req len %d offset %Ld: %m\n", err, req->len,
	   req->from);
    // PTB an -EAGAIN here is acceptable, but a 0 is better as
    // -EAGAIN will cause a half second delay
    return err;

}

static void
setsighandlers (void sighandler (int))
{
    // PTB handle all signals in one sighandler
    int k = 0;
    for (k = 1; k < 32; k++) {
#if defined(__GLIBC__) && __GLIBC__ >= 2
	struct sigaction sa = {
	#ifndef sa_handler
		sa_handler: {sighandler},
	#endif
		sa_mask: {{0}},
		sa_flags: SA_RESTART,
		//sa_restorer: NULL,
	};
	#ifdef sa_handler
		sa.sa_handler =  sighandler;
	#endif
#else
	struct sigaction sa = { {sighandler}, 0, SA_RESTART, NULL };
#endif
	sigfillset (&sa.sa_mask);
	sigdelset (&sa.sa_mask, SIGTERM);	// PTB let TERM kill handler
	sigdelset (&sa.sa_mask, SIGINT);	// PTB let INT kill handler
	sigaction (k, &sa, NULL);
    }
    signal (SIGSEGV, SIG_DFL);	// PTB handle SEGV naturally
}

static void
resetsighandlers ()
{
    // PTB restore defaults
    int k = 0;
    for (k = 1; k < 30; k++) {
#if defined(__GLIBC__) && __GLIBC__ >= 2
	struct sigaction sa = {
	#ifndef sa_handler
		sa_handler: {SIG_DFL},
	#endif
		sa_mask: {{0}},
		sa_flags: SA_RESTART,
		//sa_restorer: NULL,
	};
	#ifdef sa_handler
		sa.sa_handler =  SIG_DFL;
	#endif
#else
	struct sigaction sa = { {SIG_DFL}, 0, SA_RESTART, NULL };
#endif
	sigfillset (&sa.sa_mask);
	sigaction (k, &sa, NULL);
    }
}

static int
getdeviceslot (struct nbd_client *self)
{

    int err = 0;

    int k = self->i;

    if (self->dev < 0)
	goto fail;
    err = self->ioctl (self, ENBD_SET_SOCK, (char *)NULL);
    if (err < 0) {
	PERR ("Ioctl/2 (sign slot in kernel with %s) failed: %m\n",
	      (char *) sig);
	goto fail;
    }

    DEBUG ("client (%d) hands down socket to device %s in slot %d\n",
	   k, self->name, k);
    return self->slot = k;

  fail:
    PERR ("notice: client (%d) could not register slot %d with kernel", k, k);
    return self->slot = -1;
}

static int
getdeviceblksize (struct nbd_client *self)
{
    /* PTB if we don't yet know kernel devices blksize, get it */
    int blksize = -1;
    if (self->ioctl (self, BLKBSZGET, (char *) &blksize) >= 0) {
	goto further_tests;
    }
    if (self->ioctl (self, ENBD_GET_BLKSIZE, (char *) &blksize) >= 0) {
	goto further_tests;
    }
    goto fail;

  further_tests:
    if (blksize < 512) {
	PERR ("got too small blksize (%d) from kernel\n", blksize);
	goto fail;
    }
#ifndef PAGESIZE
#define PAGESIZE 4096
#endif
    if (blksize > PAGESIZE) {
	PERR ("got too big blksize (%d) from kernel\n", blksize);
	goto fail;
    }
    if (blksize & (blksize - 1)) {
	PERR ("got unbelievable blksize (%d) from kernel\n", blksize);
	goto fail;
    }
    return blksize;

  fail:
    PERR ("failed to get blksize from kernel\n");
    return -1;

}

static int
setdeviceblksize (struct nbd_client *self, int blksize)
{
    /* PTB set kernel blksize */
    int kblksize = 0, err;

    DEBUG ("client (%d) will set blksize to %d\n", self->i, blksize);

    if (self->ioctl (self, BLKBSZGET, (char *) &kblksize) >= 0) {
	DEBUG ("got blksize (%d) OK from kernel with BLKBSZGET\n", kblksize);
    }
    // PTB try homemade ioctl if generic not understood.
    if (!kblksize 
            && self->ioctl (self, ENBD_GET_BLKSIZE, (char *) &kblksize) >= 0) {
	DEBUG ("got blksize (%d) OK from kernel with ENBD_GET_BLKSIZE\n",
                kblksize);
    }
    if (kblksize && kblksize != blksize) {
        PERR ("client (%d) Warning! kernel says blksz is %d != %d\n",
		  self->i, kblksize, blksize);
    }
    err = -EINVAL;
    // PTB we use both setting methods as the kernel may intercept one
    if (self->ioctl (self, BLKBSZSET, (char *) &blksize) >= 0) {
	DEBUG ("set blksize (%d) OK from kernel with BLKBSZSET\n",
                blksize);
        err = 0;
    }
    // PTB try homemade ioctl too
    if (self->ioctl (self, ENBD_SET_BLKSIZE, (char *) &blksize) >= 0) {
	DEBUG ("set blksize (%d) OK from kernel with ENBD_SET_BLKSIZE\n",
                blksize);
        err = 0;
    }
    if (err < 0)
        PERR ("set blksize (%d) failed in kernel\n", blksize);
    return err;
}

static unsigned
setdevicesize (struct nbd_client *self, u64 size64)
{

    unsigned sectors;
    int err;

    // PTB if kernel device doesn't yet know its size, set it 
    if (self->ioctl (self, BLKGETSIZE, (char *) &sectors) >= 0
	&& (int) sectors >= 0) {
	if (size64 != ((u64) sectors) << 9) {
	    PERR ("client (%d) Warning! kernel says size is %Lu != %Lu\n",
		  self->i, (unsigned long long) sectors,
                  (unsigned long long) size64);
	} else {
	    DEBUG ("client (%d) kernel confirms size is %Ld\n", self->i,
		   (unsigned long long) size64);
	}
    }

    sectors = size64 >> 9;
    // PTB rounding to multiple of 512
    size64 = ((u64) sectors) << 9;
    err = self->ioctl (self, ENBD_SET_SECTORS,
		       (char *) (unsigned long) sectors);
    if (err < 0) {
	PERR ("client (%d) failed to set size %Ld\n", self->i,
	      (unsigned long long) size64);
	return -EINVAL;
    }
    MSG ("client (%d) set device size %Lu\n", self->i,
	 (unsigned long long) size64);
    self->size = size64;
    return sectors;

}

static int
setdevicemd5sum (struct nbd_client *self, int md5sum)
{
    int err = 0;
    DEBUG ("client (%d) enters setmd5sum\n", self->i);
    err = self->ioctl (self, ENBD_SET_MD5SUM, (char *) (unsigned long) md5sum);
    if (err < 0)
	return err;
    if (md5sum) {
	self->flags |= ENBD_CLIENT_MD5SUM;
	MSG ("client (%d) set md5sum flags on device\n", self->i);
    }
    else {
	self->flags &= ~ENBD_CLIENT_MD5SUM;
    }
    DEBUG ("client (%d) exits setmd5sum\n", self->i);
    return 0;
}

static int
setdevicero (struct nbd_client *self, int ro)
{
    int err = 0;
    DEBUG ("client (%d) enters setro\n", self->i);
    err = self->ioctl (self, BLKROSET, (char *) &ro);
    if (err < 0)
	return err;
    if (ro) {
	self->flags |= ENBD_CLIENT_READONLY;
	MSG ("client (%d) set ro flags on device\n", self->i);
    }
    else {
	self->flags &= ~ENBD_CLIENT_READONLY;
    }
    DEBUG ("client (%d) exits setro\n", self->i);
    return 0;
}

static int
setdeviceshowerrs (struct nbd_client *self, int show_errs)
{
    int err = 0;
    DEBUG ("client (%d) enters setshowerrs\n", self->i);
    err = self->ioctl (self, MY_NBD_SET_SHOW_ERRS,
		    (char *) (unsigned long) show_errs);
    if (err < 0)
	return err;
    if (show_errs) {
	self->flags |= ENBD_CLIENT_SHOW_ERRS;
	MSG ("client (%d) set show_errs flag on device\n", self->i);
    }
    else {
	self->flags &= ~ENBD_CLIENT_SHOW_ERRS;
    }
    DEBUG ("client (%d) exits setshowerrs\n", self->i);
    return 0;
}

static int
setdevicesvpd (struct nbd_client *self, int svpd)
{
    int err = 0;
    DEBUG ("client (%d) enters setsvpd\n", self->i);
    err = self->ioctl (self, MY_NBD_SET_SPID,
		    (char *) (unsigned long) svpd);
    if (err < 0)
	return err;
    DEBUG ("client (%d) exits setsvpd\n", self->i);
    return 0;
}

static int
setdevice_buffer_writes (struct nbd_client *self, int buffer_writes)
{
    int err = 0;
    DEBUG ("client (%d) enters set_buffer_writes\n", self->i);
    err = self->ioctl (self, MY_NBD_SET_BUFFERWR,
		    (char *) (unsigned long) buffer_writes);
    if (err < 0)
	return err;
    DEBUG ("client (%d) exits set_buffer_writes\n", self->i);
    return 0;
}

static int
setdevice_pulse_intvl (struct nbd_client *self, long pulse_intvl)
{
    int err = 0;

    DEBUG ("client (%d) enters set_pulse_intvl\n", self->i);
    if (pulse_intvl <= 0) {
	err = -EINVAL;
	goto fail;
    }

    err = self->ioctl (self, MY_NBD_SET_INTVL, (char *) pulse_intvl);
    if (err < 0)
	goto fail;

    DEBUG ("client (%d) set req_timeo %ld on device\n", self->i,
	   pulse_intvl);
    return 0;

  fail:
    DEBUG ("client (%d) failed to set req_timeo %ld on device\n", self->i,
	   pulse_intvl);
    return err;
}

static int
setdevicesig (struct nbd_client *self,
	      int sigbuf[ENBD_SIGLEN / sizeof (int)])
{

    int ioctl_sigbuf[ENBD_SIGLEN / sizeof (int) + 2];
    int err;
    DEBUG ("client (%d) enters setsig\n", self->i);

    if (self->flags & ENBD_CLIENT_USE_MAIN) {
        // PTB add our slot designator on the front
        DEBUG("set sig uses whole disk, wants slot %d\n", self->i+1);
        ioctl_sigbuf[0] = self->i + 1;
        memcpy((char *)&ioctl_sigbuf[1],(char*)sigbuf,ENBD_SIGLEN);
        ioctl_sigbuf[1+ENBD_SIGLEN / sizeof (int)] = ENBD_DEV_MAGIC;
        err = self->ioctl (self, MY_NBD_SET_SIG, (char *) ioctl_sigbuf);
    } else {
        DEBUG("set sig uses subdevice %d\n", self->i+1);
        err = self->ioctl (self, MY_NBD_SET_SIG, (char *) sigbuf);
    }

    if (err < 0) {
	// Failed sig check
	PERR ("client (%d) failed to set sig or pass sigchk\n", self->i);
	return -EINVAL;
    }
    DEBUG ("client (%d) set sig or passed sigchk\n", self->i);

    // passed or set sigcheck

    if (!(self->flags & ENBD_CLIENT_SIGNED)) {

	memcpy ((char *) self->sig, (char *) sigbuf, ENBD_SIGLEN);
	self->flags |= ENBD_CLIENT_SIGNED;
	DEBUG ("client (%d) copied sig\n", self->i);
	// set internal sig copy
	return 1;
    }
    DEBUG ("client (%d) sig matched OK\n", self->i);
    return 0;
}


static int
mainloop (struct nbd_client *self)
{

    // PTB this is the kernel/net protocol loop. Repeat until done.

    int err = 0;
    struct nbd_stream *stream = &self->stream;

    /* PTB new style user space net connex */
    err = self->ioctl (self, MY_NBD_REG_BUF, (char *) self->ioctlbuf);
    if (err < 0)
	goto fail;
    MSG ("client (%d) begins main loop\n", self->i);

    do {
	// PTB new 60s timer code
	static struct alarm my_jmp;
	const unsigned long timeout = 2 * self->data_timeout;

	DEBUG ("Client sets %lds timer for mainloop\n", timeout);
	if (catch_alarm (&my_jmp, timeout)) {
	    PERR ("Client times out waiting %lds in mainloop. "
		  "Breaking off\n", timeout);
	    stream->close (stream);
	    err = -EAGAIN;
	    break;		// PTB return -EAGAIN
	}

	err = newproto (self);

        DEBUG ("Client unsets timer for mainloop\n");
	uncatch_alarm (&my_jmp);

    } while (err >= 0);

  fail:
    // PTB an -EAGAIN here is acceptable, but a 0 is better as
    // -EAGAIN will cause a half second delay
    return err;
}

/*
 * async launch of a slave daemon to guard the client struct
 * At the time we run we have already forked off from the main
 * process so we don't share new opens with its other children.
 */
static int
start_client (struct nbd_client *self)
{

    int err = 0;
    int pid = -1;
    struct nbd_stream *stream = &self->stream;
    struct nbd_client_session *session = self->session;


    if (self->flags & ENBD_CLIENT_SESSION) {
	PERR ("client (%d) tried to launch a session master!)\n",
	      self->i);
	return -1;
    }

    // Just in case it got accidentally set .. say we haven't negotiated yet
    self->flags &= ~ENBD_CLIENT_NEGOTIATED;

    self->pid = -1;

    pid = fork ();

    // PTB open device after fork in order to register correct pid
    if (pid < 0) {
	PERR
	 ("client (%d) manager daemon %d launch failed on %s:%d\n",
	  -1, self->i, self->hostname, self->port);
	return self->pid = pid;
    }
    if (pid > 0) {		/* parent returns with child pid */
	MSG
	 ("client (%d) manager launched daemon %d (%d) for %s:%d\n",
	  -1,  self->i, pid, self->hostname, self->port);
	return self->pid = pid;
    }

    // PTB child. pid = 0

    // PTB close whole device just in case - reopen as slave shortly
    err = self->close (self);
    // PTB open subdevice - have correct self->i

    if (self->open (self) < 0) {

	PERR
	 ("client (%d) childminder failed to open subdevice %d for daemon (%d)\n",
	  -1, self->i + 1, self->i);
	goto error_out;
    }
    DEBUG ("client (%d) opens subdevice %s for child (%d)\n",
	   -1, self->name, self->i);

    // PTB change identity
    resetsighandlers ();
    session->clients[self->i] = *self;
    self = session->clients + self->i;
    stream = &self->stream;
    setself (self);

    self->pid = getpid ();
    //self->pids[self->i].pid = self->pid;	// PTB so whoami() works

    // PTB we now vamoosh a stream connection. I hope we reopen it later!

    stream->sock = -1;
    self->flags &= ~ENBD_CLIENT_SESSION;
    self->flags |= ENBD_CLIENT_SLAVE;

    // PTB go somewhere safe

    chdir ("/");

  start:
    // PTB try to set up initial child connection several times

    // PTB ignore early sigchlds
    resetsighandlers ();

    // PTB this is where we open a socket
    err = negotiate (self);

    if (err >= 0)
	goto mainloop;		// PTB all OK, go to mainloop

    if (!(self->flags & ENBD_CLIENT_NEGOTIATED)
	&& (self->max_lives <= 0 || self->count++ < self->max_lives)) {
	// PTB We were never running, so sleep 10s and try again
	microsleep (10000000);
	goto start;
    }
    goto error_out;		// PTB die die die

  mainloop:

    // PTB now need to be able to exit from select
    setsighandlers (slavesighandler);

#ifdef HAVE_MLOCKALL
    if (self->flags & ENBD_CLIENT_SWAPPING) {
	MSG ("client (%d) locking itself in memory\n", self->i);
	mlockall (MCL_CURRENT | MCL_FUTURE);
    }
#endif

    // PTB do protocol
    err = mainloop (self);

    // PTB see if we want to renegotiate when the protocol fails
    switch (err) {
      case -1:
      case -EAGAIN:
	// PTB sleep 5s and try again if told to by the err val
	microsleep (5000000);
	goto start;
	break;
      case 0:
      default:			// PTB probably a planned shutdown
	break;
    }

  error_out:
    // PTB child now will die. try and close down socket only

    stream->close (stream);
    // PTB CLEAR_SOCK done at exit

    // PTB sleep 5s
    microsleep (5000000);	// PTB delay to avoid fast SIGCHLDs in sequence

    // PTB umm, well, we return somewhere, probably to main. This
    // should be an exit?
    DEBUG ("client slave dies with last error %m\n");
    exit (0);
    return 0;
}

static int
setkernelslot (struct nbd_client *self, int *sigbuf, int svpd)
{
    int err = 0;
    int dev = self->dev;

    if (sigbuf) {
	err = setdevicesig (self, sigbuf);

	if (err < 0) {
	    PERR ("Failed sig check on fd %d\n", dev);
	    return -1;
	}
	DEBUG ("client (%d) passed or set sigcheck\n", self->i);
	if (err > 0) {
	    DEBUG ("client (%d) set internal sig copy\n", self->i);
	}
    }
    if (svpd > 0 && svpd < (1 << (sizeof (short) * 8))) {
	if (setdevicesvpd (self, svpd) < 0) {
	    // PTB comment because this ioctl should not be used yet
	    // PERR ("Ioctl/2.2 failed: %m .. continuing\n");
	    // return -1;
	}
	DEBUG ("client (%d) set device server pid  %d\n", self->i, svpd);
    }

    return 0;

}

/*
 * stuff prior to set_sock
 */
static int
setkernelnbd (struct nbd_client *self, u64 size64, 
	   unsigned blksize, int ro, long pulse_intvl, int show_errs,
	   int md5sum, int buffer_writes, int nport, int *sig)
{

    int dev = self->dev;

    // PTB - new, start with sig too, on main
    DEBUG("client (%d) trying sig <%s>\n", self->i, (char *)sig);
    if (setdevicesig (self, sig) < 0) {
	    PERR ("Error: device already signed with different signature: %m\n");
	    return -EINVAL;
    }

    if (size64 > 0) {
	if (setdevicesize (self, size64) < 0) {
	    PERR ("Failed set size %Lu on fd %d: %m\n",
			    (unsigned long long)size64, dev);
	    return -EINVAL;
	}
	DEBUG ("client (%d) set size %Lu\n", self->i,
			(unsigned long long)size64);
    }

    DEBUG ("client (%d) will set blksize %u\n", self->i, blksize);
    if (blksize > 0) {
	if (setdeviceblksize (self, blksize) < 0) {
	    PERR ("Failed set blksize %d on fd %d: %m\n", blksize, dev);
	    return -EINVAL;
	}

	DEBUG ("client (%d) set blksize %u\n", self->i, blksize);

    }

    if (ro >= 0) {
	if (buffer_writes >= 1 && ro >= 1) {
	    PERR ("Disallowed setting ro %d because caching\n", ro);
	    return -EINVAL;
	}
	if (setdevicero (self, ro) < 0) {
	    PERR ("Failed set ro %d on fd %d: %m\n", ro, dev);
	    return -EINVAL;
	}
	DEBUG ("client (%d) set device ro flag %x\n", self->i, ro);
    }

    if (md5sum >= 0) {
	if (setdevicemd5sum (self, md5sum) < 0) {
	    PERR ("Failed set md5sum %d on fd %d: %m\n", md5sum, dev);
	    return -EINVAL;
	}
	DEBUG ("client (%d) set device md5sum flag %x\n", self->i, md5sum);
    }

    if (pulse_intvl > 0) {
	if (setdevice_pulse_intvl (self, pulse_intvl) < 0) {
	    PERR ("Ioctl/2.1 failed: %m\n");
	    return -EINVAL;
	}
	DEBUG ("client (%d) set device req timeout %ld\n",
	       self->i, pulse_intvl);
    }

    if (show_errs >= 0) {
	if (setdeviceshowerrs (self, show_errs) < 0) {
	    PERR ("Ioctl/2.2 failed: %m\n");
	    return -EINVAL;
	}
	DEBUG ("client (%d) set device show_errs flag %x\n",
	       self->i, show_errs);
    }

    if (buffer_writes >= 0) {
	if (ro >= 1 && buffer_writes >= 1) {
	    PERR ("Disallowed setting caching on because ro\n");
	    return -EINVAL;
	}
	if (setdevice_buffer_writes (self, buffer_writes) < 0) {
	    PERR ("Ioctl/2.4 failed: %m\n");
	    return -EINVAL;
	}
	DEBUG ("client (%d) set device buffer_writes flag %x\n",
	       self->i, buffer_writes);
    }


    return 0;
}

#ifndef MAX
#define MAX(x,y) ((x)>(y)?(x):(y))
#endif

static char *
trim (char *ptr) {
    while (isspace (*ptr++)) ;	// trim front
    return --ptr;
}
    
static int
command (struct nbd_client *self, char *buf, int *tot, int n, char *key,
	 char *data)
{

    // read the n bytes in the buf and scan for key + space.
    // read more bytes equal to any skipped and return number skipped.
    // if error, return negative. We have n remaining bytes in buffer at
    // the end.

    // if the return value is positive and greater than the key len,
    // we got something

    // I forgot to limit the buffer size. FIXME

    char *ptr = buf;
    int offset = 0;
    int keylen = key ? strlen (key) : 0;
    int mlen;
    struct nbd_stream *stream = &self->stream;

    DEBUG ("Search for key %s\n", key);
  start:
    if (*tot - offset < MAX (n, keylen)) {
	int extra = MAX (n, keylen) - (*tot - offset);
	DEBUG ("Fetch %d more bytes from net to total %d\n",
	       extra, *tot + extra);
	if (stream->read (stream, buf + *tot, extra) < 0) {
	    PERR ("Failed/%s: %m\n", key);
	    return -1;
	}
	*tot += extra;
    }
    buf[*tot] = 0;
    // get rid of leading spaces before the command
    ptr = trim (ptr);
    offset = ptr - buf;
    DEBUG ("buffer [%4.4s]\n", ptr);
    if (!key)
	return 0;
    mlen = keylen;
    if (*tot - offset < mlen)
	mlen = *tot - offset;
    if (mlen < keylen || strncasecmp (ptr, key, mlen)) {
	if (isupper (key[0])) {
	    // if key begins with a capital letter, do a
	    // special search forwards until we find it
	    ptr++;
	    offset++;
	    goto start;
	}
	PERR ("key %s not found\n", key);
	return -1;
    }
    DEBUG ("found %s\n", key);
    // skip the match
    if (strlen (ptr) < mlen)
	mlen = strlen (ptr);
    ptr += mlen;
    if (data) {
	// write the data
    }
    // get rid of trailing spaces after the command word
    ptr = trim (ptr);
    offset = ptr - buf;
    // do we need more bytes in buffer?
    if (ptr > buf && *tot - offset < n) {
	// fill the buffer with the required extra bytes
	DEBUG ("Fetch %d more bytes from net to total %d\n",
	       n - (*tot - offset), n + offset);
	if (stream->read (stream, buf + *tot, n - (*tot - offset)) < 0) {
	    PERR ("Failed/%s: %m\n", key);
	    return -1;
	}
	*tot += n - (*tot - offset);	//  *tot = n + offset;
	buf[*tot] = 0;
    }
    return offset;
}

typedef struct {
    void *label;		// address to jump to
    char *name;			// command name to look for
    short count;		// how many times have we tried
    short size;			// length of data read/write
    void *data;			// where we read data from on write
} ITEM;

static void *
expect (struct nbd_client *self, ITEM * label_array, char *buffer,
	char *lbl)
{
    static char *buf;
    static void *label;
    static char *name;
    static int size;
    static char *data;
    static int i = 0;		// current unforced choice
    static int j = -1;		// last choice
    static int offset = 0;	// start of scan window in buffer
    static int tot = 0;		// total number of bytes in buffer
    struct nbd_stream *stream = &self->stream;

    void drop (int size) {
	// wipe out the last read
	if (size > 0) {
	    DEBUG ("shift buff %d left to total %d\n", size, tot - size);
	    if (tot > size)
		memmove (buf, buf + size, tot - size);
	    tot -= size;
	    buf[tot] = 0;
	    offset -= size;
        };
    }

    int fill (int size) {
	// get more bytes
	DEBUG ("Fetch %d more bytes from net to total %d\n",
	       size, tot + size);
	if (stream->read (stream, buf + tot, size) < 0) {
	    PERR ("Failed read %d on descriptor %d: %m\n",
		  size, stream->sock);
	    return -1;
	};
        tot += size;
	buf[tot] = 0;
	return size;
    };

    // set pointer
    buf = buffer;

    if (j < 0) {
	DEBUG ("client (%d) starts expect sequence %s\n", self->i, lbl);
    }

    // drop what we read last time
    if (j >= 0) {
	drop (offset);
	j = -1;
    }


    // look for a command
    for (j = 0; label_array[j].label; j++) {

	name = label_array[j].name;
	label = label_array[j].label;
	size = label_array[j].size;
	data = label_array[j].data;

	// skip null comamnds ... maybe exit
	if (!name)
	    continue;

	if (label_array[j].count > 0)
	    continue;

	if (!data) {
	    DEBUG ("try looking for read command %s\n", name);
	    // no write required first
	    offset = command (self, buf, &tot, size, name, NULL);
	}
	else {
	    DEBUG ("try looking for write command %s\n", name);
	    // expect only the name first
	    offset = command (self, buf, &tot, 0, name, NULL);
	    DEBUG ("result from looking for write command %s is %d\n",
		   name, offset);
	}

	if (offset < 0) {
	    goto fail;
	}

	if (offset == 0)
	    continue;		// no command

	if (offset >= strlen (name)) {
	    // command named, reposition buffer
	    DEBUG ("command %s received\n", name);
	    if (data) {
		// write command
		DEBUG ("writing data for command %s\n", name);
		if (stream->write (stream, data, size) < 0) {
		    PERR ("Failed write %s %*s on descriptor %d: %m\n",
			  name, size, data, stream->sock);
		    goto fail;
		}
		DEBUG ("client (%d) suggested %s %*s to server\n",
		       self->i, name, size, data);
		// read the return data item
		DEBUG ("have %d after offset %d, need %d more in buf\n",
		       tot - offset, offset, size - (tot - offset));
                // PTB just read until newline.
                buf[tot] = 0;
                if (!strchr(buf+offset, '\n')) {
                    while (fill(1) >= 1) {
                        if (buf[tot - 1] == '\n') {
                            break;
                        }
                    }
                }
 	    }
	    if (label) {
		DEBUG ("mark %s as actioned\n", name);
		label_array[j].count++;
		// place the data at the beginning of the buffer
		//if (buf+offset > buffer && tot > offset)
		//    memmove (buffer, buf+offset, tot - offset);
                drop(offset);
                offset = tot;
		DEBUG ("client (%d) returns with %s from expect\n",
		       self->i, name);
		return label;
	    }
	}
	// otherwise we skipped some spaces and that's all
    }				// endfor

    if (1) {

	// no command, assume next in sequence, flushleft
	offset = 0;
	while (label_array[i].count > 0)
	    i++;		// skip ones already done

	j = i;			// j is last
	label = label_array[i].label;
	name = label_array[i].name;
	size = label_array[i].size;
	data = label_array[i].data;

	if (label) {

	    DEBUG ("negotiate %s\n", name);
	    label_array[i].count++;
	    if (data) {
		// need to write before read
		if (stream->write (stream, data, size) < 0) {
		    PERR ("Failed write %s on descriptor %d: %m\n",
			  name, stream->sock);
		    goto fail;
		}
		DEBUG ("client (%d) suggested %s 0x%.8x.. to server\n",
		       self->i, name, *(unsigned *) label_array[i].data);
	    }
	    if (tot < size)
		if (fill (size - tot) < 0)
		    goto fail;
	    i++;
	    offset += size;
	    DEBUG ("client (%d) returns with %s from expect\n",
		   self->i, name);
	    return label;
	}			// endif (label)

    }

    // finished sequence. Prepare restart.
    DEBUG ("client (%d) finishes & resets expect sequence\n", self->i);
    i = 0;
    for (j = 0; label_array[j].label; j++) {
	label_array[j].count = 0;
    }
    j = -1;
    offset = 0;
    tot = 0;
    return NULL;

  fail:
    PERR ("client (%d) fails in expect sequence\n", self->i);
    return (void *) -1;
}

/*
 * This is the initial handshake by the session master daemon
 */
static int
introduction (struct nbd_client *self)
{

    //struct nbd_client * self = &session->client;
    struct nbd_client_session * session = self->session;
    int err = 0;
    unsigned long long size64;
    unsigned long long magic;
    static dev_t rdev;
    static char rdev_host[ENBD_RDEV_LEN+6];
    static char blksize_host[ENBD_BLKSIZE_LEN+6];
    unsigned long blksize;
    static struct alarm my_jmp;
    unsigned long timeout = self->data_timeout;
    static unsigned long ro_host = 0;
    int sigbuf[ENBD_SIGLEN / sizeof (int) + 1];
    static unsigned short port_host;
    static char nport_host[ENBD_NPORT_LEN+6];
    static char buf[256];
    static int count;
    static char pulse_intvl_host[ENBD_PULSE_INTVL_LEN+6];
    unsigned pulse_intvl;
    struct nbd_stream *stream = &self->stream;

    memset(sigbuf, 0, sizeof(sigbuf));

    if (count++ > 0) {
	DEBUG ("client (%d) process %d does intro %dth time\n",
	       self->i, getpid (), count);
    }

    if (catch_alarm (&my_jmp, timeout)) {
	// PTB timed out - bounced here on SIGALRM
	err = -15;
	goto exit_fail;
    }
    else {

	MSG ("client (%d) starts introduction sequence on %s:%d\n",
	     self->i, self->hostname, self->ctlp);
	// PTB join inits pgroup to spare parent SIGCHLD if we die early

	setpgid (0, getpgid (1));

	DEBUG ("client (%d) joins inits pgrp \n", self->i);

	err = 0;
	resetsighandlers ();	// PTB signals -> defaults

	DEBUG ("client (%d) resets sighandlers\n", self->i);

	if (stream->sock >= 0)
	    stream->close (stream);

	DEBUG ("client (%d) has initial socket %d\n",
	       self->i, stream->sock);

	// PTB we are the manager, making a control connection
	//self->port = self->ctlp;

	err = stream->open (stream, self->ctlp, self->hostname);
	if (err < 0) {
	    if (!(self->flags & ENBD_CLIENT_INTRODUCED))
		switch (err) {
		  case -1:
		    PERR ("Gethostname failed: %m\n");
		    err = -1;
		    goto fail;
		  case -2:
		    PERR ("Socket failed: %m\n");
		    err = -1;
		    goto fail;
		  case -3:
		    PERR ("Connect failed: %m\n");
		    err = -1;
		    goto fail;
		  case -4:
		  case -5:
		    PERR ("SSL connection failed: %m\n");
		    err = -1;
		    goto fail;
		}
	    // PTB error is sustainable and could try reintro
	    err = -1;
	    goto fail;
	}

	DEBUG ("client (%d) opened socket %d to %s:%d\n",
	       self->i, stream->sock, self->hostname, self->ctlp);

	// PTB we branched off a private socket to deal with this
	// PTB connection. We now do an intro on it.

	if (!self->blksize) {

	    /* PTB ask kernel if we don't know */

	    self->blksize = getdeviceblksize (self);
	    if (self->blksize < 0) {
		PERR ("Ioctl/1.2 failed: %m\n");
		err = -1;
		goto fail;
	    }

	    DEBUG ("client (%d) discovered blksize %u\n", self->i,
		   self->blksize);
	}


	// PTB prepare for network transmission
        sprintf(nport_host, "%*u\n", ENBD_NPORT_LEN, session->nport);
        sprintf(blksize_host, "%*u\n", ENBD_BLKSIZE_LEN, self->blksize);
        sprintf(pulse_intvl_host, "%*u\n", ENBD_PULSE_INTVL_LEN, self->pulse_intvl);
        sprintf(rdev_host, "%*.*lx\n", ENBD_RDEV_LEN, ENBD_RDEV_LEN, (unsigned long) self->rdev);

	if (1) {

	    static ITEM label_array[] = {
		{&&begin, "Helo", 0, 11, 0},
		{&&passwd, "pass", 0, ENBD_PASSWD_LEN+1, 0},
		{&&magic, "mgck", 0, ENBD_MAGIC_LEN+3, 0}
		,
		{&&rdev, "rdev", 0, ENBD_RDEV_LEN+1, rdev_host}
		,
		{&&size, "size", 0, ENBD_SIZE_LEN+1, 0}
		,
		{&&signature, "sign", 0, ENBD_SIGLEN+1, 0}
		,
		{&&ro, "ro", 0, ENBD_RO_LEN+1, 0}
		,
		{&&blksize, "bksz", 0, ENBD_BLKSIZE_LEN+1, blksize_host}
		,
		
		{&&req_to, "rqto", 0, ENBD_PULSE_INTVL_LEN+1, pulse_intvl_host}
                ,
		{&&nport, "nprt", 0, ENBD_NPORT_LEN+1, nport_host}
		,
		{&&port, "port", 0, 6, 0}
		,
		{0, 0, 0, 0, 0,}
		,
	    };

	    while (1) {
		void *label;
	      next:
		label = expect (self, label_array, buf, "intro");

		if ((long) label == -1l) {
		    err = -1;
		    goto fail;
		}
		if ((long) label == 0l) {
		    err = 0;
		    goto done;
		}
		goto *label;
	    }
	}			// endif

      begin:
	if (strncmp (buf, "nbd-client", 10)) {
	    PERR ("HELO bad (%*s)\n", 10, buf);
	    err = -1;
	    goto fail;
        }
	DEBUG ("client (%d) introduction begins\n", self->i);
	goto next;

      passwd:
	if (strncmp (buf, INIT_PASSWD, 8) && !HELPME) {
	    PERR ("INIT_PASSWD bad\n");
	    err = -1;
	    goto fail;
	}

	DEBUG ("client (%d) read passwd [%.*s] ok from %s:%d\n",
	       self->i, (int) sizeof (INIT_PASSWD) - 1, buf,
               self->hostname, self->port);
	goto next;

	// PTB we most often die here on retrying, if we die 
	// PTB and I see why! the server changed its global
	// const cliserv_magic to net order in place. Doing
	// it twice leads to confusion.

      magic:
        if (sscanf(buf, "0x%Lx\n", &magic) < 1) {
	    PERR ("Not enough cliserv_magic: %16s\n", buf);
	    err = -1;
	    goto fail;
        }
	if (magic != cliserv_magic) {
	    PERR ("Not enough cliserv_magic: %Lx instead of %Lx\n",
		  magic, (unsigned long long)cliserv_magic);
	    err = -1;
	    goto fail;
	}

	DEBUG ("client (%d) got cliserv magic ok\n", self->i);
	goto next;

      rdev:
        sscanf(buf, "%x\n", (unsigned*)&rdev);
	DEBUG ("client (%d) got rdev %#x ok\n", self->i, (unsigned)rdev);
	goto next;

      size:
        if (sscanf(buf, "%Lu\n", &size64) < 1) {
	    PERR ("Bad size parameter %*s\n", ENBD_SIZE_LEN, buf);
	    err = -1;
	    goto fail;
        }

	// PTB set object parm
	self->size = size64;

	MSG ("client (%d) got size %Lu\n", self->i, size64);
	goto next;

      signature:
	memcpy (sigbuf, buf, ENBD_SIGLEN);

	MSG ("client (%d) got signature [%.*s],"
	       " had [%.*s]\n", self->i,
	       ENBD_SIGLEN, (char *) sigbuf,
	       ENBD_SIGLEN, (char *) self->sig);
	goto next;

      blksize:

	DEBUG ("client (%d) suggested blksize %d to server\n", self->i,
	       self->blksize);

        if (sscanf(buf, "%lu\n", &blksize) < 1) {
	    PERR ("Bad blksize parameter %*s\n", ENBD_BLKSIZE_LEN, buf);
	    err = -1;
	    goto fail;
        }

	// PTB we choose the MAX of the two offered blksizes

	DEBUG ("client (%d) received suggestion blksize %ld\n",
	       self->i, blksize);

	if (blksize > self->blksize)
	    self->blksize = blksize;

	MSG ("client (%d) negotiated blksize %d\n", self->i, self->blksize);
	goto next;

      req_to:

	DEBUG ("client (%d) suggested pulse intvl %d to server\n",
	       self->i, self->pulse_intvl);

        if (sscanf(buf, "%u", &pulse_intvl) < 1) {
	    PERR ("Bad pulse_intvl parameter %4s\n", buf);
	    err = -1;
	    goto fail;
        }

	// PTB we choose the MAX of the two offered intvls

	DEBUG ("client (%d) received suggestion pulse_intvl %s\n",
	       self->i, pulse_intvl_host);

	// PTB write global
	if (pulse_intvl > self->pulse_intvl)
	    self->pulse_intvl = pulse_intvl;

	MSG ("client (%d) negotiated pulse_intvl %d\n",
	       self->i, self->pulse_intvl);
	goto next;

      ro:
        switch (*buf) {
            case 'y': case 't': case '1' :
            case 'Y': case 'T':
                ro_host = 1; 
                break;
            case 'n': case 'f': case '0' :
            case 'N': case 'F':
                ro_host = 0; 
                break;
            default:
	        PERR ("Bad ro parameter %1s\n", buf);
	        err = -1;
	        goto fail;
                break;
        }

	DEBUG ("client (%d) received remote resource ro flags %x\n",
	       self->i, (unsigned) ro_host);

	// PTB we accept what we're told from the server unless
	// we're journalling while they're readonly in which case
	// we disregard them and say we're readwrite all the same.

	if (ro_host) {
	    DEBUG ("client (%d) sets server ro flag\n", self->i);
	    self->flags |= ENBD_CLIENT_SERVER_RO;
	}

	if (ro_host && !(self->flags & ENBD_CLIENT_BUFFERWR)) {
	    DEBUG ("client (%d) sets local ro flag\n", self->i);
	    self->flags |= ENBD_CLIENT_READONLY;
	}

	DEBUG ("client (%d) negotiated device ro flags %x\n",
	       self->i, (unsigned) self->flags & ENBD_CLIENT_READONLY);
	goto next;

      nport:

	DEBUG ("client (%d) requested %d port connections\n",
	       self->i, self->nport);

	if (session->nport > 0) {
	    int i;
	    for (i = 0; i < session->nport; i++) {
		// PTB set global
		session->clients[i].port = self->port;
	    }
	}
	goto next;

      port:
        if (sscanf(buf, "%hu\n", &port_host) < 1) {
	        PERR ("Bad port parameter %*s\n", ENBD_PORT_LEN+1, buf);
	        err = -1;
	        goto fail;
        }
	MSG ("client (%d) got session port %d ok\n", self->i, port_host);
	goto next;

      fail:
	PERR ("client (%d) introduction sequence ends FAIL\n", self->i);
	uncatch_alarm (&my_jmp);
	err = -1;
	goto exit_fail;

      done:
	MSG ("client (%d) introduction sequence ends ok\n", self->i);
	uncatch_alarm (&my_jmp);

    }				//endif for timeout

    // ------------------------------------
    // PTB start implementing results of introduction

    self->flags |= ENBD_CLIENT_INTRODUCED;

#ifndef MIN
#define MIN(x,y) ((x)<(y)?(x):(y))
#endif

    DEBUG("client (%d) will call setkernel with blksize %d, size %Ld\n", self->i,
            self->blksize, size64);

    // PTB this sets the kernel pid to getpid(). We have to go on to
    // run setsock using the same pid to terminate.
    err = setkernelnbd (self, size64, self->blksize,
		     self->flags & ENBD_CLIENT_READONLY,
		     self->pulse_intvl,
		     self->flags & ENBD_CLIENT_SHOW_ERRS,
		     self->flags & ENBD_CLIENT_MD5SUM,
		     self->flags & ENBD_CLIENT_BUFFERWR,
                     session->nport,
                     sigbuf);

    MSG ("client (%d) sets session slots to to %d-%d\n",
	   self->i, 0, session->nport - 1);

    if (err < 0) {
	PERR ("Failed/5: %m\n");
	err = -1;
	goto exit_fail;
    }

    DEBUG ("client (%d) sets session port %d to %d\n",
	   self->i, self->port, port_host);
    self->port = port_host;

    setpgid (0, self->pgid);	// PTB rejoin parents process group
    resetsighandlers ();	// PTB signals -> defaults

    // PTB close the primary connection on which intro was
    // PTB done. Its port is the old one.

    stream->close (stream);

    // PTB success, soo ...

    return 0;

  exit_fail:
    resetsighandlers ();	// PTB signals -> defaults

    PERR ("client (%d) introduction bails out on %s:%d\n",
	  self->i, self->hostname, self->port);

    stream->close (stream);
    return err;			// PTB renogiate if -1
}

/*
 * This is the subsequent minor handshake by slave daemons.
 */
static int
negotiate (struct nbd_client *self)
{

    int err = 0;
    unsigned long long magic;
    int svpd = 0;
    static char buf[256];
    static struct alarm my_jmp;
    unsigned long timeout = self->data_timeout;
    int sigbuf[ENBD_SIGLEN / sizeof (int)];
    struct nbd_stream *stream = &self->stream;

    memset(sigbuf, 0, sizeof(sigbuf));
    memset(buf, 0, sizeof(buf));

    if (catch_alarm (&my_jmp, timeout)) {
	// PTB timed out - bounced here on SIGALRM
	err = -15;
	goto exit_fail;
    }
    else {

	DEBUG ("client slave %d (%d) starts negotiation on %s:%d\n",
	       self->pid, self->i, self->hostname, self->port);
	// PTB join inits pgroup to spare parent SIGCHLD if we die early

	setpgid (0, getpgid (1));

	DEBUG ("client (%d) joins inits pgrp \n", self->i);

	// PTB we no longer handle minor errors in the first
	//     negotiate seq internally

	err = 0;
	resetsighandlers ();	// PTB signals -> defaults

	DEBUG ("client (%d) resets sighandlers\n", self->i);

	// PTB our socket should have been closed already when
	// we got here, but just in case it's not we close it
	// again

	if (stream->sock >= 0)
	    stream->close (stream);

	DEBUG ("client (%d) has initial socket %d\n",
	       self->i, stream->sock);

	// PTB now we open a fresh socket

	err = stream->open (stream, self->port, self->hostname);
	if (err < 0) {
	    if (!(self->flags & ENBD_CLIENT_NEGOTIATED))
		switch (err) {
		  case -1:
		    PERR ("Gethostname failed: %m\n");
		    err = -1;
		    goto fail;
		  case -2:
		    PERR ("Socket failed: %m\n");
		    err = -1;
		    goto fail;
		  case -3:
		    PERR ("Connect failed: %m\n");
		    err = -1;
		    goto fail;
		  case -4:
		  case -5:
		    PERR ("SSL connection failed: %m\n");
		    err = -1;
		    goto fail;
		}
	    // PTB error is sustainable and could try renegotiate
	    err = -1;
	    goto fail;
	}

	MSG ("client (%d) opened socket %d to %s:%d\n",
	     self->i, stream->sock, self->hostname, self->port);

	if (1) {
	    // PTB you really should guess this is a state machine

	    static ITEM label_array[] = {
		{&&begin, "Helo", 0, 11, 0},
		{&&passwd, "pass",  0, ENBD_PASSWD_LEN+1, 0},
		{&&magic, "mgck",   0, ENBD_MAGIC_LEN+3, 0}
		,
		{&&signature, "sign", 0, ENBD_SIGLEN+1, 0}
		,
		//{&&svpd,    "svpd",  0, sizeof(int), 0 },
		{0, 0, 0, 0, 0,}
		,
	    };

	    void *label;

	  next:
            memset(buf, 0, sizeof(buf));
	    label = expect (self, label_array, buf, "neg");
	    if (label == (void *) -1) {
		err = -1;
		goto fail;
	    }
	    if (label == NULL) {
		err = 0;
		goto done;
	    }
	    goto *label;

	}			// endif

      begin:
	if (strncmp (buf, "old-friend", 10)) {
            // not a renegotiation!
	    if (strncmp (buf, "nbd-client", 10)) {
                // need to do a restart
                kill(0, SIGPWR);
            }
	    PERR ("HELO bad (%*s)\n", 10, buf);
	    err = -1;
	    goto fail;
        }
	DEBUG ("client (%d) negotiation begins\n", self->i);
	goto next;

      passwd:
	if (strcmp (buf, INIT_PASSWD) && !HELPME) {
	    PERR ("INIT_PASSWD bad\n");
	    err = -1;
	    goto fail;
	}

	DEBUG ("client (%d) read passwd [%.*s] ok from %s:%d\n",
	       self->i, (int) sizeof (INIT_PASSWD) - 1, buf,
               self->hostname, self->port);
	MSG ("client (%d) read passwd ok from %s:%d\n", self->i,
	     self->hostname, self->port);
	goto next;

	/* PTB we most often die here on retrying, if we die */

      magic:
        if (sscanf(buf, "0x%Lx\n", &magic) < 1) {
	    PERR ("Not enough cliserv_magic: %16s\n", buf);
	    err = -1;
	    goto fail;
        }
	if (magic != cliserv_magic) {
	    PERR ("Not enough cliserv_magic: %Lx instead of %Lx\n",
		  magic, (unsigned long long)cliserv_magic);
	    err = -1;
	    goto fail;
	}

	MSG ("client (%d) got cliserv magic ok from %s:%d\n",
	     self->i, self->hostname, self->port);
	goto next;

      signature:
	memcpy (sigbuf, buf, ENBD_SIGLEN);

	DEBUG ("client (%d) got signature [%.*s],"
	       " had [%.*s]\n", self->i,
	       ENBD_SIGLEN, (char *) sigbuf,
	       ENBD_SIGLEN, (char *) self->sig);
	MSG ("client (%d) got a signature ok from %s:%d\n", self->i,
	     self->hostname, self->port);
	goto next;

/*
             svpd:
                memcpy(&svpd, buf, sizeof(svpd));
                svpd = ntohl (svpd);
                DEBUG ("client got server pid: %x (%d)\n", svpd, svpd);
                if (svpd < 0 || svpd >= 1<<(sizeof(short)*8)) {
                      PERR ("Server pid out of 16bit range: %d \n", svpd);
                      err = -1; goto fail;
                }
                goto next;
*/

      fail:
	PERR ("client (%d) negotiation ends FAIL\n", self->i);
	uncatch_alarm (&my_jmp);
	goto exit_fail;

      done:
	DEBUG ("client (%d) negotiation ends OK\n", self->i);
	uncatch_alarm (&my_jmp);

    }

    // ------------------------------------
    // PTB start implementing results of negotiation

    err = setkernelslot (self, sigbuf, svpd);
    if (err < 0) {
	PERR ("Failed to confirm signature with kernel: %m\n");
	err = -1;
	goto exit_fail;
    }

    // ------------------------------------
    // PTB tell the kernel we're here

    if (self->i >= 0) {
	static int count;
	while (getdeviceslot (self) < 0) {
	    PERR ("Failed to get our slot from kernel: %m\n");
	    if (count++ < 3) {
	        PERR ("slot request (%d) failed %m on %s:%d"
	           " so clear socket\n", self->i, self->hostname, self->port);
                unplug(self);
		DEBUG ("client cleared kernel socket for next time\n");
		continue;
	    }
	    err = -1;
	    goto exit_fail;
	}
	count = 0;
	DEBUG
	 ("client (%d) set device %s to use our socket %d in slot %d\n",
	  self->i, self->name, stream->sock, self->slot);
    }

    /* PTB - try and set termination action to clear socket */

    atexit (unplug_from_kernel);

    // ------------------------------------
    /* Go daemon */

    self->flags |= ENBD_CLIENT_NEGOTIATED;

    setpgid (0, self->pgid);	// PTB rejoin parents process group
    setsighandlers (slavesighandler);	// PTB handle CHLD USR1 HUP TERM

    //self->ioctl (self, BLKRRPART, NULL /* dummy */ );
    return 0;

  exit_fail:
    resetsighandlers ();	// PTB signals -> defaults

    PERR ("client (%d) negotiation bails out on %s:%d\n",
	  self->i, self->hostname, self->port);

    stream->close (stream);
    return err;

}

/*
 * PTB check and restart an unknown slave pid
 */
static int
restart_unknown (struct nbd_client_session *session)
{

    static int lock;

    int i = 0;
    int newly_launched = 0;
    int newly_eliminated = 0;
    int eligible = 0;

    if (++lock > 1) {
	lock--;
	return 0;
    }

    for (i = 0; i < session->nport; i++) {

	int pid = session->pids[i].pid;
	struct timeval tv;

	if (pid <= 0)
	    continue;

	eligible++;
	DEBUG ("client (%d) childminder checking pid %d (%d)\n", -1, pid,
	       i);

	if (pid >= 0 && kill (pid, 0) >= 0)
            // PTB OK
	    continue;

	// PTB we may have to restart this client slave

	mygettimeofday (&tv, NULL);

	session->pids[i].pid = -1;

	DEBUG ("client (%d) childminder verifies pid %d (%d) is down\n",
	       -1, pid, i);

	if (session->pids[i].count >= ENBD_MAX_LIVES) {
	    MSG
	     ("client (%d) childminder sees slot (%d) used all %d lives"
	      " so restart skipped\n", -1, i, session->pids[i].count);
	    newly_eliminated++;
	    continue;
	}

	session->clients[i].flags &= ~ENBD_CLIENT_SESSION;
	session->clients[i].flags |= ENBD_CLIENT_SLAVE;

	pid = start_client (&session->clients[i]);
	session->pids[i].pid = pid > 0 ? pid : -1;

	// PTB count rapid attempts to launch within 10mins
	if (tv.tv_sec < session->pids[i].tv.tv_sec + 600) {
	    session->pids[i].count++;
	} else {
	    // PTB OK, we presume the last launch was successful
	    session->pids[i].count = 0;
	    // PTB record time of new attempted launch 
	    mygettimeofday (&session->pids[i].tv, NULL);
	}

	// PTB we want to shut down the slave if too many attempts/min

	if (pid <= 0) {
	    PERR
	     ("client (%d) childminder failed to launch slave (%d)\n",
	      -1, i);
            continue;
	}
	
        MSG ("client (%d) childminder launched pid %d (%d)\n", -1, pid, i);
        newly_launched++;

    }				// efor
    lock--;
    //if (newly_eliminated >= eligible) {
    //  MSG("client (%d) skipped restart on all slaves so reborn\n", -1);
    //  // PTB we should restart as though from a SIGPWR
    //  siglongjmp(jmp_start,1);
    //}
    return newly_launched;
}

static void
sigchldhandler (struct nbd_client_session *session)
{

    // PTB reap and restart dead children.

    struct nbd_client * self = &session->client;
    int died = 0;
    int count = 0;

    static int lock;

    if (!self)
        return;

    if (self->dev <= 0)
	return;

    if (++lock > 1) {
	lock--;
	return;
    }

    while (died = wait4 (-1, NULL, WNOHANG, NULL), died > 0) {
	int i;
	MSG ("client (%d) reaped dead child %d\n", self->i, died);
	for (i = 0; i < session->nport; i++) {
	    int pid = session->pids[i].pid;
	    if (pid == died) {
		DEBUG ("child %d is known and will be restarted shortly\n",
		     died);
		break;
	    }
	}
	if (i >= session->nport) {
	    PERR ("child %d is unknown to us! Fixing table\n", died);
	    for (i = 0; i < session->nport; i++) {
		int pid = session->pids[i].pid;
		if (pid >= 0 && session->pids[i].count < 0)
		    // PTB we just put it in the table
		    continue;
		if (pid < 0 || kill (pid, 0) < 0) {
		    session->pids[i].pid = died;
		    session->pids[i].count = -1;
		    break;
		}
	    }
	    if (i >= session->nport) {
		PERR ("couldn't find a space for child %d\n", died);
	    }
	}
    }

    if (count <= 0)
	// PTB a mysterious unknown child died so check all
	count += restart_unknown (session);

    lock--;
}

#ifdef USING_SSL
int
init_ssl (struct nbd_client *self)
{
    /* AMARIN initializing SSL stuff */

    meth = SSLv23_client_method ();
    apps_startup ();
    SSLeay_add_ssl_algorithms ();
    if (!bio_err)
	bio_err = BIO_new_fp (stderr, BIO_NOCLOSE);

    ctx = SSL_CTX_new (meth);
    if (!ctx) {
	PERR ("client manager could not open an SSL context\n");
	return -1;
    }
    SSL_CTX_set_options (ctx, 0);

    if (cipher)
	SSL_CTX_set_cipher_list (ctx, cipher);

    SSL_CTX_set_verify (ctx, verify, verify_callback);
    if (!set_cert_stuff (ctx, cert_file, key_file)) {
	SSL_CTX_free (ctx);
	return -2;
    }

    if (!SSL_CTX_load_verify_locations (ctx, CAfile, CApath) ||
	!SSL_CTX_set_default_verify_paths (ctx)) {

	PERR ("client manager no se pudo verificar el certificado\n");
	SSL_CTX_free (ctx);
	return -3;
    }
    DEBUG ("client manager completes SSL initializations\n");

    return 0;
}
#endif

static
 int
ioctl_client (struct nbd_client *self, int request, char *argp)
{

    int err;

    if (self->dev < 0)
	return -ENODEV;

    err = ioctl (self->dev, request, argp);

    return err;

}

static
 int
close_client (struct nbd_client *self)
{

    self->flags &= ~ENBD_CLIENT_OPENWANTED;
    if (!self->flags & ENBD_CLIENT_OPEN) {
	return 0;
    }
    if (self->dev < 0) {
	self->flags &= ~ENBD_CLIENT_OPEN;
	return -EBADF;
    }
    DEBUG ("client (%d) about to run close on device with fd %d \n",
	   self->i, self->dev);
    if (close (self->dev) < 0) {
	self->dev = -1;
	self->flags &= ~ENBD_CLIENT_OPEN;
	return -EBADF;
    }
    self->dev = -1;
    self->flags &= ~ENBD_CLIENT_OPEN;
    return 0;
}

static
 int
open_client (struct nbd_client *self)
{

    const int buflen = 1024;
    char buffer[buflen+1];
    struct stat statbuf;
    int subdev_nr = self->i >= 0 ? self->i + 1 : 0;
    char namelastchar = 0;
    int namelen = self->name ? strlen (self->name) : -1;
    int sufflen = 0;
    int suffval = 0; // 16 for /dev/ndb16 
    unsigned long openflags = O_RDWR;

    if (namelen <= 0) {
        PERR("client (%d) null device name %s\n", self->i, self->name);
	return -EINVAL;
    }

    namelastchar = self->name[namelen - 1];
    self->flags |= ENBD_CLIENT_OPENWANTED;
    // we test below to see if we are forced
    self->flags &= ~ENBD_CLIENT_USE_MAIN;
    memset(buffer, 0, buflen);

    // PTB try the same name first, to allow reopens without memleak
    if (namelen < buflen - 4
    && sprintf (buffer, "%.*s", namelen, self->name) < buflen
    && stat (buffer, &statbuf) >= 0
    && S_ISBLK (statbuf.st_mode)) {
        int n = namelen;
	if (self->flags & ENBD_CLIENT_ASYNC) {
	    openflags |= O_NONBLOCK;
        }
        while (n > 0) {
	    namelastchar = self->name[n - 1];
	    if (!isdigit (namelastchar)) {
                break;
            }
	    n--;
	    // PTB don't zero the trailer - keep for cmp below
        }
        if (n < namelen) {
            int intval;
            char * strerr;
            intval = strtol(&self->name[n], &strerr, 10);
            if (!*strerr) {
                suffval = intval;
            }
        }
        if (suffval % ENBD_MAXCONN == subdev_nr) {
	    self->dev = open (buffer, openflags);
	    if (self->dev >= 0) {
                MSG("client (%d) opens device %s again\n", self->i, buffer);
	        goto ok;
	    }
	}
    }

    // PTB wipe off the current trailer on the name
    while (namelen > 0) {
	namelastchar = self->name[namelen - 1];
	if (isdigit (namelastchar) || namelastchar == '/') {
	    namelen--;
	    sufflen++;
	    // PTB don't zero the trailer - keep for cmp below
	}
	else
	    break;
    }
    if (sufflen > 0) {
        int n = namelen;
        while (self->name[n] == '/' && n < namelen + sufflen)
            n++;
        if (self->name[n]) {
            int intval;
            char * strerr;
            intval = strtol(&self->name[n], &strerr, 10);
            if (!*strerr) {
                suffval = intval;
            }
        }
    }

    // PTB look for /dev/nd/a/0 /dev/nd/a/1 etc.
    if (namelen < buflen - 4
    && sprintf (buffer, "%.*s", namelen, self->name) < buflen
    && stat (buffer, &statbuf) >= 0
    && S_ISDIR (statbuf.st_mode)
    && sprintf (buffer, "%.*s/%d", namelen, self->name, subdev_nr) < buflen
    && stat (buffer, &statbuf) >= 0
    && S_ISBLK (statbuf.st_mode)) {
        MSG("client (%d) opens device %s\n", self->i, buffer);
        self->dev = open (buffer, O_RDWR);
	if (self->dev >= 0) {
	    char *name = self->name;
	    if (strcmp (self->name, buffer)) {
		self->name = strdup (buffer);
            }
            if (!self->name) {
		// PTB restore
		self->name = name;
		return -ENOMEM;
	    }
            MSG("client (%d) opened device %s\n", self->i, buffer);
	    goto ok;
        }
    }

    // PTB look for /dev/nda1 /dev/nda2  etc.
    if (namelen < buflen - 3
    && sprintf (buffer, "%.*s", namelen, self->name) < buflen
    && stat (buffer, &statbuf) >= 0
    && S_ISBLK (statbuf.st_mode)
    && sprintf (buffer, "%.*s%d", namelen, self->name, subdev_nr) < buflen
    && stat (buffer, &statbuf) >= 0
    && S_ISBLK (statbuf.st_mode)) {
        MSG("client (%d) opens device %s\n", self->i, buffer);
	self->dev = open (buffer, O_RDWR);
	if (self->dev >= 0) {
	    char *name = self->name;
	    if (strcmp (self->name, buffer)) {
		self->name = strdup (buffer);
            }
	    if (!self->name) {
		// PTB restore
		self->name = name;
		return -ENOMEM;
	    }
            MSG("client (%d) opened device %s\n", self->i, buffer);
	    goto ok;
	}
    }

    if (namelen < buflen - 5
	&& sprintf (buffer, "%.*s/../%d", namelen, self->name,
		    subdev_nr) < buflen) {
	if (stat (buffer, &statbuf) >= 0) {
	    if (S_ISBLK (statbuf.st_mode)) {
                MSG("client (%d) opens device %s\n", self->i, buffer);
		self->dev = open (buffer, O_RDWR);
		if (self->dev >= 0) {
		    char *name = self->name;
		    if (strcmp (self->name, buffer))
			self->name = strdup (buffer);
		    if (!self->name) {
			// PTB restore
			self->name = name;
			return -ENOMEM;
		    }
		    MSG ("client (%d) opened device %s ok\n", self->i,
			 self->name);
		    goto ok;
		}
	    }
	}
    }

    // PTB search for something in same /dev dir with the right major and minor
    if (
         (namelen + sufflen < buflen - 1
	  && sprintf (buffer, "%.*s", namelen+sufflen, self->name) < buflen
	  && stat (buffer, &statbuf) >= 0
	  && S_ISBLK (statbuf.st_mode))
    ||
         (namelen < buflen - 1
	  && sprintf (buffer, "%.*s", namelen, self->name) < buflen 
	  && stat (buffer, &statbuf) >= 0
	  && S_ISBLK (statbuf.st_mode))
    ) {
        // get the major and minor
        int major_nr = major(statbuf.st_rdev);
        int minor_nr = minor(statbuf.st_rdev);
        char * p;
        DIR *d;

        MSG("LOOKING for subdev %d of %s\n", subdev_nr, buffer);
        minor_nr += subdev_nr;
        MSG("LOOKING for major %d minor %d\n", major_nr, minor_nr);
        p = strrchr(buffer, '/');
        if (!p) {
            p = buffer;
        }
        *p = 0;
        if (!*buffer){
            strcpy(buffer, "/");
        }
        d = opendir(buffer);
        if (d) {
            struct dirent * de;
            int l = strlen(buffer);
            while ((de = readdir(d)) != NULL) {
                if (l + 1 + strlen(de->d_name) < buflen -1) {
                    buffer[l] = '/';
                    strcpy(buffer+l+1, de->d_name);
	            if (stat (buffer, &statbuf) >= 0) {
	                if (S_ISBLK (statbuf.st_mode)
                        && major_nr == major(statbuf.st_rdev) 
                        && minor_nr == minor(statbuf.st_rdev)) {
                            // PTB found it
	                    self->dev = open (buffer, O_RDWR);
	                    if (self->dev >= 0) {
               	                char *name = self->name;
		                if (strcmp (self->name, buffer)) {
			            self->name = strdup (buffer);
                                }
		                if (!self->name) {
			            // PTB restore
			            self->name = name;
                                    closedir(d);
			            return -ENOMEM;
		                }
		                MSG ("client (%d) opened device %s ok\n",
                                        self->i, self->name);
                                closedir(d);
                                goto ok;
                            } 
                            DEBUG("FAILED OPEN device %s: %m\n", buffer);
                        }
                    }
                }
            }
            closedir(d);
        }
    }


    // PTB - last attempt. Open the main device and hope to do funny
    // ioctls to it!
    if (namelen < buflen - 4
	&& sprintf (buffer, "%.*s", namelen, self->name) < buflen) {
	if (stat (buffer, &statbuf) >= 0) {
	    if (S_ISBLK (statbuf.st_mode)) {
		unsigned long openflags = O_RDWR;
		if (self->flags & ENBD_CLIENT_ASYNC)
		    openflags |= O_NONBLOCK;
		if (!isdigit (namelastchar) || namelastchar == '0') {
                    MSG("client (%d) opens device %s\n",
                        self->i, buffer);
		    self->dev = open (buffer, openflags);
		    if (self->dev >= 0) {
                        self->flags |= ENBD_CLIENT_USE_MAIN;
			goto ok;
		    }
		}
	    }
	}
    }

    // PTB hurr - also try on full original name 
    if (namelen + sufflen < buflen - 4
	&& sprintf (buffer, "%.*s", namelen + sufflen, self->name) < buflen) {
	if (stat (buffer, &statbuf) >= 0) {
	    if (S_ISBLK (statbuf.st_mode)) {
		unsigned long openflags = O_RDWR;
		if (self->flags & ENBD_CLIENT_ASYNC)
		    openflags |= O_NONBLOCK;
		if (!isdigit (namelastchar) || namelastchar == '0') {
                    MSG("client (%d) opens device %s\n",
                        self->i, buffer);
		    self->dev = open (buffer, openflags);
		    if (self->dev >= 0) {
                        self->flags |= ENBD_CLIENT_USE_MAIN;
			goto ok;
		    }
		}
	    }
	}
    }

  ok:
    if (self->dev < 0) {
	//PERR ("Cannot open NBD device %s: %m\n",buffer);
	return -EBADF;
    }
    self->rdev = statbuf.st_rdev;
    self->flags |= ENBD_CLIENT_OPEN;
    self->flags &= ~ENBD_CLIENT_OPENWANTED;
    return 0;
}

// PTB small init, for opens. This is completely generic
void
init_client (struct nbd_client *self)
{
    self->open = open_client;
    self->ioctl = ioctl_client;
    self->close = close_client;
    self->start = start_client;
}

// PTB really this is a slave client
static void
init_slave (struct nbd_client *self, int i, int dev, int *sig, int ppid,
	    int pgid, int flags, u64 size,
	    char *ioctlbuf, unsigned short port, unsigned short ctlp,
	    int negotiate_timeout, int pulse_intvl, dev_t rdev, char *cid,
	    char *name,
            struct nbd_client_session *session,
            char *hostname)
{

    struct nbd_stub_server *server = &self->server;

    self->i = i;
    self->dev = dev;
    self->sig = sig;
    self->ppid = ppid;
    self->pgid = pgid;
    self->flags = flags;
    if (using_ssl)
	self->flags |= ENBD_CLIENT_USING_SSL;
    else
	self->flags &= ~ENBD_CLIENT_USING_SSL;
    self->size = size;

    self->ioctlbuf = ioctlbuf;
    self->port = port;
    self->ctlp = ctlp;
    self->count = 0;
    self->negotiate_timeout = negotiate_timeout;
    self->data_timeout = 3 * pulse_intvl;
    self->pulse_intvl = pulse_intvl;
    self->rdev = rdev;
    self->cid = cid;
    self->name = name;
    self->session = session;
    self->hostname = hostname;
    self->sigvector = 0;
    self->sighandler = slave_sighandler;

    //session->nport = 0;

    if (strncmp("file$",self->name,5) != 0) {

        struct nbd_stream *stream = &self->stream;

#ifdef USING_SSL
        if (using_ssl)
	        initstream (stream, self->data_timeout, &ctx);
        else
#endif
	        initstream (stream, self->data_timeout, NULL);

        // PTB just a direct connection to the net
        init_netserver (server, self->i, self->size, stream,
			0 != (self->flags & ENBD_CLIENT_SERVER_RO), NULL,
			0 != (self->flags & ENBD_CLIENT_ASYNC));
    } else {

        struct nbd_file *file = malloc(sizeof(*file));
        unsigned long caddr = 0;
        char * names[] = { self->name, NULL };
        s64 sizes[] = { self->size, 0 };

        self->name += 5; // file$

        init_file(file, 1, 0644, sizes,
                self->blksize, names,
		0 != (self->flags & ENBD_CLIENT_SERVER_RO),
                0,
		0 != (self->flags & ENBD_CLIENT_ASYNC),
                0, 0,  &caddr);

                        
        init_fileserver (server, self->i,
                        self->size, file,
			0 != (self->flags & ENBD_CLIENT_SERVER_RO),
                        NULL);
    }

    init_client(self);
}

#ifdef HAVE_BDFLUSH
static void
set_bdflush (struct nbd_client *client)
{
# define BDFLUSH_NFRACT_SYNC  6
# define BDFLUSH_NFRACT  0

    // PTB train VM to avoid deadlocks - really should never be sync
    int bdflush ();
    long bdflush_nfract_sync;
    long nfract_sync = 80;
    long nfract = 25;
    long bdflush_nfract;

    // PTB adjust bdflush sync point to at least 80%
    if (bdflush ((BDFLUSH_NFRACT_SYNC << 1) + 2, &bdflush_nfract_sync)
	< 0) {
	PERR ("Failed to read kernel bdflush sync boundary parameter: %m\n");
	goto nfract;
    }
    if (bdflush_nfract_sync >= 80) {
	MSG ("client (-1) left kernel bdflush sync boundary at %ld%%\n",
	  bdflush_nfract_sync);
	goto nfract;
    }
    if (bdflush ((BDFLUSH_NFRACT_SYNC << 1) + 3, nfract_sync) < 0) {
	PERR ("Could not set kernel bdflush sync boundary >=80%%,"
	      " remains at %ld%%: %m\n", bdflush_nfract_sync);
	goto nfract;
    }
    MSG ("client (-1) set kernel bdflush sync boundary to %ld%% from %ld%%\n",
      nfract_sync, bdflush_nfract_sync);
nfract:
    // PTB lower bdflush async point to 25%
    if (bdflush ((BDFLUSH_NFRACT << 1) + 2, &bdflush_nfract) < 0) {
	PERR ("Failed to read kernel bdflush async boundary parameter: %m\n");
	goto end;
    }
    if (bdflush_nfract <= 25) {
	MSG ("client (-1) left kernel bdflush async boundary at %ld%%\n",
	  bdflush_nfract);
	goto end;
    }
    if (bdflush ((BDFLUSH_NFRACT << 1) + 3, nfract) < 0) {
	PERR ("Could not set kernel bdflush async boundary <=25%%,"
	      " remains at %ld%%: %m\n", bdflush_nfract);
	goto end;
    }
    MSG ("client (-1) set kernel bdflush async boundary to %ld%% from %ld%%\n",
      nfract, bdflush_nfract);
end:
    return;
}
#endif		/* HAVE_BDFLUSH */

int
main (int argc, char *argv[])
{
    int err = 0;		// returned error from routines
    int i;
// PTB manager is the initial client session manager
//static // static in orer to zero the undefed fields
struct nbd_client_session manager = {
  client: {
    hostname:NULL,		/* PTB hostname */
    ctlp:0,			/* PTB master control port */
    port:-1,			/* PTB port */
    saddr:0,			/* PTB local addr */
    daddr:0,			/* PTB dest addr */
    pid:-1,			/* PTB daemon pid */
    slot:-1,			/* PTB kernel slot */
    sig:sig,			/* PTB slot signature */
    i:-1,			/* PTB client identifier */
    stream:{
      sock:-1,
#ifdef USING_SSL
      con: NULL,
#endif
    },
    dev:-1,			/* PTB nbd device descriptor */
    /* PTB what kind of client daemon are we? */
    flags:ENBD_CLIENT_SESSION,
    ppid:0,			/* PTB parent pid */
    pgid:0,			/* PTB group pid */
    size:0L,			/* PTB size of device in bytes */
    type:0,
    ioctlbuf:NULL,
    count:0,			/* PTB how often have we renegotiated */
    negotiate_timeout:120,
    /* PTB how long allowed per neg attempt */
    pulse_intvl:10,		/* PTB heartbeat */
    data_timeout:30,		/* PTB how long to wait for server */
    server:{0,},
    blksize:1024,		/* PTB device blocksize - shared, dflt 1024 */
    cid:NULL,			/* PTB client identifier (-i flag) */
    pidfile:{NULL,},		/* PTB where we keep our chiefs pid */
    pfd:-1,			/* PTB descriptor for where we keep the above */
    session: &manager,          /* PTB parent */
  },
  /* PTB list of child pids, if any */
  pids: {{pid: -1, count: 0, tv:{0, 0,}},},
  nport:0,			/* PTB total number of ports - shared */
  //hostname: { NULL, },
  //ctlp: { -1, },
  //ngroup: 0,                    /* PTB total number of mirror groups */
};
    struct nbd_client_session * session = &manager;
    struct nbd_client *client = &session->client;
    const int ioctlbufsize = ENBD_MAX_SECTORS * 512 + 1000;	/* AMARIN */
    struct nbd_stream *stream = &client->stream;
    struct timeval start_tv;

    if (err = cmdline (session, argc, argv), err) {
	PERR ("client cmdline returned error\n");
	usage (err);
	return err;
    }
#ifdef USING_SSL
    if (using_ssl)
	initstream (stream, client->data_timeout, &ctx);
    else
#endif
	initstream (stream, client->data_timeout, NULL);

    logging ();

    /* PTB - mod_inc count will be = 1, but that's correct (dup fd) */

    client->ioctlbuf = calloc (1, ioctlbufsize);	/* AMARIN */
    if (!client->ioctlbuf) {
	PERR ("Out of memory for ioctl buffer!\n");
	exit (2);
    }

    // make sure we walk it once PTB
    memset (client->ioctlbuf, 0, ioctlbufsize);
    DEBUG ("Created NBD ioctl buffer size %d\n", ioctlbufsize);

    // PTB we return here on SIGPWR for refork

    if (sigsetjmp (jmp_start, 1)) {

	// PTB we arrive here from a longjmp
	int i;
	DEBUG ("client (%d) restarts and closes net\n", client->i);
	stream->close (stream);
	// PTB don't close the device. We hang. (why???)
	// PTB but close all other nonessential descriptors 
	DEBUG ("client (%d) closes all descriptors except dev %d\n",
	       client->i, client->dev);
	for (i = 3; i < 256; i++) {
	    if (i != client->dev)
		close (i);
	}
	// PTB try and avoid treating sigchld etc for the moment!
	setsighandlers (SIG_DFL);
	// PTB clean up children
	DEBUG ("client (%d) waits for dead children 1\n", client->i);
	while (wait4 (-1, NULL, WNOHANG, NULL) > 0) ;
	propagate_signal (session, SIGTERM);
	microsleep (500000);
	// PTB clean up children
	DEBUG ("client (%d) waits for dead children 2\n", client->i);
	while (wait4 (-1, NULL, WNOHANG, NULL) > 0) ;
	propagate_signal (session, SIGKILL);
	microsleep (500000);
	// PTB clean up children
	DEBUG ("client (%d) waits for dead children 3\n", client->i);
	while (wait4 (-1, NULL, WNOHANG, NULL) > 0) ;
	MSG ("client (%d) restarts\n", client->i);
    }

    // PTB "little init and open" which we really want to defer.
    // it only gives us the methods. name is already set.
    init_client (client);
    // PTB only open the device if this is a first time start
    if (client->dev < 0) {
	if (client->open (client) < 0) {
	    PERR ("Cannot open NBD device %s: %m\n", client->name);
	    exit (1);
	}
	else {
	    MSG ("client (%d) manager opened NBD device %s (%x)\n",
		 -1, client->name, (unsigned) client->rdev);
	}
    }

#ifdef HAVE_BDFLUSH
    set_bdflush(client);
#endif		/* HAVE_BDFLUSH */

#ifdef USING_SSL
    if (using_ssl) {
	err = init_ssl (client);
	if (err < 0)
	    exit (1 - err);
    }
#endif

    //PTB set up our pidfile before fork
    if (client->cid) {
	char format[] = PIDDIR "/" "enbd-client-%s.pid";
	char *pidfile = malloc (strlen (client->cid) + strlen (format));

	if (pidfile) {
	    sprintf (pidfile, format, client->cid);
	}
	else {
	    PERR ("could not allocate %lu bytes of memory\n",
		  (unsigned long)(strlen (self->cid) + strlen (format)));
	    exit (1);
	}
	initpidfile (&client->pidfile, pidfile);
	DEBUG ("Set pid file %s d\n", pidfile);
    }
    else {
	DEBUG ("No pid file set\n");
    }

    MSG ("client (%d) manager detaches from shell\n", -1);
    if (fork ()) {	/* PTB detach */
        DEBUG("client manager parent %d tries to die\n", getpid());
	return 0;
    }

    /* manager */

    setsid ();
    // PTB register parent daemon
    client->pid = getpid ();
    // PTB tell the whole group who is the boss
    //ppid = self->pid;
    setpgrp ();			// PTB - this can appear to prevent connect via a make race
    client->pgid = getpgid (0);

    // PTB (re) make our (session master) pidfile with our pid inside after fork
    if (client->cid) {
        struct nbd_pidfile * pidfile = &client->pidfile;
	void removepidfile () {
	    pidfile->unlock (pidfile);
	};
        
	if (pidfile->lock (pidfile) < 0) {
	    PERR ("There is already a enbd-client with id %s running."
		  " Check in %s!\n", client->cid, pidfile->pidfile);
	    exit (5);
	}
        MSG("client (%d) manager made pidfile %s with pid %d\n", -1,
                pidfile->pidfile, getpid());
	atexit (removepidfile);
    }

    // PTB set the start time
    mygettimeofday (&start_tv, NULL);

    DEBUG ("client (%d) manager started new process group %d\n", -1,
	   client->pgid);

    // PTB open control socket and interchange params
    atexit (report_last_error);

    while (1) {
	// PTB retry intro at delay s intervals until exceed negotiate_timeout
	static int count;
	const long retry_delay = 10;
	struct timeval curr_tv;
	long elapsed_time = 0, naptime = retry_delay;
        int j, k, l;
        struct { 
            char * hostname;
            int ctlp;
            int targets[ENBD_MAXCONN];
            int count;
            int port;
        } list[2 /*ENBD_MAXGROUP*/];
        int nerrs = 0;

        for (j = 0, k = 0; k < session->nport; k++) {

                struct nbd_client * c = &session->clients[k];

                DEBUG ("client (%d) manager sees target %d is %s:%d(%d)\n", -1,
                        k, c->hostname, c->port, c->ctlp);
                for (l = 0; l < j; l++) {
                    if (strcmp(list[l].hostname, c->hostname) == 0
                            && list[l].ctlp == c->ctlp)
                        break;
                }
                if (l >= j) {
                    list[j].hostname = c->hostname;
                    list[j].ctlp = c->ctlp;
                    j++;
                }
        }
        // j is the number of distinct targets 
        DEBUG ("client (%d) manager sees %d distinct targets\n", -1, j);

        // list the members of each target group
        for (l = 0; l < j; l++) {
                list[l].count = 0;
                for (k = 0; k < session->nport; k++) {
                        struct nbd_client * c = &session->clients[k];
                        if (strcmp(list[l].hostname, c->hostname) == 0
                            && list[l].ctlp == c->ctlp) {
                            list[l].targets[list[l].count++] = k;
                        }
                }
        }

        for (l = 0; l < j; l++) {
                client->hostname = list[l].hostname;
                client->port = client->ctlp = list[l].ctlp;
                DEBUG ("client (%d) manager tries target %s:%d\n", -1,
                        client->hostname, client->ctlp);

                err = introduction (client);

                if (err < 0) {
                    nerrs++;
                    continue;
                }

                DEBUG ("client (%d) manager got port %d on target %s:%d\n", -1,
                        client->port, client->hostname, client->ctlp);
                // PTB store the result
                list[l].port = client->port;

        }

        // PTB otherwise read back the results
        for (l = 0; l < j; l++) {
            for (k = 0; k < list[l].count; k++) {
                struct nbd_client * c =
                        &session->clients[list[l].targets[k]];

                c->port = list[l].port;
                c->ctlp = list[l].ctlp;

                DEBUG ("client (%d) manager sets port %d on target %s:%d for daemon %d\n", -1,
                        c->port, c->hostname, c->ctlp, list[l].targets[k]);
                    
            }
        }
 
	if (nerrs <= 0) {	// PTB this was >= -1 but -1 means renegotiate!
	    count = 0;
	    goto launch;	// PTB success
	}

	count++;		// PTB fail

	mygettimeofday (&curr_tv, NULL);
	elapsed_time = curr_tv.tv_sec - start_tv.tv_sec;

	if (client->negotiate_timeout > 0
	    && elapsed_time > client->negotiate_timeout) {

	    PERR
	     ("client (%d) manager gave up on intro after %d tries and %lds\n",
	      client->i, count, elapsed_time);
	    exit (1);		// PTB fail
	}

	naptime = count * retry_delay - elapsed_time;
	if (naptime < retry_delay / 2)
	    naptime = retry_delay / 2;

	MSG ("client (%d) manager waits %lds after %lds before retry %d\n",
	     client->i, naptime, elapsed_time, count);
	// PTB delay by the promised delay before retry
	microsleep (naptime * 1000000);

    }				// PTB end of while try intro loop

  launch:
    // PTB begin main loop launching children

    for (i = 0; i < session->nport; i++) {

	int pid = 0;
        struct nbd_client * c = &session->clients[i];

	// PTB pass on the global data from main

	init_slave (c,
                    i,
                    client->dev,
                    client->sig,
                    client->pid,
                    client->pgid,
                    (client->flags & ~ENBD_CLIENT_SESSION) | ENBD_CLIENT_SLAVE,
                    client->size,
                    client->ioctlbuf,
                    c->port,
                    c->ctlp,
		    client->negotiate_timeout,
                    client->pulse_intvl,
                    client->rdev,
                    client->cid,
                    client->name,
                    session,
                    c->hostname
                    );

        DEBUG ("client (%d) manager sees port %d on target %s:%d for daemon %d\n", -1,
                        c->port, c->hostname, c->ctlp, i);
	// we seem to close our device inside. Not a good idea.
	pid = c->start (c);

	if (pid > 0)
	    session->pids[i].pid = pid;

    }				// PTB end main loop launching children

    client->slot = -1;		// PTB to stop atexit stomping on us

    setself (client);

    // PTB main ancestor guards children at end of launches

    // child (or parent if fork failed) guards the children

    // PTB delay 5s
    microsleep (5000000);

    setsighandlers (managersighandler);	// PTB handle CHLD USR1 etc

    DEBUG ("client (%d) manager set signal handlers\n", client->i);

    if (client->dev < 0)
	goto end;

    // PTB handler loop. We wait for interrupts and sync in the kernel.
    if (!(client->flags & ENBD_CLIENT_ASYNC)) {
	// PTB we reopen the device NONBLOCK to do ioctls only
	client->close (client);
	client->flags |= ENBD_CLIENT_ASYNC;
	if (client->open (client) < 0) {
	    PERR ("failed to reopen device nonblock for ioctls\n");
	    goto end;
	}
    }

    for (;;) {

            struct timeval tv0;        // last time
            struct timeval tv1;        // current time
            long usleep;               // microtimediff
            static short tv0_known;

        // PTB signals set global flags that we look at here
	if (sigchldhandler_needed > 0) {
	    DEBUG ("main client loop handles sigchld\n");
	    sigchldhandler (session);
	    sigchldhandler_needed--;
	}
	if (longjmp_needed > 0) {
	    DEBUG ("main client loop handles longjmp (sigpwr)\n");
	    longjmp_needed--;
	    siglongjmp (jmp_start, 1);
	}

	if (SYNC_INTVL <= 0) {
            // PTB even if no SYNC_INTVL delay a bit per cycle.
	    microsleep (1000000);
            continue;
        }

        // PTB async sync the kernel every second

        DEBUG ("Client syncs kernel driver\n");

        /*
         * PTB do this ASYNC in the kernel. Arg 0. To avoid
         * deadlock when we have no live clients or a dead
         * server and there are requests outstanding.
         * Bug (#1) identified by Christian
         * Schmid (webmaster@rapidforum.com). Thanks!
         */

        tv0_known = (mygettimeofday(&tv0, NULL) >= 0);

	client->ioctl (client, MY_NBD_SYNC, (char *) 0);

        if (mygettimeofday(&tv1, NULL) < 0 || !tv0_known) {
            microsleep(SYNC_INTVL * 1000000);
            continue;
        }

        usleep = tv1.tv_sec - tv0.tv_sec;
        usleep *= 1000000;
        usleep += tv1.tv_usec - tv0.tv_usec;

        // PTB how many us still to wait
        usleep = SYNC_INTVL * 1000000 - usleep;

        if (usleep <= 0) {
            // PTB this is arithmetic error. Let the cpu cool down
            usleep = 1000000;
        }
        microsleep(usleep);

    }

  end:
    MSG ("main client loop exited!\n");

    return 0;
}
